/**
* @file DDD2004BallSpecialist.cpp
* This file contains a class for Image Processing.
* @author <A href=mailto:juengel@informatik.hu-berlin.de>Matthias Juengel</A>
* @author Max Risler
* @author Ronnie Brunn
* @author Michael Kunz
*/

#include "DDD2004BallSpecialist.h"

#include "Tools/FieldDimensions.h"
#include "Tools/Math/Common.h"
#include "Tools/Math/MVTools.h"
#include "Tools/Math/Matrix_nxn.h"
#include "Tools/Math/Vector_n.h"

#include "Tools/Debugging/DebugDrawings.h"
#include "DDD2004ImageProcessorTools.h"

DDD2004BallSpecialist::DDD2004BallSpecialist
(
 const DDD2004ColorCorrector& colorCorrector
)
:
 colorCorrector(colorCorrector)
{
}

void DDD2004BallSpecialist::searchBall
(
 const Image& image, 
 const ColorTable& colorTable,
 const CameraMatrix& cameraMatrix,
 int x, int y,
 BallPercept& ballPercept
)
{
  BallPointList ballPoints;
  Vector2<int> center;
  double radius;
  int countOrange = 0;
  int maxOrangePerLine = 0;
  int countPixel = 0;
  CameraInfo bwCameraInfo = image.cameraInfo;
  bwCameraInfo.resolutionHeight*=2;
  bwCameraInfo.resolutionWidth*=2;
  bwCameraInfo.opticalCenter.x*=2;
  bwCameraInfo.opticalCenter.y*=2;
  bwCameraInfo.focalLength*=2;
  bwCameraInfo.focalLengthInv/=2;

  scanForBallPoints(image, bwCameraInfo, colorTable, x, y, ballPoints, countOrange, maxOrangePerLine, countPixel);

  //ignore ball if not enough points are orange
  //OUTPUT (idText, text, "DDD2004BallSpecialist: " << countOrange * 100 / countPixel << "% orange");
  int i;
  int numberOfHardEdgePoints = 0;
  int numberOfPointsInYellow = 0;

  for(i = 0; i < ballPoints.number; i++)
  {
    if (ballPoints[i].yellowIsClose && !ballPoints[i].atBorder) numberOfPointsInYellow++;
    if (ballPoints[i].hardEdge) numberOfHardEdgePoints++;
  }
  //OUTPUT (idText, text, "NOPoints: " << ballPoints.number << " GhostYellow: " << numberOfPointsInYellow << " HardEdge: " << numberOfHardEdgePoints);
  
  if ( (countOrange > countPixel / 6) && (numberOfHardEdgePoints * 5 >= ballPoints.number * 3) && (numberOfPointsInYellow * 4 <= ballPoints.number * 3) )
  {
    //try only points near green first
    BallPointList testPoints;
    for(i = 0; i < ballPoints.number; i++)
      if (ballPoints[i].greenIsClose) 
        testPoints.add(ballPoints[i]);
    if (
      testPoints.number >= ballPoints.number / 2 &&
      createBallPerceptLevenbergMarquardt(testPoints, center, radius) &&
      checkIfPointsAreInsideBall(ballPoints, center, radius))
    {
      addBallPercept(image, bwCameraInfo, cameraMatrix, center, radius, ballPercept);
    }
    else
    {
      //now all points if not at border
      testPoints.number = 0;
      for(i = 0; i < ballPoints.number; i++)
        if (!ballPoints[i].atBorder) 
          testPoints.add(ballPoints[i]);
      if (
        createBallPerceptLevenbergMarquardt(testPoints, center, radius) &&
        checkIfPointsAreInsideBall(ballPoints, center, radius))
      {
        addBallPercept(image, bwCameraInfo, cameraMatrix, center, radius, ballPercept);
      }
      else
      {
        //take all points if nothing else works
        if (createBallPerceptLevenbergMarquardt(ballPoints, center, radius))
        {
          addBallPercept(image, bwCameraInfo, cameraMatrix, center, radius, ballPercept);
        }
      }
    }
  }

  DEBUG_DRAWING_FINISHED(imageProcessor_ball1);
  DEBUG_DRAWING_FINISHED(imageProcessor_ball2);
}

void DDD2004BallSpecialist::scanForBallPoints
(
 const Image& image,
 const CameraInfo& bwCameraInfo,
 const ColorTable& colorTable,
 int x, int y,
 BallPointList& ballPoints,
 int& countOrange,
 int& maxOrangePerLine,
 int& countPixel
)
{
  // search for ball variables
  BallPoint north;
  BallPoint east;
  BallPoint south;
  BallPoint west;

  BallPoint start;
  Vector2<int>step;
  BallPoint destination;
  
//  int xStep;
//  int yStep;

  start.x = x * 2; start.y = y * 2;
  BallPoint start2;
  
  DOT(imageProcessor_ball2, x, y, Drawings::black, Drawings::white);

  //find north ///////////////////////////////////////////
  step.x = 0; step.y = -1;
  findEndOfBall(image, bwCameraInfo, colorTable, start, step, north, countOrange, maxOrangePerLine, countPixel);
  if(north.atBorder)
  {
    start2 = north - step;
    //find east
    step.x = 1; step.y = 0;
    if(findEndOfBall(image, bwCameraInfo, colorTable, start2, step, destination, countOrange, maxOrangePerLine, countPixel))
    {
      ballPoints.add(destination);
    }
    //find west
    step.x = -1; step.y = 0;
    if(findEndOfBall(image, bwCameraInfo, colorTable, start2, step, destination, countOrange, maxOrangePerLine, countPixel))
    {
      ballPoints.add(destination);
    }
  }
  else
  {
    ballPoints.add(north);
  }

  //find east ///////////////////////////////////////////
  step.x = 1; step.y = 0;
  findEndOfBall(image, bwCameraInfo, colorTable, start, step, east, countOrange, maxOrangePerLine, countPixel);
  if(east.atBorder)
  {
    start2 = east - step;
    //find north
    step.x = 0; step.y = -1;
    if(findEndOfBall(image, bwCameraInfo, colorTable, start2, step, destination, countOrange, maxOrangePerLine, countPixel))
    {
      ballPoints.add(destination);
    }
    //find south
    step.x = 0; step.y = 1;
    if(findEndOfBall(image, bwCameraInfo, colorTable, start2, step, destination, countOrange, maxOrangePerLine, countPixel))
    {
      ballPoints.add(destination);
    }
  }
  else
  {
    ballPoints.add(east);
  }

  //find south ///////////////////////////////////////////
  step.x = 0; step.y = 1;
  findEndOfBall(image, bwCameraInfo, colorTable, start, step, south, countOrange, maxOrangePerLine, countPixel);
  if(south.atBorder)
  {
    start2 = south - step;
    //find east
    step.x = 1; step.y = 0;
    if(findEndOfBall(image, bwCameraInfo, colorTable, start2, step, destination, countOrange, maxOrangePerLine, countPixel))
    {
      ballPoints.add(destination);
    }
    //find west
    step.x = -1; step.y = 0;
    if(findEndOfBall(image, bwCameraInfo, colorTable, start2, step, destination, countOrange, maxOrangePerLine, countPixel))
    {
      ballPoints.add(destination);
    }
  }
  else
  {
    ballPoints.add(south);
  }

  //find west ///////////////////////////////////////////
  step.x = -1; step.y = 0;
  findEndOfBall(image, bwCameraInfo, colorTable, start, step, west, countOrange, maxOrangePerLine, countPixel);
  if(west.atBorder)
  {
    start2 = west - step;
    //find north
    step.x = 0; step.y = -1;
    if(findEndOfBall(image, bwCameraInfo, colorTable, start2, step, destination, countOrange, maxOrangePerLine, countPixel))
    {
      ballPoints.add(destination);
    }
    //find south
    step.x = 0; step.y = 1;
    if(findEndOfBall(image, bwCameraInfo, colorTable, start2, step, destination, countOrange, maxOrangePerLine, countPixel))
    {
      ballPoints.add(destination);
    }
  }
  else
  {
    ballPoints.add(west);
  }

  //
  if( (south.y - north.y) > (east.x - west.x) )
  {
    if ((north.y + south.y) / 2 != start.y)
    {
      start.y = (north.y + south.y) / 2;
      //find east
      step.x = 1; step.y = 0;
      findEndOfBall(image, bwCameraInfo, colorTable, start, step, east, countOrange, maxOrangePerLine, countPixel);
      if(!east.atBorder)
      {
        ballPoints.add(east);
      }
      //find west
      step.x = -1; step.y = 0;
      findEndOfBall(image, bwCameraInfo, colorTable, start, step, west, countOrange, maxOrangePerLine, countPixel);
      if (!west.atBorder)
      {
        ballPoints.add(west);
      }
      //////////
      start.x = (west.x + east.x) / 2;
    }
  }
  else
  {
    if ((west.x + east.x) / 2 != start.x)
    {
      start.x = (west.x + east.x) / 2;
      //find north
      step.x = 0; step.y = -1;
      findEndOfBall(image, bwCameraInfo, colorTable, start, step, north, countOrange, maxOrangePerLine, countPixel);
      if(!north.atBorder)
      {
        ballPoints.add(north);
      }
      //find south
      step.x = 0; step.y = 1;
      findEndOfBall(image, bwCameraInfo, colorTable, start, step, south, countOrange, maxOrangePerLine, countPixel);
      if(!south.atBorder)
      {
        ballPoints.add(south);
      }
      //////////
      start.y = (north.y + south.y) / 2;
    }
  }
  //

  // find in diagonal
  
//  for(xStep = -1; xStep <= 1; xStep += 2)
  for (step.x = -1; step.x <= 1; step.x += 2)
  {
    //for(yStep = -1; yStep <= 1; yStep += 2)
	for (step.y = -1; step.y <= 1; step.y += 2)
    {
      //step.x = xStep; step.y = yStep;
      findEndOfBall(image, bwCameraInfo, colorTable, start, step, destination, countOrange, maxOrangePerLine, countPixel);
      if (!destination.atBorder)
      {
        ballPoints.add(destination);
      }
    } //for(int yStep ...
  } //for(int xStep ...
}

bool DDD2004BallSpecialist::findEndOfBall
(
 const Image& image,
 const CameraInfo& bwCameraInfo,
 const ColorTable& colorTable,
 const BallPoint& start,
 const Vector2<int>& step,
 BallPoint& destination,
 int& countOrange,
 int& maxOrangePerLine,
 int& countPixel
)
{
  /*
  stopColors are colors indicating the end of the ball:
    green, yellow, skyblue
  more than 3 pixels of a stopColor indicate the end of the ball
  */
  int stopColorCounter = 0;
  int currentOrange = 0;
  int len = 0;
  int stopLen = 0;
  
  colorClass currentColorClass;

  Vector2<int> firstStopColor = start;
  unsigned char lastGoodColorOrangeSim = 0;
  unsigned char currentOrangeSim = 0;
  unsigned char prevOrangeSim = 0;
  Vector2<int> lastPoint = start;
  destination = start;
  destination.greenIsClose = false;
  destination.yellowIsClose = false;
  destination.atBorder = false;
  destination.hardEdge = true;

  bool isOrange = false;
 
  bool goOn = true;
  while(goOn)
  {
    lastPoint = destination;
    destination += step;
    /////////////
    if(destination.x < 0 || destination.x >= bwCameraInfo.resolutionWidth||
      destination.y < 0 || destination.y >= bwCameraInfo.resolutionHeight)
    {
      destination.atBorder = true;
      countPixel += len;
      goOn = false;
    }
    else
    {
      currentColorClass = CORRECTED_COLOR_CLASS(
        destination.x / 2,destination.y / 2,
        image.getHighResY(destination.x,destination.y),
        image.image[destination.y / 2][1][destination.x / 2],
        image.image[destination.y / 2][2][destination.x / 2]);
      
      // counting all orange pixels on the scanned horz./vert./diag. lines
      len++;
      if ( currentColorClass == orange )
      {
        currentOrange++;
      }
      LINE(imageProcessor_ball1,destination.x / 2,destination.y / 2,destination.x / 2 + 1,destination.y / 2,1,Drawings::ps_solid,Drawings::green);
      
      prevOrangeSim = currentOrangeSim;
      currentOrangeSim = getSimilarityToOrange(
        colorCorrector.correctY(destination.x / 2,destination.y / 2,image.getHighResY(destination.x,destination.y)),
        colorCorrector.correctU(destination.x / 2,destination.y / 2,image.image[destination.y / 2][1][destination.x / 2]),
        colorCorrector.correctV(destination.x / 2,destination.y / 2,image.image[destination.y / 2][2][destination.x / 2])
        );
      isOrange = (currentOrangeSim > 30) || (currentColorClass == orange);
      
      if(currentColorClass == green || 
        currentColorClass == yellow ||
        currentColorClass == skyblue ||
        currentColorClass == red ||
        currentColorClass == blue ||
        !isOrange
        )
      {
        if (stopColorCounter == 0)
        {
          firstStopColor = destination;
          lastGoodColorOrangeSim = prevOrangeSim;
          stopLen = len;
        }
        if (currentColorClass == green) destination.greenIsClose = true;
        if (currentColorClass == yellow) destination.yellowIsClose = true;
        stopColorCounter++;

        if (isOrange)
        {
          LINE(imageProcessor_ball1,destination.x / 2,destination.y / 2,destination.x / 2 + 1,destination.y / 2,1,Drawings::ps_solid,Drawings::orange);
        }
        else
        {
          LINE(imageProcessor_ball1,destination.x / 2,destination.y / 2,destination.x / 2 + 1,destination.y / 2,1,Drawings::ps_solid,Drawings::red);
        }
        
        if(stopColorCounter > 8)
        {
          destination = firstStopColor;
          countPixel += stopLen;
          goOn = false;
        }
      }
      else
      {
        destination.greenIsClose = false;
        destination.yellowIsClose = false;
        stopColorCounter = 0;
      }
      
    } // else destination in range
  } //  while(goOn)

  destination -= step;

  // compute sum of all orange pixels and max-pixels-per-line
  countOrange += currentOrange;
  maxOrangePerLine = max ( maxOrangePerLine, currentOrange );
  if (destination.greenIsClose) destination.yellowIsClose = false;

/*  if (!destination.atBorder)
  {
    OUTPUT(idText, text, "Orange in: " << lastGoodColorOrangeSim << " OrangeOut: " << currentOrangeSim);
  }*/
  if ( lastGoodColorOrangeSim > 2 * currentOrangeSim ) destination.hardEdge = true;
    else destination.hardEdge = false;

  if (!destination.atBorder)
  {
    if (destination.greenIsClose)
    {
      LINE(imageProcessor_ball1,destination.x / 2,destination.y / 2,destination.x / 2 + 1,destination.y / 2,1,Drawings::ps_solid,Drawings::blue);
    }
    else
      if (destination.yellowIsClose)
      {
        LINE(imageProcessor_ball1,destination.x / 2,destination.y / 2,destination.x / 2 + 1,destination.y / 2,1,Drawings::ps_solid,Drawings::skyblue);
      }
      else
      {
        LINE(imageProcessor_ball1,destination.x / 2,destination.y / 2,destination.x / 2 + 1,destination.y / 2,1,Drawings::ps_solid,Drawings::pink);
      }
  }
  return true;
}



bool DDD2004BallSpecialist::createBallPerceptLevenbergMarquardt
(
 const BallPointList& ballPoints,
 Vector2<int>& center,
 double& radius
)
{
	if (ballPoints.number < 3)
		return false;

	double Mx = 0.0;
  double My = 0.0;
  double Mxx = 0.0;
  double Myy = 0.0;
  double Mxy = 0.0;
  double Mz = 0.0;
  double Mxz = 0.0;
  double Myz = 0.0;
  
	for (int i = 0; i < ballPoints.number; ++i)
	{
    double x = ballPoints[i].x;
    double y = ballPoints[i].y;
    double xx = x*x;
    double yy = y*y;
    double z = xx + yy;

    Mx += x;
    My += y;
    Mxx += xx;
    Myy += yy;
    Mxy += x*y;
    Mz += z;
    Mxz += x*z;
    Myz += y*z;
	}

	try
  {
    Matrix_nxn<double, 3> M;
	  double Matrix[9] = 
    {
      Mxx, Mxy, Mx,
      Mxy, Myy, My,
      Mx, My, ballPoints.number
    };
	  M = Matrix;

	  Vector_n<double, 3> v;
	
	  v[0] = -Mxz;
	  v[1] = -Myz;
	  v[2] = -Mz;

	  Vector_n<double, 3> BCD;
    BCD = M.solve(v);

	  center.x = (int)(-BCD[0] / 2.0);
	  center.y = (int)(-BCD[1] / 2.0);

	  double tmpWurzel = BCD[0]*BCD[0]/4.0 + BCD[1]*BCD[1]/4.0 - BCD[2];
	
	  if (tmpWurzel < 0.0)
      return false;

  	radius = sqrt(tmpWurzel);
  }
  catch (MVException)
  {
    return false;
  }
  catch (...)
  {
    OUTPUT(idText, text, "Unknown exception in DDD2004BallSpecialist::createBallPerceptsFromXPoints");
    return false;
  }

	return true;
}

bool DDD2004BallSpecialist::checkIfPointsAreInsideBall(const BallPointList& ballPoints, Vector2<int>& center, double radius)
{
  for(int i = 0; i < ballPoints.number; i++)
    if (Geometry::distance(center, ballPoints[i]) > radius * 1.1)
      return false;

  return true;
}

void DDD2004BallSpecialist::addBallPercept
  (
    const Image& image,
    const CameraInfo& bwCameraInfo,
    const CameraMatrix& cameraMatrix,
    const Vector2<int>& center,
    double radius,
    BallPercept& ballPercept
  )
{
  // test if ball is below horizon
  //~ double xFactor = tan(bwCameraInfo.openingAngleWidth / 2) / (bwCameraInfo.resolutionWidth / 2),
    //~ yFactor = tan(bwCameraInfo.openingAngleHeight / 2) / (bwCameraInfo.resolutionHeight / 2);
  double factor = bwCameraInfo.focalLengthInv;
  Vector3<double> 
    vectorToCenter(1, (bwCameraInfo.opticalCenter.x - center.x) * factor, (bwCameraInfo.opticalCenter.y - center.y) * factor);
  Vector3<double> 
    vectorToCenterWorld = cameraMatrix.rotation * vectorToCenter;
  
  //Is the point above the horizon ? - return
//  if(vectorToCenterWorld.z <= -5 * yFactor) //for perfect horizon
    if(vectorToCenterWorld.z < 1 * factor)
  {
    Vector2<double>angles;
    Geometry::calculateAnglesForPoint(center, cameraMatrix, bwCameraInfo, angles);
    ballPercept.add(
      bwCameraInfo,
      center,
      radius,
      angles, 
      Geometry::pixelSizeToAngleSize(radius, bwCameraInfo), 
      cameraMatrix.translation, 
      cameraMatrix.isValid);
  }
}

/*
* Change log :
* 
* $Log: DDD2004BallSpecialist.cpp,v $
* Revision 1.22  2004/05/07 15:16:24  nistico
* All geometry calculations are making use of intrinsic functions.
* I updated most of the images processors to make correct use of this.
* Please test if there are any problems, because i'm going to remove the
* old code soon.
*
* Revision 1.21  2004/04/07 12:28:56  risler
* ddd checkin after go04 - first part
*
* Revision 1.15  2004/04/06 13:19:36  risler
* cleaned up and improved high resolution image support
*
* Revision 1.14  2004/04/02 19:33:01  mkunz
* lets see higher balls
*
* Revision 1.13  2004/04/02 18:06:52  mkunz
* even less ghost balls
*
* Revision 1.12  2004/04/02 16:59:13  Schmitt
* Removed unneccessary variables
*
* Revision 1.11  2004/04/02 15:16:31  mkunz
* hard edge detection basics
*
* Revision 1.10  2004/04/02 01:51:03  mkunz
* better handling og highlights and ghost balls in yellow goal
*
* Revision 1.9  2004/03/30 20:42:34  park
* added optical center and focal length info to bwCameraInfo, for use
* in INTRINSIC methods
*
* Revision 1.8  2004/03/30 11:10:39  mkunz
* bwCameraInfo intrduced
*
* Revision 1.7  2004/03/29 20:46:01  risler
* use high resolution
*
* Revision 1.6  2004/03/29 15:21:13  mkunz
* add all data to ballPercept
*
* Revision 1.5  2004/03/29 12:50:02  risler
* added ball percept horizon test
*
* Revision 1.4  2004/03/29 12:41:26  park
* no message
*
* Revision 1.3  2004/03/29 11:27:57  park
* no message
*
* Revision 1.2  2004/03/29 11:14:18  park
* added refiner for Ball-Percept
*
* Revision 1.1.1.1  2004/03/29 08:28:47  Administrator
* initial transfer from tamara
*
* Revision 1.20  2004/03/28 19:36:04  risler
* removed the now obsolete createBallPercept function
*
* Revision 1.19  2004/03/28 18:52:12  uhrig
* Removed function createBallPerceptFrom3Points (and all functions called exclusively by that function) because the ball percept can be created by the Levenberg-Marquardt algorithm, too.
*
* Revision 1.18  2004/03/28 17:08:14  risler
* added counting all scanned pixels
* added test for number of orange pixels
*
* Revision 1.17  2004/03/28 14:04:50  mkunz
* higher orange threshold (30)
*
* Revision 1.16  2004/03/27 18:44:35  risler
* removed unnecesary double scans
*
* Revision 1.15  2004/03/27 17:40:08  risler
* take only green points only if not less than half of all points
*
* Revision 1.14  2004/03/27 17:17:00  risler
* ball perception revisited
*
* Revision 1.13  2004/03/27 14:54:46  risler
* dots corrected
*
* Revision 1.12  2004/03/27 14:39:36  risler
* moved scan to scanForBallPoints
*
* Revision 1.11  2004/03/27 14:32:54  mkunz
* All ball points "dotted"
*
* Revision 1.10  2004/03/27 13:50:32  risler
* searchBall border scan fixed
*
* Revision 1.9  2004/03/27 12:24:44  risler
* findEndOfBall return value behavior changed
*
* Revision 1.8  2004/03/26 19:43:33  mkunz
* atBorder and greenIsNear added to findEndOfBall
*
* Revision 1.7  2004/03/26 18:56:45  risler
* no message
*
* Revision 1.6  2004/03/26 18:45:48  risler
* removed old ball recognition stuff
*
* Revision 1.5  2004/03/26 16:56:26  brunn
* removed obsolete variables
*
* Revision 1.4  2004/03/26 16:36:02  brunn
* integrated orange similarity in end of ball detection
*
* Revision 1.3  2004/03/26 15:24:35  risler
* ball point debug drawing added
*
* Revision 1.2  2004/03/26 14:25:17  risler
* ball specialist drawing fixed
*
* Revision 1.1  2004/03/26 09:12:42  risler
* added DDD2004BallSpecialist
* moved ball related stuff from ImageProcessor to BallSpecialist
*
* Revision 1.3  2003/05/20 15:30:37  risler
* removed remains of color table hack
*
* Revision 1.2  2003/05/12 00:03:29  dueffert
* doxygen bugs fixed
*
* Revision 1.1  2003/04/15 17:19:43  risler
* moved DDDGO2003 ImageProcessor to own module
* removed ContinuousBasicBehaviorTester
*
* Revision 1.14  2003/04/15 15:52:09  risler
* DDD GO 2003 code integrated
*
* Revision 1.17  2003/04/10 21:46:01  max
* hack: colortable access via non virtual function
*
* Revision 1.16  2003/04/08 20:19:26  dthomas
* modified: prevent detection of "ghost"-balls
*
* Revision 1.15  2003/04/08 17:02:49  dthomas
* modified: "ghost"-balls should no more be detected as ball-percept
*
* Revision 1.14  2003/04/08 00:59:41  max
* tamara update
*
* Revision 1.13  2003/04/05 21:02:29  max
* increased minimum ball radius
*
* Revision 1.12  2003/03/31 18:17:41  max
* tamara update
*
* Revision 1.13  2003/04/06 21:37:03  juengel
* Ball distance is calculated by size and not by angle, if cameraMatrix is not valid.
*
* Revision 1.12  2003/03/31 10:30:54  juengel
* no message
*
* Revision 1.11  2003/03/18 16:46:54  mkunz
* calculateSingleBallPercept:
* calculation of SingleBallPercept improved
* division by zero avoided
*
* Revision 1.10  2003/02/21 16:34:16  dueffert
* common pi in own code, warnings removed, platform compatibility restored
*
* Revision 1.9  2003/02/19 14:59:55  roefer
* pColorTable -> colorTable
*
* Revision 1.8  2003/02/18 21:29:16  osterhues
* Changed all instances of ColorTable64 to new base class ColorTable
*
* Revision 1.7  2002/11/20 13:37:30  dueffert
* missing or wrong doxygen file corrected
*
* Revision 1.6  2002/11/12 23:00:47  dueffert
* started restore greenhills compatibility
*
* Revision 1.5  2002/11/11 11:27:11  juengel
* First Step: New debug drawing macros.
*
* Revision 1.4  2002/09/23 13:54:48  risler
* abs replaced by fabs/labs
*
* Revision 1.3  2002/09/22 18:40:51  risler
* added new math functions, removed GTMath library
*
* Revision 1.2  2002/09/19 23:38:53  juengel
* Changed debug image mechanisms.
*
* Revision 1.1  2002/09/10 15:36:15  cvsadm
* Created new project GT2003 (M.L.)
* - Cleaned up the /Src/DataTypes directory
* - Removed challenge related source code
* - Removed processing of incoming audio data
* - Renamed AcousticMessage to SoundRequest
*
* Revision 1.1  2002/09/07 13:36:57  loetzsch
* unified the vision modules into one module "ImageProcessor"
* - FloodFillRLEImageProcessor, BallPerceptor, LandmarksPerceptor
*   and PlayersPerceptor were are combined to the new solution
*   "BlobImageProcessor"
* - The GridImageProcessor and the SubPixelGradientCalculator became
*   a solution of "ImageProcessor"
*
* Revision 1.10  2002/09/03 16:01:41  juengel
* Some comments added.
*
* Revision 1.9  2002/08/30 14:02:22  juengel
* no message
*
* Revision 1.8  2002/07/23 13:33:39  loetzsch
* new streaming classes
*
* removed many #include statements
*
* Revision 1.7  2002/06/28 10:26:18  roefer
* OUTPUT is possible in constructors
*
* Revision 1.6  2002/06/22 08:02:02  Thomas Rfer
* No balls behind
*
* Revision 1.5  2002/06/11 15:32:28  juengel
* If orange is everywhere, the ball is assumbed to be 60mm in front of the robot.
*
* Revision 1.4  2002/06/08 17:01:35  juengel
* Removed close ball bug.
*
* Revision 1.3  2002/05/25 17:51:27  juengel
* Preparations for Ball Collection Challenge.
*
* Revision 1.2  2002/05/13 13:19:20  juengel
* findEndOfBall modified.
*
* Revision 1.1.1.1  2002/05/10 12:40:14  cvsadm
* Moved GT2002 Project from ute to tamara.
*
* Revision 1.11  2002/05/06 17:52:35  juengel
* Changed searchEndOfFlags
*
* Revision 1.10  2002/05/03 11:52:08  juengel
* ...
*
* Revision 1.9  2002/05/02 12:12:34  kallnik
* GTMath
*
* Revision 1.8  2002/04/25 20:29:57  roefer
* New BallPercept and BallPosition, GTMath errors in SimGT2002 fixed
*
* Revision 1.7  2002/04/16 16:55:12  dueffert
* no message
*
* Revision 1.2  2002/04/11 21:57:32  Joscha Bach
* Changed so that only size is used in distance calculation
*
* Revision 1.6  2002/04/04 18:43:58  juengel
* FlagSpecialist improved.
*
* Revision 1.5  2002/04/03 13:25:41  risler
* changed doubles to double/Angle, tan to tan
*
* Revision 1.4  2002/04/03 12:50:20  juengel
* Flags and ball.
*
* Revision 1.3  2002/04/02 10:30:34  juengel
* GridImageProcessor enhanced.
*
* Revision 1.2  2002/03/18 09:45:36  kallnik
* GTMathTable updated
* GTMathConfig updated
* several doubles changed in GTMathValue
*
* Revision 1.1  2002/03/16 13:43:37  juengel
* GridImageProcessor created.
*
* Revision 1.8  2002/02/24 10:02:21  juengel
* TimeDiagramDlgBar completed.
*
* Revision 1.7  2002/02/11 11:13:06  roefer
* BallPerceptor and BallLocator inserted
*
* Revision 1.6  2002/02/11 00:57:05  loetzsch
* now returns validity 0 if ball is not seen
*
* Revision 1.5  2002/02/05 04:01:03  loetzsch
* replaced direct member access in WorldState
* and PerceptCollection members by
* inline const VALUE& get...() const   and
* inline void set...(const Value&) methods.
*
* Revision 1.4  2002/02/03 20:42:26  juengel
* Ball perception in improved.
*
* Revision 1.3  2002/02/03 14:50:09  juengel
* Drawing of the world state removed from Berlin2001BehaviorControl.
* Drawing method for world states added to PaintMethods.
* Drawing of the world state added to the Processes with BehaviorControl.
*
* Revision 1.2  2002/02/03 14:37:58  juengel
* Drawing of the world state removed from Berlin2001BehaviorControl.
* Drawing method for world states added to PaintMethods.
* Drawing of the world state added to the Processes with BehaviorControl.
*
* Revision 1.1  2002/01/22 14:53:30  juengel
* ImageToPerceptCollection eingefhrt
*
*
*/
