/**
* @file ColorTable32KImageProcessor.h
*
* Definition of class ColorTable32KImageProcessor
*
* @author <a href="mailto:juengel@informatik.hu-berlin.de">Matthias Juengel</a>
* @author <a href="mailto:roefer@tzi.de">Thomas Rfer</a>
* @author <a href=mailto:walter.nistico@uni-dortmund.de>Walter Nistico</a>
*/

#ifndef __ColorTable32KImageProcessor_h_
#define __ColorTable32KImageProcessor_h_

#include "Modules/ImageProcessor/ImageProcessor.h"
#include "../GT2003ImageProcessor/GT2003FlagSpecialist.h"
#include "../ImageProcessorTools/GoalRecognizer.h"
#include "../ImageProcessorTools/FastSUSANNoiseReduction.h"
#include "../ImageProcessorTools/MSH2004ColorCorrector.h"
#include "Tools/Debugging/DebugDrawings.h"
#include "Tools/Math/Geometry.h"
#include "Tools/Debugging/DebugImages.h"
#include "Tools/RobotConfiguration.h"
#include "Platform/SystemCall.h"

/**
* @class ColorTable32KImageProcessor
*
* The lines image processor recognizes characteristic lines in the image.
* Four types of lines are distinguished:
* edges between the skyblue goal and the field, edges between the yellow goal 
* and the field, edges between the border and the field, and edges between the
* field lines and the field.
*
* The module scans vertical and horizontal lines in the image from top to bottom
* and from left to right. As the green of the field is very dark, all edges are
* characterized by a big difference of the y-channel of adjacent pixels. An
* increase in the y-channel followed by a decrease is an indication for an edge.
*
* The projection of the pixels on the field plane is used to determine their
* relative position to the robot.
*
* @author <a href="mailto:juengel@informatik.hu-berlin.de">Matthias Juengel</a>
* @author <a href="mailto:roefer@tzi.de">Thomas Rfer</a>
* @author <a href=mailto:walter.nistico@uni-dortmund.de>Walter Nistico</a>
*/ 

/**
* This paramater controls the maximum level of noise filtered, as well as the minimum edge strength that is preserved 
* (the higher this value, the more this filter performs similarly to a linear one (blur))
*/
const int SMOOTHING_THRESHOLD = 12;

class ColorTable32KImageProcessor : public ImageProcessor
{
public:
  /** 
  * Constructor.
  * @param interfaces The paramters of the ColorTable32KImageProcessor module.
  */
  ColorTable32KImageProcessor(ImageProcessorInterfaces& interfaces);

  /** Executes the module */
  virtual void execute();
  virtual bool handleMessage(InMessage& message);

  DECLARE_DEBUG_COLOR_CLASS_IMAGE(segmentedImage1);
  //~ DECLARE_DEBUG_COLOR_CLASS_IMAGE(segmentedImage2);

private:

  /** Radiometric color distortion correction for ERS7 */
  MSH2004ColorCorrector colorCorrector;

  /**
  * An instance of a noise reduction filter; "filteredImage" is a buffer for the result
  * at the moment, they are just used for displaying a debug image as segmented_image_2
  */
	FastSUSANNoiseReduction noiseReductionFilter;
  Image filteredImage;

  /**
  * The struct represents a ball point.
  */
  struct BallPoint : Vector2<int>
  {
    bool isBottom, /**< Is the point at the lower side of the ball? */
         greenIsClose;
  };

  double xFactor, /**< Factor to convert the pixel coordinate space to the anglular coordinate space. */
         yFactor; /**< Factor to convert the pixel coordinate space to the anglular coordinate space. */
  int yThreshold; /**< Brightness increase threshold. */
  int vThreshold; /**< Brightness decrease threshold. */
  Geometry::Line horizon; /**< The horizon. */
  int orangeCount,  /**< Number of columns with ball points. */
      noOrangeCount, /**< Number of columns without a ball point. */
      noRedCount, /**< Number of columns without a red robot point. */
      noBlueCount, /**< Number of columns without a blue robot point. */
      noGoalCount, /**< Number of columns without a opponent goal seen. */
      closestBottom; /**< Closest bottom point on the grid. */
  Vector2<int> firstRed, /**< First red robot point in a cluster. */
               closestRed, /**< Closest red robot point in a cluster. */
               lastRed, /**< Last red robot point in a cluster. */
               firstBlue, /**< First blue robot point in a cluster. */
               closestBlue, /**< Closest blue robot point in a cluster. */
               lastBlue, /**< Last blue robot point in a cluster. */
               firstBall, /**< First center between top and bottom of a ball. */
               lastBall, /**< Last center between top and bottom of a ball. */
               firstFlag, /**< First flag point in a cluster. */
               lastFlag; /**< Last flag point in a cluster. */
  bool goalAtBorder; /**< Is the first goal point at the image border? */
  enum {maxNumberOfPoints = 400}; /**< The maximum number of ball points. */
  BallPoint ballPoints[maxNumberOfPoints]; /**< The ball points. */
  int numberOfBallPoints; /**< The actual number of ball points. */
  int flagCount[6]; /**< Histogram for flag points. */

  CameraMatrix cmBall, /**< Camera matrix without ball radius. */
               cmTricot; /**< Camera matrix without tricot height. */

  GT2003FlagSpecialist flagSpecialist; /**< The flag specialist. */
//  GT2003GoalSpecialist goalSpecialist; /**< The goal specialist. */

  /**
  * The function scans columns for line points.
  */
  void scanColumns();

  /** 
  * The function scans rows for line points. 
  */
  void scanRows();

  /** 
  * The function scans a line for line points. 
  * @param start The start point of the line.
  * @param end The end point of the line.
  * @param vertical Vertical lines are scanned for more information.
  * @param noLines Should the line not be scanned for points on field lines or borders?
  */
  void scan(const Vector2<int>& start, const Vector2<int>& end,
            bool vertical, bool noLines);

  /** 
  * The function clusters points of balls.
  * @param topPoint The top point of the current scan column.
  * @param bottomPoint The bottom point of the current scan column.
  * @param first The upper ball point. If 0, no ball has been found.
  * @param last The lower ball point. If 0, no ball has been found.
  * @param greenAboveOrange Was there green above the top point?
  * @param greenBelowOrange Was there green below the bottom point?
  */
  void clusterBalls(const unsigned char* topPoint,
                    const unsigned char* bottomPoint,
                    const unsigned char* first,
                    const unsigned char* last,
                    bool greenAboveOrange,
                    bool greenBelowOrange);

  /**
  * The function creates a ball percept from the ball points stored in "ballPoints".
  * It replaces the previous percept if the new ball is larger than the larger one 
  * found so far.
  */
  void createBallPercept();

  /**
  * The function filters outlier points.
  */
  void filterBallPoints();

  /**
  * The functions tries to calculate the ball percept by calculating the center of the ball.
  * @param onlyIfGreenIsClose Use only points that are close to green.
  * @return Has a percept been generated?
  */
  bool createBallPerceptFrom3Points(bool onlyIfGreenIsClose);

  /**
  * The function selects three points from the array "ballPoints".
  * The points will be used to determine the center of the ball.
  * @param point1 The index of the first point is returned to this variable.
  * @param point2 The index of the second point is returned to this variable.
  * @param point3 The index of the third point is returned to this variable.
  * @param onlyIfGreenIsClose Use only points that are close to green.
  * @return Were there enough adequate points?
  */
  bool select3Points(int& point1, int& point2, int& point3, bool onlyIfGreenIsClose) const;

  /**
  * The function determines the center of the ball by cutting the middle perpendiculars.
  * @param v1 The first point on the border of the ball.
  * @param v2 The second point on the border of the ball.
  * @param v3 The third point on the border of the ball.
  * @return The center of the ball.
  */
  Vector2<int> cutMiddlePerpendiculars(Vector2<int>& v1,
                                       Vector2<int>& v2,
                                       Vector2<int>& v3) const;

  /**
  * The function checks whether a ball percept is plausible and will add it if so.
  * @param centerX The x-coordinate of the center of the ball.
  * @param centerY The y-coordinate of the center of the ball.
  * @param radius The radius of the ball.
  */
  void addBallPercept(
    int centerX,
    int centerY,
    double radius
    );  


  /**
  * The function checks whether a ball percept is plausible.
  * @param offset The offset of the ball.
  * @return Is enough orange inside the ball? 
  */
  bool checkOrangeInsideBall(const Vector2<double>& offset);

  /**
  * The function checks whether a ball percept is plausible.
  * @param circle A circle in the image.
  * @return Is enough orange inside the ball? 
  */
  bool checkOrangeInsideBall(const Geometry::Circle& circle);

  
  /** 
  * The function clusters points of red and blue robots.
  * @param bottomPoint The bottom point of the current scan column.
  * @param redFound Has a red robot point been found? In that case, the last
  *                 entry in the lines percept is that point.
  * @param blueFound Has a blue robot point been found? In that case, the last
  *                  entry in the lines percept is that point.
  */
  void clusterRobots(const unsigned char* bottomPoint, bool redFound, bool blueFound);

  /** 
  * The function clusters points of the opponent's goal.
  */
  void clusterGoal();

  /** 
  * The function adds or replaces the direction to the free part of the opponent's goal.
  */
  void addGoal(const Pose2D& camera, 
               const Vector2<double>& first, const Vector2<double>& last);

  /** 
  * The function clusters flag points.
  * @param flagType The type of the flag.
  * @param point The center of the pink area.
  */
  void clusterFlags(Flag::FlagType flagType, const Vector2<int>& point);

  /**
  * The function filters the percepts, i.e. it removes potential misreadings.
  */
  void filterPercepts();

  /** 
  * The function converts an address to pixel coordinates.
  * @param p An address in image.image.
  * @return The x- and y-coordinates of the corresponding pixel.
  */
  Vector2<int> getCoords(const unsigned char* p) const;

  //!@name Helpers for grid drawing
  //!@{
  const unsigned char* last;
  Drawings::Color lineColor;
  void plot(const unsigned char* p,Drawings::Color color);
  //!@}

  /**The image processor interfaces */
  ImageProcessorInterfaces& interfaces;

  //obstacle buffers
  Vector2<int> obstaclesNearPointOnField2;
  bool obstaclesFarPointOnField2IsOnImageBorder;
  
  Vector2<int> obstaclesFarPointOnField1;
  Vector2<int> obstaclesFarPointOnField2;

  double angleBetweenDirectionOfViewAndGround;

  int numberOfScannedPixels;

  DECLARE_DEBUG_IMAGE(imageProcessorPlayers);
};
#endif// __ColorTable32KImageProcessor_h_

/*
* Change log :
* 
* $Log: ColorTable32KImageProcessor.h,v $
* Revision 1.7  2004/03/04 17:51:17  nistico
* -Bug with MSH2004ColorCorrection fixed
*
* Revision 1.6  2004/03/01 12:54:09  nistico
* -Added MSH2004ColorCorrector
* -Integrated MSH2004ColorCorrector into ColorTable32KImageProcessor (through a hack! :-) this has to be improved...
*
* Revision 1.5  2004/02/12 14:21:49  nistico
* Calculated refined (4 robots used) intrinsic camera parameters for ERS-7
* Fixed bug with ERS-7 logfiles and intrinsic camera parameters (formerly, they weren't correctly associated
* with the higher resolution image from the new robot)
*
* Revision 1.4  2004/02/03 11:35:35  nistico
* ColorTable32KImageProcessor is again bound to ColorTable32K, for performance testings
* ColorTable32K now supports unpacked access format, for performance testings
* Under evaluation the performance impact of higher cache hit rate Vs fewer instructions per pixel
*
* Revision 1.3  2004/01/23 17:07:33  nistico
* Added FastSUSANNoiseReduction.*, a non-linear image noise reduction filter, to ImageProcessorTools
* Added debug visualization of segmented image in ColorTable32KImageProcessor, and experimental support of FastSUSANNoiseReduction
*
* Revision 1.2  2004/01/21 14:11:07  nistico
* Added change log string to ColorTable32KImageProcessor.*
*
*
*/
