/**
* @file MSH2004HeadControl.h
*
* This file contains a class for head-behavior.
*
* @author Jan Hoffmann
*/

#ifndef __MSH2004HeadControl_h__
#define __MSH2004HeadControl_h__

#include "HeadControl.h"
#include "Tools/Math/PIDsmoothedValue.h"


//in urad per frame: 4000 = 28.6/s
#define minHeadSpeed 4000

/**
* @class MSH2004HeadControl
*
* A solution of the HeadControl module.
*/
class MSH2004HeadControl : public HeadControl
{
public:
/**
* Constructor.
* @param interfaces The paramters of the HeadControl module.
  */
  MSH2004HeadControl(const HeadControlInterfaces& interfaces);
  
  /** Destructor. */
  ~MSH2004HeadControl() {}
  
  /** Executes the module */
  virtual void execute();
  
  enum headControlStates
  {
    otherModes,
      lookAtBall,
      lookAtBallGoalie,
      ballJustLost,
      ballLost,
      returnToBall,
      searchAutoUpperScan,
      searchAutoLowerScan,
      numOfStates
  } headControlState, lastHeadControlState;
  
  bool lastScanWasLeft,leftWatch;
  double scanPan;
  
private:
  CameraInfo cameraInfo;
  
  void executeLookAtBall();
  void executeLookAtBallGoalie();
  void executeBallJustLost();
  void executeBallLost();
  void executeReturnToBall();
  void executeSearchAutoUpperScan();
  void executeSearchAutoLowerScan();
  
  void searchForLandmarks();
  void searchForLines();
  void searchForSpecialBall();
  void searchForObstacles();
  void searchForObstaclesInFront();
  void lookAroundWhenBallJustCaught();
  void lookAroundWhenBallCaught();
  void stayAsForced();
  void followTail();
  void lookLeft();
  void lookRight();
  void lookJustBelowHorizon();
  void doOpenChallengeCheckBite();
  
  /** should be obsolete
  2do: CameraMatrix woanders herbekommen!
  */
  void buildCameraMatrix(const SensorData& data, const HeadState& headState);
  unsigned long lastFrame;
  
  /** 3d transformation from world coordinates to camera coordinates using the camera matrix */
  void fromWorld2CameraCoordinates(const Vector3<double> pointIn3D, Vector3<double> &pointInCameraCoordinates);
  
  /** 3d transformation from world coordinates to robot coordinates */
  void fromWorld2RobotCoordinates(const Vector3<double> pointIn3D, Vector3<double> &pointInRobotCoordinates);
  
  /** look at 3d-point on field with offset point in camera image */
  void lookAtPoint(const Vector3<double> &pos,const Vector2<int> &offset,double& tilt,double& pan,double& roll);
  /** robot model specific core of the look-at-point calculation for ERS210 */
  void lookAtPointCore210(const Vector3<double> &aim, const int& xoff, const int& yoff, double& tilt, double& pan);
  /** robot model specific core of the look-at-point calculation for ERS7 */
  void lookAtPointCore7(const Vector3<double> &aim, const int& xoff, const int& yoff, double& tilt, double& pan, double& tilt2);
  
  /** calculate the angles for looking at the seen ball with a certain camera pixel depending on ball distance */
  void getLookAtBallAngles(double& tilt,double& pan,double& roll);
  
  /** calculate the angles for looking at the seen ball with a certain camera pixel depending on ball distance */
  void getLookAtBallAngles2(long& tilt,long& pan,long& roll);

  /** just a fake for compatibility, no smoothing yet (speed is ignored) */
  void setJoints(long tilt, long pan, long roll, long speed=100, long mouth=0);
  
  /* direct HeadControl with speed parameter (smooting) */
  void moveHead(long tilt=0, long pan=0, long roll=0, long speed=400, long mouth=0);
  
  /** Indicates if this head control mode was active the last time execute was called */
  HeadControlMode headControlModeExecutedLastTime;
  
  /** This struct can calculate a smooth series of head joint angles from a gives set of way points and an overall duration */
  struct HeadPathPlanner
  {
  public:
  /**
  * Initializes a set of points to visit in a certain time
  * @param vectors set of arcs to visit
  * @param numberOfVectors number of Vector3s in param vectors
  * @param duration time in ms to reach the last position in param vecors
  * @param delay time in ms to delay at each position in param vecors
    */
//    void init(const Vector3<long>* vectors, int numberOfVectors, unsigned long duration, unsigned long delay = 0);
    void init(const Vector3<long>* vectors, int numberOfVectors, unsigned long duration);
    
    /**
    * default constructor
    */
    HeadPathPlanner():lastTilt(0),lastPan(0),lastRoll(0),currentPoint(0),currentFrame(0),numberOfFrames(0),numberOfPoints(0) {}
    
    /**
    * Initializes a set of points to visit in a certain time. These points are
    * assumed to be cyclic. The first and last point to visit is the one closest
    * to the current head position.
    * @param vectors set of arcs to visit
    * @param numberOfVectors number of Vector3s in param vectors
    * @param duration time in ms to reach the last position in param vecors
    * @param delay time in ms to delay at each position in param vecors
    */
//    void initLoop(const Vector3<long>* vectors, int numberOfVectors, unsigned long duration, unsigned long delay = 0);
    void initLoop(const Vector3<long>* vectors, int numberOfVectors, unsigned long duration);
    
    /**
    * Calculates the angles for tilt pan and roll
    * @return true if the last point is reached.
    */
    bool getAngles(long& tilt, long& pan, long& roll);
    
    /**
    * Return whether the last initialized path is already finished
    * @return true if last path is finished.
    */
    inline bool isLastPathFinished() { return (currentFrame>=numberOfFrames); }
    
    long lastTilt;
    long lastPan;
    long lastRoll;
    
  private:
    long currentPoint;
    long currentFrame;
    long numberOfFrames;
    
    enum {maxNumberOfPoints = 20};
    long numberOfPoints;
    Vector3<long> points[maxNumberOfPoints*2];
    double firstFrame[maxNumberOfPoints*2];
  };
  
  HeadPathPlanner headPathPlanner;
    
  // needed for transformations and for simple look at point
  CameraMatrix cameraMatrix2;
  double currentTilt, currentPan, currentRoll;
  
};

#endif //__MSH2004HeadControl_h__

/*
* Change log :
*
* $Log: MSH2004HeadControl.h,v $
* Revision 1.11  2004/06/18 14:59:11  kerdels
* new headcontrolmode for open challenge
*
* Revision 1.10  2004/06/03 14:02:36  kerdels
* repaired HeadControl
*
* Revision 1.9  2004/05/17 19:21:51  loetzsch
* renamed all Variables "cameraMatrix" to "cameraMatrix2"
*
* Revision 1.8  2004/03/17 18:03:46  kerdels
* worked on headcontrol
*
* Revision 1.7  2004/03/17 03:19:29  kerdels
* added delay parameter to headPathPlanner to be able to stop "delay"-ms at each point of the path
*
* Revision 1.5  2004/03/08 01:38:55  roefer
* Interfaces should be const
*
* Revision 1.4  2004/03/04 10:05:23  jhoffman
* - motion process now uses odometry to propagate the robot pose while no new robot pose is being sent (this makes headcontrol a little better)
* - removed headcontrol member variable "propagatedPose" from headcontrol and cognition->motion-sender
*
* Revision 1.3  2004/03/03 08:33:15  dueffert
* doxygen bug fixed
*
* Revision 1.2  2004/02/29 13:37:15  dueffert
* doxygen bugs fixed and beautified
*
* Revision 1.1  2004/02/27 15:03:52  wachter
* cloned GT2003HeadControl as MSH2004HeadContol
*
*
*/


