/**
* @file MSH2004HeadControl.cpp
*
* Implementation of class MSH2004HeadControl
* 
*/

#ifdef _WIN32
static const unsigned int timeAfterWhichBallIsConsideredLost = 250;
#else
static const unsigned int timeAfterWhichBallIsConsideredLost = 250;
#endif


#include "Platform/GTAssert.h"
#include "MSH2004HeadControl.h"
#include "Tools/Math/Geometry.h"
#include "Tools/Math/Common.h"
#include "Platform/SystemCall.h"
#include "Tools/FieldDimensions.h"
#include "Tools/Debugging/DebugDrawings.h"


/*
void MSH2004HeadControl::HeadPathPlanner::init(const Vector3<long>* vectors, int numberOfVectors, unsigned long duration, unsigned long delay)
{
  const RobotDimensions& robotDimensions = getRobotConfiguration().getRobotDimensions();
  if (numberOfVectors>maxNumberOfPoints-1) numberOfVectors=maxNumberOfPoints-1;
  if (duration<8) duration=8;
  
  //start from current position
  points[0].x = lastTilt;
  points[0].y = lastPan;
  points[0].z = lastRoll;
  // to enable a delay
  points[1].x = lastTilt;
  points[1].y = lastPan;
  points[1].z = lastRoll;
  currentPoint = 0;
  numberOfPoints = numberOfVectors;
  currentFrame = 0;
  numberOfFrames = duration/8;
  long delayFrames = delay/8;
  
  const Range<long> jointRangeHeadTilt(toMicroRad(robotDimensions.jointLimitHeadTiltN),toMicroRad(robotDimensions.jointLimitHeadTiltP));
  const Range<long> jointRangeHeadPan( toMicroRad(robotDimensions.jointLimitHeadPanN), toMicroRad(robotDimensions.jointLimitHeadPanP));
  const Range<long> jointRangeHeadRoll(toMicroRad(robotDimensions.jointLimitHeadRollN),toMicroRad(robotDimensions.jointLimitHeadRollP));
  
  //calculate total distance and speed
  double distance=0;
  int i;
  for (i=0;i<numberOfVectors;i++)
  {
    //clip arcs to physical limits
    points[(i+1)*2].x     = jointRangeHeadTilt.limit(vectors[i].x);
    points[((i+1)*2)+1].x = points[(i+1)*2].x;
    points[(i+1)*2].y     = jointRangeHeadPan.limit(vectors[i].y);
    points[((i+1)*2)+1].y = points[(i+1)*2].y;
    points[(i+1)*2].z     = jointRangeHeadRoll.limit(vectors[i].z);
    points[((i+1)*2)+1].z = points[(i+1)*2].z;

    
    distance += sqrt(
      (double)(points[(i+1)*2].x-points[i*2].x)*(double)(points[(i+1)*2].x-points[i*2].x)+
      (double)(points[(i+1)*2].y-points[i*2].y)*(double)(points[(i+1)*2].y-points[i*2].y)+
      (double)(points[(i+1)*2].z-points[i*2].z)*(double)(points[(i+1)*2].z-points[i*2].z));
  }
  double speed = distance/(double)numberOfFrames; //in urad per frame
  if (speed<minHeadSpeed) speed=minHeadSpeed;
  
  //calculate duration for each part of the route
  double sum=0;
  long delayAdd=0;
  for (i=0;i<=numberOfVectors;i++)
  {
    firstFrame[i*2]     = (sum/speed) + delayAdd;
    delayAdd += delayFrames;
    firstFrame[(i*2)+1] = (sum/speed) + delayAdd;
    sum += sqrt(
      (double)(points[(i+1)*2].x-points[i*2].x)*(double)(points[(i+1)*2].x-points[i*2].x)+
      (double)(points[(i+1)*2].y-points[i*2].y)*(double)(points[(i+1)*2].y-points[i*2].y)+
      (double)(points[(i+1)*2].z-points[i*2].z)*(double)(points[(i+1)*2].z-points[i*2].z));
  }
  numberOfFrames += delayAdd;
}



void MSH2004HeadControl::HeadPathPlanner::initLoop(const Vector3<long>* vectors, int numberOfVectors, unsigned long duration, unsigned long delay)
{
  Vector3<long> newVec[maxNumberOfPoints];
  double min=1e20;
  int minIndex=0;
  int i;
  if (numberOfVectors>maxNumberOfPoints-1) numberOfVectors=maxNumberOfPoints-1;
  for (i=0;i<numberOfVectors;i++)
  {
    double diff2 =
      (double)(vectors[i].x-lastTilt)*(double)(vectors[i].x-lastTilt)+
      (double)(vectors[i].y-lastPan)*(double)(vectors[i].y-lastPan)+
      (double)(vectors[i].z-lastRoll)*(double)(vectors[i].z-lastRoll);
    if (diff2<min)
    {
      min=diff2;
      minIndex=i;
    }
  }
  for (i=0;i<=numberOfVectors;i++)
  {
    newVec[i]=vectors[(minIndex+i)%numberOfVectors];
  }
  init(newVec,numberOfVectors+1,duration,delay);
}
*/

void MSH2004HeadControl::HeadPathPlanner::init(const Vector3<long>* vectors, int numberOfVectors, unsigned long duration)
{
  const RobotDimensions& robotDimensions = getRobotConfiguration().getRobotDimensions();
  if (numberOfVectors>maxNumberOfPoints-1) numberOfVectors=maxNumberOfPoints-1;
  if (duration<8) duration=8;
  
  //start from current position
  points[0].x = lastTilt;
  points[0].y = lastPan;
  points[0].z = lastRoll;
  currentPoint = 0;
  numberOfPoints = numberOfVectors;
  currentFrame = 0;
  numberOfFrames = duration/8;
  
  const Range<long> jointRangeHeadTilt(toMicroRad(robotDimensions.jointLimitHeadTiltN),toMicroRad(robotDimensions.jointLimitHeadTiltP));
  const Range<long> jointRangeHeadPan( toMicroRad(robotDimensions.jointLimitHeadPanN), toMicroRad(robotDimensions.jointLimitHeadPanP));
  const Range<long> jointRangeHeadRoll(toMicroRad(robotDimensions.jointLimitHeadRollN),toMicroRad(robotDimensions.jointLimitHeadRollP));
  
  //calculate total distance and speed
  double distance=0;
  int i;
  for (i=0;i<numberOfVectors;i++)
  {
    //clip arcs to physical limits
    points[i+1].x = jointRangeHeadTilt.limit(vectors[i].x);
    points[i+1].y = jointRangeHeadPan.limit(vectors[i].y);
    points[i+1].z = jointRangeHeadRoll.limit(vectors[i].z);
    
    distance += sqrt(
      (double)(points[i+1].x-points[i].x)*(double)(points[i+1].x-points[i].x)+
      (double)(points[i+1].y-points[i].y)*(double)(points[i+1].y-points[i].y)+
      (double)(points[i+1].z-points[i].z)*(double)(points[i+1].z-points[i].z));
  }
  double speed = distance/(double)numberOfFrames; //in urad per frame
  if (speed<minHeadSpeed) speed=minHeadSpeed;
  
  //calculate duration for each part of the route
  double sum=0;
  for (i=0;i<=numberOfVectors;i++)
  {
    firstFrame[i] = sum/speed;
    sum += sqrt(
      (double)(points[i+1].x-points[i].x)*(double)(points[i+1].x-points[i].x)+
      (double)(points[i+1].y-points[i].y)*(double)(points[i+1].y-points[i].y)+
      (double)(points[i+1].z-points[i].z)*(double)(points[i+1].z-points[i].z));
  }
}

void MSH2004HeadControl::HeadPathPlanner::initLoop(const Vector3<long>* vectors, int numberOfVectors, unsigned long duration)
{
  Vector3<long> newVec[maxNumberOfPoints];
  double min=1e20;
  int minIndex=0;
  int i;
  if (numberOfVectors>maxNumberOfPoints-1) numberOfVectors=maxNumberOfPoints-1;
  for (i=0;i<numberOfVectors;i++)
  {
    double diff2 =
      (double)(vectors[i].x-lastTilt)*(double)(vectors[i].x-lastTilt)+
      (double)(vectors[i].y-lastPan)*(double)(vectors[i].y-lastPan)+
      (double)(vectors[i].z-lastRoll)*(double)(vectors[i].z-lastRoll);
    if (diff2<min)
    {
      min=diff2;
      minIndex=i;
    }
  }
  /** @todo check if opposite loop direction wouldnt be better after visit of first point from current position */
  for (i=0;i<=numberOfVectors;i++)
  {
    newVec[i]=vectors[(minIndex+i)%numberOfVectors];
  }
  init(newVec,numberOfVectors+1,duration);
}



bool MSH2004HeadControl::HeadPathPlanner::getAngles(long& tilt, long& pan, long& roll)
{
  if (currentFrame<numberOfFrames)
  {
    currentFrame++;
    while ((currentFrame>firstFrame[currentPoint+1])&&
           (currentPoint<numberOfPoints-1)&&
           (currentFrame<numberOfFrames))
    {
      currentPoint++;
    }
    
    double dist=0;
    if (currentPoint<numberOfPoints)
    {
      dist=firstFrame[currentPoint+1]-firstFrame[currentPoint];
    }
    if (dist==0)
    {
      tilt = points[currentPoint].x;
      pan  = points[currentPoint].y;
      roll = points[currentPoint].z;
    }
    else
    {
      double leftFactor = (firstFrame[currentPoint+1]-currentFrame)/dist;
      double rightFactor = 1-leftFactor;
      tilt = (long)(leftFactor*points[currentPoint].x + rightFactor*points[currentPoint+1].x);
      pan  = (long)(leftFactor*points[currentPoint].y + rightFactor*points[currentPoint+1].y);
      roll = (long)(leftFactor*points[currentPoint].z + rightFactor*points[currentPoint+1].z);
    }
  }
  return (currentFrame>=numberOfFrames);
}




MSH2004HeadControl::MSH2004HeadControl(const HeadControlInterfaces& interfaces)
: HeadControl(interfaces),headControlState(otherModes),lastHeadControlState(otherModes),
lastScanWasLeft(true)
{
  headPathPlanner.lastTilt = sensorDataBuffer.lastFrame().data[SensorData::headTilt];
  headPathPlanner.lastPan  = sensorDataBuffer.lastFrame().data[SensorData::headPan];
  headPathPlanner.lastRoll = sensorDataBuffer.lastFrame().data[SensorData::headRoll];
  scanPan = 0.0;
  leftWatch= true;
}

void MSH2004HeadControl::setJoints(long tilt, long pan, long roll, long speed, long mouth)
{
  headMotionRequest.tilt = tilt;
  headMotionRequest.pan  = pan;
  headMotionRequest.roll = roll;
  headMotionRequest.mouth = mouth;
}


void MSH2004HeadControl::moveHead(long tilt, long pan, long roll, long speed, long mouth)
{
  //tilt = tilt + sensorDataBuffer.lastFrame().data[SensorData::headTilt];
  //pan = pan + sensorDataBuffer.lastFrame().data[SensorData::headPan];
  //roll = roll + sensorDataBuffer.lastFrame().data[SensorData::headRoll];
  //mouth = mouth + sensorDataBuffer.lastFrame().data[SensorData::mouth];
  
  if (speed >= 400)
  {
    if (headMotionRequest.tilt != tilt) {
      if (headMotionRequest.tilt > toMicroRad(jointLimitHeadTiltP))
        headMotionRequest.tilt = toMicroRad(jointLimitHeadTiltP);
      else {
        if (headMotionRequest.tilt < toMicroRad(jointLimitHeadTiltN))
          headMotionRequest.tilt = toMicroRad(jointLimitHeadTiltN);
        else
          headMotionRequest.tilt = tilt;
      }
    }
    if (headMotionRequest.pan != pan) {
      if (headMotionRequest.pan > toMicroRad(jointLimitHeadPanP))
        headMotionRequest.pan = toMicroRad(jointLimitHeadPanP);
      else {
        if (headMotionRequest.pan < toMicroRad(jointLimitHeadPanN))
          headMotionRequest.pan = toMicroRad(jointLimitHeadPanN);
        else
          headMotionRequest.pan = pan;
      }
    }
    if (headMotionRequest.roll != roll) {
      if (headMotionRequest.roll > toMicroRad(jointLimitHeadRollP))
        headMotionRequest.roll = toMicroRad(jointLimitHeadRollP);
      else {
        if (headMotionRequest.roll > toMicroRad(jointLimitHeadRollN))
          headMotionRequest.roll = toMicroRad(jointLimitHeadRollN);
        else
          headMotionRequest.roll = roll;
      }
    }
    if (headMotionRequest.mouth != mouth)
      headMotionRequest.mouth = mouth;
  }
  else
  {
    if (headPathPlanner.isLastPathFinished())
    {
      Vector3<long> point(tilt,pan,roll);
      Vector3<long> points[1]={point};
      headPathPlanner.init(points,1, 2000);
    }
    headPathPlanner.getAngles(headMotionRequest.tilt, headMotionRequest.pan, headMotionRequest.roll);
  }
}

void MSH2004HeadControl::lookAtPointCore210(const Vector3<double> &aim, const int& xoff, const int& yoff, double& tilt, double& pan)
{
  //the following calculation is specific to ERS110/210 anatomy
  
  //Looking with a certain xoffset and yoffset gives us a ray from camera into space.
  //Looking around with different pans produces a cone (xoffset==0) or a plane (yoffset==0)
  //or a more complex shape in general. The only important thing about this shape is its
  //radius r(h) around the pan axis in a certain height h.
  double xtan = tan(cameraInfo.openingAngleWidth / 2)*xoff/(cameraInfo.resolutionWidth / 2);
  double ytan = tan(cameraInfo.openingAngleHeight / 2)*(0.01+yoff)/(cameraInfo.resolutionHeight / 2);
  double xfactor=1/ytan;
  double yfactor=-xtan/ytan;
  
  //r(z) is the radius of the ray of look into space around the pan axis in height z above the neck:
  //Now we have r(z)=sqrt((yfactor*(z-distanceNeckToPanCenter))^2 + (distancePanCenterToCameraX+xfactor*(z-distanceNeckToPanCenter))^2)
  
  //The hole shape of rays potentially reachable by moving pan can be rotated by tilt through space.
  //Now look at the intersection between the xz-plane through the point (aim.x,aim.y,aim.z) and this shape.
  //We already know the distance from the tilt axis, its distance^2 = aim.x^2+aim.z^2.
  //The same distance from the tilt axis has to have our intersection point (still to
  //be rotated by tilt) to match the given aim. There we have distance^2 = x^2+z^2.
  //y is constant (y=aim.y) and we know the radius of our shape in height z.
  //So its distance^2 = (r(z)^2-y^2)+z^2. This can be resolved (by Mathematica) to z:
  double a = 2*distancePanCenterToCameraX*xfactor;
  double b = 2*distanceNeckToPanCenter*(xfactor*xfactor+yfactor*yfactor);
  double divisor = 2*(1+xfactor*xfactor+yfactor*yfactor);
  double radikant = -divisor*(distanceNeckToPanCenter*(-2*a+b)+
    2*(distancePanCenterToCameraX*distancePanCenterToCameraX-aim.x*aim.x-aim.y*aim.y-aim.z*aim.z))
    +(a-b)*(a-b);
  
  double z;
  if (radikant<0)
  {
    //the aim is totally unreachable, i.e. within us, so return the last known tilt / pan
    tilt=headPathPlanner.lastTilt;
    pan=headPathPlanner.lastPan;
    return;
  }
  else
  {
    if (xfactor>0)
    {
      z = (b-a + sqrt(radikant))/divisor;
    }
    else
    {
      z = (b-a - sqrt(radikant))/divisor;
    }
  }
  
  //Now we can calculate (r and) x from z:
  double newx = distancePanCenterToCameraX+xfactor*(z-distanceNeckToPanCenter);
  double newy = yfactor*(z-distanceNeckToPanCenter);
  double x2 = newx*newx + newy*newy - aim.y*aim.y;
  double x;
  if (x2<=0)
  {
    //the aim is within a unvisible "pipe" from left to right, so look left or right instead
    x=0;
  }
  else if (aim.x>0)
  {
    //the aim is infront of us / the invisible pipe
    x=sqrt(x2);
  }
  else
  {
    //the aim is behind us / the invisible pipe
    x=-sqrt(x2);
  }
  
  //Now we have (x,z) forming an arc and (aim.x,aim.z) forming an arc in the xz-plane.
  //And the arc between these two is the tilt we are looking for:
  tilt = atan2(x, z) - atan2(aim.x, aim.z);
  
  //Its pretty the same with pan: in the xy-plane our aim is (x,y) and we currently look at
  //(distancePanCenterToCameraX+xfactor*(z-distanceNeckToPanCenter), yfactor*(z-distanceNeckToPanCenter))
  //So the pan we are looking for is
  pan = atan2(distancePanCenterToCameraX+xfactor*(z-distanceNeckToPanCenter), yfactor*(z-distanceNeckToPanCenter))
    - atan2(x, aim.y);
  
  if (pan>=pi)
  {
    pan -= pi2;
  }
  else if (pan<-pi)
  {
    pan += pi2;
  }
  
  /*
  // just a check to see that everything went right:
  RotationMatrix rot2;
  rot2.rotateY(tilt);
  Vector3<double> aim2;
  aim2 = rot2*aim;
  RotationMatrix rot3;
  rot3.rotateZ(-pan);
  Vector3<double> aim3;
  aim3 = rot3*aim2;
  Vector3<double> aim4 = aim3-Vector3<double>(distancePanCenterToCameraX,0,distanceNeckToPanCenter);
  double realxoff=-aim4.y/aim4.x*cameraInfo.resolutionWidth / 2/tan(cameraInfo.openingAngleWidth_2);
  double realyoff=aim4.z/aim4.x*cameraInfo.resolutionHeight / 2/tan(cameraInfo.openingAngleHeight_2);
  if ((aim3.x<=0)||(fabs(realxoff-xoff)>0.1)||(fabs(realyoff-yoff)>0.1))
  {
  bool mistake=true;
  }
  */
}

void MSH2004HeadControl::lookAtPointCore7(const Vector3<double> &aim, const int& xoff, const int& yoff, double& tilt, double& pan, double& tilt2)
{
  //the following calculation is specific to ERS7 anatomy
  
  //Looking with a certain xoffset and yoffset gives us a ray from camera into space.
  //Looking around with different pans produces a cone (xoffset==0) or a plane (yoffset==0)
  //or a more complex shape in general. The only important thing about this shape is its
  //radius r(h) around the pan axis in a certain height h.
  double xtan = tan(cameraInfo.openingAngleWidth / 2)*xoff/(cameraInfo.resolutionWidth / 2);
  double ytan = tan(cameraInfo.openingAngleHeight / 2)*(0.01+yoff)/(cameraInfo.resolutionHeight / 2);
  double xfactor=1/ytan;
  double yfactor=-xtan/ytan;
  
  //r(z) is the radius of the ray of look into space around the pan axis in height z above
  //the pan center (this is different than for ERS210)
  
  //Now we have r(z)=sqrt((yfactor*z)^2 + (distancePanCenterToCameraX+xfactor*z)^2)
  
  //due to ERS7 anatomy we can now change tilt1 and/or tilt2 to reach our aim. There
  //are 3 possibly useful strategies for that:
  //1) change tilt1 and tilt2 synchronously: this would look cool but will be uncalculatable
  //2) maximize tilt1: try to find tilt2 solution for max tilt1, if there is none try to find
  //   tilt1 solution for minimal tilt2
  //3) maximize tilt2: try to find tilt1 solution for max tilt2, if there is none try to find
  //   tilt2 solution for minimal tilt1
  //unfortunately, even 2) and 3) are uncalculatable as long as the the fix tilt is not 0 ->
  //4) first try tilt1=0, and if we dont find a solution that way try tilt2=0:
  
  bool solutionFound=false;
  //4.1) try tilt1=0, find tilt2+pan in range
  
  //The hole shape of rays potentially reachable by moving pan can be rotated by tilt2
  //through space. Now look at the intersection between the xz-plane through the point
  //(aim.x,aim.y,aim.z) and this shape. We already know the distance from the neck,
  //its distance^2 = aim.x^2+aim.z^2.
  //The same distance from the neck has to have our intersection point (still to be rotated by
  //tilt2) to match the given aim. There we have distance^2 = x^2+(z+distanceNeckToPanCenter)^2.
  //y is constant (y=aim.y) and we know the radius of our shape in height z.
  //So its distance^2 = (r(z)^2-y^2)+z^2. This can be resolved (by Mathematica) to z:
  double a = distanceNeckToPanCenter+distancePanCenterToCameraX*xfactor;
  double divisor = 1+xfactor*xfactor+yfactor*yfactor;
  double radikant = a*a+divisor*(aim.x*aim.x+aim.y*aim.y+aim.z*aim.z
    -distanceNeckToPanCenter*distanceNeckToPanCenter
    -distancePanCenterToCameraX*distancePanCenterToCameraX);
  double z;
  if (radikant>=0)
  {
    //the aim is possibly reachable, i.e. it is not within us
    if (xfactor>0)
    {
      z = (-a + sqrt(radikant))/divisor;
    }
    else
    {
      z = (-a - sqrt(radikant))/divisor;
    }
    
    //Now we can calculate (r and) x from z:
    double newx = distancePanCenterToCameraX+xfactor*z;
    double newy = yfactor*z;
    double x2 = newx*newx + newy*newy - aim.y*aim.y;
    double x;
    if (x2<=0)
    {
      //the aim is within a unvisible "pipe" from left to right, so look left or right instead
      x=0;
    }
    else
    {
      if (aim.x>0)
      {
        //the aim is infront of us / the invisible pipe
        x=sqrt(x2);
      }
      else
      {
        //the aim is behind us / the invisible pipe
        x=-sqrt(x2);
      }
      
      //Now we have (x,z) forming an arc and (aim.x,aim.z) forming an arc in the xz-plane.
      //And the arc between these two is the tilt we are looking for:
      tilt2 = atan2(x, z) - atan2(aim.x, aim.z);
      if ((tilt2>=jointLimitHeadRollN)&&(tilt2<=jointLimitHeadRollP))
      {
        //Its pretty the same with pan: in the xy-plane our aim is (x,y) and we currently look at
        //(distancePanCenterToCameraX+xfactor*z, yfactor*z)
        //So the pan we are looking for is
        pan = atan2(distancePanCenterToCameraX+xfactor*z, yfactor*z) - atan2(x, aim.y);
        
        if (pan>=pi)
        {
          pan -= pi2;
        }
        else if (pan<-pi)
        {
          pan += pi2;
        }
        if ((pan>=jointLimitHeadPanN)&&(pan<=jointLimitHeadPanP))
        {
          solutionFound=true;
        }
      }
    }
  }
  
  
  if (!solutionFound)
  {
    //4.2) try tilt2=0, find tilt1+pan in range
    tilt2=0;
    //well, let me think... hm, thats really the same as ERS210:
    lookAtPointCore210(aim,xoff,yoff,tilt,pan);
  }
}

void MSH2004HeadControl::lookAtPoint(const Vector3<double> &pos,const Vector2<int> &offset,double& tilt,double& pan,double& roll)
{
  Vector3<double> posInCameraCoordinates;
  posInCameraCoordinates = cameraMatrix2.rotation.invert() * (pos - cameraMatrix2.translation);
  Vector2<int> pointToLookTo;
  Vector2<double> angles;
  angles.x = atan2(posInCameraCoordinates.x, posInCameraCoordinates.y);
  angles.y = atan2(posInCameraCoordinates.x, posInCameraCoordinates.z);
  Geometry::calculatePointByAngles(angles, cameraMatrix2, cameraInfo, pointToLookTo);
  //debugging...
  LINE(sketch, offset.x, 0, offset.x, 144, 1, Drawings::ps_solid, Drawings::gray);
  LINE(sketch, 0,offset.y, 176, offset.y, 1, Drawings::ps_solid, Drawings::gray);
  DOT(sketch, pointToLookTo.x - 88, pointToLookTo.y - 72, Drawings::white, Drawings:: black);
  DEBUG_DRAWING_FINISHED(sketch);
  /*
  PAINT_TO_DEBUGDRAWING(crosshairs,
  crosshairsDrawing.line(0, offset.y + 71, 175, offset.y + 71, DebugDrawing::ps_solid, 0, DebugDrawing::Color(255,255,255) );
  crosshairsDrawing.line(offset.x + 87, 0, offset.x + 87, 143, DebugDrawing::ps_solid, 0, DebugDrawing::Color(255,255,255) );
  );
  
  GTMath::Vector2<int> onScreen;
  onScreen.x = (int)(- transformedPoint.y*fx/(transformedPoint.x));
  onScreen.y = (int)(- transformedPoint.z*fy/(transformedPoint.x));
  
  PAINT_TO_DEBUGDRAWING(crosshairs,
  crosshairsDrawing.dot(87+onScreen.x, 71+onScreen.y, DebugDrawing::Color(255,255,255), DebugDrawing::Color(127,127,255));
  );
  
  SEND_DEBUGDRAWING(crosshairs);
  */
  
  int xoff = offset.x;
  int yoff = offset.y;
  
  //Is the offset in range?
  if (yoff > cameraInfo.resolutionHeight / 2) yoff = cameraInfo.resolutionHeight / 2;
  else if (yoff < -cameraInfo.resolutionHeight / 2) yoff = -cameraInfo.resolutionHeight / 2;
  
  if (xoff > cameraInfo.resolutionWidth / 2) xoff = cameraInfo.resolutionWidth / 2;
  else if (xoff < -cameraInfo.resolutionWidth / 2) xoff = -cameraInfo.resolutionWidth / 2;
  
  Pose2D robotPose = this->robotPose.getPose();
  Vector3<double> aim1(
    pos.x-robotPose.translation.x,
    pos.y-robotPose.translation.y,
    pos.z-headState.neckHeight
    );
  
  RotationMatrix rot;
  Vector3<double> aim;
  rot.rotateY(-headState.bodyTilt).
    rotateX(headState.bodyRoll).
    rotateZ(-robotPose.getAngle());
  aim = rot*aim1;
  
  //now we have rotated the coordinates in a way, that the neck
  //is (0,0,0) and should look without any body rotation to aim:
  //aim.x in front, aim.y left, aim.z above
  
  if (getRobotConfiguration().getRobotDesign()==RobotDesign::ERS7)
  {
    lookAtPointCore7(aim,xoff,yoff,tilt,pan,roll);
    const Range<double> jointRangeHeadTilt2( jointLimitHeadRollN, jointLimitHeadRollP);
    roll = jointRangeHeadTilt2.limit(roll);
  }
  else
  {
    lookAtPointCore210(aim,xoff,yoff,tilt,pan);
    roll=0;
  }
  
  const Range<double> jointRangeHeadTilt(jointLimitHeadTiltN,jointLimitHeadTiltP);
  const Range<double> jointRangeHeadPan( jointLimitHeadPanN, jointLimitHeadPanP);
  tilt = jointRangeHeadTilt.limit(tilt);
  pan = jointRangeHeadPan.limit(pan);
}



void MSH2004HeadControl::execute()
{
  if(headIsBlockedBySpecialActionOrWalk)
  {
    headControlModeExecutedLastTime.headControlMode = HeadControlMode::none;
    headControlState=otherModes;
    lastHeadControlState=otherModes;
    headPathPlanner.lastTilt = sensorDataBuffer.lastFrame().data[SensorData::headTilt];
    headPathPlanner.lastPan  = sensorDataBuffer.lastFrame().data[SensorData::headPan];
    headPathPlanner.lastRoll = sensorDataBuffer.lastFrame().data[SensorData::headRoll];
  }
  
  headMotionRequest.mouth=0;
  
  buildCameraMatrix(sensorDataBuffer.lastFrame(),headState);
    
  long timeSinceBallSeenLast = SystemCall::getTimeSince(ballPosition.seen.timeWhenLastSeen);
  
  //Transitions between states
  switch (headControlMode.headControlMode)
  {
  case HeadControlMode::openChallengeReset:
		  pidData.setToDefaults();
      pidData.setValues(JointData::mouth,0,0,0);
    break;
  case HeadControlMode::openChallengeJoysickMode:
		  pidData.setToDefaults();
	  	pidData.setValues(JointData::headPan,0,0,0);
  		pidData.setValues(JointData::headRoll,0,0,0);
      pidData.setValues(JointData::legFR1,0,0,0);
      pidData.setValues(JointData::legFR2,0,0,0);
      pidData.setValues(JointData::legFR3,0,0,0);
      pidData.setValues(JointData::legFL1,0,0,0);
      pidData.setValues(JointData::legFL2,0,0,0);
      pidData.setValues(JointData::legFL3,0,0,0);

      pidData.setValues(JointData::legHR1,0,0,0);
      pidData.setValues(JointData::legHR2,0,0,0);
      pidData.setValues(JointData::legHR3,0,0,0);
      pidData.setValues(JointData::legHL1,0,0,0);
      pidData.setValues(JointData::legHL2,0,0,0);
      pidData.setValues(JointData::legHL3,0,0,0);

      pidData.setValues(JointData::mouth,0,0,0);
      pidData.setValues(JointData::tailPan,0,0,0);
      pidData.setValues(JointData::tailTilt,0,0,0);
    break;
  case HeadControlMode::openChallengePullBridge:
		    pidData.setToDefaults();
	  	  pidData.setValues(JointData::headPan,0,0,0);
  		  pidData.setValues(JointData::headRoll,0,0,0);
//        pidData.setValues(JointData::mouth,0,0,0);
        headMotionRequest.tilt=153256;
        this->headMotionRequest.mouth = 0;
		  break;
  case HeadControlMode::openChallengeTest:
	  	pidData.setToDefaults();
	  	pidData.setToDefaults();
      setJoints(-518453,0,-460000,200, 56217);
//      setJoints(toMicroRad(jointLimitHeadTiltP),0,toMicroRad(jointLimitHeadRollP),200, -1400000);
		break;
  case HeadControlMode::openChallengeTest2:
    {
	  	pidData.setToDefaults();
      setJoints(-773022,0,872700,200, -156217);
    }
	break;

  case HeadControlMode::openChallengeGoToBridge:
   	  pidData.setToDefaults();
      headMotionRequest.roll=363721;
      headMotionRequest.pan=0;
      headMotionRequest.tilt=153256;
      headMotionRequest.mouth=-2000000;
      break;
    /*
    {
    const int tiltTolerance = 60000;
    const int panTolerance  = 60000;
  
    long tiltDiff = sensorDataBuffer.lastFrame().data[SensorData::headTilt] - headMotionRequest.tilt;
    long panDiff  = sensorDataBuffer.lastFrame().data[SensorData::headPan]  - headMotionRequest.pan;
  
    if (tiltDiff > tiltTolerance)
    {
      headMotionRequest.tilt += tiltDiff-tiltTolerance;
    }
    else if (tiltDiff < -tiltTolerance)
    {
      headMotionRequest.tilt += tiltDiff+tiltTolerance;
    }
  
    if (panDiff > panTolerance)
    {
      headMotionRequest.pan += panDiff-panTolerance;
    }
    else if (panDiff < -panTolerance)
    {
     headMotionRequest.pan += panDiff+panTolerance;
    }
    }
    */
    headMotionRequest.roll=496055;
    headMotionRequest.mouth=-2000000;
    break;
  case HeadControlMode::openChallengeLookAtRedLineStart:
		    pidData.setToDefaults();
        pidData.setValues(JointData::mouth,0,0,0);
    {
      // setting the joints depending on the distance of the start of the redline
      double tilt = 0; // 0= weit entfernt, -452960=nah
      if ((specialPercept.ocRedLine.lineStart.x < 500) &&
          (SystemCall::getTimeSince(specialPercept.ocRedLine.timeWhenLastSeen) < 250))
      {
        tilt = -452960 * (1-(specialPercept.ocRedLine.lineStart.x / 500));
      }
      setJoints(long(tilt),0,0);

    } break;
  case HeadControlMode::openChallengeLookAtBitePoint1:
 		    pidData.setToDefaults();
   {
      if (SystemCall::getTimeSince(specialPercept.ocBridge.bitePoint[BitePoint::frontleft].timeWhenLastSeen) < 250)
      {
        // setting the joints depending on the angle of the bitePoint1
        double pan = 0; // 0= gerade aus, -1623200 = -90 (rechts), 1623200 = 90 (links)
        pan = toDegrees(specialPercept.ocBridge.bitePoint[BitePoint::frontleft].angleTo) * 18035;
        setJoints(0,long(pan),0);
      }
    } break;
  case HeadControlMode::openChallengeLookAtBitePoint2:
		    pidData.setToDefaults();
    {
      if (SystemCall::getTimeSince(specialPercept.ocBridge.bitePoint[BitePoint::frontright].timeWhenLastSeen) < 250)
      {
        // setting the joints depending on the angle of the bitePoint1
        double pan = 0; // 0= gerade aus, -1623200 = -90 (rechts), 1623200 = 90 (links)
        pan = toDegrees(specialPercept.ocBridge.bitePoint[BitePoint::frontright].angleTo) * 18035;
        setJoints(0,long(pan),0);
      }
    } break;
  case HeadControlMode::openChallengeLookAtBitePoint3:
		    pidData.setToDefaults();
    {
      if (SystemCall::getTimeSince(specialPercept.ocBridge.bitePoint[BitePoint::behindleft].timeWhenLastSeen) < 250)
      {
        // setting the joints depending on the angle of the bitePoint1
        double pan = 0; // 0= gerade aus, -1623200 = -90 (rechts), 1623200 = 90 (links)
        pan = toDegrees(specialPercept.ocBridge.bitePoint[BitePoint::behindleft].angleTo) * 18035;
        setJoints(0,long(pan),0);
      }
    } break;
  case HeadControlMode::openChallengeLookAtBitePoint4:
		    pidData.setToDefaults();
    {
      if (SystemCall::getTimeSince(specialPercept.ocBridge.bitePoint[BitePoint::behindright].timeWhenLastSeen) < 250)
      {
        // setting the joints depending on the angle of the bitePoint1
        double pan = 0; // 0= gerade aus, -1623200 = -90 (rechts), 1623200 = 90 (links)
        pan = toDegrees(specialPercept.ocBridge.bitePoint[BitePoint::behindright].angleTo) * 18035;
        setJoints(0,long(pan),0);
      }
    } break;
  case HeadControlMode::searchForBall:
  case HeadControlMode::searchAuto:
  case HeadControlMode::searchAutoForGoalie:
		    pidData.setToDefaults();
        pidData.setValues(JointData::mouth,0,0,0);
    switch (headControlState)
    {
    case otherModes:
      if (timeSinceBallSeenLast<250)
      {
        headControlState=returnToBall;
      }
      else
      {
        headControlState=ballLost;
      }
      break;
    case lookAtBall:
    case lookAtBallGoalie:
		    pidData.setToDefaults();
        pidData.setValues(JointData::mouth,0,0,0);
      if (SystemCall::getTimeSince(ballPosition.seen.timeWhenLastSeen) > timeAfterWhichBallIsConsideredLost)
      {
        headControlState=ballJustLost;
      }
      else if (headControlMode.headControlMode == HeadControlMode::searchAuto)
      {
        long consecTime = ballPosition.seen.getConsecutivelySeenTime();
        if (/*(consecTime>2000) || ((*/consecTime>300/*)&&(ballPosition.seen.speed.abs()<150))*/)
        {
          headControlState=searchAutoUpperScan;
        }
      }
      else if (headControlMode.headControlMode == HeadControlMode::searchAutoForGoalie)
      {
        long consecTime = ballPosition.seen.getConsecutivelySeenTime();
        if ((consecTime>3000) && (ballPosition.seen.x > xPosHalfWayLine))
        {
          headControlState=searchAutoUpperScan;
        }
        else if (consecTime>6000)
        {
          headControlState=searchAutoUpperScan;
        }
      }
      break;
    case ballJustLost:
      if (timeSinceBallSeenLast<250)
      {
        if (headControlMode.headControlMode == HeadControlMode::searchAutoForGoalie)
          headControlState=lookAtBallGoalie;
        else
          headControlState=lookAtBall;
      }
      else if (headPathPlanner.isLastPathFinished())
      {
        headControlState=ballLost;
      }
      break;
    case ballLost:
      if (timeSinceBallSeenLast<250)
      {
        if (headControlMode.headControlMode == HeadControlMode::searchAutoForGoalie)
          headControlState=lookAtBallGoalie;
        else
          headControlState=lookAtBall;
      }
      break;
    case returnToBall:
      if (timeSinceBallSeenLast<250)
      {
        if (headControlMode.headControlMode == HeadControlMode::searchAutoForGoalie)
          headControlState=lookAtBallGoalie;
        else
          headControlState=lookAtBall;
      }
      else if (headPathPlanner.isLastPathFinished())
      {
        headControlState=ballJustLost;
      }
      break;
    case searchAutoUpperScan:
      if (headControlMode.headControlMode == HeadControlMode::searchForBall)
      {
        headControlState=returnToBall;
      }
      else if (headPathPlanner.isLastPathFinished())
      {
        headControlState=searchAutoLowerScan;
      }
      break;
    case searchAutoLowerScan:
      if (timeSinceBallSeenLast<250)
      {
        headControlState=lookAtBall;
      }
      else if (headControlMode.headControlMode == HeadControlMode::searchForBall)
      {
        headControlState=returnToBall;
      }
      else if (headPathPlanner.isLastPathFinished())
      {
        headControlState=ballJustLost;
      }
      break;
    }
    break;
    default:
      headControlState=otherModes;
  }
  
  //Execution
  switch (headControlState)
  {
  case lookAtBall:
    executeLookAtBall();
    break;
  case ballJustLost:
    executeBallJustLost();
    break;
  case ballLost:
    executeBallLost();
    break;
  case returnToBall:
    executeReturnToBall();
    break;
  case searchAutoUpperScan:
    executeSearchAutoUpperScan();
    break;
  case searchAutoLowerScan:
    executeSearchAutoLowerScan();
    break;
  case lookAtBallGoalie:
    executeLookAtBallGoalie();
    break;
  case otherModes:
    {
      switch (headControlMode.headControlMode)
      {
      case HeadControlMode::lookLeft:
        lastScanWasLeft = true;
        lookLeft();
        break;
      case HeadControlMode::lookRight:
        lastScanWasLeft = false;
        lookRight();
        break;
      case HeadControlMode::searchForLandmarks:
		    pidData.setToDefaults();
        pidData.setValues(JointData::mouth,0,0,0);
        lastScanWasLeft = (headPathPlanner.lastPan>0);
        searchForLandmarks();
        break;
      case HeadControlMode::openChallengeCheckBite:
		    pidData.setToDefaults();
        pidData.setValues(JointData::mouth,0x06,0x00,0x04);
//	  	  pidData.setValues(JointData::headPan,0x06,0x04,0x02);
        lastScanWasLeft = (headPathPlanner.lastPan>0);
        doOpenChallengeCheckBite();
        headMotionRequest.mouth = 0;
        break;
      case HeadControlMode::searchForLines:
        lastScanWasLeft = (headPathPlanner.lastPan>0);
        searchForLines();
        break;
      case HeadControlMode::searchForSpecialBall:
        lastScanWasLeft = (headPathPlanner.lastPan>0);
        searchForSpecialBall();
        break;
      case HeadControlMode::searchForObstacles:
        lastScanWasLeft = (headPathPlanner.lastPan>0);
        searchForObstacles();
        break;
      case HeadControlMode::searchForObstaclesInFront:
        lastScanWasLeft = (headPathPlanner.lastPan>0);
        searchForObstaclesInFront();
        break;
      case HeadControlMode::lookBetweenFeet:
        setJoints(-1000000, 0, 0, 200);
        break;
      case HeadControlMode::lookAroundWhenBallCaught:
        lastScanWasLeft = (headPathPlanner.lastPan>0);
        lookAroundWhenBallCaught();
        break;
      case HeadControlMode::lookAroundWhenBallJustCaught:
        lastScanWasLeft = (headPathPlanner.lastPan>0);
        lookAroundWhenBallJustCaught();
        break;
      case HeadControlMode::lookStraightAhead:
		    pidData.setToDefaults();
        pidData.setValues(JointData::mouth,0,0,0);
        setJoints(-48000,0,233000,200);
        break;
      case HeadControlMode::catchBall:
        if(motionRequest.motionType == MotionRequest::walk
          && (motionRequest.walkParams.translation.y > 50.0 ||
          motionRequest.walkParams.rotation > 0.78))
        {
          setJoints(-975000,-175000,0,400,-700000);
          lastScanWasLeft = false;
        }
        else if(motionRequest.motionType == MotionRequest::walk
          && (motionRequest.walkParams.translation.y < -50.0 ||
          motionRequest.walkParams.rotation < -0.78))
        {
          setJoints(-975000,175000,0,400,-700000);
          lastScanWasLeft = true;
        }
        else
        {
          setJoints(-975000,0,0,400,-700000);
        }
        break;
      case HeadControlMode::stayAsForced:
        stayAsForced();
        break;
      case HeadControlMode::followTail:
        lastScanWasLeft = (headPathPlanner.lastPan>0);
        followTail();
        break;
      case HeadControlMode::lookToStars:
        
        if(getRobotConfiguration().getRobotDesign()== RobotConfiguration::ERS7)
        {
          setJoints(toMicroRad(jointLimitHeadTiltP),0,toMicroRad(jointLimitHeadRollP),200, -700000);
        }
        
        else
        {
          setJoints(toMicroRad(jointLimitHeadTiltP),0,0,200, -700000);
        }
        break;
      case HeadControlMode::lookAtPoint:
        {
          double tilt,pan,roll;
          Vector3<double> point(headControlMode.point.x,headControlMode.point.y,headControlMode.point.z);
          lookAtPoint(point,headControlMode.offset,tilt,pan,roll);
          setJoints(toMicroRad(tilt),toMicroRad(pan),toMicroRad(roll), 200,0);
          break;
        }
      case HeadControlMode::watchOrigin:
        {
          double tilt,pan,roll;
          Vector3<double> point(0,0,180);
          //2do: this is not optimal: we start scanning after a single missing image:
          if (robotPose.getValidity()<0.5)
          {
            if (robotPose.rotation<-0.9)
            {
              //if we can/will see the origin, then left
              lastScanWasLeft=true;
              if(getRobotConfiguration().getRobotDesign()== RobotConfiguration::ERS7)
              {
                setJoints(0,1500000 ,0, 200,0);
              }
              else
              {
                setJoints(300000,1500000 ,0, 200,0);
              }
            }
            else if (robotPose.rotation>0.9)
            {
              //if we can/will see the origin, then right
              lastScanWasLeft=false;
              if(getRobotConfiguration().getRobotDesign()== RobotConfiguration::ERS7)
              {
                setJoints(0,-1500000 ,0, 200,0);
              }
              else
              {
                setJoints(300000,-1500000 ,0, 200,0);
              }
            }
            else
            {
              //scan for origin
              if(headPathPlanner.isLastPathFinished())
              {
                if(lastScanWasLeft)
                {
                  lookAtPoint(point,Vector2<int>(-65,0),tilt,pan,roll);
                  Vector3<long> right(toMicroRad(tilt),toMicroRad(pan),toMicroRad(roll));
                  Vector3<long> points[1]={right};
                  headPathPlanner.init(points,1, 550);
                }
                else
                {
                  lookAtPoint(point,Vector2<int>(65,0),tilt,pan,roll);
                  Vector3<long> left(toMicroRad(tilt),toMicroRad(pan),toMicroRad(roll));
                  Vector3<long> points[1]={left};
                  headPathPlanner.init(points,1, 550);
                }
                lastScanWasLeft=!lastScanWasLeft;
              }
              headPathPlanner.getAngles(headMotionRequest.tilt, headMotionRequest.pan, headMotionRequest.roll);
            }
          }
          else
          {
            double rota=-robotPose.speedbyDistanceToGoal*30;
            lookAtPoint(point,Vector2<int>((int)rota,0),tilt,pan,roll);
            setJoints(toMicroRad(tilt),toMicroRad(pan),toMicroRad(roll), 200,0);
          }
          break;
        }
      case HeadControlMode::lookParallelToGround:
        if(getRobotConfiguration().getRobotDesign()== RobotConfiguration::ERS7)
        {
          if (headState.bodyTilt>0)
          {
            setJoints(0,0,toMicroRad(headState.bodyTilt));
          }
          else
          {
            setJoints(toMicroRad(headState.bodyTilt),0,0);
          }
        }
        else
        {
          setJoints(toMicroRad(headState.bodyTilt),0,-toMicroRad(headState.bodyRoll));
        }
        break;
      case HeadControlMode::scanMediumWidth:
        setJoints(toMicroRad(headState.bodyTilt),0,-toMicroRad(headState.bodyRoll));
        break;
      case HeadControlMode::lookJustBelowHorizon:
        setJoints(
          toMicroRad(headState.bodyTilt - cameraInfo.openingAngleHeight*.5),
          toMicroRad(cameraInfo.openingAngleWidth/4*sin((double )SystemCall::getCurrentSystemTime()/200.0)),
          0);
        break;
      case HeadControlMode::direct:
        moveHead(headControlMode.tilt, headControlMode.pan, headControlMode.roll, headControlMode.speed, headControlMode.mouth);
      default: /* if none is selected, do nothing so direction can be set manually through the head motion tester */
        break;
      }
    }
  }
  
  headControlModeExecutedLastTime.headControlMode = headControlMode.headControlMode;
  lastHeadControlState = headControlState;
  headPathPlanner.lastTilt=headMotionRequest.tilt;
  headPathPlanner.lastPan=headMotionRequest.pan;
  headPathPlanner.lastRoll=headMotionRequest.roll;
}


void MSH2004HeadControl::getLookAtBallAngles(double& tilt,double& pan,double& roll)
{
  double distanceToBall = Geometry::distanceTo(robotPose.getPose(), ballPosition.seen);
  
  // calculate where the ball should be in the image:
  // far away balls are small and close to the horizon,
  // close balls appear under horizon...
  if (distanceToBall > 800) distanceToBall = 800;
  int yOffset = (int)(0 - 60.0*(800 - distanceToBall)/800.0);
  
  lookAtPoint(Vector3<double>(ballPosition.seen.x, ballPosition.seen.y, ballRadius), Vector2<int>(0, yOffset), tilt, pan, roll);
}

void MSH2004HeadControl::getLookAtBallAngles2(long& tilt,long& pan,long& roll)
{

  double distanceToBall = Geometry::distanceTo(robotPose.getPose(), ballPosition.seen);
  if (distanceToBall > 1000) distanceToBall = 1000;
  if (distanceToBall <  100) distanceToBall =  100;
  double angleToBall = toDegrees(Geometry::angleTo(robotPose.getPose(),ballPosition.seen));
  if (angleToBall >  90) angleToBall =  90;
  if (angleToBall < -90) angleToBall = -90;

  tilt = -313000 + (long)(((distanceToBall - 100.0) / 900.0)*365400.0);

  pan = (long)(angleToBall * (1623200.0/90.0));

  roll = -349100 + (long)(((distanceToBall - 100.0) / 900.0)*265609.0);

}


void MSH2004HeadControl::executeLookAtBallGoalie()
{
  /*
  double tilt, pan, roll,stepWidth= 2 * pi / 180 ,scanBorder = pi/7;
  
  getLookAtBallAngles(tilt,pan,roll);
  
  // clipped to a minimum angle to prevent from touching the ball
  if (tilt < -0.86) tilt=-0.86;
  
  if (leftWatch)
  {
    if (fabs(scanPan) < scanBorder)
      scanPan += stepWidth;
    else
    {
      scanPan -= stepWidth;
      leftWatch = false;
    }
  }
  else
  {
    if (fabs(scanPan) < scanBorder)
      scanPan -= stepWidth;
    else
    {
      scanPan += stepWidth;
      leftWatch = true;
    }
    
  }
  
  setJoints(toMicroRad(tilt), toMicroRad(pan+scanPan), toMicroRad(roll));
  */
  executeLookAtBall();
}

void MSH2004HeadControl::executeLookAtBall()
{

  /*
   * previous Version
   *
  double tilt, pan, roll;
  getLookAtBallAngles(tilt,pan,roll);
  // clipped to a minimum angle to prevent from touching the ball
  if (tilt < -0.86) tilt=-0.86;
  setJoints(toMicroRad(tilt), toMicroRad(pan+scanPan), toMicroRad(roll));
  */

  long tilt, pan, roll;

  getLookAtBallAngles2(tilt,pan,roll);

  setJoints(tilt,pan,roll);
}

void MSH2004HeadControl::executeBallJustLost()
{
  if (lastHeadControlState!=headControlState)
  {
    // scan around propagated ball position can look pretty ugly in easy cases, therefore we scan around seen ball
    //Vector2<double> propagatedBallPos = ballPosition.propagated.getPropagatedPosition(SystemCall::getCurrentSystemTime());
    /*
    double tilt,pan,roll;
    getLookAtBallAngles(tilt,pan,roll);
    Vector3<long> littleLeft( toMicroRad(tilt),        toMicroRad(pan)+300000,toMicroRad(roll));
    Vector3<long> littleRight(toMicroRad(tilt),        toMicroRad(pan)-300000,toMicroRad(roll));
    Vector3<long> littleUp(   toMicroRad(tilt)+120000, toMicroRad(pan),toMicroRad(roll));
    Vector3<long> littleDown( toMicroRad(tilt)-280000, toMicroRad(pan),toMicroRad(roll));
    Vector3<long> littlePoints[4]={littleDown,littleLeft,littleUp,littleRight};
    headPathPlanner.init(littlePoints,4, 500);
    */

    long tilt, pan, roll;

    getLookAtBallAngles2(tilt,pan,roll);

    Vector3<long> littleLeft( tilt,        pan+300000,roll);
    Vector3<long> littleRight(tilt,        pan-300000,roll);
    Vector3<long> littleUp(   tilt+120000, pan,roll);
    Vector3<long> littleDown( tilt-280000, pan,roll);
    Vector3<long> littlePoints[4]={littleDown,littleLeft,littleUp,littleRight};
    headPathPlanner.init(littlePoints,4, 500);

  }
  headPathPlanner.getAngles(headMotionRequest.tilt, headMotionRequest.pan, headMotionRequest.roll);
}

void MSH2004HeadControl::executeBallLost()
{
  Vector3<long> left;
  Vector3<long> right;
  Vector3<long> halfLeftDown;
  Vector3<long> halfRightDown;
  Vector3<long> up;
  Vector3<long> down;
  
  
  if(getRobotConfiguration().getRobotDesign()== RobotConfiguration::ERS7)
  {
    /*
    left.x =  toMicroRad(headState.bodyTilt - cameraInfo.openingAngleHeight / 4.0);
    left.y = 1500000;
    left.z = toMicroRad(fromDegrees(-10));
    
    right.x = toMicroRad(headState.bodyTilt - cameraInfo.openingAngleHeight / 4.0);
    right.y = -1500000;
    right.z = toMicroRad(fromDegrees(-10));
    
    halfLeftDown.x = -600000;
    halfLeftDown.y =  800000;
    halfLeftDown.z = toMicroRad(fromDegrees(15));
    
    halfRightDown.x = -600000;
    halfRightDown.y = -800000;
    halfRightDown.z = toMicroRad(fromDegrees(15));
    
    up.x = toMicroRad(headState.bodyTilt - cameraInfo.openingAngleHeight / 6.0);
    up.y = 0;
    up.z = toMicroRad(fromDegrees(25));
    
    down.x = -960000;
    down.y = 0;
    down.z = 0;
    */

    left.x = 52400;
    left.y = 1623200;
    left.z = 97123;
    
    right.x = 52400;
    right.y = -1623200;
    right.z = 97123;
    
    halfLeftDown.x = 52400;
    halfLeftDown.y = 800000;
    halfLeftDown.z = -83491;
    
    halfRightDown.x = 52400;
    halfRightDown.y = -800000;
    halfRightDown.z = -83491;
    
    up.x = 52400;
    up.y = 0;
    up.z = 97123;
    
    down.x = -31300;
    down.y = 0;
    down.z = -349100;

  }
  
  else
  {
    left.x = toMicroRad(headState.bodyTilt - cameraInfo.openingAngleHeight / 4.0);
    left.y = 1500000;
    left.z = 0;
    
    right.x = toMicroRad(headState.bodyTilt - cameraInfo.openingAngleHeight / 4.0);
    right.y = -1500000;
    right.z = 0;
    
    halfLeftDown.x = -600000;
    halfLeftDown.y =  800000;
    halfLeftDown.z = 0;
    
    halfRightDown.x = -600000;
    halfRightDown.y = -800000;
    halfRightDown.z = 0;
    
    up.x = toMicroRad(headState.bodyTilt - cameraInfo.openingAngleHeight / 6.0);
    up.y = 0;
    up.z = 0;
    
    down.x = -960000;
    down.y = 0;
    down.z = 0;
  }
  if ((lastHeadControlState!=headControlState)||headPathPlanner.isLastPathFinished())
  {
    if (lastScanWasLeft)
    {
      Vector3<long> points[4]={down,halfRightDown,right,up};
      headPathPlanner.init(points,4, 800);
    }
    else
    {
      Vector3<long> points[4]={up,left,halfLeftDown,down};
      headPathPlanner.init(points,4, 800);
    }
    lastScanWasLeft = !lastScanWasLeft;
  }
  headPathPlanner.getAngles(headMotionRequest.tilt, headMotionRequest.pan, headMotionRequest.roll);
}

void MSH2004HeadControl::executeReturnToBall()
{
  if (lastHeadControlState!=headControlState)
  {
    /*
    double tilt, pan, roll;
    getLookAtBallAngles(tilt,pan,roll);
    if (tilt < -0.86) tilt=-0.86;
    
    Vector3<long> ball(toMicroRad(tilt), toMicroRad(pan),toMicroRad(roll));
    //calculated time proportional to effort to reach ball angles: 400ms for rotation of pi:
    long time = (long)(sqrt(
      (double)(headPathPlanner.lastTilt-tilt)*(double)(headPathPlanner.lastTilt-tilt)+
      (double)(headPathPlanner.lastPan-pan)*(double)(headPathPlanner.lastPan-pan)+
      (double)(headPathPlanner.lastRoll)*(double)(headPathPlanner.lastRoll))/pi/1000000*400);
    headPathPlanner.init(&ball,1, time);

    */

    long tilt, pan, roll;

    getLookAtBallAngles2(tilt,pan,roll);

    Vector3<long> ball(tilt, pan, roll);

    long time = (long)(sqrt(
      (double)(headPathPlanner.lastTilt-tilt)*(double)(headPathPlanner.lastTilt-tilt)+
      (double)(headPathPlanner.lastPan-pan)*(double)(headPathPlanner.lastPan-pan)+
      (double)(headPathPlanner.lastRoll)*(double)(headPathPlanner.lastRoll))/pi/1000000*400);
    headPathPlanner.init(&ball,1, time);

  }
  headPathPlanner.getAngles(headMotionRequest.tilt, headMotionRequest.pan, headMotionRequest.roll);
}

void MSH2004HeadControl::executeSearchAutoUpperScan()
{
  if (lastHeadControlState!=headControlState)
  {
    if (lastScanWasLeft)
    {
      //initialize upper right scan, maximum 90
      Vector3<long> centerTop(    160000, headPathPlanner.lastPan-70000,0);
      Vector3<long> rightTop(     160000, max(-1550000,headPathPlanner.lastPan-1550000),0);
      Vector3<long> rightBottom( -400000, max(-1550000,headPathPlanner.lastPan-1550000),0);
      Vector3<long> points[3]={centerTop,rightTop,rightBottom};
      headPathPlanner.init(points,3, 480);
    }
    else
    {
      //initialize upper left scan, maximum 90
      Vector3<long> centerTop(    160000, headPathPlanner.lastPan+70000,0);
      Vector3<long> leftTop(      160000, min(1550000,headPathPlanner.lastPan+1550000),0);
      Vector3<long> leftBottom(  -400000, min(1550000,headPathPlanner.lastPan+1550000),0);
      Vector3<long> points[3]={centerTop,leftTop,leftBottom};
      headPathPlanner.init(points,3, 480);
    }
    lastScanWasLeft = !lastScanWasLeft;
  }
  headPathPlanner.getAngles(headMotionRequest.tilt, headMotionRequest.pan, headMotionRequest.roll);
}

void MSH2004HeadControl::executeSearchAutoLowerScan()
{
  if (lastHeadControlState!=headControlState)
  {
    double tilt,pan,roll;
    getLookAtBallAngles(tilt,pan,roll);
    if(lastScanWasLeft) // == (headPathPlanner.lastPan > 0)
    {
      //initialize lower left scan
      Vector3<long> centerBottom(-400000, toMicroRad(pan)+70000,toMicroRad(roll));
      Vector3<long> ball(toMicroRad(tilt),toMicroRad(pan)-50000,toMicroRad(roll));
      Vector3<long> points[2]={centerBottom,ball};
      headPathPlanner.init(points,2, 400);
    }
    else
    {
      //initialize lower right scan
      Vector3<long> centerBottom(-400000, toMicroRad(pan)-70000,toMicroRad(roll));
      Vector3<long> ball(toMicroRad(tilt),toMicroRad(pan)+50000,toMicroRad(roll));
      Vector3<long> points[2]={centerBottom,ball};
      headPathPlanner.init(points,2, 400);
    }
  }
  headPathPlanner.getAngles(headMotionRequest.tilt, headMotionRequest.pan, headMotionRequest.roll);
}



void MSH2004HeadControl::lookLeft()
{
  if(headControlModeExecutedLastTime.headControlMode != headControlMode.headControlMode)
  {
    Vector3<long> down(-1200000,headPathPlanner.lastPan,0);
    Vector3<long> left(-1300000,1300000,0);
    Vector3<long> points[2]={down,left};
    headPathPlanner.init(points,2, 400);
  }
  headPathPlanner.getAngles(headMotionRequest.tilt, headMotionRequest.pan, headMotionRequest.roll);
}

void MSH2004HeadControl::lookRight()
{
  if(headControlModeExecutedLastTime.headControlMode != headControlMode.headControlMode)
  {
    Vector3<long> down(-1200000,headPathPlanner.lastPan,0);
    Vector3<long> right(-1300000,-1300000,0);
    Vector3<long> points[2]={down,right};
    headPathPlanner.init(points,2, 400);
  }
  headPathPlanner.getAngles(headMotionRequest.tilt, headMotionRequest.pan, headMotionRequest.roll);
}


void MSH2004HeadControl::searchForLandmarks()
{
  //just like last year, but
  /** @todo may be we would like to see lines too */
  
  Vector3<long> left(toMicroRad(headState.bodyTilt - cameraInfo.openingAngleHeight / 6.0),  1500000,0);
  Vector3<long> right(toMicroRad(headState.bodyTilt - cameraInfo.openingAngleHeight / 6.0), -1500000,0);
  
  if(headControlModeExecutedLastTime.headControlMode != headControlMode.headControlMode)
  {
    if(headPathPlanner.lastPan < 0)
    {
      Vector3<long> points[2]={right,left};
      headPathPlanner.init(points,2, 1600);
    }
    else
    {
      Vector3<long> points[2]={left,right};
      headPathPlanner.init(points,2, 1600);
    }
  }
  
  if(headPathPlanner.getAngles(headMotionRequest.tilt, headMotionRequest.pan, headMotionRequest.roll))
  {
    if(headPathPlanner.lastPan < 0)
    {
      Vector3<long> points[1]={left};
      headPathPlanner.init(points,1, 1600);
    }
    else
    {
      Vector3<long> points[1]={right};
      headPathPlanner.init(points,1, 1600);
    }
  }
}

void MSH2004HeadControl::doOpenChallengeCheckBite()
{
  //just like last year, but
  /** @todo may be we would like to see lines too */
  
  Vector3<long> left(363721,  600000, 153256);
  Vector3<long> right(363721, -600000, 153256);
  
  if(headControlModeExecutedLastTime.headControlMode != headControlMode.headControlMode)
  {
    if(headPathPlanner.lastPan < 0)
    {
      Vector3<long> points[2]={right,left};
      headPathPlanner.init(points,2, 1600);
    }
    else
    {
      Vector3<long> points[2]={left,right};
      headPathPlanner.init(points,2, 1600);
    }
  }
  
  if(headPathPlanner.getAngles(headMotionRequest.tilt, headMotionRequest.pan, headMotionRequest.roll))
  {
    if(headPathPlanner.lastPan < 0)
    {
      Vector3<long> points[1]={left};
      headPathPlanner.init(points,1, 1600);
    }
    else
    {
      Vector3<long> points[1]={right};
      headPathPlanner.init(points,1, 1600);
    }
  }
}



void MSH2004HeadControl::searchForLines()
{
  Vector3<long> left(0, 1500000, 0);
  Vector3<long> right(0, -1500000,0);
  
  if(headControlModeExecutedLastTime.headControlMode != headControlMode.headControlMode)
  {
    if(headPathPlanner.lastPan < 0)
    {
      Vector3<long> points[2]={right,left};
      headPathPlanner.init(points,2, 800);
    }
    else
    {
      Vector3<long> points[2]={left,right};
      headPathPlanner.init(points,2, 800);
    }
  }
  
  if(headPathPlanner.getAngles(headMotionRequest.tilt, headMotionRequest.pan, headMotionRequest.roll))
  {
    if(headPathPlanner.lastPan < 0)
    {
      Vector3<long> points[1]={left};
      headPathPlanner.init(points,1, 800);
    }
    else
    {
      Vector3<long> points[1]={right};
      headPathPlanner.init(points,1, 800);
    }
  }
}

void MSH2004HeadControl::searchForSpecialBall()
{
  
  //Vector3<long> centerTop(    160000, headPathPlanner.lastPan-70000,0);
  Vector3<long> rightTop(     160000, -1500000,0);
  Vector3<long> rightCenter( -120000, -1500000,0);
  Vector3<long> down(        -900000,        0,0);
  Vector3<long> leftCenter(  -120000,  1500000,0);
  Vector3<long> leftTop(      160000,  1500000,0);
  
  if(headControlModeExecutedLastTime.headControlMode != headControlMode.headControlMode)
  {
    if(headPathPlanner.lastPan < 0)
    {
      Vector3<long> points[3]={rightTop,down,leftTop};
      headPathPlanner.init(points,3, 8000);
    }
    else
    {
      Vector3<long> points[3]={leftTop,down,rightTop};
      headPathPlanner.init(points,3, 8000);
    }
  }
  
  if(headPathPlanner.getAngles(headMotionRequest.tilt, headMotionRequest.pan, headMotionRequest.roll))
  {
    if(headPathPlanner.lastPan < 0)
    {
      Vector3<long> points[3]={leftTop,down,rightTop};
      headPathPlanner.init(points,3, 8000);
    }
    else
    {
      Vector3<long> points[3]={rightTop,down,leftTop};
      headPathPlanner.init(points,3, 8000);
    }
  }
}

void MSH2004HeadControl::searchForObstacles()
{
  Vector3<long> left(toMicroRad(headState.bodyTilt - cameraInfo.openingAngleHeight / 6.0),  1500000,0);
  Vector3<long> right(toMicroRad(headState.bodyTilt - cameraInfo.openingAngleHeight / 6.0), -1500000,0);
  Vector3<long> center(toMicroRad(headState.bodyTilt - cameraInfo.openingAngleHeight / 2.0), 0,0);
  
  if(headControlModeExecutedLastTime.headControlMode != headControlMode.headControlMode)
  {
    if(headPathPlanner.lastPan > 0)
    {
      Vector3<long> points[2]={left,right};
      headPathPlanner.init(points, 2, 800);
    }
    else
    {
      Vector3<long> points[3]={right, center, left};
      headPathPlanner.init(points, 3, 800);
    }
  }
  
  if(headPathPlanner.getAngles(headMotionRequest.tilt, headMotionRequest.pan, headMotionRequest.roll))
  {
    if(headPathPlanner.lastPan > 0)
    {
      Vector3<long> points[1]={right};
      headPathPlanner.init(points, 1, 800);
    }
    else
    {
      Vector3<long> points[2]={center, left};
      headPathPlanner.init(points, 2, 800);
    }
  }
}

void MSH2004HeadControl::searchForObstaclesInFront()
{
  Vector3<long> left(toMicroRad(headState.bodyTilt - cameraInfo.openingAngleHeight / 6.0), 1400000,0);
  Vector3<long> right(toMicroRad(headState.bodyTilt - cameraInfo.openingAngleHeight / 6.0), -1400000,0);
  Vector3<long> center(toMicroRad(headState.bodyTilt - cameraInfo.openingAngleHeight / 1.5), 0,0);
  
  if(headControlModeExecutedLastTime.headControlMode != headControlMode.headControlMode)
  {
    if(headPathPlanner.lastPan > 0)
    {
      Vector3<long> points[2]={left,right};
      headPathPlanner.init(points, 2, 800);
    }
    else
    {
      Vector3<long> points[3]={right, center, left};
      headPathPlanner.init(points, 3, 800);
    }
  }
  
  if(headPathPlanner.getAngles(headMotionRequest.tilt, headMotionRequest.pan, headMotionRequest.roll))
  {
    if(headPathPlanner.lastPan > 0)
    {
      Vector3<long> points[1]={right};
      headPathPlanner.init(points, 1, 800);
    }
    else
    {
      Vector3<long> points[2]={center, left};
      headPathPlanner.init(points, 2, 800);
    }
  }
}


void MSH2004HeadControl::lookAroundWhenBallCaught()
{
  Vector3<long> left(  300000, 1200000,0);
  Vector3<long> middle(     0,       0,0);
  Vector3<long> right( 300000,-1200000,0);
  //The mode is active the first time after another one
  if(headControlModeExecutedLastTime.headControlMode != headControlMode.headControlMode)
  {
    //If current pan is left:
    if(headPathPlanner.lastPan > 0)
    {
      //from current position to left, center and right.
      Vector3<long> points[3]={left,middle,right};
      headPathPlanner.init(points,3, 800);
    }
    else
    {
      //from current position to right, center and left.
      Vector3<long> points[3]={right,middle,left};
      headPathPlanner.init(points,3, 800);
    }
  }
  
  if(headPathPlanner.getAngles(headMotionRequest.tilt, headMotionRequest.pan, headMotionRequest.roll))
  {
    //init the next loop
    if(headPathPlanner.lastPan > 0)
    {
      Vector3<long> points[2]={middle,right};
      headPathPlanner.init(points,2, 800);
    }
    else
    {
      Vector3<long> points[2]={middle,left};
      headPathPlanner.init(points,2, 800);
    }
  }
}

void MSH2004HeadControl::lookAroundWhenBallJustCaught()
{
  Vector3<long> left(  100000, 500000,0);
  Vector3<long> right( 100000,-500000,0);
  
  //The mode is active the first time after another one
  if(headControlModeExecutedLastTime.headControlMode != headControlMode.headControlMode)
  {
    //If current pan is left:
    if(headPathPlanner.lastPan > 0)
    {
      //from current position to left, center and right.
      Vector3<long> points[2]={left,right};
      headPathPlanner.init(points,2, 350);
    }
    else
    {
      //from current position to right, center and left.
      Vector3<long> points[3]={right,left};
      headPathPlanner.init(points,2, 350);
    }
  }
  
  if(headPathPlanner.getAngles(headMotionRequest.tilt, headMotionRequest.pan, headMotionRequest.roll))
  {
    //init the next loop
    if(headPathPlanner.lastPan > 0)
    {
      Vector3<long> points[1]={right};
      headPathPlanner.init(points,1, 350);
    }
    else
    {
      Vector3<long> points[1]={left};
      headPathPlanner.init(points,1, 350);
    }
  }
}

void MSH2004HeadControl::stayAsForced()
{
  const int tiltTolerance = 60000;
  const int panTolerance  = 60000;
  const int rollTolerance = 50000;
  
  long tiltDiff = sensorDataBuffer.lastFrame().data[SensorData::headTilt] - headMotionRequest.tilt;
  long panDiff  = sensorDataBuffer.lastFrame().data[SensorData::headPan]  - headMotionRequest.pan;
  long rollDiff = sensorDataBuffer.lastFrame().data[SensorData::headRoll] - headMotionRequest.roll;
  
  if (tiltDiff > tiltTolerance)
  {
    headMotionRequest.tilt += tiltDiff-tiltTolerance;
  }
  else if (tiltDiff < -tiltTolerance)
  {
    headMotionRequest.tilt += tiltDiff+tiltTolerance;
  }
  
  if (panDiff > panTolerance)
  {
    headMotionRequest.pan += panDiff-panTolerance;
  }
  else if (panDiff < -panTolerance)
  {
    headMotionRequest.pan += panDiff+panTolerance;
  }
  
  if (rollDiff > rollTolerance)
  {
    headMotionRequest.roll += rollDiff-rollTolerance;
  }
  else if (rollDiff < -rollTolerance)
  {
    headMotionRequest.roll += rollDiff+rollTolerance;
  }
}

void MSH2004HeadControl::followTail()
{
  long tilt, pan, tilt2;
  if(getRobotConfiguration().getRobotDesign() == RobotConfiguration::ERS7)
  {
    tilt2 = sensorDataBuffer.lastFrame().data[SensorData::headTilt2];
    if(sensorDataBuffer.lastFrame().data[SensorData::backF] > 15)
    {
      if(sensorDataBuffer.lastFrame().data[SensorData::backR] > 15) tilt2 = 0;
      else tilt2 -= toMicroRad(fromDegrees(20));
    }
    else if(sensorDataBuffer.lastFrame().data[SensorData::backR] > 15) tilt2 += toMicroRad(fromDegrees(20));
  }
  else // not ERS7
  {
    tilt2 = 0;
  }
  
  pan = toMicroRad(
    fromMicroRad(sensorDataBuffer.lastFrame().data[SensorData::tailPan]) / jointLimitTailPanN * jointLimitHeadPanN);
  
  tilt = toMicroRad(
    (fromMicroRad(sensorDataBuffer.lastFrame().data[SensorData::tailTilt]) - jointLimitTailTiltN) /
    (jointLimitTailTiltP - jointLimitTailTiltN) * (jointLimitHeadTiltP - jointLimitHeadTiltN)
    +jointLimitHeadTiltN
    );
  
  setJoints(tilt, pan, tilt2);
}

void MSH2004HeadControl::buildCameraMatrix(const SensorData& data, const HeadState& headState)
{
  if(lastFrame != data.frameNumber)
  {
    lastFrame = data.frameNumber;
    currentTilt = fromMicroRad(data.data[SensorData::headTilt]);
    currentPan  = fromMicroRad(data.data[SensorData::headPan]);
    currentRoll = fromMicroRad(data.data[SensorData::headRoll]);
    
    Geometry::calculateCameraMatrix(currentTilt, currentPan, currentRoll, headState, cameraMatrix2);
  }
}

void MSH2004HeadControl::fromWorld2CameraCoordinates
(
 const Vector3<double> pointIn3D,
 Vector3<double> &pointInCameraCoordinates
 )
{
  RotationMatrix bodyRotation;
  Vector3<double> bodyTranslation;
  //  double phi = -this->robotPose.getPose().getAngle();
  bodyRotation.fromKardanRPY(0, 0, -robotPose.rotation);
  bodyTranslation.x = robotPose.translation.x;
  bodyTranslation.y = robotPose.translation.y;
  bodyTranslation.z = 0;
  
  pointInCameraCoordinates =
    cameraMatrix2.rotation.invert()*((bodyRotation.invert()*(pointIn3D - bodyTranslation) - cameraMatrix2.translation));
}

void MSH2004HeadControl::fromWorld2RobotCoordinates
(
 const Vector3<double> pointIn3D,
 Vector3<double> &pointInRobotCoordinates
 )
{
  RotationMatrix bodyRotation;
  Vector3<double> bodyTranslation;
  
  bodyRotation.fromKardanRPY(0, 0, -robotPose.rotation);
  
  bodyTranslation.x = robotPose.translation.x;
  bodyTranslation.y = robotPose.translation.y;
  bodyTranslation.z = 0;
  
  pointInRobotCoordinates = bodyRotation.invert()*(pointIn3D - bodyTranslation);
}



/*
* Change log :
*
* $Log: MSH2004HeadControl.cpp,v $
* Revision 1.27  2004/06/21 11:52:53  hamerla
* OpenChallenge
*
* Revision 1.26  2004/06/18 14:59:11  kerdels
* new headcontrolmode for open challenge
*
* Revision 1.25  2004/06/17 15:10:39  hamerla
* oc
*
* Revision 1.24  2004/06/14 12:14:15  rossdeutscher
* new fiddlings in bite bridge behaviour
*
* Revision 1.23  2004/06/04 09:18:05  hamerla
* OC headcontrol
*
* Revision 1.22  2004/06/04 07:10:36  kerdels
* fixed HeadControls lookAtBytePoint1-4
*
* Revision 1.21  2004/06/04 06:41:49  kerdels
* added HeadControls lookAtBytePoint1-4
*
* Revision 1.20  2004/06/04 05:18:06  kerdels
* slowed down search-for-landmarks
*
* Revision 1.19  2004/06/03 21:27:35  kerdels
* modified headcontrol
*
* Revision 1.18  2004/06/03 16:10:53  kerdels
* added new headcontrolmode
*
* Revision 1.17  2004/06/03 14:02:36  kerdels
* repaired HeadControl
*
* Revision 1.16  2004/06/01 16:49:49  hamerla
* oc_master goes to ramp and climb
* oc-align-to-ramp uses headcontroll oc-pull-bridge ist only hack to test
*
* Revision 1.15  2004/06/01 15:08:24  hamerla
* climb ramp (parameter work -0,1 0.6 1 1 180)
*
* Revision 1.14  2004/05/28 18:08:01  koh
* Head-Controll "openChallengeTest2" created
*
* Revision 1.13  2004/05/17 19:21:51  loetzsch
* renamed all Variables "cameraMatrix" to "cameraMatrix2"
*
* Revision 1.12  2004/05/17 15:34:37  hamerla
* Open Challenge add bite bridge
*
* Revision 1.11  2004/05/13 21:56:57  hamerla
* OpenChallenge new Headmode
*
* Revision 1.10  2004/05/08 16:18:14  hamerla
* Open Challenge
*
* Revision 1.9  2004/05/04 09:05:35  dueffert
* headPathPlanner bug fixed in all copies
*
* Revision 1.8  2004/04/08 15:33:05  wachter
* GT04 checkin of Microsoft-Hellounds
*
* Revision 1.7  2004/03/17 18:03:46  kerdels
* worked on headcontrol
*
* Revision 1.6  2004/03/17 03:19:29  kerdels
* added delay parameter to headPathPlanner to be able to stop "delay"-ms at each point of the path
*
* Revision 1.4  2004/03/08 01:38:54  roefer
* Interfaces should be const
*
* Revision 1.3  2004/03/04 10:05:20  jhoffman
* - motion process now uses odometry to propagate the robot pose while no new robot pose is being sent (this makes headcontrol a little better)
* - removed headcontrol member variable "propagatedPose" from headcontrol and cognition->motion-sender
*
* Revision 1.2  2004/02/29 13:37:15  dueffert
* doxygen bugs fixed and beautified
*
* Revision 1.1  2004/02/27 15:03:50  wachter
* cloned GT2003HeadControl as MSH2004HeadContol
*
*
*/
