/**
* @file BB2004HeadControl.h
* 
* This file contains a class for head-behavior.
*
* @author Jan Hoffmann
*/

#ifndef __BB2004HeadControl_h__
#define __BB2004HeadControl_h__

#include "HeadControl.h"
#include "Tools/Math/PIDsmoothedValue.h"


//in urad per frame: 4000 = 28.6/s
#define minHeadSpeed 4000

/**
* @class BB2004HeadControl
*
* A solution of the HeadControl module.
*/
class BB2004HeadControl : public HeadControl
{
public:
	/** 
	* Constructor.
	* @param interfaces The paramters of the HeadControl module.
	*/
	BB2004HeadControl(const HeadControlInterfaces& interfaces);

	/** Destructor. */
	~BB2004HeadControl() {}

	/** Executes the module */
	virtual void execute();

	enum headControlStates
	{
		otherModes,
		lookAtBall,
		lookAtBallGoalie,
		ballJustLost,
		ballLost,
		returnToBall,
		searchAutoUpperScan,
		searchAutoLowerScan,
		numOfStates
	} headControlState, lastHeadControlState;

	bool lastScanWasLeft,leftWatch;
	double scanPan;

private:
  CameraInfo cameraInfo;
	int autoCatchBallLost;
	bool executingCatch;

	//timestamps
	unsigned long timeCatch, timeUncatch, timeSinceCatch ;


	void executeLookAtBall();
	void executeLookAtBallGoalie();
	void executeBallJustLost();
	void executeBallLost();
	void executeReturnToBall();
	void executeSearchAutoUpperScan();
	void executeSearchAutoLowerScan();
	void executeCatchBall();
	void executeAutoCatchBall();

	void searchForLandmarks();
	void searchForLines();
	void searchForSpecialBall();
	void searchForObstacles();
	void searchForObstaclesInFront();
	void lookAroundWhenBallJustCaught();
	void lookAroundWhenBallCaught();
	void stayAsForced();
	void followTail();
	void lookLeft();
	void lookRight();
	void lookJustBelowHorizon();
	void punchLeft();

	/** Set the headjoints so that the head has an absolute roll of @param absRoll and tilt @param neckTilt*/
	void absoluteStraightRoll(double neckTilt, double absRoll);
	
	/** Set the headjoints so that the head has an absolute roll of
	@param absRoll and tilt 
	@param neckTilt
	like StraightRoll but with the new engine*/
	void absoluteStraightRolla(double neckTilt, double absRoll);
	
	/** Same as StraightRoll but with all HeadParameters. 2do: no execute but transform of joints */
	void absoluteRoll(double neckTilt, double pan,double absTilt, double mouth);

	/** 3d transformation from world coordinates to camera coordinates using the camera matrix */
	void fromWorld2CameraCoordinates(const Vector3<double> pointIn3D, Vector3<double> &pointInCameraCoordinates);

	/** 3d transformation from world coordinates to robot coordinates */
	void fromWorld2RobotCoordinates(const Vector3<double> pointIn3D, Vector3<double> &pointInRobotCoordinates);

	/** look at 3d-point on field with offset point in camera image */
	void lookAtPoint(const Vector3<double> &pos,const Vector2<int> &offset,double& tilt,double& pan,double& roll);

  /** look at 3d-point subroutine trying to find tilt1 solution for given tilt2/roll. this is useful on ERS210 and ERS7
  *@return true when matching angles were found */
  bool lookAtPointFixTilt2(const Vector3<double> &aim, const double& xtan, const double& ytan, double& tilt, double& pan, const double& tilt2);

  /** look at 3d-point subroutine trying to find tilt2 solution for given tilt1. this is only useful on ERS7
  *@return true when matching angles were found */
  bool lookAtPointFixTilt1(const Vector3<double> &aim, const double& xtan, const double& ytan, const double& tilt, double& pan, double& tilt2);

	/** calculate the angles for looking at the seen ball with a certain camera pixel depending on ball distance */
	void getLookAtBallAngles(double& tilt,double& pan,double& roll);

	/** just a fake for compatibility, no smoothing yet (speed is ignored) */
	void setJoints(long tilt, long pan, long roll, long speed=100, long mouth=0);

	/* direct HeadControl with speed parameter (smooting) */
	void moveHead(long tilt=0, long pan=0, long roll=0, long speed=400, long mouth=0);

  /** reset the values of the PIDs */
  void resetPIDs();

  /** update all information related to the body pose. */
  void updateBodyPose();
  
  /** The time when the state machine changed */
  unsigned long timeOfLastStateChange;

  /** Indicates if this head control mode was active the last time execute was called */
	HeadControlMode headControlModeExecutedLastTime;

	/** This struct can calculate a smooth series of head joint angles from a gives set of way points and an overall duration */
	struct HeadPathPlanner
	{
	public:
		/**
		* Initializes a set of points to visit in a certain time
		* @param vectors set of arcs to visit
		* @param numberOfVectors number of Vector3s in param vectors
		* @param duration time in ms to reach the last position in param vecors
		*/
		void init(const Vector3<long>* vectors, int numberOfVectors, unsigned long duration);

		/**
		* Initializes a set of points to visit in a certain time
		* @param vectors set of arcs to visit (with absolute Roll)
		* @param numberOfVectors number of Vector3s in param vectors
		* @param duration time in ms to reach the last position in param vecors
		*/
		void initAbs(Vector3<long>* vectors, int numberOfVectors, unsigned long duration);

		/**
		* Initializes a set of points to visit in a certain time
		* @param vectors set of arcs to visit (with absolute Roll)
		* @param numberOfVectors number of Vector3s in param vectors
		* @param duration time in ms to reach the last position in param vecors
		*/
		void initAbsDouble(Vector3<double>* vectors, int numberOfVectors, unsigned long duration);

		/**
		* default constructor
		*/
		HeadPathPlanner(const HeadState& headState)
    : lastTilt(0),
      lastPan(0),
      lastRoll(0),
      headState(headState),
      currentPoint(0),
      currentFrame(0),
      numberOfFrames(0),
      numberOfPoints(0)
    {}

		/**
		* Initializes a set of points to visit in a certain time. These points are
		* assumed to be cyclic. The first and last point to visit is the one closest
		* to the current head position.
		* @param vectors set of arcs to visit
		* @param numberOfVectors number of Vector3s in param vectors
		* @param duration time in ms to reach the last position in param vecors
		*/
		void initLoop(const Vector3<long>* vectors, int numberOfVectors, unsigned long duration);

		/**
		* Calculates the angles for tilt pan and roll
		* @return true if the last point is reached.
		*/
		bool getAngles(long& tilt, long& pan, long& roll);

		/**
		* Return whether the last initialized path is already finished
		* @return true if last path is finished.
		*/
		inline bool isLastPathFinished() { return (currentFrame>=numberOfFrames); }

		long lastTilt;
		long lastPan;
		long lastRoll;

	private:
    const HeadState& headState;	
    long currentPoint;
		long currentFrame;
		long numberOfFrames;

		enum {maxNumberOfPoints = 20};
		long numberOfPoints;
		Vector3<long> points[maxNumberOfPoints];
		Vector3<double> dpoints[maxNumberOfPoints];
		double firstFrame[maxNumberOfPoints];
	};

	HeadPathPlanner headPathPlanner;

	// used for interpolating while no new robot pose
	// is sent from cognition
	RobotPose posePropagatedByOdometry;

	// needed for transformations and for simple look at point
	CameraMatrix cameraMatrix2;
};

#endif //__BB2004HeadControl_h__

/*
* Change log :
* 
* $Log: BB2004HeadControl.h,v $
* Revision 1.11  2004/05/17 19:21:51  loetzsch
* renamed all Variables "cameraMatrix" to "cameraMatrix2"
*
* Revision 1.10  2004/04/09 11:35:51  roefer
* Bremen Byters German Open check-in
*
* Revision 1.9  2004/03/20 09:55:24  roefer
* Preparation for improved odometry
*
* Revision 1.8  2004/03/18 18:07:21  hodie
* Some more absolute HeadControlModes
* Absolute HeadPathPlanner
*
* Revision 1.7  2004/03/17 16:21:12  hodie
* Some more absolute HeadControlModes
* Absolute HeadPathPlanner
*
* Revision 1.6  2004/03/15 15:14:57  hodie
* shorter straightAbsRoll
*
* Revision 1.5  2004/03/15 15:09:46  hodie
* new absRoll and absStraightRoll
* some new functions and tools
* modified lookArounds
*
* Revision 1.4  2004/03/08 01:38:53  roefer
* Interfaces should be const
*
* Revision 1.3  2004/03/04 16:01:49  fritsche
* added BBautoCatchBall to BB2004HeadControl
*
* Revision 1.2  2004/02/20 12:27:32  hodie
* added catchBall and beginning of new lookArounds
*
* Revision 1.1  2004/02/04 13:41:32  roefer
* Some place holders for BB2004 modules added
*
*/
