/**
* @file GT2003HeadControl.h
* 
* This file contains a class for head-behavior.
*
* @author Jan Hoffmann
*/

#ifndef __ATH2004ERS7HeadControl_h__
#define __ATH2004ERS7HeadControl_h__

#include "HeadControl.h"
#include "Tools/Math/PIDsmoothedValue.h"


//in urad per frame: 4000 = 28.6/s
#define minHeadSpeed 4000

/**
* @class ATH2004ERS7HeadControl
*
* A solution of the HeadControl module.
*/
class ATH2004ERS7HeadControl : public HeadControl
{
public:
/** 
* Constructor.
* @param interfaces The paramters of the HeadControl module.
  */
  ATH2004ERS7HeadControl(HeadControlInterfaces& interfaces);
  
  /** Destructor. */
  ~ATH2004ERS7HeadControl() {}
  
  /** Executes the module */
  virtual void execute();
  
  enum headControlStates
  {
    otherModes,
      lookAtBall,
      lookAtBallGoalie,
      ballJustLost,
      ballLost,
      returnToBall,
      searchAutoUpperScan,
      searchAutoLowerScan,
      waitUntilStepIsFinished,
      lookToStarsWhenStepIsFinished,
      numOfStates
  } headControlState, lastHeadControlState;
  
  bool lastScanWasLeft,leftWatch;
  double scanPan;
  
private:
  CameraInfo cameraInfo;
  
  void executeLookAtBall();
  void executeLookAtBallGoalie();
  void executeBallJustLost();
  void executeBallLost();
  void executeReturnToBall();
  void executeSearchAutoUpperScan();
  void executeSearchAutoLowerScan();
  
  void searchForLandmarks();
  void searchForLines();
  void searchForSpecialBall();
  void searchForObstacles();
  void searchForObstaclesInFront();
  void lookAroundWhenBallJustCaught();
  void lookAroundWhenBallCaught();
  void stayAsForced();
  void followTail();
  void kickLeft();
  void kickRight();
  void lookLeft();
  void lookRight();
  void lookJustBelowHorizon();
  
  /** should be obsolete
  2do: CameraMatrix woanders herbekommen!
  */
  void buildCameraMatrix(const SensorData& data, const HeadState& headState);
  unsigned long lastFrame;

  /** The time when the state machine changed */
  unsigned long timeOfLastStateChange;

  /** 3d transformation from world coordinates to camera coordinates using the camera matrix */
  void fromWorld2CameraCoordinates(const Vector3<double> pointIn3D, Vector3<double> &pointInCameraCoordinates);
  
  /** 3d transformation from world coordinates to robot coordinates */
  void fromWorld2RobotCoordinates(const Vector3<double> pointIn3D, Vector3<double> &pointInRobotCoordinates);
  
  /** look at 3d-point on field with offset point in camera image */
  void lookAtPoint(const Vector3<double> &pos,const Vector2<int> &offset,double& tilt,double& pan,double& roll);
  /** look at 3d-point subroutine trying to find tilt1 solution for given tilt2/roll. this is useful on ERS210 and ERS7
  *@return true when matching angles were found */
  bool lookAtPointFixTilt2(const Vector3<double> &aim, const double& xtan, const double& ytan, double& tilt, double& pan, const double& tilt2);
  /** look at 3d-point subroutine trying to find tilt2 solution for given tilt1. this is only useful on ERS7
  *@return true when matching angles were found */
  bool lookAtPointFixTilt1(const Vector3<double> &aim, const double& xtan, const double& ytan, const double& tilt, double& pan, double& tilt2);
  
  /** calculate the angles for looking at the seen ball with a certain camera pixel depending on ball distance */
  void getLookAtBallAngles(double& tilt,double& pan,double& roll);
  
  /** just a fake for compatibility, no smoothing yet (speed is ignored) */
  void setJoints(long tilt, long pan, long roll, long speed=100, long mouth=0);
  
  /* direct HeadControl with speed parameter (smooting) */
  void moveHead(long tilt=0, long pan=0, long roll=0, long speed=400, long mouth=0);
  
  /** Indicates if this head control mode was active the last time execute was called */
  HeadControlMode headControlModeExecutedLastTime;
  
  /** current odometry data of last call to head control */
  OdometryData lastOdometryData;
  /** time of last good self localization */
  unsigned long lastTimeOfGoodSL;
  
  /** This struct can calculate a smooth series of head joint angles from a gives set of way points and an overall duration */
  struct HeadPathPlanner
  {
  public:
  /**
  * Initializes a set of points to visit in a certain time
  * @param vectors set of arcs to visit
  * @param numberOfVectors number of Vector3s in param vectors
  * @param duration time in ms to reach the last position in param vecors
    */
    void init(const Vector3<long>* vectors, int numberOfVectors, unsigned long duration);
    
    /**
    * default constructor
    */
    HeadPathPlanner():lastTilt(0),lastPan(0),lastRoll(0),currentPoint(0),currentFrame(0),numberOfFrames(0),numberOfPoints(0) {}
    
    /**
    * Initializes a set of points to visit in a certain time. These points are
    * assumed to be cyclic. The first and last point to visit is the one closest
    * to the current head position.
    * @param vectors set of arcs to visit
    * @param numberOfVectors number of Vector3s in param vectors
    * @param duration time in ms to reach the last position in param vecors
    */
    void initLoop(const Vector3<long>* vectors, int numberOfVectors, unsigned long duration);
    
    /**
    * Calculates the angles for tilt pan and roll
    * @return true if the last point is reached.
    */
    bool getAngles(long& tilt, long& pan, long& roll);
    
    /**
    * Return whether the last initialized path is already finished
    * @return true if last path is finished.
    */
    inline bool isLastPathFinished() { return (currentFrame>=numberOfFrames); }
    
    long lastTilt;
    long lastPan;
    long lastRoll;
    
  private:
    long currentPoint;
    long currentFrame;
    long numberOfFrames;
    
    enum {maxNumberOfPoints = 20};
    long numberOfPoints;
    Vector3<long> points[maxNumberOfPoints];
    double firstFrame[maxNumberOfPoints];
  };
  
  HeadPathPlanner headPathPlanner;
  
  // needed for transformations and for simple look at point
  CameraMatrix cameraMatrix2;
  double currentTilt, currentPan, currentRoll;

  /** update all information related to the body pose. */
  void updateBodyPose();
};

#endif //__ATH2004ERS7HeadControl_h__

/*
* Change log :
* 
* $Log: ATH2004ERS7HeadControl.h,v $
* Revision 1.4  2004/05/17 19:21:50  loetzsch
* renamed all Variables "cameraMatrix" to "cameraMatrix2"
*
* Revision 1.3  2004/05/10 19:10:16  juengel
* Added states waitUntilStepIsFinished and lookToStarsWhenStepIsFinished
*
* Revision 1.2  2004/05/04 09:05:02  dueffert
* watchOrigin improved; headPathPlanner bug fixed in all copies
*
* Revision 1.1  2004/03/28 14:06:41  jhoffman
* renamed headcontrolmode ATH2004 in ATH2004ERS7
*
* Revision 1.4  2004/03/24 18:39:40  juengel
* Added updateBodyPose()
*
* Revision 1.3  2004/03/23 16:58:17  loetzsch
* improved
*
* Revision 1.2  2004/03/17 19:43:02  juengel
* Added kickLeft and kickRight.
*
* Revision 1.1  2004/03/16 14:00:21  juengel
* Integrated Improvments from "Gnne"
* -ATH2004ERS7Behavior
* -ATHHeadControl
* -KickSelectionTable
* -KickEditor
*
* Revision 1.1  2004/03/15 17:11:39  hoffmann
* - added ATH2004ERS7HeadControl
* - added ATH2004LEDControl
* - headmotiontester shows "tilt2"
* - motion process updates odometry while no new robotPose is received, added to motion request
* - some ui adjustments
* - added member function to "field" to find out if robot is in own penalty area for use in the obstacles locator
*
*
*/
