/**
* @file MotionRatingBehaviorControl.cpp
* 
* Implementation of class MotionRatingBehaviorControl.
*
* @author Dimitri Idessis
* @author Eduardo Tsen
* @author Sebastian Petters
* @author Dirk Thomas
*/

#include "MotionRatingBehaviorControl.h"

MotionRatingBehaviorControl::MotionRatingBehaviorControl(BehaviorControlInterfaces& interfaces)
: BehaviorControl(interfaces)
{
  lastMessage = 0;
  pingTimeout = 2000;
  reset();
  path = 0;
  pathOffset = 0;
  startlocalise = 0;
  state = sleep;

  distTolerance = 50;
  angleTolerance = 0.1;

  //paths[0].start = Pose2D (  pi_2, -1500, -750 );
  //paths[0].end   = Pose2D ( -pi_2,  -1500, 750 );
  paths[0].start = Pose2D (  0, -1500, -500 );
  paths[0].end   = Pose2D ( pi,  1500, -500 );
  paths[1].start = Pose2D (  0, -1500,  500 );
  paths[1].end   = Pose2D ( pi,  1500,  500 );
  paths[2].start = Pose2D (  pi_2,  1200, -1000 );
  paths[2].end   = Pose2D ( -pi_2,  1200,  1000 );
}

MotionRatingBehaviorControl::~MotionRatingBehaviorControl()
{
}

bool MotionRatingBehaviorControl::handleMessage(InMessage& msg)
{
  if ( msg.getMessageID() != idMotionRatingBehaviorMessage ) return false;

  MotionRatingBehaviorMessage message;
  msg.bin >> message;
  switch ( message.type )
  {
    case MotionRatingBehaviorMessage::init:
      path = message.data;
      paramSetId = 0;
      walkCount = 0;
      currentWalk = 0;
      state = prepare;
      setPoints();
      break;
    case MotionRatingBehaviorMessage::request:
      paramSetId = message.data;
      walkCount = 2;
      currentWalk = 0;
      setPoints();
      state = prepare;
      break;
    case MotionRatingBehaviorMessage::abort:
      paramSetId = 0;
      state = wait;
      break;
    default:
      OUTPUT ( idText, text, "unhandled message received" );
  }

  MotionRatingBehaviorMessage outmessage;

  outmessage.type = MotionRatingBehaviorMessage::ping;
  outmessage.state = encodeState();
  outmessage.paramSetId = paramSetId;
  outmessage.data = 0;
  outmessage.currentSpeed = 0.0;
  outmessage.averageSpeed = 0.0;
  outmessage.currentWalk = currentWalk + 1;
  outmessage.walkCount = walkCount;
  outmessage.remainingPower = SystemCall::getRemainingPower();

  OUTPUT ( idMotionRatingBehaviorMessage, bin, outmessage );

  lastMessage = SystemCall::getCurrentSystemTime();

  return true;
}


void MotionRatingBehaviorControl::execute()
{

  //headControlMode.headControlMode = HeadControlMode::searchForLines;
  //headControlMode.headControlMode = HeadControlMode::searchForLandmarks;
  headControlMode.headControlMode = HeadControlMode::optimizerMode;


  
  bool notify = false;

  switch (state)
  {
  case sleep:
    playDead();
    break;
  case prepare:
	  if ( gotoPoint ( startpoint ) )
    {
      notify = true;
      startlocalise = SystemCall::getCurrentSystemTime();
      state = ( paramSetId != 0 ) ? localisepre : wait;
    }
    break;
  case wait:
    stand();
    state = ( paramSetId != 0 ) ? prepare : wait;
    break;
  case localisepre:
    if ( localise() || ( SystemCall::getTimeSince(startlocalise) > 10000 ) )
    {
      notify = true;
      p1 = robotPose.translation;
      state = walk;
      starttime = SystemCall::getCurrentSystemTime();
    }
    break;
  case walk:
    motionRequest.walkType = MotionRequest::debug;
    if ( gotoPoint ( endpoint.translation ) )
    {
      stand();
      //endtime = SystemCall::getCurrentSystemTime();
      state = stop;
    }
    break;
  case stop:
    //endtime = SystemCall::getCurrentSystemTime();
    stand();
    if ( executedMotionRequest.motionType == MotionRequest::stand )
    {
      endtime = SystemCall::getCurrentSystemTime();
      startlocalise = SystemCall::getCurrentSystemTime();
      state = localiseaft;
      motionRequest.walkType = MotionRequest::normal;
    }
    break;
  case localiseaft:
    if ( localise() || ( SystemCall::getTimeSince(startlocalise) > 10000 ) )
    {
      p2 = robotPose.translation;
      state = report;
    }
    break;
  case report:
	  {
      double distance = (p2 - p1).abs();
      unsigned long time = (endtime - starttime);

      totalDistance += distance;
      totalTime += time;

      /*
      char note[50];
      sprintf ( note, "current: t: %.2fs d: %.2fm v: %.2fm/s",
        time/1000.0,
        distance/1000.0,
        100.0*(distance/time)
      );
      OUTPUT(idText,text, note );
      sprintf ( note, "total: t: %.2fs d: %.2fm v: %.2fm/s",
        totalTime/1000.0,
        totalDistance/1000.0,
        100.0*(totalDistance/totalTime)
      );
      OUTPUT(idText,text, note );
      */

      currentWalk++;
      bool finished = walkCount <= currentWalk;

      pathOffset = 1 - pathOffset;
      setPoints();
      state = prepare;
      
      MotionRatingBehaviorMessage message;
      message.remainingPower = SystemCall::getRemainingPower();
      message.type = MotionRatingBehaviorMessage::report;
      message.state = encodeState();
      message.paramSetId = paramSetId;
      message.currentSpeed = distance / time;
      message.averageSpeed = totalDistance / totalTime;
      message.currentWalk = currentWalk + 1;
      message.walkCount = walkCount;
      message.finished = finished;
      message.distance=distance;
      message.time=time;

      OUTPUT ( idMotionRatingBehaviorMessage, bin, message );
      lastMessage = SystemCall::getCurrentSystemTime();

      if ( finished ) reset();
	  }
    break;
  default:
    OUTPUT(idText,text,"unknown state");
    break;
  }

  if ( notify || ( SystemCall::getTimeSince(lastMessage) > pingTimeout ) )
  {
    MotionRatingBehaviorMessage message;
    message.type = MotionRatingBehaviorMessage::ping;
    message.state = encodeState();
    message.paramSetId = paramSetId;
    message.currentWalk = currentWalk + 1;
    message.walkCount = walkCount;
    message.data = 0;
    message.remainingPower = SystemCall::getRemainingPower();

    OUTPUT ( idMotionRatingBehaviorMessage, bin, message );

    lastMessage = SystemCall::getCurrentSystemTime();
  }

}

void MotionRatingBehaviorControl::reset()
{
  paramSetId = 0;
  currentWalk = 0;
  walkCount = 0;
  totalDistance = 0.0;
  totalTime = 0;
  paused = false;
}

void MotionRatingBehaviorControl::setPoints()
{
  int p = path - 1;
  startpoint = ( pathOffset == 0 ) ? paths[p].start : paths[p].end;
  endpoint = ( pathOffset == 0 ) ? paths[p].end : paths[p].start;
}

bool MotionRatingBehaviorControl::gotoPoint ( Pose2D pose )
{
  Vector2<double> v = Pose2D( -robotPose.rotation ) * ( pose.translation - robotPose.translation );

  double dist = v.abs();
  double theta = normalize ( v.angle() );
  double phi = normalize ( pose.getAngle() - robotPose.getAngle() );

  double rotationDist = 600;
  double t = 1.0;
  if ( dist < rotationDist && dist > distTolerance ) t = ( dist - distTolerance ) / ( rotationDist - distTolerance );
  if ( dist <= distTolerance ) t = 0.0;
  double alpha = t * theta + (1 - t) * phi;

  setWalkMotionRequest ( dist, theta, alpha );

  return dist < distTolerance && fabs ( phi ) < angleTolerance;
}

bool MotionRatingBehaviorControl::gotoPoint ( Vector2<double> point )
{
  Vector2<double> v = Pose2D( -robotPose.rotation ) * ( point - robotPose.translation );

  double dist = v.abs();
  double theta = normalize ( v.angle() );

  setWalkMotionRequest ( walkMaxForwardSpeed, theta, theta );

  return dist < distTolerance;
}

void MotionRatingBehaviorControl::setWalkMotionRequest ( double distance, double theta, double phi )
{
  double speed = distance > walkMaxForwardSpeed ? walkMaxForwardSpeed : distance;

  if ( phi >  walkMaxRotationSpeed ) phi =  walkMaxRotationSpeed;
  if ( phi < -walkMaxRotationSpeed ) phi = -walkMaxRotationSpeed;

  motionRequest.motionType = MotionRequest::walk;
  motionRequest.walkParams.rotation = phi;
  motionRequest.walkParams.translation.x = cos ( theta ) * speed;
  motionRequest.walkParams.translation.y = sin ( theta ) * speed;
}

void MotionRatingBehaviorControl::stand()
{
  motionRequest.motionType = MotionRequest::stand;
}

void MotionRatingBehaviorControl::playDead()
{
  motionRequest.motionType = MotionRequest::specialAction;
  motionRequest.specialActionType = MotionRequest::playDead;
}


bool MotionRatingBehaviorControl::localise()
{
  /*
  char message[50];
  sprintf ( message, "pos: x:%.2f, y:%.2f phi:%.2f s:%.4f",
    robotPose.translation.x,
    robotPose.translation.y,
    robotPose.rotation,
    robotPose.getValidity()
  );
  OUTPUT(idText,text, message );
  */
  //headControlMode.headControlMode = HeadControlMode::optimizerMode;
  stand();
  return robotPose.getValidity() > 0.90;
}

MotionRatingBehaviorMessage::States MotionRatingBehaviorControl::encodeState()
{
  switch (state)
  {
    case sleep:
      return MotionRatingBehaviorMessage::sleeping;
    case wait:
      return MotionRatingBehaviorMessage::waiting;
    case prepare:
      return ( paramSetId != 0 ) ? MotionRatingBehaviorMessage::testing : MotionRatingBehaviorMessage::preparing;
    case localisepre:
    case walk:
    case stop:
    case localiseaft:
    case report:
      return MotionRatingBehaviorMessage::testing;
    default:
      return MotionRatingBehaviorMessage::unknown;
  }
}







/*
* Change log :
* 
* $Log: MotionRatingBehaviorControl.cpp,v $
* Revision 1.1  2004/03/17 16:18:49  thomas
* added preversion of motion optimisation with behaviour, selflocator, headcontrol and robotcontrol dialog
*
*
*/
