/**
* @file MotionRatingBehaviorControl.cpp
* 
* Implementation of class MotionRatingBehaviorControl.
*
* @author Dimitri Idessis
* @author Eduardo Tsen
* @author Sebastian Petters
* @author Dirk Thomas
*/

#include "MotionRating2BehaviorControl.h"

MotionRating2BehaviorControl::MotionRating2BehaviorControl(BehaviorControlInterfaces& interfaces)
: BehaviorControl(interfaces)
{
  lastMessage = 0;
  pingTimeout = 2000;
  reset();
  path = 0;
  pathOffset = 0;
  startlocalise = 0;
  startrotate=0;
  state = sleep;

  distTolerance = 750.0;
  angleTolerance = 0.1;

  paths[0].start = Pose2D (  0, 0, 0 );
  paths[0].end   = Pose2D ( -pi,  0, 0 );
  //paths[0].start = Pose2D (  pi_2, -1500, -750 );
  //paths[0].end   = Pose2D ( -pi_2,  -1500, 750 );
  //paths[0].start = Pose2D (  0, -1500, -500 );
  //paths[0].end   = Pose2D ( pi,  1500, -500 );
  paths[1].start = Pose2D (  0, -1500,  500 );
  paths[1].end   = Pose2D ( pi,  1500,  500 );
  paths[2].start = Pose2D (  pi_2,  1200, -1000 );
  paths[2].end   = Pose2D ( -pi_2,  1200,  1000 );
}

MotionRating2BehaviorControl::~MotionRating2BehaviorControl()
{
}

bool MotionRating2BehaviorControl::handleMessage(InMessage& msg)
{
  if ( msg.getMessageID() != idMotionRatingBehaviorMessage ) return false;

  MotionRatingBehaviorMessage message;
  msg.bin >> message;
  switch ( message.type )
  {
    case MotionRatingBehaviorMessage::init:
      path = message.data;
      paramSetId = 0;
      walkCount = 0;
      currentWalk = 0;
      state = prepare;
      setPoints();
      break;
    case MotionRatingBehaviorMessage::request:
      paramSetId = message.data;
      walkCount = 10;
      currentWalk = 0;
      setPoints();
      state = prepare;
      break;
    case MotionRatingBehaviorMessage::abort:
      paramSetId = 0;
      state = wait;
      break;
    default:
      OUTPUT ( idText, text, "unhandled message received" );
  }

  MotionRatingBehaviorMessage outmessage;

  outmessage.type = MotionRatingBehaviorMessage::ping;
  outmessage.state = encodeState();
  outmessage.paramSetId = paramSetId;
  outmessage.data = 0;
  outmessage.currentSpeed = 0.0;
  outmessage.averageSpeed = 0.0;
  outmessage.currentWalk = currentWalk + 1;
  outmessage.walkCount = walkCount;
  outmessage.remainingPower = SystemCall::getRemainingPower();

  OUTPUT ( idMotionRatingBehaviorMessage, bin, outmessage );

  lastMessage = SystemCall::getCurrentSystemTime();

  return true;
}


void MotionRating2BehaviorControl::execute()
{
  /////////////////////////////
    char message[50];
    sprintf ( message,"(%.2f,%.2f) -> %.2f\r\n\r\n",robotPose.translation.x,robotPose.translation.y,robotPose.rotation);
    OUTPUT(idText,text, message );
/////////////////////////////

  //headControlMode.headControlMode = HeadControlMode::searchForLines;
  //headControlMode.headControlMode = HeadControlMode::searchForLandmarks;
  //headControlMode.headControlMode = HeadControlMode::optimizerMode;
  headControlMode.headControlMode = HeadControlMode::lookParallelToGround;
  //headControlMode.headControlMode = HeadControlMode::lookJustBelowHorizon;
    //headControlMode.headControlMode = HeadControlMode::lookAtPoint;
    //headControlMode.point.x=(int)robotPose.translation.x;
    //headControlMode.point.y=(int)robotPose.translation.y;
    //headControlMode.point.z=165;


  
  bool notify = false;

  switch (state)
  {
  case sleep:
    playDead();
    break;
  case prepare:
	  //if ( gotoPoint ( startpoint ) )
    //if ( goTo(startpoint.translation))
    if ( rotateTo(startpoint.rotation))
    {
      notify = true;
      startlocalise = SystemCall::getCurrentSystemTime();
      state = ( paramSetId != 0 ) ? localisepre : wait;
    }
    break;
  case wait:
    stand();
    state = ( paramSetId != 0 ) ? prepare : wait;
    break;
  case localisepre:
    if ( localise() || ( SystemCall::getTimeSince(startlocalise) > 10000 ) )
    {
      notify = true;
      p1 = robotPose.translation;
      state = walk;
      starttime = SystemCall::getCurrentSystemTime();
    }
    break;
  case walk:
    //if ( gotoPoint ( endpoint.translation ) )
    if (goTo(startpoint/*.translation*/))
    {
      stand();
      //endtime = SystemCall::getCurrentSystemTime();
      state = stop;
    }
    break;
  case stop:
    //endtime = SystemCall::getCurrentSystemTime();
    stand();
    if ( executedMotionRequest.motionType == MotionRequest::stand )
    {
      endtime = SystemCall::getCurrentSystemTime();
      startlocalise = SystemCall::getCurrentSystemTime();
      state = localiseaft;
    }
    break;
  case localiseaft:
    if ( localise() || ( SystemCall::getTimeSince(startlocalise) > 10000 ) )
    {
      p2 = robotPose.translation;
      state = report;
    }
    break;
  case report:
	  {
      double distance = (p2 - p1).abs();
      unsigned long time = (endtime - starttime);

      totalDistance += distance;
      totalTime += time;

      /*
      char message[50];
      sprintf ( message, "current: t: %.2fs d: %.2fm v: %.2fm/s",
        time/1000.0,
        distance/1000.0,
        100.0*(distance/time)
      );
      OUTPUT(idText,text, message );
      sprintf ( message, "total: t: %.2fs d: %.2fm v: %.2fm/s",
        totalTime/1000.0,
        totalDistance/1000.0,
        100.0*(totalDistance/totalTime)
      );
      OUTPUT(idText,text, message );
      */

      currentWalk++;
      bool finished = walkCount <= currentWalk;

      pathOffset = 1 - pathOffset;
      setPoints();
      state = prepare;
      
      MotionRatingBehaviorMessage message;
      message.remainingPower = SystemCall::getRemainingPower();
      message.type = MotionRatingBehaviorMessage::report;
      message.state = encodeState();
      message.paramSetId = paramSetId;
      message.currentSpeed = distance / time;
      message.averageSpeed = totalDistance / totalTime;
      message.currentWalk = currentWalk + 1;
      message.walkCount = walkCount;
      message.finished = finished;
      message.distance=distance;
      message.time=time;

      OUTPUT ( idMotionRatingBehaviorMessage, bin, message );
      lastMessage = SystemCall::getCurrentSystemTime();

      if ( finished ) reset();
	  }
    break;
  default:
    OUTPUT(idText,text,"unknown state");
    break;
  }

  if ( notify || ( SystemCall::getTimeSince(lastMessage) > pingTimeout ) )
  {
    MotionRatingBehaviorMessage message;
    message.type = MotionRatingBehaviorMessage::ping;
    message.state = encodeState();
    message.paramSetId = paramSetId;
    message.currentWalk = currentWalk + 1;
    message.walkCount = walkCount;
    message.data = 0;
    message.remainingPower = SystemCall::getRemainingPower();

    OUTPUT ( idMotionRatingBehaviorMessage, bin, message );

    lastMessage = SystemCall::getCurrentSystemTime();
  }

}

void MotionRating2BehaviorControl::reset()
{
  paramSetId = 0;
  currentWalk = 0;
  walkCount = 0;
  totalDistance = 0.0;
  totalTime = 0;
  paused = false;
}

void MotionRating2BehaviorControl::setPoints()
{
  int p = path - 1;
  startpoint = ( pathOffset == 0 ) ? paths[p].start : paths[p].end;
  endpoint = ( pathOffset == 0 ) ? paths[p].end : paths[p].start;
}

bool MotionRating2BehaviorControl::gotoPoint ( Pose2D pose )
{
  char message[50];


  Vector2<double> v = Pose2D( -robotPose.rotation ) * ( pose.translation - robotPose.translation );
    
  sprintf ( message, "Vektor v: (%.2f,%.2f)",v.x,v.y);
  OUTPUT(idText,text, message );

  double dist = v.abs();
  double theta = normalize ( v.angle() );
  double phi = normalize ( pose.getAngle() - robotPose.getAngle() );

  sprintf ( message, "dist: %.2f",dist);
  OUTPUT(idText,text, message );
  sprintf ( message, "theta: %.2f",theta);
  OUTPUT(idText,text, message );
  sprintf ( message, "phi: %.2f",phi);
  OUTPUT(idText,text, message );



  double rotationDist = 600;
  double t = 1.0;
  if ( dist < rotationDist && dist > distTolerance ) t = ( dist - distTolerance ) / ( rotationDist - distTolerance );
  if ( dist <= distTolerance ) t = 0.0;
  double alpha = t * theta + (1 - t) * phi;

  sprintf ( message, "alpha: %.2f\r\n",alpha);
  OUTPUT(idText,text, message );

  setWalkMotionRequest ( dist, theta, alpha );

  return dist < distTolerance && fabs ( phi ) < angleTolerance;
}

bool MotionRating2BehaviorControl::gotoPoint ( Vector2<double> point )
{
  Vector2<double> v = Pose2D( -robotPose.rotation ) * ( point - robotPose.translation );

  double dist = v.abs();
  double theta = normalize ( v.angle() );

  setWalkMotionRequest ( walkMaxForwardSpeed, theta, theta );

  return dist < distTolerance;
}

void MotionRating2BehaviorControl::setWalkMotionRequest ( double distance, double theta, double phi )
{
  double speed = distance > walkMaxForwardSpeed ? walkMaxForwardSpeed : distance;

  if ( phi >  walkMaxRotationSpeed ) phi =  walkMaxRotationSpeed;
  if ( phi < -walkMaxRotationSpeed ) phi = -walkMaxRotationSpeed;

  motionRequest.motionType = MotionRequest::walk;
  motionRequest.walkType = MotionRequest::normal;
  motionRequest.walkParams.rotation = phi;
  motionRequest.walkParams.translation.x = cos ( theta ) * speed;
  motionRequest.walkParams.translation.y = sin ( theta ) * speed;
}

bool MotionRating2BehaviorControl::stand()
{
  motionRequest.motionType = MotionRequest::stand;
  return (executedMotionRequest.motionType==MotionRequest::stand);
}

void MotionRating2BehaviorControl::playDead()
{
  motionRequest.motionType = MotionRequest::specialAction;
  motionRequest.specialActionType = MotionRequest::playDead;
}


bool MotionRating2BehaviorControl::localise()
{
  /*
  char message[50];
  sprintf ( message, "pos: x:%.2f, y:%.2f phi:%.2f s:%.4f",
    robotPose.translation.x,
    robotPose.translation.y,
    robotPose.rotation,
    robotPose.getValidity()
  );
  OUTPUT(idText,text, message );
  */
  //headControlMode.headControlMode = HeadControlMode::optimizerMode;
  if (!stand()) return false;
  return robotPose.getValidity() > 0.90;
}

MotionRatingBehaviorMessage::States MotionRating2BehaviorControl::encodeState()
{
  switch (state)
  {
    case sleep:
      return MotionRatingBehaviorMessage::sleeping;
    case wait:
      return MotionRatingBehaviorMessage::waiting;
    case prepare:
      return ( paramSetId != 0 ) ? MotionRatingBehaviorMessage::testing : MotionRatingBehaviorMessage::preparing;
    case localisepre:
    case walk:
    case stop:
    case localiseaft:
    case report:
      return MotionRatingBehaviorMessage::testing;
    default:
      return MotionRatingBehaviorMessage::unknown;
  }
}


bool MotionRating2BehaviorControl::goTo ( Pose2D target /*Vector2<double> point */)
{
  if ((target.translation-robotPose.translation).abs()<distTolerance)
  {
   return true;
  }
  else
  {
        double theta=robotPose.rotation-target.rotation;
        if (fabs(theta)<angleTolerance)
        {
          theta=0.0;
        }
        motionRequest.motionType = MotionRequest::walk;
        motionRequest.walkType = MotionRequest::normal;
        motionRequest.walkParams = Pose2D(theta,walkMaxForwardSpeed,0);
    return false;
  }
}

bool MotionRating2BehaviorControl::rotateTo ( double angle )
{
      if (startrotate==0) startrotate=SystemCall::getCurrentSystemTime();
      double theta=robotPose.rotation-angle;
      if (fabs(theta)<angleTolerance 
        || ( SystemCall::getTimeSince(startlocalise) > 10000 ))
      {
        startrotate=0;
        return true;
      }
      else
      {
        motionRequest.motionType = MotionRequest::walk;
        motionRequest.walkType = MotionRequest::normal;
        //if (fabs(theta)>pi_2) theta=(theta<0.0)?-pi_2:pi_2;
        motionRequest.walkParams = Pose2D(theta,0,0);
        return false;
      }

}



/*
* Change log :
* 
* $Log: MotionRating2BehaviorControl.cpp,v $
* Revision 1.1  2004/03/17 16:18:49  thomas
* added preversion of motion optimisation with behaviour, selflocator, headcontrol and robotcontrol dialog
*
*
*/
