/** 
* @file MSH2004StrategySymbols.cpp
*
* Implementation of class MSH2004StrategySymbols.
*
* @author Martin Ltzsch
*/

#include "MSH2004StrategySymbols.h"
#include "Tools/FieldDimensions.h"
#include "Tools/StringFunctions.h"
#include "Tools/Math/Geometry.h"
#include "Tools/Location.h"
#include "Tools/Streams/InStreams.h"
 
//----------------------------------------------------------------------------
MSH2004StrategySymbols::MSH2004StrategySymbols(const BehaviorControlInterfaces& interfaces)
: BehaviorControlInterfaces(interfaces),
timeWhenBallWasStartedToCatch(0), timeUntilBallWasCaught(0)
{
  role = BehaviorTeamMessage::goalie;
  teamcolor = getPlayer().getTeamColor();
  estimatedTimeToReachBall = 0.0;

	// Loading the Sector Map
  InBinaryFile stream(getLocation().getFilename("sectmap.dat"));
  if (stream.exists())
  {
    unsigned char tmp;
    // i messed up something with the map, so just fixing it while reading
    for (int y = 0; y < 27; ++y)
      for (int x = 41; x >= 0; --x) {
        stream >> tmp;
        sectormap[x][y] = tmp;
      }
  }
  else
  {
		OUTPUT(idText,text,"StrategySymbols : couldn't load SectorMap");
  }

  sectNr = sector1;

  successfulCarries = 1;
  failedCarries     = 1;

  carryIsRunning = false;
  carryDisabled  = false;

  earsStripped = false;
  isPenaltyShooter = false;
}
//----------------------------------------------------------------------------
void MSH2004StrategySymbols::registerSymbols(Xabsl2Engine& engine)
{
  // "robot-number"
  engine.registerDecimalInputSymbol("robot-number", this,
    (double(Xabsl2FunctionProvider::*)())&MSH2004StrategySymbols::getRobotNumber);
  
  // "role"
  engine.registerEnumeratedInputSymbol("role", (int *)&role);
  engine.registerEnumeratedInputSymbolEnumElement("role", "goalie", BehaviorTeamMessage::goalie);
  engine.registerEnumeratedInputSymbolEnumElement("role", "striker", BehaviorTeamMessage::striker);
  engine.registerEnumeratedInputSymbolEnumElement("role", "defensive-supporter", BehaviorTeamMessage::defensiveSupporter);
  engine.registerEnumeratedInputSymbolEnumElement("role", "offensive-supporter", BehaviorTeamMessage::offensiveSupporter);
  
  // "teamcolor"
  engine.registerEnumeratedInputSymbol("teamcolor", (int *)&teamcolor);
  engine.registerEnumeratedInputSymbolEnumElement("teamcolor", "teamcolor.red"               , Player::red);
  engine.registerEnumeratedInputSymbolEnumElement("teamcolor", "teamcolor.blue"              , Player::blue);
  engine.registerEnumeratedInputSymbolEnumElement("teamcolor", "teamcolor.undefinedTeamColor", Player::undefinedTeamColor);
  
  // "teamcolor"
  engine.registerEnumeratedInputSymbol("sector-nr", (int *)&sectNr);
  engine.registerEnumeratedInputSymbolEnumElement("sector-nr", "sector-nr.1"  , sector1);
  engine.registerEnumeratedInputSymbolEnumElement("sector-nr", "sector-nr.2"  , sector2);
  engine.registerEnumeratedInputSymbolEnumElement("sector-nr", "sector-nr.3"  , sector3);
  engine.registerEnumeratedInputSymbolEnumElement("sector-nr", "sector-nr.4"  , sector4);
  engine.registerEnumeratedInputSymbolEnumElement("sector-nr", "sector-nr.5"  , sector5);
  engine.registerEnumeratedInputSymbolEnumElement("sector-nr", "sector-nr.6"  , sector6);
  engine.registerEnumeratedInputSymbolEnumElement("sector-nr", "sector-nr.7"  , sector7);
  engine.registerEnumeratedInputSymbolEnumElement("sector-nr", "sector-nr.8"  , sector8);
  engine.registerEnumeratedInputSymbolEnumElement("sector-nr", "sector-nr.9"  , sector9);
  engine.registerEnumeratedInputSymbolEnumElement("sector-nr", "sector-nr.10" , sector10);
  engine.registerEnumeratedInputSymbolEnumElement("sector-nr", "sector-nr.11" , sector11);
  engine.registerEnumeratedInputSymbolEnumElement("sector-nr", "sector-nr.12" , sector12);
  engine.registerEnumeratedInputSymbolEnumElement("sector-nr", "sector-nr.13" , sector13);
  engine.registerEnumeratedInputSymbolEnumElement("sector-nr", "sector-nr.14" , sector14);
  engine.registerEnumeratedInputSymbolEnumElement("sector-nr", "sector-nr.15" , sector15);
  engine.registerEnumeratedInputSymbolEnumElement("sector-nr", "sector-nr.16" , sector16);
  engine.registerEnumeratedInputSymbolEnumElement("sector-nr", "sector-nr.17" , sector17);
  engine.registerEnumeratedInputSymbolEnumElement("sector-nr", "sector-nr.18" , sector18);
  engine.registerEnumeratedInputSymbolEnumElement("sector-nr", "sector-nr.19" , sector19);
  
  // "estimated-time-to-reach-ball"
  engine.registerDecimalInputSymbol("estimated-time-to-reach-ball", (double *)&estimatedTimeToReachBall);
  
  // "sent-game-state"
  engine.registerEnumeratedOutputSymbol("sent-game-state",(int*)&outgoingBehaviorTeamMessage.gameState);
  engine.registerEnumeratedOutputSymbolEnumElement("sent-game-state","sent-game-state.sleep",BehaviorTeamMessage::initial);
  engine.registerEnumeratedOutputSymbolEnumElement("sent-game-state","sent-game-state.initial",BehaviorTeamMessage::initial);
  engine.registerEnumeratedOutputSymbolEnumElement("sent-game-state","sent-game-state.ready",BehaviorTeamMessage::ready);
  engine.registerEnumeratedOutputSymbolEnumElement("sent-game-state","sent-game-state.playing",BehaviorTeamMessage::playing);
  engine.registerEnumeratedOutputSymbolEnumElement("sent-game-state","sent-game-state.penalized",BehaviorTeamMessage::penalized);
  engine.registerEnumeratedOutputSymbolEnumElement("sent-game-state","sent-game-state.finished",BehaviorTeamMessage::finished);
  engine.registerEnumeratedOutputSymbolEnumElement("sent-game-state","sent-game-state.set",BehaviorTeamMessage::set);
  
  // "another-player-is-in-ready-state"
  engine.registerBooleanInputSymbol("another-player-is-in-ready-state",
    this,(bool(Xabsl2FunctionProvider::*)())&MSH2004StrategySymbols::getAnotherPlayerIsInReadyState);
  
  // "team-message"
  engine.registerEnumeratedOutputSymbol("team-message",(int*)&outgoingBehaviorTeamMessage.message);
  engine.registerEnumeratedOutputSymbolEnumElement("team-message","team-message.none", BehaviorTeamMessage::none);
  engine.registerEnumeratedOutputSymbolEnumElement("team-message","team-message.just-performed-a-kick",BehaviorTeamMessage::justPerformedAKick);
  engine.registerEnumeratedOutputSymbolEnumElement("team-message","team-message.performing-a-kick",BehaviorTeamMessage::performingAKick);
  engine.registerEnumeratedOutputSymbolEnumElement("team-message","team-message.preparing-a-kick",BehaviorTeamMessage::preparingAKick);
  
  // "another-teammate-is-preparing-a-kick"
  engine.registerBooleanInputSymbol("another-teammate-is-preparing-a-kick",
    this,(bool(Xabsl2FunctionProvider::*)())&MSH2004StrategySymbols::getAnotherTeammateIsPreparingAKick);

  // "another-teammate-is-performing-a-kick"
  engine.registerBooleanInputSymbol("another-teammate-is-performing-a-kick",
    this,(bool(Xabsl2FunctionProvider::*)())&MSH2004StrategySymbols::getAnotherTeammateIsPerformingAKick);
  
  // "another-teammate-just-performed-a-kick"
  engine.registerBooleanInputSymbol("another-teammate-just-performed-a-kick",
    this,(bool(Xabsl2FunctionProvider::*)())&MSH2004StrategySymbols::getAnotherTeammateJustPerformedAKick);

  // "the-striker-is-playing-near-the-opponent-goal"
  engine.registerBooleanInputSymbol("the-striker-is-playing-near-the-opponent-goal",
    this,(bool(Xabsl2FunctionProvider::*)())&MSH2004StrategySymbols::getTheStrikerIsPlayingNearTheOpponentGoal);

  // "the-striker-is-not-playing-near-the-opponent-goal"
  engine.registerBooleanInputSymbol("the-striker-is-not-playing-near-the-opponent-goal",
    this,(bool(Xabsl2FunctionProvider::*)())&MSH2004StrategySymbols::getTheStrikerIsNotPlayingNearTheOpponentGoal);

  // "the-striker-is-playing-near-the-own-goal"
  engine.registerBooleanInputSymbol("the-striker-is-playing-near-the-own-goal",
    this,(bool(Xabsl2FunctionProvider::*)())&MSH2004StrategySymbols::getTheStrikerIsPlayingNearTheOwnGoal);

  // "the-striker-is-not-playing-near-the-own-goal"
  engine.registerBooleanInputSymbol("the-striker-is-not-playing-near-the-own-goal",
    this,(bool(Xabsl2FunctionProvider::*)())&MSH2004StrategySymbols::getTheStrikerIsNotPlayingNearTheOwnGoal);

  // "catch-ball"
  engine.registerBooleanInputSymbol("catch-ball", this,
    (bool (Xabsl2FunctionProvider::*)())&MSH2004StrategySymbols::getCaught);
  
  // "catch-ball-time"
  engine.registerDecimalInputSymbol("catch-ball-time",this,
    (double (Xabsl2FunctionProvider::*)())&MSH2004StrategySymbols::getCatchTime);

  // "robot-is-stuck"
  engine.registerBooleanInputSymbol("robot-is-stuck", &robotIsStuck);

  // "obstacles-are-close"
  engine.registerBooleanInputSymbol("obstacles-are-close", &obstaclesAreClose);

  engine.registerDecimalInputSymbol("search-ball.x", this,
    (double (Xabsl2FunctionProvider::*)())&MSH2004StrategySymbols::getSearchBallX);

  engine.registerDecimalInputSymbol("fieldpos.angle-to-teammate1",this,
    (double (Xabsl2FunctionProvider::*)())&MSH2004StrategySymbols::getAngleToTeammate1);
  engine.registerDecimalInputSymbol("fieldpos.angle-to-teammate2",this,
    (double (Xabsl2FunctionProvider::*)())&MSH2004StrategySymbols::getAngleToTeammate2);
  engine.registerDecimalInputSymbol("fieldpos.angle-to-teammate3",this,
    (double (Xabsl2FunctionProvider::*)())&MSH2004StrategySymbols::getAngleToTeammate3);
  engine.registerDecimalInputSymbol("fieldpos.distance-to-teammate1",this,
    (double (Xabsl2FunctionProvider::*)())&MSH2004StrategySymbols::getDistanceToTeammate1);
  engine.registerDecimalInputSymbol("fieldpos.distance-to-teammate2",this,
    (double (Xabsl2FunctionProvider::*)())&MSH2004StrategySymbols::getDistanceToTeammate2);
  engine.registerDecimalInputSymbol("fieldpos.distance-to-teammate3",this,
    (double (Xabsl2FunctionProvider::*)())&MSH2004StrategySymbols::getDistanceToTeammate3);


  engine.registerDecimalInputSymbol("block-position.x",this,
    (double (Xabsl2FunctionProvider::*)())&MSH2004StrategySymbols::getBlockPositionBetweenBallAndGoalX);
  engine.registerDecimalInputSymbol("block-position.y",this,
    (double (Xabsl2FunctionProvider::*)())&MSH2004StrategySymbols::getBlockPositionBetweenBallAndGoalY);
  engine.registerDecimalInputSymbol("angle-to-opponent-goal",this,
    (double (Xabsl2FunctionProvider::*)())&MSH2004StrategySymbols::getAngleToOpponentGoal);
  engine.registerDecimalInputSymbol("angle-to-own-goal",this,
    (double (Xabsl2FunctionProvider::*)())&MSH2004StrategySymbols::getAngleToOwnGoal);
  
  
  
  // "carry.result"
  engine.registerEnumeratedOutputSymbol("carry.result", this, (void (Xabsl2FunctionProvider::*)(int))&MSH2004StrategySymbols::setCarryResult );
  engine.registerEnumeratedOutputSymbolEnumElement("carry.result","carry.result.success", 1);
  engine.registerEnumeratedOutputSymbolEnumElement("carry.result","carry.result.failure", 0);

  // "carry.status"
  engine.registerEnumeratedOutputSymbol("carry.status", this, (void (Xabsl2FunctionProvider::*)(int))&MSH2004StrategySymbols::setCarryStatus );
  engine.registerEnumeratedOutputSymbolEnumElement("carry.status","carry.status.running",  1);
  engine.registerEnumeratedOutputSymbolEnumElement("carry.status","carry.status.stopped",  0);
  engine.registerEnumeratedOutputSymbolEnumElement("carry.status","carry.status.punished", 2);

  // "carry-disabled"
  engine.registerBooleanInputSymbol("carry-disabled", &carryDisabled);

  // "carry-running"
  engine.registerBooleanInputSymbol("carry-running", &carryIsRunning);

  // "ears.status"
  engine.registerEnumeratedOutputSymbol("ears.status", this, (void (Xabsl2FunctionProvider::*)(int))&MSH2004StrategySymbols::setEarsStatus );
  engine.registerEnumeratedOutputSymbolEnumElement("ears.status","ears.status.stripped", 0);
  engine.registerEnumeratedOutputSymbolEnumElement("ears.status","ears.status.mounted" , 1);

   // "setpenaltyshooter"
  engine.registerEnumeratedOutputSymbol("set-penalty-shooter", this, (void (Xabsl2FunctionProvider::*)(int))&MSH2004StrategySymbols::setPenaltyShooter);
  engine.registerEnumeratedOutputSymbolEnumElement("set-penalty-shooter","set-penalty-shooter.false", 0);
  engine.registerEnumeratedOutputSymbolEnumElement("set-penalty-shooter","set-penalty-shooter.true" , 1);

 engine.registerBooleanInputSymbol("is-penalty-shooter", &isPenaltyShooter);

  // "ears-already-stripped"
  engine.registerBooleanInputSymbol("ears-already-stripped", &earsStripped);

}
//----------------------------------------------------------------------------
void MSH2004StrategySymbols::update()
{
  estimateTimeToReachBall();
  computeRole();
  teamcolor = getPlayer().getTeamColor();

  // get the sector of the ballPosition
  int x = (int)(ballPosition.seen.x + 2100);
  int y = (int)(ballPosition.seen.y + 1350);
  if (x >= 4200) x = 4100;
  if (x < 0)     x = 0;
  if (y >= 2700) y = 2600;
  if (y < 0)     y = 0;
  sectNr = SectorNr(sectormap[x/100][y/100]);

  // if the ballPosition is in our own goal, then apply "special" sector 13
  if ((ballPosition.seen.x < -2100) &&
      (ballPosition.seen.y > -1050) &&
      (ballPosition.seen.y <  1050)    )
    sectNr = sector13;

  
  outgoingBehaviorTeamMessage.estimatedTimeToReachBall = estimatedTimeToReachBall;
  outgoingBehaviorTeamMessage.dynamicRole = role;

  if (((executedMotionRequest.motionType == MotionRequest::specialAction
    && 
    (executedMotionRequest.specialActionType == MotionRequest::catchBall2
    || executedMotionRequest.specialActionType == MotionRequest::lookForLandmarksWithBallCaught)
    )
    || 
    (executedMotionRequest.motionType == MotionRequest::walk 
    && executedMotionRequest.walkType == MotionRequest::turnWithBall)
    )
    || headControlMode.headControlMode == HeadControlMode::catchBall)
  {
    if (SystemCall::getTimeSince(timeUntilBallWasCaught) > 1000)
    {
      timeWhenBallWasStartedToCatch = SystemCall::getCurrentSystemTime();
    }

    timeUntilBallWasCaught = SystemCall::getCurrentSystemTime();
  }

  // robot is stuck
  if(obstaclesModel.getPercentageOfLowDistanceObstaclesInRange(0, pi_2, 300) > 0.2)
  {
    robotIsStuck = true;
    obstaclesAreClose = true;
  }
  else
  {
    robotIsStuck = false;
    if(obstaclesModel.getPercentageOfLowDistanceObstaclesInRange(0, pi_2, 500) > 0.15)
      obstaclesAreClose = true;
    else
      obstaclesAreClose = false;
  }

  if (failedCarries > (successfulCarries*2))
    carryDisabled = true;

}
//----------------------------------------------------------------------------
void MSH2004StrategySymbols::computeRole()
{
  int i;
  if (getPlayer().getPlayerNumber() == Player::one)
  {
    // no role changes for the goalie
    role = BehaviorTeamMessage::goalie;
    return;
  }
  
  // fall-back if no wlan
  if (SystemCall::getTimeSince(teamMessageCollection[0].lastReceivedTimeStamp) > 5000 
   && SystemCall::getTimeSince(teamMessageCollection[1].lastReceivedTimeStamp) > 5000
   && SystemCall::getTimeSince(teamMessageCollection[2].lastReceivedTimeStamp) > 5000)
  {
    switch(getPlayer().getPlayerNumber())
    {
      case Player::two:
        if (SystemCall::getTimeSince (ballPosition.seen.timeWhenLastSeen) < 5000 &&
          ballPosition.seen.x < 75)
          role = BehaviorTeamMessage::striker;
        else
          role = BehaviorTeamMessage::defensiveSupporter;
        break;
      case Player::three:
        if (SystemCall::getTimeSince (ballPosition.seen.timeWhenLastSeen) < 5000 &&
          ballPosition.seen.x > 500)
          role = BehaviorTeamMessage::striker;
        else
          role = BehaviorTeamMessage::offensiveSupporter;
        break;
      case Player::four:
      default:
        if (SystemCall::getTimeSince (ballPosition.seen.timeWhenLastSeen) < 5000 &&
          ballPosition.seen.x > -75)
          role = BehaviorTeamMessage::striker;
        else
          role = BehaviorTeamMessage::offensiveSupporter;
        break;
    }
    return;
  }
  
  // with wlan
  
  // estimate the closest other teammate to the ball
  double minTeammateTime = 100000.0;
  for (i = 0; i < teamMessageCollection.numberOfTeamMessages; i++ ) 
  {
    if ( teamMessageCollection[i].isActual() )
    {
      if ( teamMessageCollection[i].behaviorTeamMessage.dynamicRole != BehaviorTeamMessage::goalie 
        && teamMessageCollection[i].behaviorTeamMessage.gameState == BehaviorTeamMessage::playing) 
      {
        double time = 
          teamMessageCollection[i].behaviorTeamMessage.estimatedTimeToReachBall;
        
        // bonus for current striker
        if ( teamMessageCollection[i].behaviorTeamMessage.dynamicRole == BehaviorTeamMessage::striker ) 
          time -= 500.0;
        
        if (time < minTeammateTime ) 
        {
          minTeammateTime = time;
        }
      }
    }
  }
  
  // bonus for current striker
  if (role == BehaviorTeamMessage::striker)
    minTeammateTime += 1000.0;
  
  // assign striker-role if nearest to the ball
  if ( estimatedTimeToReachBall < minTeammateTime )
  {
    role = BehaviorTeamMessage::striker;
    return;
  }
  
  // assign supporting roles
  double maxTeammateX = (double)xPosOwnGroundline;
  for (i = 0; i < teamMessageCollection.numberOfTeamMessages; i++ ) 
  {
    if ( teamMessageCollection[i].isActual() )
    {
      if ( teamMessageCollection[i].behaviorTeamMessage.dynamicRole != BehaviorTeamMessage::striker 
        && teamMessageCollection[i].behaviorTeamMessage.dynamicRole != BehaviorTeamMessage::goalie 
        && teamMessageCollection[i].behaviorTeamMessage.gameState == BehaviorTeamMessage::playing)
      {
        double teammateX = 
          teamMessageCollection[i].robotPose.translation.x;
        
        // bonus for current offensive supporter
        if ( teamMessageCollection[i].behaviorTeamMessage.dynamicRole == BehaviorTeamMessage::offensiveSupporter ) 
          teammateX += 300.0;
        
        if (teammateX > maxTeammateX ) 
        {
          maxTeammateX = teammateX;
        }
      }
    }
  }
  
  // bonus for current offensive supporter
  if ( role == BehaviorTeamMessage::offensiveSupporter ) 
    maxTeammateX -= 300.0;
  
  if ( robotPose.translation.x >= maxTeammateX ) 
    role = BehaviorTeamMessage::offensiveSupporter;
  else 
    role = BehaviorTeamMessage::defensiveSupporter;
}



//----------------------------------------------------------------------------
void MSH2004StrategySymbols::estimateTimeToReachBall()
{
  // stay striker if ball is caught
  if (getCaught())
  {
    estimatedTimeToReachBall = 0.0;
    return;
  }

  // account for distance to ball
  estimatedTimeToReachBall = Geometry::distanceTo(robotPose.translation, ballPosition.seen) / 0.2;
  
  // account if the robot is between the ball and the opponent goal

  // the position of the robot
  Vector2<double> robotPosition = robotPose.translation;
  double angleToLeftOpponentGoalPost = Geometry::angleTo(robotPosition, Vector2<double>(xPosOpponentGroundline,yPosLeftGoal));
  double angleToRightOpponentGoalPost = Geometry::angleTo(robotPosition, Vector2<double>(xPosOpponentGroundline,yPosRightGoal));
  if(angleToLeftOpponentGoalPost < angleToRightOpponentGoalPost)
  {
    angleToLeftOpponentGoalPost += pi2;
  }
  double angleToOpponentGoal= (angleToLeftOpponentGoalPost + angleToRightOpponentGoalPost) / 2.0;
  double absoluteAngleToBall = Geometry::angleTo(robotPosition,ballPosition.seen);
  double angleBetweenBallAndOpponentGoal = normalize(angleToOpponentGoal - absoluteAngleToBall);

  estimatedTimeToReachBall += 400.0 * fabs(angleBetweenBallAndOpponentGoal);

  // longer if ball not seen
  estimatedTimeToReachBall += 2.0 * SystemCall::getTimeSince(ballPosition.seen.timeWhenLastSeen);
  
  // test for obstacles
  /*
  if ( isOpponentBetweenRobotAndBall(robotPose) || 
  isOwnBetweenRobotAndBall (robotPose) )
  time += 5000.0;
  */
  
}

//----------------------------------------------------------------------------
bool MSH2004StrategySymbols::getAnotherPlayerIsInReadyState()
{
  for (int i=0; i<3;i++)
  {
    if (teamMessageCollection[i].isActual())
    {
      if (teamMessageCollection[i].behaviorTeamMessage.gameState 
        == BehaviorTeamMessage::ready)
      {
        return true;
      }
    }
  }
  return false;
}
//----------------------------------------------------------------------------
bool MSH2004StrategySymbols::getAnotherTeammateIsPreparingAKick()
{
  for (int i=0; i<3;i++)
  {
    if (teamMessageCollection[i].isActual())
    {
      if (teamMessageCollection[i].behaviorTeamMessage.message
        == BehaviorTeamMessage::preparingAKick)
      {
        return true;
      }
    }
  }
  return false;
}
//----------------------------------------------------------------------------
bool MSH2004StrategySymbols::getAnotherTeammateIsPerformingAKick()
{
  for (int i=0; i<3;i++)
  {
    if (teamMessageCollection[i].isActual())
    {
      if (teamMessageCollection[i].behaviorTeamMessage.message
        == BehaviorTeamMessage::performingAKick)
      {
        return true;
      }
    }
  }
  return false;
}
//----------------------------------------------------------------------------
bool MSH2004StrategySymbols::getAnotherTeammateJustPerformedAKick()
{
  for (int i=0; i<3;i++)
  {
    if (teamMessageCollection[i].isActual())
    {
      if (teamMessageCollection[i].behaviorTeamMessage.message
        == BehaviorTeamMessage::justPerformedAKick)
      {
        return true;
      }
    }
  }
  return false;
}
//----------------------------------------------------------------------------
bool MSH2004StrategySymbols::getTheStrikerIsPlayingNearTheOpponentGoal()
{
  for (int i=0; i<3;i++)
  {
    if (teamMessageCollection[i].isActual())
    {
      if (teamMessageCollection[i].behaviorTeamMessage.dynamicRole == BehaviorTeamMessage::striker
        && teamMessageCollection[i].robotPose.translation.x > 1400)
      {
        return true;
      }
    }
  }
  return false;
}
//----------------------------------------------------------------------------
bool MSH2004StrategySymbols::getTheStrikerIsNotPlayingNearTheOpponentGoal()
{
  for (int i=0; i<3;i++)
  {
    if (teamMessageCollection[i].isActual())
    {
      if (teamMessageCollection[i].behaviorTeamMessage.dynamicRole == BehaviorTeamMessage::striker
        && teamMessageCollection[i].robotPose.translation.x < 1200)
      {
        return true;
      }
    }
  }
  return false;
}
//----------------------------------------------------------------------------
bool MSH2004StrategySymbols::getTheStrikerIsPlayingNearTheOwnGoal()
{
  for (int i=0; i<3;i++)
  {
    if (teamMessageCollection[i].isActual())
    {
      if (teamMessageCollection[i].behaviorTeamMessage.dynamicRole == BehaviorTeamMessage::striker
        && teamMessageCollection[i].robotPose.translation.x < -1200)
      {
        return true;
      }
    }
  }
  return false;
}
//----------------------------------------------------------------------------
bool MSH2004StrategySymbols::getTheStrikerIsNotPlayingNearTheOwnGoal()
{
  for (int i=0; i<3;i++)
  {
    if (teamMessageCollection[i].isActual())
    {
      if (teamMessageCollection[i].behaviorTeamMessage.dynamicRole == BehaviorTeamMessage::striker
        && teamMessageCollection[i].robotPose.translation.x > -1000)
      {
        return true;
      }
    }
  }
  return false;
}
//----------------------------------------------------------------------------
bool MSH2004StrategySymbols::getCaught()
{
  return (SystemCall::getTimeSince(timeUntilBallWasCaught) < 500);  
}
//----------------------------------------------------------------------------
double MSH2004StrategySymbols::getCatchTime()
{
  return (SystemCall::getTimeSince(timeUntilBallWasCaught) < 500?
    timeUntilBallWasCaught - timeWhenBallWasStartedToCatch : 0);
}
//----------------------------------------------------------------------------
double MSH2004StrategySymbols::getSearchBallX()
{
  // fall-back if no wlan
  if (SystemCall::getTimeSince(teamMessageCollection[0].lastReceivedTimeStamp) > 5000 
   && SystemCall::getTimeSince(teamMessageCollection[1].lastReceivedTimeStamp) > 5000
   && SystemCall::getTimeSince(teamMessageCollection[2].lastReceivedTimeStamp) > 5000)
  {
    switch(getPlayer().getPlayerNumber())
    {
      case Player::two:
        return -1000;
      case Player::three:
        return 0;
      case Player::four:
      default:
        return 1000;
    }
  }
  else
  {
    switch (role)
    {
      case BehaviorTeamMessage::defensiveSupporter:
        return -1000;
      case BehaviorTeamMessage::offensiveSupporter:
        return 1000;
      case BehaviorTeamMessage::striker:
      default:
        return 0;
    }
  }
}
//----------------------------------------------------------------------------

double MSH2004StrategySymbols::getAngleToTeammate1()
{
  if(1 <= playerPoseCollection.numberOfOwnPlayers)
  {
/*
    return toDegrees(Geometry::angleTo(robotPose,
      Vector2<double>((playerPoseCollection.getOwnPlayerPose(0).getPose().translation.x + xPosOpponentGroundline) / 2,
                      (playerPoseCollection.getOwnPlayerPose(0).getPose().translation.y + yPosCenterGoal) / 2)));
*/    
      return toDegrees(Geometry::angleTo(robotPose,
      Vector2<double>(playerPoseCollection.getOwnPlayerPose(0).getPose().translation.x,
                      playerPoseCollection.getOwnPlayerPose(0).getPose().translation.y )));
  }
  else
  {
    return toDegrees(Geometry::angleTo(robotPose,
      Vector2<double>(xPosOpponentGroundline,yPosCenterGoal)));
  }
}

double MSH2004StrategySymbols::getDistanceToTeammate1()
{
  if(1 <= playerPoseCollection.numberOfOwnPlayers)
  {
/*    return Geometry::distanceTo(robotPose,
      Vector2<double>((playerPoseCollection.getOwnPlayerPose(0).getPose().translation.x + xPosOpponentGroundline) / 2,
                      (playerPoseCollection.getOwnPlayerPose(0).getPose().translation.y + yPosCenterGoal) / 2));
*/
    return Geometry::distanceTo(robotPose,
      Vector2<double>(playerPoseCollection.getOwnPlayerPose(0).getPose().translation.x,
                      playerPoseCollection.getOwnPlayerPose(0).getPose().translation.y ));
  }
  else
  {
    return Geometry::distanceTo(robotPose,
      Vector2<double>(xPosOpponentGroundline,yPosCenterGoal));
  }
}

double MSH2004StrategySymbols::getAngleToTeammate2()
{
  if(2 <= playerPoseCollection.numberOfOwnPlayers)
  {
/*    return toDegrees(Geometry::angleTo(robotPose,
      Vector2<double>((playerPoseCollection.getOwnPlayerPose(1).getPose().translation.x + xPosOpponentGroundline) / 2,
                      (playerPoseCollection.getOwnPlayerPose(1).getPose().translation.y + yPosCenterGoal) / 2)));
*/
    return toDegrees(Geometry::angleTo(robotPose,
      Vector2<double>(playerPoseCollection.getOwnPlayerPose(1).getPose().translation.x,
                      playerPoseCollection.getOwnPlayerPose(1).getPose().translation.y)));

  }
  else
  {
    return toDegrees(Geometry::angleTo(robotPose,
      Vector2<double>(xPosOpponentGroundline,yPosCenterGoal)));
  }
}

double MSH2004StrategySymbols::getDistanceToTeammate2()
{
  if(2 <= playerPoseCollection.numberOfOwnPlayers)
  {
/*
    return Geometry::distanceTo(robotPose,
      Vector2<double>((playerPoseCollection.getOwnPlayerPose(1).getPose().translation.x + xPosOpponentGroundline) / 2,
                      (playerPoseCollection.getOwnPlayerPose(1).getPose().translation.y + yPosCenterGoal) / 2));
*/
    return Geometry::distanceTo(robotPose,
      Vector2<double>(playerPoseCollection.getOwnPlayerPose(1).getPose().translation.x,
                      playerPoseCollection.getOwnPlayerPose(1).getPose().translation.y));

  }
  else
  {
    return Geometry::distanceTo(robotPose,
      Vector2<double>(xPosOpponentGroundline,yPosCenterGoal));
  }
}


double MSH2004StrategySymbols::getAngleToTeammate3()
{
  if(3 <= playerPoseCollection.numberOfOwnPlayers)
  {
/*
    return toDegrees(Geometry::angleTo(robotPose,
      Vector2<double>((playerPoseCollection.getOwnPlayerPose(2).getPose().translation.x + xPosOpponentGroundline) / 2,
                      (playerPoseCollection.getOwnPlayerPose(2).getPose().translation.y + yPosCenterGoal) / 2)));
*/
    return toDegrees(Geometry::angleTo(robotPose,
      Vector2<double>(playerPoseCollection.getOwnPlayerPose(2).getPose().translation.x,
                      playerPoseCollection.getOwnPlayerPose(2).getPose().translation.y)));

  }
  else
  {
    return toDegrees(Geometry::angleTo(robotPose,
      Vector2<double>(xPosOpponentGroundline,yPosCenterGoal)));
  }
}

double MSH2004StrategySymbols::getDistanceToTeammate3()
{
  if(3 <= playerPoseCollection.numberOfOwnPlayers)
  {
/*    return Geometry::distanceTo(robotPose,
      Vector2<double>((playerPoseCollection.getOwnPlayerPose(2).getPose().translation.x + xPosOpponentGroundline) / 2,
                      (playerPoseCollection.getOwnPlayerPose(2).getPose().translation.y + yPosCenterGoal) / 2));
*/
    return Geometry::distanceTo(robotPose,
      Vector2<double>(playerPoseCollection.getOwnPlayerPose(2).getPose().translation.x,
                      playerPoseCollection.getOwnPlayerPose(2).getPose().translation.y));

  }
  else
  {
    return Geometry::distanceTo(robotPose,
      Vector2<double>(xPosOpponentGroundline,yPosCenterGoal));
  }
}


void MSH2004StrategySymbols::setCarryResult(int number)
{
  if (number == 1)
    successfulCarries++;
  else
    failedCarries++;
}

void MSH2004StrategySymbols::setCarryStatus(int number)
{
  if (number < 2)
  {
    if (number == 1)
      carryIsRunning = true;
    else
      carryIsRunning = false;
  }
  else 
  { 
    carryDisabled  = true;
    carryIsRunning = false;
  }
}

void MSH2004StrategySymbols::setEarsStatus(int number)
{
  if (number == 0)
    earsStripped = true;
  else
    earsStripped = false;

}

void MSH2004StrategySymbols::setPenaltyShooter(int number)
{
   isPenaltyShooter = (number == 1);
}

double MSH2004StrategySymbols::getBlockPositionBetweenBallAndGoalX()
{
  Vector2<double> goal;
  goal.x=xPosOwnGoalpost;
  goal.y=yPosCenterGoal;
  Vector2<double> ballToGoal=goal-ballPosition.seen;
  ballToGoal.normalize();
  ballToGoal*=200;

  return (ballToGoal+ballPosition.seen).x;
}
double MSH2004StrategySymbols::getBlockPositionBetweenBallAndGoalY()
{
  Vector2<double> goal;
  goal.x=xPosOwnGoalpost;
  goal.y=yPosCenterGoal;
  Vector2<double> ballToGoal=goal-ballPosition.seen;
  ballToGoal.normalize();
  ballToGoal*=200;

  return (ballToGoal+ballPosition.seen).y;
}

double MSH2004StrategySymbols::getAngleToOpponentGoal()
{
  Vector2<double> goal;
  goal.x=xPosOpponentGoalpost;
  goal.y=yPosCenterGoal;
  Vector2<double> robotToGoal=goal-robotPose.translation;
  return toDegrees(robotToGoal.angle()-robotPose.rotation);
}

double MSH2004StrategySymbols::getAngleToOwnGoal()
{
  Vector2<double> goal;
  goal.x=xPosOwnGoalpost;
  goal.y=yPosCenterGoal;
  Vector2<double> robotToGoal=goal-robotPose.translation;
  return toDegrees(robotToGoal.angle()-robotPose.rotation);
}


/*
* Change Log
* 
* $Log: MSH2004StrategySymbols.cpp,v $
* Revision 1.23  2004/05/12 19:33:13  kerdels
* merged the behavior changes during australian, american and japan open
*
* Revision 1.22  2004/04/08 15:33:04  wachter
* GT04 checkin of Microsoft-Hellounds
*

* Revision 1.27  2004/04/04 00:23:27  wachter
* MISMIP penalty-shooter
*
* Revision 1.26  2004/04/03 22:32:44  wachter
* Added symbol
*
* Revision 1.25  2004/04/02 23:01:50  schumann
* Turning with ball improved
*
* Revision 1.24  2004/04/01 22:47:28  schumann
* Border handling near own goal
*
* Revision 1.23  2004/03/30 01:48:35  pg_joke
* created some symbols to detect wether the ears are already stripped or not, so the ears-off action is only executed once.
* added a "logical-switch" to the carry-option in simple-kick.xml. At the moment the carry option is disabled.
*
* Revision 1.22  2004/03/28 22:15:37  pg_joke
* added carry-ball-to-opponent-goal option and included it into simple-kick.
* the option is executed when the player is not the goalie, the carry option was not penalized before, the carry option was not bad performing before and our score is lower than the score of the opponent team. fair enough, isn't it ?
* btw: at the moment the option uses some fake special-actions, because the real special-actions used for this aren't implemented yet
*
* Revision 1.21  2004/03/25 16:29:40  pg_miwa
* Added symbol to set team-color


* Revision 1.21  2004/03/25 17:40:15  loetzsch
* adaptations to changes in the game controller and the led request
*
* Revision 1.20  2004/03/16 17:41:32  kerdels
* corrected comment
*
* Revision 1.19  2004/03/16 02:19:09  kerdels
* finalized basic work on penalty shooter
*
* Revision 1.18  2004/03/15 18:30:13  kerdels
* removed error, sorry folks
*
* Revision 1.17  2004/03/15 18:07:12  kerdels
* fixed conflict
*
* Revision 1.16  2004/03/15 15:59:45  schumann
* splitted symbols for kick engine from other symbols
*
* Revision 1.15  2004/03/14 18:51:19  kerdels
* worked on penalty shooter
*
* Revision 1.14  2004/03/10 09:01:52  schumann
* changed origin of coordinate system
*
* Revision 1.13  2004/03/09 14:13:33  schumann
* changed training for kickengine
*
* Revision 1.12  2004/03/08 01:07:13  roefer
* Interfaces should be const
*
* Revision 1.11  2004/03/04 08:37:41  schumann
* added neighbourhood search
*
* Revision 1.10  2004/03/02 19:04:33  kerdels
* some minor fix
*
* Revision 1.9  2004/03/01 22:27:21  kerdels
* *** empty log message ***
*
* Revision 1.8  2004/03/01 21:58:39  kerdels
* further work on kickengine integration
*
* Revision 1.7  2004/03/01 20:53:11  kerdels
* worked on kickengine integration
*
* Revision 1.6  2004/03/01 17:14:05  kerdels
* added robot-pose.distance-to-opponent-goal,
* moved robot-pose.angle-to-teammate1-3 to MSH2004StrategySymbols as fieldpos.angle-to-teammate1-3,
* added fieldpos.distance-to-teammate1-3,
* added DTT-Options newKickToGoal, newKickToClear, newKickToTeammate1-3,
* added DTT-OptionClass newKickToTeammate,
* added kickToPosRating function in DefaultOptionRating
*
* Revision 1.5  2004/02/27 16:20:45  schumann
* improved kickengine training
*
* Revision 1.4  2004/02/25 11:53:20  schumann
* extended range of kick engine
*
* Revision 1.3  2004/02/23 12:52:58  schumann
* added MSH2004KickEngine
*
* Revision 1.2  2003/12/06 17:45:33  loetzsch
* replaced Player::playerRole (goalie, defender, striker1, striker2)
* by Player::playerNumber (one, two, three, four)
*
* Revision 1.1  2003/10/26 22:49:40  loetzsch
* created ATH2004BehaviorControl from GT2003BehaviorControl
*  - strongly simplified option graph
*  - moved some symbols from GT2003 to CommonXabsl2Symbols
*  - moved some basic behaviors from GT2003 to CommonXabsl2BasicBehaviors
*
* cloned ATH2004 three times (BB2004, DDD2004, MSH2004)
*
*/

