/**
* @file MSH2004KickEngineEntry.cpp
* 
* Implementation of class MSH2004KickEngineEntry.
*
* @author Carsten Schumann (schumann@tempus-vivit.net)
*/
 
#include "MSH2004KickEngineEntry.h"
//----------------------------------------------------------------------------
MSH2004KickEngineEntry::MSH2004KickEngineEntry()
{
  this->kicks=new List<MSH2004KickEngineKick*>();
}
//----------------------------------------------------------------------------
MSH2004KickEngineEntry::~MSH2004KickEngineEntry()
{
  List<MSH2004KickEngineKick*>::Pos iterator=this->kicks->getFirst();
  while(iterator)
  {
    if ((*kicks)[iterator]!=NULL)
    {
      delete((*kicks)[iterator]);
      (*kicks)[iterator]=NULL;
    }
    iterator++;
  }
  delete(this->kicks);
  this->kicks=NULL;
}
//----------------------------------------------------------------------------
void MSH2004KickEngineEntry::addKick(int kickSector, 
                                     int kickDistance, 
                                     MotionRequest::SpecialActionID kick)
{
  
  List<MSH2004KickEngineKick*>::Pos iterator=this->kicks->getFirst();
  MSH2004KickEngineKick* kickStats=NULL;
  while(iterator && !kickStats)
  {
    if ((*kicks)[iterator]->getKickID()==kick)
    {
      kickStats=(*kicks)[iterator];
    }
    iterator++;
  }

  if (!kickStats)
  {
    kickStats=new MSH2004KickEngineKick(kick);
    kicks->insert(kickStats);
  }

  kickStats->addKick(kickSector,kickDistance);
}
//----------------------------------------------------------------------------
void MSH2004KickEngineEntry::clear()
{
  List<MSH2004KickEngineKick*>::Pos iterator=this->kicks->getFirst();
  while(iterator)
  {
    delete((*kicks)[iterator]);
    (*kicks)[iterator]=NULL;
    iterator++;
  }
  this->kicks->clear();
}
//----------------------------------------------------------------------------
MSH2004KickEngineSearchResult MSH2004KickEngineEntry::getPreferredKick(unsigned short int kickSector, 
                                                                        unsigned short int kickDistance){
  double maxSuccessRate=0.0;
  List<MSH2004KickEngineKick*>::Pos maxSuccessIterator=NULL;
  List<MSH2004KickEngineKick*>::Pos iterator=this->kicks->getFirst();
  while(iterator)
  {
    double currentSuccessRate=(*kicks)[iterator]->getSuccessRate(kickSector,kickDistance);

    if (currentSuccessRate>maxSuccessRate)
    {
      maxSuccessRate=currentSuccessRate;
      maxSuccessIterator=iterator;
    }
    iterator++;

  }

  MSH2004KickEngineSearchResult result;

  result.successRate=maxSuccessRate;
  if (maxSuccessRate==0.0)
  {
    result.kick=MotionRequest::sit;
  } else {
    result.kick=(*kicks)[maxSuccessIterator]->getKickID();
  }
  return result;
}
//----------------------------------------------------------------------------
MSH2004KickEngineSearchResult MSH2004KickEngineEntry::getNeighbourhoodKick(unsigned short int kickSector, 
                                                                           unsigned short int kickDistance,
                                                                           double maxKickTolerance,
                                                                           double distanceForKickSector,
                                                                           double distanceForKickDistance)
{
  double maxSuccessRate=0.0;
  List<MSH2004KickEngineKick*>::Pos maxSuccessIterator=NULL;
  List<MSH2004KickEngineKick*>::Pos iterator=this->kicks->getFirst();
  while(iterator)
  {
    double currentSuccessRate=(*kicks)[iterator]->getSuccessRate(kickSector,kickDistance,maxKickTolerance,distanceForKickSector,distanceForKickDistance);

    if (currentSuccessRate>maxSuccessRate)
    {
      maxSuccessRate=currentSuccessRate;
      maxSuccessIterator=iterator;
    }
    iterator++;

  }

  MSH2004KickEngineSearchResult result;

  result.successRate=maxSuccessRate;
  if (maxSuccessRate==0.0)
  {
    result.kick=MotionRequest::sit;
  } else {
    result.kick=(*kicks)[maxSuccessIterator]->getKickID();
    result.foundKickDistance=kickDistance;
    result.foundKickSector=kickSector;
  }
  return result;
}
//----------------------------------------------------------------------------
List<MSH2004KickEngineKick*>::Pos MSH2004KickEngineEntry::findEntry(MotionRequest::SpecialActionID kick)
{
  List<MSH2004KickEngineKick*>::Pos iterator=this->kicks->getFirst();
  while(iterator)
  {
    if ((*kicks)[iterator]->getKickID()==kick)
    {
      return iterator;
    }
    iterator ++;
  }
  return iterator;
}
    
//----------------------------------------------------------------------------
bool MSH2004KickEngineEntry::kickKnown(MotionRequest::SpecialActionID kick)
{
  List<MSH2004KickEngineKick*>::Pos iterator=findEntry(kick);
  if (iterator)
  {
    return (*kicks)[iterator]->kickKnown();
  }
  return false;
}
//----------------------------------------------------------------------------
bool MSH2004KickEngineEntry::kickDataKnown(MotionRequest::SpecialActionID kick)
{
  List<MSH2004KickEngineKick*>::Pos iterator=findEntry(kick);
  if (iterator)
  {
    return true;
  }
  return false;
}
//----------------------------------------------------------------------------
bool MSH2004KickEngineEntry::goodResultKnown(MotionRequest::SpecialActionID kick)
  {
  List<MSH2004KickEngineKick*>::Pos iterator=findEntry(kick);
  if(iterator)
    {
      return (*kicks)[iterator]->goodResultKnown();
    }
  return false;
}
//----------------------------------------------------------------------------
void MSH2004KickEngineEntry::registerTry(MotionRequest::SpecialActionID kick)
{
  List<MSH2004KickEngineKick*>::Pos iterator=findEntry(kick);
  if (iterator)
    {
      (*kicks)[iterator]->registerTry();
  } else {
    MSH2004KickEngineKick* kickStats=new MSH2004KickEngineKick(kick);
  kickStats->registerTry();
  kicks->insert(kickStats);
}
}
//----------------------------------------------------------------------------
bool MSH2004KickEngineEntry::positiveClusterPoint(unsigned short int kickDistance,
                            unsigned short int kickSector,
                            MotionRequest::SpecialActionID kick)
{
  List<MSH2004KickEngineKick*>::Pos iterator=findEntry(kick);
  if(iterator)
  {
    return (*kicks)[iterator]->goodResultKnown();
  }
  return false;
}
//----------------------------------------------------------------------------
bool MSH2004KickEngineEntry::negativeClusterPoint(unsigned short int kickDistance,
                            unsigned short int kickSector,
                            MotionRequest::SpecialActionID kick)
  {
  List<MSH2004KickEngineKick*>::Pos iterator=findEntry(kick);
  if(iterator)
  {
    return !(*kicks)[iterator]->goodResultKnown();
  }
  return false;
  }

//----------------------------------------------------------------------------
void MSH2004KickEngineEntry::loadKickResults(MotionRequest::SpecialActionID kick,In& stream,char* buffer)
{
  List<MSH2004KickEngineKick*>::Pos iterator=findEntry(kick);
  if (iterator){
    (*kicks)[iterator]->loadResults(stream,buffer);
  } else {
    MSH2004KickEngineKick* newEntry=new MSH2004KickEngineKick(kick);
    newEntry->loadResults(stream,buffer);
    kicks->insert(newEntry);
  }
}
//----------------------------------------------------------------------------
void MSH2004KickEngineEntry::loadKickTries(MotionRequest::SpecialActionID kick,In& stream,char* buffer)
{
  List<MSH2004KickEngineKick*>::Pos iterator=findEntry(kick);
  if (iterator){
    (*kicks)[iterator]->loadTries(stream,buffer);
  } else {
    MSH2004KickEngineKick* newEntry=new MSH2004KickEngineKick(kick);
    newEntry->loadTries(stream,buffer);
    kicks->insert(newEntry);
  }
}
//----------------------------------------------------------------------------
void MSH2004KickEngineEntry::saveKick(MotionRequest::SpecialActionID kick, Out& stream){
  List<MSH2004KickEngineKick*>::Pos iterator=findEntry(kick);
  if(iterator)
  {
    (*kicks)[iterator]->save(stream);
  }
}

/*
 * Change log :
 * 
 * $Log: MSH2004KickEngineEntry.cpp,v $
 * Revision 1.10  2004/04/08 15:33:04  wachter
 * GT04 checkin of Microsoft-Hellounds
 *
 * Revision 1.13  2004/03/26 15:31:34  pg_cars
 * Improved data file format, improved debugging
 *
 * Revision 1.12  2004/03/26 09:21:49  pg_cars
 * datafile changed, aggressive kick engine
 *
 * Revision 1.11  2004/03/25 21:05:24  pg_cars
 * change of datafile, before backup
 *
 * Revision 1.10  2004/03/22 17:53:08  pg_cars
 * improved training for kick engine
 *
 * Revision 1.9  2004/03/15 15:59:46  schumann
 * splitted symbols for kick engine from other symbols
 *
 * Revision 1.8  2004/03/09 14:13:32  schumann
 * changed training for kickengine
 *
 * Revision 1.7  2004/03/04 08:37:40  schumann
 * added neighbourhood search
 *
 * Revision 1.6  2004/02/27 16:20:45  schumann
 * improved kickengine training
 *
 * Revision 1.5  2004/02/26 10:54:01  jhoffman
 * - hopefully fixed ath-mof name problems
 *
 * Revision 1.4  2004/02/25 13:43:27  jhoffman
 * ATH kick renamed
 *
 * Revision 1.3  2004/02/24 16:03:04  schumann
 * New behavior for training
 *
 * Revision 1.2  2004/02/24 13:51:05  schumann
 * Changed structure of kick engine
 *
 * Revision 1.1  2004/02/23 12:51:37  schumann
 * Added MSH2004KickEngine
 *
*/
