/**
* @file MSH2004KickEngine.cpp
* 
* Implementation of class MSH2004KickEngine.
*
* @author Carsten Schumann (schumann@tempus-vivit.net)
*/
 
#include "MSH2004KickEngine.h"
#include "Tools/Streams/InStreams.h"
#include "Tools/Streams/OutStreams.h"
#include "Tools/Debugging/Debugging.h"
//----------------------------------------------------------------------------
MSH2004KickEngine::MSH2004KickEngine(const char* myKickEngineFile)
{
  strcpy(this->kickEngineFile,myKickEngineFile);

  for (int i=MSH2004KickEngineXMin; i<=MSH2004KickEngineXMax; i++)
  {
    for (int j=MSH2004KickEngineYMin; j<=MSH2004KickEngineYMax; j++)
    {
        this->ballPosition[i-MSH2004KickEngineXMin][j-MSH2004KickEngineYMin]=NULL;
        this->possibleBallPosition[i-MSH2004KickEngineXMin][j-MSH2004KickEngineYMin]=0;
    }
  }

  for(int k=0; k<MotionRequest::numOfSpecialAction; k++){
    this->knownKicks[k]=false;
  }

  this->load();
/*  InBinaryFile stream(kickEngineFile);
  if (stream.exists() && !stream.eof())
  {
    OUTPUT(idText,text,"Reading kick engine metrics from "<<kickEngineFile);
    stream>> *this;
  } else {
    OUTPUT(idText,text,"Cannot read kick engine metrics from "<<kickEngineFile);
  }
  */
  this->distanceForBallY=1;
  this->distanceForBallX=1;
  this->distanceForKickDistance=0;
  this->distanceForKickSector=1;
  this->possibleBallPositionCount=0;
  this->unsavedChanges=0;

}
//----------------------------------------------------------------------------
void MSH2004KickEngine::save()
{
  OutTextRawFile stream(kickEngineFile);
//  OutBinaryFile stream();
  if (stream.exists())
  {
    stream <<"[KnownKicks]\r\n";
    int k;
    for(k=0; k<MotionRequest::numOfSpecialAction; k++){
      if (this->knownKicks[k]){
        stream << MotionRequest::getSpecialActionName((MotionRequest::SpecialActionID) k);
        stream << "\r\n";
      }
    }
    stream <<"\r\n";

    for(k=0; k<MotionRequest::numOfSpecialAction; k++){
      if (this->knownKicks[k]){
        stream << "["<<MotionRequest::getSpecialActionName((MotionRequest::SpecialActionID) k)<<"]";
        stream << "\r\n";
        
        for (short int i=MSH2004KickEngineXMin; i<=MSH2004KickEngineXMax; i++)
        {
          for (short int j=MSH2004KickEngineYMin; j<=MSH2004KickEngineXMax; j++)
          {
            if (ballPosition[i-MSH2004KickEngineXMin][j-MSH2004KickEngineYMin] != NULL)
            {
              if (ballPosition[i-MSH2004KickEngineXMin][j-MSH2004KickEngineYMin]->kickDataKnown((MotionRequest::SpecialActionID)k))
              {
                stream<<"Position "<<i<<" "<<j<<"\r\n";
                ballPosition[i-MSH2004KickEngineXMin][j-MSH2004KickEngineYMin]->saveKick((MotionRequest::SpecialActionID)k,stream);
                stream<<"PositionEnd\r\n";
              }
            }
          }
        }
        stream <<"\r\n";
      }
    }

    stream << "[PossiblePositions]\r\n";
    stream << "# Format: \r\n";
    stream << "# Position [X] [Y] [Count]\r\n\r\n";

    for (short int i=MSH2004KickEngineXMin; i<=MSH2004KickEngineXMax; i++)
    {
      for (short int j=MSH2004KickEngineYMin; j<=MSH2004KickEngineXMax; j++)
      {
        if (possibleBallPosition[i-MSH2004KickEngineXMin][j-MSH2004KickEngineYMin] >0)
        {
          stream << "Position "<<i<<" "<<j<<" "<<possibleBallPosition[i-MSH2004KickEngineXMin][j-MSH2004KickEngineYMin]<<"\r\n";
        }
      }
    }




  }
}
//----------------------------------------------------------------------------
void MSH2004KickEngine::load()
{

  InConfigFile stream(kickEngineFile,"KnownKicks");
  if (!stream.exists() || stream.eof()) {
    OUTPUT(idText,text,"Unable to load text based kick metrics");
  } else {
    bool eod=false; //end of data
    char buffer[256];
    while(!stream.eof() && !eod) {
      stream >> buffer;
      if (buffer[0] == '['){
        eod=true; //next section
      } else {
        MotionRequest::SpecialActionID sid;
        sid=MotionRequest::getSpecialActionIDFromName(buffer);
        if (sid != MotionRequest::numOfSpecialAction){
          this->knownKicks[sid]=true;
        } else {
          OUTPUT(idText,text,"Error: Unknown SpecialAction "<<buffer);
        }
      }
    }
  }


  for(int k=0; k<MotionRequest::numOfSpecialAction; k++){
    if (this->knownKicks[k]){
      MotionRequest::SpecialActionID sid=(MotionRequest::SpecialActionID) k;
      InConfigFile kickstream(kickEngineFile,MotionRequest::getSpecialActionName(sid));
      if (!kickstream.exists() || kickstream.eof()) {
        OUTPUT(idText,text,"Unable to load text based kick metrics for kick " << MotionRequest::getSpecialActionName(sid));
      } else {
        bool eod=false; //end of data
        short int i;
        short int j;
        char buffer[256];
        kickstream >> buffer;
        while(!kickstream.eof() && !eod) {
          if (buffer[0] == '[') eod=true; //next section
          if (!strcmp(buffer,"Position")){ //means that they are equal
            kickstream >> i;
            kickstream >> j;
            INFO(sendKickEngineInfo, idText,text,"ReadingBallLocation:"<<i<<", "<<j);
            if (!kickstream.eof() && !eod){
              kickstream>>buffer;
            }
          }

          if (!strcmp(buffer,"Tries")){
            if (ballPosition[i-MSH2004KickEngineXMin][j-MSH2004KickEngineYMin] == NULL)
            {
              ballPosition[i-MSH2004KickEngineXMin][j-MSH2004KickEngineYMin]= new MSH2004KickEngineEntry();
            }
            ballPosition[i-MSH2004KickEngineXMin][j-MSH2004KickEngineYMin]->loadKickTries(sid,kickstream,buffer);
          }    
          if (!strcmp(buffer,"Result")){
            if (ballPosition[i-MSH2004KickEngineXMin][j-MSH2004KickEngineYMin] == NULL)
            {
              ballPosition[i-MSH2004KickEngineXMin][j-MSH2004KickEngineYMin]= new MSH2004KickEngineEntry();
            }
            ballPosition[i-MSH2004KickEngineXMin][j-MSH2004KickEngineYMin]->loadKickResults(sid,kickstream,buffer);
          }    

          if (!kickstream.eof() && !eod && !strcmp(buffer,"PositionEnd")){
            kickstream>>buffer;
          }
        }
      }
    }    
  }
  
  InConfigFile stream2(kickEngineFile,"PossiblePositions");
  if (!stream2.exists() || stream2.eof()) {
    INFO(sendKickEngineInfo, idText,text,"Unable to load text based kick metrics");
  } else {
    bool eod=false; //end of data
    while(!stream2.eof() && !eod) {
      char buffer[256];
      stream2 >> buffer;
      if (buffer[0] == '[') eod=true; //next section
      if (!strcmp(buffer,"Position")){ //means that they are equal
        short int i;
        short int j;
        short int count;

        stream2 >> i;
        stream2 >> j;
        stream2 >> count;

        possibleBallPosition[i-MSH2004KickEngineXMin][j-MSH2004KickEngineYMin]=count;
        possibleBallPositionCount+=count;
      }
    }
  }
}
//----------------------------------------------------------------------------
MSH2004KickEngine::~MSH2004KickEngine()
{
  this->save();
  this->clearDistribution();
}
//----------------------------------------------------------------------------
void MSH2004KickEngine::clearDistribution()
{
  for (int i=MSH2004KickEngineXMin; i<=MSH2004KickEngineXMax; i++)
  {
    for (int j=MSH2004KickEngineYMin; j<=MSH2004KickEngineYMax; j++)
    {
      if (this->ballPosition[i-MSH2004KickEngineXMin][j-MSH2004KickEngineYMin] != NULL)
      {
        delete(this->ballPosition[i-MSH2004KickEngineXMin][j-MSH2004KickEngineYMin]);
      }
    }
  } 
}

//----------------------------------------------------------------------------
short unsigned int MSH2004KickEngine::getKickSector(double angle)
{
  while (angle>=2*pi) 
  {
    angle-=2*pi;
  }
  double sector=((angle+(pi/32))*16/pi);
  if (sector<0) 
  {
    sector+=32;
  }
  return (int)sector;
}
//----------------------------------------------------------------------------
short int MSH2004KickEngine::getBallYSector(double y)
{
  //double y=sin(angle)*distance;
  // AXIS: Y
  // distance  -27 -25 -23 -21 -19 -17 -15 -13 -11  -9  -7  -5  -3  -1  +1  +3  +5  +7  +9
  // sector      | 0 | 1 | 2 | 3 | 4 | 5 | 6 | 7 | 8 | 9 |10 |11 |12 |13 |14 |15 |16 |   |   |
  // neu         |-13|-12|-11|-10| -9| -8| -7|-6 | -5| -4|-3 |-2 | -1|0  | 1 | 2 | 3 | 4 | 5 |

  // -0.99....0 => 0    0.49...0   =>0
  // -2.99...-1 => 1    1.49...0.5 =>1
  int sector=0;

  if (y<0){
    sector=(int)((y-1)/20);
    if (sector<MSH2004KickEngineYMin) { sector=MSH2004KickEngineYMin; }
  } else {
    sector=(int)((y+1)/20);
    if (sector>MSH2004KickEngineYMax) { sector=MSH2004KickEngineYMax; }
  }
  return sector;
}
//----------------------------------------------------------------------------
short int MSH2004KickEngine::getBallXSector(double x)
{
  //double x=cos(angle)*distance;

  int sector;
  if (x<0){
    sector=(int)((x-1)/20);
    if (sector<MSH2004KickEngineXMin) { sector=MSH2004KickEngineXMin; }
  } else {
    sector=(int)((x+1)/20);
    if (sector>MSH2004KickEngineXMax) { sector=MSH2004KickEngineXMax; }
  }

  return sector;
}
//----------------------------------------------------------------------------
short int MSH2004KickEngine::getBallXSectorWithoutBounds(double x)
{
  int sector;
  if (x<0){
    sector=(int)((x-1)/20);
  } else {
    sector=(int)((x+1)/20);
  }
  return sector;
}
//----------------------------------------------------------------------------
short int MSH2004KickEngine::getBallYSectorWithoutBounds(double y)
{
  int sector=0;

  if (y<0){
    sector=(int)((y-1)/20);
  } else {
    sector=(int)((y+1)/20);
  }
  return sector;
}
//----------------------------------------------------------------------------
short unsigned int MSH2004KickEngine::getKickDistance(double distance)
{
  short unsigned int distanceKick = 0;
  if (distance<500)
  {
    distanceKick=0;
  } else if (distance<1000)
  {
    distanceKick=1;
  } else if (distance<1500)
  {
    distanceKick=2;
  } else if (distance<2000)
  {
    distanceKick=3;
  } else if (distance<2500)
  {
    distanceKick=4;
  } else {
    distanceKick=5;
  }
  return distanceKick;
}


//----------------------------------------------------------------------------
void MSH2004KickEngine::addKick(double ballXmm, double ballYmm, double kickAngle, double kickDistance,MotionRequest::SpecialActionID kick)
{
  short int bx=this->getBallXSector(ballXmm);
  short int by=this->getBallYSector(ballYmm);  
  short unsigned int ks=this->getKickSector(kickAngle);
  short unsigned int kd=this->getKickDistance(kickDistance);
  this->knownKicks[kick]=true;




  if (this->ballPosition[bx-MSH2004KickEngineXMin][by-MSH2004KickEngineYMin]==NULL)
  {
    this->ballPosition[bx-MSH2004KickEngineXMin][by-MSH2004KickEngineYMin] = new MSH2004KickEngineEntry();
  }
  this->ballPosition[bx-MSH2004KickEngineXMin][by-MSH2004KickEngineYMin]->addKick(ks,kd,kick);
  this->unsavedChanges++;
  if (this->unsavedChanges>4){
    this->save();
    this->unsavedChanges=0;
  }
}

//----------------------------------------------------------------------------
MSH2004KickEngineSearchResult MSH2004KickEngine::getPreferredKick(double ballXmm, 
                                                                   double ballYmm, 
                                                                   double kickAngle, 
                                                                   double kickDistance,
                                                                   double maxKickTolerance)
{
  short int bx=this->getBallXSector(ballXmm);
  short int by=this->getBallYSector(ballYmm);  
  short unsigned int ks=this->getKickSector(kickAngle);
  short unsigned int kd=this->getKickDistance(kickDistance);


  if (this->ballPosition[bx-MSH2004KickEngineXMin][by-MSH2004KickEngineYMin]!=NULL)
  {
    return this->ballPosition[bx-MSH2004KickEngineXMin][by-MSH2004KickEngineYMin]->getNeighbourhoodKick(ks,kd,maxKickTolerance,distanceForKickSector,distanceForKickDistance);
    
//    return this->ballPosition[bx-MSH2004KickEngineXMin][by-MSH2004KickEngineYMin]->getPreferredKick(ks,kd);
  } else {
    MSH2004KickEngineSearchResult falseResult;
    falseResult.successRate=0.0;
    falseResult.kick=MotionRequest::sit;
    return falseResult;
  }
}


//----------------------------------------------------------------------------
double MSH2004KickEngine::getDistance(int x1, int y1, int x2, int y2)
{
  return abs(x1-x2)+abs(y1-y2);

//  return ((double)abs(alter_bx))*(double)distanceForBallX+
//         ((double)abs(alter_by))*distanceForBallY;
}
//----------------------------------------------------------------------------
MSH2004KickEngineSearchResult MSH2004KickEngine::getNeighbourhoodKick(double currentBallXmm, 
                                                                       double currentBallYmm, 
                                                                       double kickAngle, 
                                                                       double kickDistance, 
                                                                       double maxSearchDistance,
                                                                       double maxKickTolerance)
{

/*
  currentBallXmm=50;
  currentBallYmm=-15;
  kickAngle=-0.43;
  kickDistance=2180;
*/
//  short int base_bx=this->getBallXSector(currentBallXmm);
//  short int base_by=this->getBallYSector(currentBallYmm);  
  short unsigned int base_ks=this->getKickSector(kickAngle);
  short unsigned int base_kd=this->getKickDistance(kickDistance);

  MSH2004KickEngineSearchResult bestResult;
  bestResult.successRate=0.0;
  bestResult.negativeClusterPoints=1;
  bestResult.positiveClusterPoints=0;
  bestResult.kick=MotionRequest::sit;

  short int currentBallX=this->getBallXSectorWithoutBounds(currentBallXmm);
  short int currentBallY=this->getBallYSectorWithoutBounds(currentBallYmm);


  for (short int searchDistance=0; searchDistance<=maxSearchDistance; searchDistance++){
    for (short int x=MSH2004KickEngineXMin; x<=MSH2004KickEngineXMax; x++){ 
      for (short int y=MSH2004KickEngineYMin; y<=MSH2004KickEngineYMax; y++){ 
        if (getDistance(x,y,currentBallX,currentBallY)==searchDistance){
          if (this->ballPosition[x-MSH2004KickEngineXMin][y-MSH2004KickEngineYMin]!=NULL){
            MSH2004KickEngineSearchResult result=this->ballPosition[x-MSH2004KickEngineXMin][y-MSH2004KickEngineYMin]->getNeighbourhoodKick(base_ks,base_kd,maxKickTolerance,distanceForKickSector,distanceForKickDistance);
            if (result.successRate>0)
            {
              result.foundSearchDistance=searchDistance;
              result.foundBallX=x;
              result.foundBallY=y;
              this->calculateClusterPoints(result);
              if ((result.successRate>bestResult.successRate) && ((bestResult.positiveClusterPoints-bestResult.negativeClusterPoints)<(result.positiveClusterPoints-result.negativeClusterPoints))){
                  bestResult=result;
                }
            }
          }
        }
      }
    }
  }


  return bestResult;
              }
//----------------------------------------------------------------------------
void MSH2004KickEngine::calculateClusterPoints(MSH2004KickEngineSearchResult& result)
{
  result.positiveClusterPoints=0;
  result.negativeClusterPoints=0;

  for (int testx=result.foundBallX-1; testx<=result.foundBallX+1; testx++)
  {
    for(int testy=result.foundBallY-1; testy<=result.foundBallY+1; testy++)
    {
      if (this->inBounds(testx,testy) && result.foundBallX!=testx && result.foundBallY!=testy)
      {
        if (this->ballPosition[testx-MSH2004KickEngineXMin][testy-MSH2004KickEngineYMin]!=NULL)
        {
          if (this->ballPosition[testx-MSH2004KickEngineXMin][testy-MSH2004KickEngineYMin]->positiveClusterPoint(result.foundKickDistance,result.foundKickSector,result.kick))
          {
            result.positiveClusterPoints++;
            }
          if (this->ballPosition[testx-MSH2004KickEngineXMin][testy-MSH2004KickEngineYMin]->negativeClusterPoint(result.foundKickDistance,result.foundKickSector,result.kick))
          {
            result.negativeClusterPoints++;
          }
        }
      }
    }
  }
} 
//----------------------------------------------------------------------------
/** Sets the distance metrics for a neighbourhood search
*   
*@param forBallX distance for the change in 1 X-Sector
*@param forBallY distance for the change in 1 Y-Sector
*/
void MSH2004KickEngine::setSearchDistanceMetrics(double forBallX, 
                                                 double forBallY)
{
  this->distanceForBallX=forBallX;
  this->distanceForBallY=forBallY;
}
//----------------------------------------------------------------------------
/** Sets the tolerance metrics for a neighbourhood search
*   
*@param forKickSector distance for the change in 1 kick sector
*@param forKickDistance distance for the change in 1 distance
*/
void MSH2004KickEngine::setSearchToleranceMetrics(double forKickSector, 
                                                  double forKickDistance)
{
  this->distanceForKickSector=forKickSector;
  this->distanceForKickDistance=forKickDistance;
}
//----------------------------------------------------------------------------
bool MSH2004KickEngine::inBounds(short int x,short int y)
{
  if (x>MSH2004KickEngineXMax) return false;
  if (x<MSH2004KickEngineXMin) return false;
  if (y>MSH2004KickEngineYMax) return false;
  if (y<MSH2004KickEngineYMin) return false;
  return true;
}
//----------------------------------------------------------------------------
/** Searches a specific position for known results
*   
*@param x x-Sector to search
*@param y y-Sector to search
*@param kick kick to perform the search for
*@return if there are any known results, return true
*/
bool MSH2004KickEngine::resultKnown(short int x, short int y,MotionRequest::SpecialActionID kick)
{
  if (this->ballPosition[x-MSH2004KickEngineXMin][y-MSH2004KickEngineYMin]==NULL) return false;
  return this->ballPosition[x-MSH2004KickEngineXMin][y-MSH2004KickEngineYMin]->kickKnown(kick);
}
//----------------------------------------------------------------------------
/** Searches the neighbourhood of an entry if there are known results.
*   
*@param cx x-Sector to start
*@param y y-Sector to start
*@param searchDistance max search distance in each dimension
*@param kick kick to perform the search for
*@return if there are any known results, return true
*/
bool MSH2004KickEngine::resultKnownInNeighbourhood(short int cx, short int cy, int searchDistance,MotionRequest::SpecialActionID kick)
{
  int x;
  int y;
  for (x=cx; x<=cx+searchDistance; x++)
  {
    for (y=cy; y<=cy+searchDistance; y++)
    {
      if (this->inBounds(x,y) && this->resultKnown(x,y,kick))
      {
        return true;
      }
    }
    for (y=cy; y>=cy-searchDistance; y--)
    {
      if (this->inBounds(x,y) && this->resultKnown(x,y,kick))
      {
        return true;
      }
    }
  }
  for (x=cx; x>=cx-searchDistance; x--)
  {
    for (y=cy; y<=cy+searchDistance; y++)
    {
      if (this->inBounds(x,y) && this->resultKnown(x,y,kick))
      {
        return true;
      }
    }
    for (y=cy; y>=cy-searchDistance; y--)
    {
      if (this->inBounds(x,y) && this->resultKnown(x,y,kick))
      {
        return true;
      }
    }
  }
  return (this->inBounds(cx,cy) && resultKnown(cx,cy,kick));
}
//----------------------------------------------------------------------------
/** Register a training try. if there are n tries for a specific position
*   try another position.
*@param x x-Sector of the try
*@param y y-Sector of the try
*@param kick tried kick
*/
void MSH2004KickEngine::registerTry(int x,int y,MotionRequest::SpecialActionID kick)
{
  INFO(sendKickEngineInfo, idText,text,"RegisterTry X="<<x<<" Y="<<y);
  if (this->inBounds(x,y)){
    if (this->ballPosition[x-MSH2004KickEngineXMin][y-MSH2004KickEngineYMin]==NULL) 
    {
      this->ballPosition[x-MSH2004KickEngineXMin][y-MSH2004KickEngineYMin] = new MSH2004KickEngineEntry();
    }
    this->ballPosition[x-MSH2004KickEngineXMin][y-MSH2004KickEngineYMin]->registerTry(kick);
  }

}
//----------------------------------------------------------------------------
bool MSH2004KickEngine::goodResultKnown(short int x, short int y, MotionRequest::SpecialActionID kick)
{
  if (this->ballPosition[x-MSH2004KickEngineXMin][y-MSH2004KickEngineYMin]!=NULL) 
  {
    return this->ballPosition[x-MSH2004KickEngineXMin][y-MSH2004KickEngineYMin]->goodResultKnown(kick);
  }
  return false;
}
//----------------------------------------------------------------------------
/** Search for a non trained position.
*   First perform a raw search with a wide grid, then if grid is filled
*   perform a fine search.
*@param x x-Sector of the try
*@param y y-Sector of the try
*@param kick tried kick
*@return Search result. foundBallX and foundBallY contain the sectors, 
*        successRate is 1 if search was successful
*/
MSH2004KickEngineSearchResult MSH2004KickEngine::getUnknownPosition(MotionRequest::SpecialActionID kick)
{
  int x;
        MSH2004KickEngineSearchResult result;
  result.successRate=0;
  result.count=0;
  //search near successful kicks
  for (x=MSH2004KickEngineXMin; x<=MSH2004KickEngineXMax; x++)
  {
    for (int y=MSH2004KickEngineYMin; y<=MSH2004KickEngineYMax; y++)
    {
      if (this->goodResultKnown(x,y,kick))
      {
        INFO(sendKickEngineInfo, idText,text,"Exploring from: X="<<x<<" Y="<<y);
        for(int testx=x-1; testx<=x+1; testx++)
        {
          for(int testy=y-1; testy<=y+1; testy++)
          {
            if (inBounds(testx,testy))
            {
              if (this->ballPositionPossible(testx,testy))
              {
              if(!this->resultKnown(testx,testy,kick))
              {
                  if (result.successRate==0)
                  {
                    result.foundBallX=testx;
                    result.foundBallY=testy;
                result.successRate=1;
              }
                  INFO(sendKickEngineInfo, idText,text,"  TrainingPos: X="<<testx<<" Y="<<testy);
                  result.count++;
                }
              } else {
                INFO(sendKickEngineInfo, idText,text,"  ImpossibleTrainingPos: X="<<testx<<" Y="<<testy);
            }
          }
        }
      }
    }
  }
  }
  //raw search

/*
  for (x=MSH2004KickEngineXMax; x >= 2; x--)
  {
    for (int y=0; y<=MSH2004KickEngineXMax; y++)
    {
      if (!this->resultKnownInNeighbourhood(x,y,4,kick))
      {
        if (this->ballPositionPossible(x,y)){
        MSH2004KickEngineSearchResult result;
        result.foundBallX=x;
        result.foundBallY=y;
        result.successRate=1;
        return result;
      }
      }
      if (!this->resultKnownInNeighbourhood(x,-y,4,kick))
      {
        if (this->ballPositionPossible(x,-y))
      {
        MSH2004KickEngineSearchResult result;
        result.foundBallX=x;
        result.foundBallY=-y;
        result.successRate=1;
        return result;
      }
    }
  }
}

*/

  return result;
}
//----------------------------------------------------------------------------
void MSH2004KickEngine::reportPossibleBallPosition(double x, double y)
    {
  int sectorX=this->getBallXSector(x);
  int sectorY=this->getBallYSector(y);
  if (this->possibleBallPosition[sectorX-MSH2004KickEngineXMin][sectorY-MSH2004KickEngineYMin] < 10000){
    this->possibleBallPosition[sectorX-MSH2004KickEngineXMin][sectorY-MSH2004KickEngineYMin]++;
    this->possibleBallPositionCount++;
    if (this->possibleBallPosition[sectorX-MSH2004KickEngineXMin][sectorY-MSH2004KickEngineYMin]==1)
      {
      INFO(sendKickEngineInfo, idText,text,"NPBP X="<<sectorX<<" Y="<<sectorY);  
    }
  }
}
//----------------------------------------------------------------------------
bool MSH2004KickEngine::ballPositionPossible(int sectorX, int sectorY)
{
  int minrequired=(int)(this->possibleBallPositionCount/MSH2004KickEngineSectors/10);
  if (minrequired<2)
  {
    minrequired=2;
    }
  if (this->possibleBallPosition[sectorX-MSH2004KickEngineXMin][sectorY-MSH2004KickEngineYMin] >=minrequired)
  {
    return true;
  } else {
    return false;
  }
}



/*
 * Change log :
 * 
 * $Log: MSH2004KickEngine.cpp,v $
 * Revision 1.14  2004/04/08 15:33:04  wachter
 * GT04 checkin of Microsoft-Hellounds
 *
 * Revision 1.20  2004/04/01 14:45:29  kindler
 * - added debug keys for walking- and kick-engine
 *
 * Revision 1.19  2004/03/29 14:44:21  pg_cars
 * Kick engine works now ;-)
 * Use PID Ball Locator and MSH2004-test behavior for testing
 *
 * Revision 1.18  2004/03/27 13:58:44  pg_cars
 * Bug fixed
 *
 * Revision 1.17  2004/03/26 15:31:34  pg_cars
 * Improved data file format, improved debugging
 *
 * Revision 1.16  2004/03/26 09:21:49  pg_cars
 * datafile changed, aggressive kick engine
 *
 * Revision 1.15  2004/03/25 21:05:24  pg_cars
 * change of datafile, before backup
 *
 * Revision 1.14  2004/03/22 17:53:07  pg_cars
 * improved training for kick engine
 *
 * Revision 1.13  2004/03/15 15:59:46  schumann
 * splitted symbols for kick engine from other symbols
 *
 * Revision 1.12  2004/03/10 10:05:15  schumann
 * changed origin of coordinate system
 *
 * Revision 1.11  2004/03/10 09:21:39  schumann
 * changed origin of coordinate system
 *
 * Revision 1.10  2004/03/10 09:01:52  schumann
 * changed origin of coordinate system
 *
 * Revision 1.9  2004/03/09 14:13:31  schumann
 * changed training for kickengine
 *
 * Revision 1.8  2004/03/04 08:37:40  schumann
 * added neighbourhood search
 *
 * Revision 1.7  2004/03/03 23:44:50  kerdels
 * worked on kickengine integration
 *
 * Revision 1.6  2004/03/01 22:27:19  kerdels
 * *** empty log message ***
 *
 * Revision 1.5  2004/03/01 20:53:10  kerdels
 * worked on kickengine integration
 *
 * Revision 1.4  2004/02/25 11:53:20  schumann
 * extended range of kick engine
 *
 * Revision 1.3  2004/02/24 16:03:04  schumann
 * New behavior for training
 *
 * Revision 1.2  2004/02/24 13:51:05  schumann
 * Changed structure of kick engine
 *
 * Revision 1.1  2004/02/23 12:51:37  schumann
 * Added MSH2004KickEngine
 *
*/

