/**
* @file MSH2004BehaviorConfiguration.h
* 
* Implementation of class MSH2004BehaviorConfiguration.
*
* @author Max Risler
*/

#ifndef __MSH2004BehaviorConfiguration_h_
#define __MSH2004BehaviorConfiguration_h_

#include "Tools/Math/Pose2D.h"
#include "Tools/ContinuousBasicBehaviors/ContinuousBasicBehavior.h"

/**
* @class MSH2004BehaviorConfiguration
*
* This class contains all tuneable configuration parameters for MSH2004BehaviorControl.
* 
* @author Max Risler
*/ 
class MSH2004BehaviorConfiguration : public MessageHandler
{
public:
  
  /** number of different kickoff variants */
  enum { numOfKickoff = 2 };

  /** the max speed with which the robot walks to its kickoff position */
  double goToKickoffMaxSpeed;

  /** the goalie start position */
  Vector2<double> goalie;

  /** the defensive supporter start position */
  Vector2<double> defensiveSupporter;

  /** the offensive supporter start position at opponent kickoff */
  Vector2<double> offensiveSupporterOpponentKickoff;
  /** the striker start position at opponent kickoff */
  Vector2<double> strikerOpponentKickoff;

  /** the offensive supporter start position at own kickoff */
  Vector2<double> offensiveSupporterOwnKickoff[numOfKickoff];
  /** the orientation of the offensive supporter at own kickoff in degrees */
  double offensiveSupporterOwnKickoffAngle[numOfKickoff];
  /** the striker start position at own kickoff */
  Vector2<double> strikerOwnKickoff[numOfKickoff];
  /** the orientation of the striker at own kickoff in degrees */
  double strikerOwnKickoffAngle[numOfKickoff];


	/* The distance where behaviors near the ball should be activated */
  double ballNearDistance;
	
  /* The distance where behaviors far from the ball should be activated */
  double ballNotNearDistance;
	
  /* The distance where behaviors at the ball should be activated */
  double atBallDistance;
	
  /* The distance where behaviors near the ball should be activated */
  double notAtBallDistance;

  
  /* The evaluation mode for continuous basic behaviors.
   * This determines how continuous basic behaviors will be evaluated.
   * See ContinuousBasicBehavior.h for details.
   */
  ContinuousBasicBehavior::EvaluationMode contEvaluationMode;

  /** 
  * Constructor.
  */
  MSH2004BehaviorConfiguration()
  {
    goToKickoffMaxSpeed = 250.0;

    // start position for goalie (both kickoffs)
    goalie.x = -2000.0;
    goalie.y = 50;
    // start position for defensive supporter (both kickoffs)
    defensiveSupporter.x = -1000.0;
    defensiveSupporter.y = -200.0;

    // start position for offensive supporter (opponent kickoffs)
    offensiveSupporterOpponentKickoff.x = -1000.0;
    offensiveSupporterOpponentKickoff.y = -800.0;
    // start position for striker (opponent kickoffs)
    strikerOpponentKickoff.x = -1000.0;
    strikerOpponentKickoff.y = 500.0;

    // long kickoff to the left
    offensiveSupporterOwnKickoff[0].x = -80.0;
    offensiveSupporterOwnKickoff[0].y = 1100.0;
    offensiveSupporterOwnKickoffAngle[0] = 0.0;
    strikerOwnKickoff[0].x = -130.0;
    strikerOwnKickoff[0].y = -130.0;
    strikerOwnKickoffAngle[0] = 45.0;

    // long kickoff to the right
    offensiveSupporterOwnKickoff[1].x = -80.0;
    offensiveSupporterOwnKickoff[1].y = -1100.0;
    offensiveSupporterOwnKickoffAngle[1] = 0.0;
    strikerOwnKickoff[1].x = -130.0;
    strikerOwnKickoff[1].y = 130.0;
    strikerOwnKickoffAngle[1] = -45.0;

    // direct long kickoff
    //offensiveSupporterOwnKickoff[2].x = -80.0;
    //offensiveSupporterOwnKickoff[2].y = -550.0;
    //offensiveSupporterOwnKickoffAngle[2] = 0;
    //strikerOwnKickoff[2].x = -170.0;
    //strikerOwnKickoff[2].y = 0.0;
    //strikerOwnKickoffAngle[2] = 0.0;

    ballNearDistance = 700.0;
    ballNotNearDistance = 1000.0;
    atBallDistance = 280.0;
    notAtBallDistance = 320.0;

    contEvaluationMode = ContinuousBasicBehavior::discrete;
  }

  /** 
  * Is called for every incoming debug message.
  * @param message An interface to read the message from the queue
  * @return if the message was read
  */
  virtual bool handleMessage(InMessage& message);
};


#endif// __MSH2004BehaviorConfiguration_h_

/*
 * Change log :
 * 
 * $Log: MSH2004BehaviorConfiguration.h,v $
 * Revision 1.1  2003/10/26 22:49:39  loetzsch
 * created ATH2004BehaviorControl from GT2003BehaviorControl
 *  - strongly simplified option graph
 *  - moved some symbols from GT2003 to CommonXabsl2Symbols
 *  - moved some basic behaviors from GT2003 to CommonXabsl2BasicBehaviors
 *
 * cloned ATH2004 three times (BB2004, DDD2004, MSH2004)
 *
 */
