/** 
* @file MSH2004SimpleBasicBehaviors.h
*
* Declaration of basic behaviors defined in simple-basic-behaviors.xml.
*
* @author Uwe Dffert
* @author Jan Hoffmann
* @author Matthias Jngel
* @author Martin Ltzsch
* @author Max Risler
*/

#ifndef __MSH2004SimpleBasicBehaviors_h_
#define __MSH2004SimpleBasicBehaviors_h_

#include "../../BehaviorControl.h"
#include "Tools/Xabsl2/Xabsl2Engine/Xabsl2Engine.h"
#include "Tools/Xabsl2/Xabsl2Engine/Xabsl2BasicBehavior.h"
#include "Tools/Actorics/WalkAccelerationRestrictor.h"
#include "Tools/Evolution/Population.h"
#include "Tools/Math/PIDsmoothedValue.h"
#include "Modules/WalkingEngine/InvKinWalkingEngine.h"
#include "Modules/WalkingEngine/InvKinWalkingParameterSets.h"
#include "Tools/SlidingAverageValue.h"



/**
* A Basic Behavior for going to the ball
* @author Matthias Jngel
* @author Martin Ltzsch
*/
class MSH2004BasicBehaviorGoToBall : public Xabsl2BasicBehavior, public BehaviorControlInterfaces
{
public:
/*
* Constructor.
* @param errorHandler Is invoked when errors occur
* @param interfaces The paramters of the BehaviorControl module.
  */
  MSH2004BasicBehaviorGoToBall(const BehaviorControlInterfaces& interfaces,
    Xabsl2ErrorHandler& errorHandler)
    : Xabsl2BasicBehavior("go-to-ball", errorHandler),
    BehaviorControlInterfaces(interfaces),
    accelerationRestrictor(motionRequest)
  {
    registerParameter("go-to-ball.distance",distanceAtEnd);
    registerParameter("go-to-ball.max-speed", maxSpeed);
  }
  
  /**  Executes the basic behavior. */
  virtual void execute();
  
private:
  /** parameter "go-to-ball.distance" */
  double distanceAtEnd;
  
  /** parameter "go-to-ball.max-speed" */
  double maxSpeed;
  
  /** Restricts the walk acceleration to maximum values */
  WalkAccelerationRestrictor accelerationRestrictor;
};

/**
* A Basic Behavior for going to the ball and reaching it with certain distance (x and y relative to robot) and direction
* @author Uwe Dffert
* @author Matthias Jngel
* @author Martin Ltzsch
*/
class MSH2004BasicBehaviorGoToBallWithDirection : public Xabsl2BasicBehavior, public BehaviorControlInterfaces
{
public:
/*
* Constructor.
* @param errorHandler Is invoked when errors occur
* @param interfaces The paramters of the BehaviorControl module.
  */
  MSH2004BasicBehaviorGoToBallWithDirection(const BehaviorControlInterfaces& interfaces,
    Xabsl2ErrorHandler& errorHandler)
    : Xabsl2BasicBehavior("go-to-ball-with-direction", errorHandler),
    BehaviorControlInterfaces(interfaces),
    accelerationRestrictor(motionRequest)
  {
    registerParameter("go-to-ball-with-direction.distance.x",distanceX);
    registerParameter("go-to-ball-with-direction.distance.y",distanceY);
    registerParameter("go-to-ball-with-direction.max-speed", maxSpeed);
    registerParameter("go-to-ball-with-direction.direction", direction);
  }
  
  /**  Executes the basic behavior. */
  virtual void execute();
  
private:
  /** parameter "go-to-ball-with-direction.distance.x" */
  double distanceX;
  /** parameter "go-to-ball-with-direction.distance.y" */
  double distanceY;
  /** parameter "go-to-ball-with-direction.max-speed" */
  double maxSpeed;
  /** parameter "go-to-ball-with-direction.direction" */
  double direction;
  
  /** Restricts the walk acceleration to maximum values */
  WalkAccelerationRestrictor accelerationRestrictor;
};


/**
* A Basic Behavior for avoiding obstacles
* @author Matthias Jngel
*/
class MSH2004BasicBehaviorAvoidObstacles : public Xabsl2BasicBehavior, public BehaviorControlInterfaces
{
public:
/*
* Constructor.
* @param errorHandler Is invoked when errors occur
* @param interfaces The paramters of the BehaviorControl module.
  */
  MSH2004BasicBehaviorAvoidObstacles(const BehaviorControlInterfaces& interfaces,
    Xabsl2ErrorHandler& errorHandler)
    : Xabsl2BasicBehavior("avoid-obstacles", errorHandler),
    BehaviorControlInterfaces(interfaces)
  {
    registerParameter("avoid-obstacles.turn-distance",turnDistance);
    registerParameter("avoid-obstacles.stop-distance",stopDistance);
    registerParameter("avoid-obstacles.walk-speed", walkSpeed);
    registerParameter("avoid-obstacles.turn-speed", turnSpeed);
  }
  
  /**  Executes the basic behavior. */
  virtual void execute();
  
private:
  /** parameter "avoid-obstacles.turn-distance" */
  double turnDistance;
  
  /** parameter "avoid-obstacles.stop-distance" */
  double stopDistance;
  
  /** parameter "avoid-obstacles.walk-speed" */
  double walkSpeed;
  
  /** parameter "avoid-obstacles.turn-speed" */
  double turnSpeed;
};

/**
* A Basic Behavior for going to a point on the field
* @author Uwe Dffert
* @author Martin Ltzsch
*/
class MSH2004BasicBehaviorGoToPoint : public Xabsl2BasicBehavior, public BehaviorControlInterfaces
{
public:
/*
* Constructor.
* @param errorHandler Is invoked when errors occur
* @param interfaces The paramters of the BehaviorControl module.
  */
  MSH2004BasicBehaviorGoToPoint(const BehaviorControlInterfaces& interfaces,
    Xabsl2ErrorHandler& errorHandler)
    : Xabsl2BasicBehavior("go-to-point", errorHandler),
    BehaviorControlInterfaces(interfaces),
    accelerationRestrictor(motionRequest)
  {
    registerParameter("go-to-point.x", x);
    registerParameter("go-to-point.y", y);
    registerParameter("go-to-point.destination-angle", destinationAngle);
    registerParameter("go-to-point.max-speed", maxSpeed);
  }
  
  /**  Executes the basic behavior. */
  virtual void execute();
  
protected:
  /** parameter "go-to-point.x" */
  double x;
  
  /** parameter "go-to-point.y" */
  double y;
  
  /** parameter "go-to-point.destination-angle" */
  double destinationAngle;
  
  /** parameter "go-to-point.max-speed" */
  double maxSpeed;
  
  /** Restricts the walk acceleration to maximum values */
  WalkAccelerationRestrictor accelerationRestrictor;

  friend class MSH2004BasicBehaviorGetBehindBall;
};

/**
* A Basic Behavior for precisely going to a point on the field
* @author Uwe Dffert
* @author Martin Ltzsch
* @author Thomas Rfer
*/
class MSH2004BasicBehaviorGoToPointPrecisely : public Xabsl2BasicBehavior, public BehaviorControlInterfaces
{
public:
/*
* Constructor.
* @param errorHandler Is invoked when errors occur
* @param interfaces The paramters of the BehaviorControl module.
  */
  MSH2004BasicBehaviorGoToPointPrecisely(const BehaviorControlInterfaces& interfaces,
    Xabsl2ErrorHandler& errorHandler)
    : Xabsl2BasicBehavior("go-to-point-precisely", errorHandler),
    BehaviorControlInterfaces(interfaces),
    accelerationRestrictor(motionRequest)
  {
    registerParameter("go-to-point-precisely.x", x);
    registerParameter("go-to-point-precisely.y", y);
    registerParameter("go-to-point-precisely.destination-angle", destinationAngle);
    registerParameter("go-to-point-precisely.max-speed", maxSpeed);
  }
  
  /**  Executes the basic behavior. */
  virtual void execute();
  
protected:
  /** parameter "go-to-point-precisely.x" */
  double x;
  
  /** parameter "go-to-point-precisely.y" */
  double y;
  
  /** parameter "go-to-point-precisely.destination-angle" */
  double destinationAngle;
  
  /** parameter "go-to-point-precisely.max-speed" */
  double maxSpeed;
  
  /** Restricts the walk acceleration to maximum values */
  WalkAccelerationRestrictor accelerationRestrictor;
};

/**
* A Basic Behavior for going to a point on the field and avoiding obstacles on the way there.
* @author Jan Hoffmann
*/
class MSH2004BasicBehaviorGoToPointAndAvoidObstacles : public Xabsl2BasicBehavior, public BehaviorControlInterfaces
{
public:
/*
* Constructor.
* @param errorHandler Is invoked when errors occur
* @param interfaces The paramters of the BehaviorControl module.
  */
  MSH2004BasicBehaviorGoToPointAndAvoidObstacles(const BehaviorControlInterfaces& interfaces,
    Xabsl2ErrorHandler& errorHandler)
    : Xabsl2BasicBehavior("go-to-point-and-avoid-obstacles", errorHandler),
    BehaviorControlInterfaces(interfaces),
	  speedPhi(0, .1, 0, 0, -pi_4, pi_4, pi/10),
	  speedX(0, .6, 0, 0, -80, 200, 100),
	  speedY(0, .1, 0, 0, -100, 100, 100),
		goToDestCounter(0),
		stickToBackOffDirection(0)
  {
    registerParameter("go-to-point-and-avoid-obstacles.x", x);
    registerParameter("go-to-point-and-avoid-obstacles.y", y);
    registerParameter("go-to-point-and-avoid-obstacles.max-speed", maxSpeed);
  }
  
  /**  Executes the basic behavior. */
  virtual void execute();
  
private:
  /** parameter "go-to-point-and-avoid-obstacles.x" */
  double x;
  
  /** parameter "go-to-point-and-avoid-obstacles.y" */
  double y;
  
  /** parameter "go-to-point-and-avoid-obstacles.max-speed" */
  double maxSpeed;
    
  /** smoothes the rotation */  
  PIDsmoothedValue speedPhi;

  /** smoothes the speed in forward direction */  
  PIDsmoothedValue speedX;

  /** smoothes the speed in sideways direction */  
  PIDsmoothedValue speedY;

  /** history that holds information in which direction the robot is/was turning */
  double turnDirection;

  /** history that holds information in which direction (left/right) the robot is/was moving  */
  double sideways;

	/** parameter "goToDestCounter" used for hysteresis*/
  long goToDestCounter;

	/** parameter "go-to-point-avoiding" */
  double backOffAngle;

	/** */
	long stickToBackOffDirection;
};


/**
* A Basic Behavior for getting behind the ball
* @author Max Risler
* @author Martin Ltzsch
*/
class MSH2004BasicBehaviorGetBehindBall : public Xabsl2BasicBehavior, public BehaviorControlInterfaces
{
public:
/*
* Constructor.
* @param errorHandler Is invoked when errors occur
* @param interfaces The paramters of the BehaviorControl module.
  */
  MSH2004BasicBehaviorGetBehindBall(const BehaviorControlInterfaces& interfaces,
    Xabsl2ErrorHandler& errorHandler)
    : Xabsl2BasicBehavior("get-behind-ball", errorHandler),
    BehaviorControlInterfaces(interfaces),
    accelerationRestrictor(motionRequest)
  {
    registerParameter("get-behind-ball.distance",distance);
    registerParameter("get-behind-ball.left-right", leftRight);
    registerParameter("get-behind-ball.max-speed", maxSpeed);
    registerParameter("get-behind-ball.angle", angle);
  }
  
  /**  Executes the basic behavior. */
  virtual void execute();
  
private:
  /** parameter "get-behind-ball.distance" */
  double distance;
  
  /** parameter "get-behind-ball.left-right" */
  double leftRight;
  
  /** parameter "get-behind-ball.max-speed" */
  double maxSpeed;
  
  /** parameter "get-behind-ball.angle" */
  double angle;
  
  /** Restricts the walk acceleration to maximum values */
  WalkAccelerationRestrictor accelerationRestrictor;
  
  /** Walks to a point */
  void gotoPoint(double x, double y, double maxSpeed);
};

/**
* A Basic Behavior for dribbling the ball
* @author Arthur Cesarz
*/
class MSH2004BasicBehaviorDribbleBall : public Xabsl2BasicBehavior, public BehaviorControlInterfaces
{
public:
/*
 * Constructor.
 * @param errorHandler Is invoked when errors occure
 * @param interfaces The paramters of the BehaviorControl module.
 */
  MSH2004BasicBehaviorDribbleBall(const BehaviorControlInterfaces& interfaces,
    Xabsl2ErrorHandler& errorHandler)
    : Xabsl2BasicBehavior("dribble-ball", errorHandler),
    BehaviorControlInterfaces(interfaces),
    accelerationRestrictor(motionRequest)
  {
    registerParameter("dribble-ball.x", targetPos.x);
    registerParameter("dribble-ball.y", targetPos.y);
    registerParameter("dribble-ball.destination-angle", destinationAngle);
    registerParameter("dribble-ball.avoidance-radius", avoidanceRadius);
    registerParameter("dribble-ball.max-speed", maxSpeed);
  }
  
  /**  Executes the basic behavior. */
  virtual void execute();

  void calcTangent( Vector2<double> &tangentPoint,
                    const Vector2<double> &circlePos, 
                    const double &circleRadius,
                    const double &leftRight);
  
  void setWalkingParameters();

protected:
  /** parameter "dribble-ball.x" */
  /** parameter "dribble-ball.y" */
  Vector2<double> targetPos;
  
  /** parameter "dribble-ball.destination-angle" */
  double destinationAngle;

  /** parameter "dribble-ball.avoidance-radius" */
  double avoidanceRadius;

  /** parameter "dribble-ball.max-speed" */
  double maxSpeed;
  
  /** Restricts the walk acceleration to maximum values */
  WalkAccelerationRestrictor accelerationRestrictor;

  double robotDistanceToCenterOfCircle, relXDistToTarget, relYDistToTarget;
  Vector2<double> walkingDirection;
  Vector2<double> mainCirclePos, rightCirclePos, 
                  leftCirclePos, actCirclePos, tangentPoint;

};

/**
* A Basic Behavior for going turning around a point on the field
* @author Jan Hoffmann
*/
class MSH2004BasicBehaviorTurnAroundPoint : public Xabsl2BasicBehavior, public BehaviorControlInterfaces
{
public:
/*
* Constructor.
* @param errorHandler Is invoked when errors occur
* @param interfaces The paramters of the BehaviorControl module.
  */
  MSH2004BasicBehaviorTurnAroundPoint(const BehaviorControlInterfaces& interfaces,
    Xabsl2ErrorHandler& errorHandler)
    : Xabsl2BasicBehavior("turn-around-point", errorHandler),
    BehaviorControlInterfaces(interfaces),
    accelerationRestrictor(motionRequest)
  {
    registerParameter("turn-around-point.x", x);
    registerParameter("turn-around-point.y", y);
    registerParameter("turn-around-point.radius", radius);
    registerParameter("turn-around-point.left-right", leftRight);
    registerParameter("turn-around-point.forward-component", forwardComponent);
  }
  
  /**  Executes the basic behavior. */
  virtual void execute();
  
protected:
  /** parameter "turn-around-point.x" */
  double x;
  
  /** parameter "turn-around-point.y" */
  double y;
  
  /** parameter "turn-around-point.radius" */
  double radius;
  
  /** parameter "go-to-point.left-right" */
  double leftRight;
  
  /** parameter "go-to-point.forward-component" */
  double forwardComponent;

  /** Restricts the walk acceleration to maximum values */
  WalkAccelerationRestrictor accelerationRestrictor;
};

/**
* A Basic Behavior for going to a point on the field and avoiding obstacles on the way there.
* @author 
*/
class MSH2004BasicBehaviorMoveTheBridge : public Xabsl2BasicBehavior, public BehaviorControlInterfaces
{
public:
/*
* Constructor.
* @param errorHandler Is invoked when errors occur
* @param interfaces The paramters of the BehaviorControl module.
  */
  MSH2004BasicBehaviorMoveTheBridge(const BehaviorControlInterfaces& interfaces,
    Xabsl2ErrorHandler& errorHandler)
    : Xabsl2BasicBehavior("move-the-bridge", errorHandler),
    BehaviorControlInterfaces(interfaces)
  {
    registerParameter("move-the-bridge.type", type);
    registerParameter("move-the-bridge.x", x);
    registerParameter("move-the-bridge.y", y);
    registerParameter("move-the-bridge.rotation", rotation);

    sax.reset(30);
    say.reset(30);
    sarotation.reset(30);
  }
  
  /**  Executes the basic behavior. */
  virtual void execute();
  
private:
  /** parameter "move-the-bridge.x" */
  double x;
  SlidingAverageValue sax;  
  
  /** parameter "move-the-bridge.y" */
  double y;
  SlidingAverageValue say;
  
  /** parameter "move-the-bridge.type" */
  double type;
  
  /** parameter "move-the-bridge.rotation" */
  double rotation;
  SlidingAverageValue sarotation;
  
};


/**
* @class MSH2004SimpleBasicBehaviors
*
* Creates and registers simple basic behaviors
*/
class MSH2004SimpleBasicBehaviors : public BehaviorControlInterfaces
{
public:
  /**
  * Constructor
  */
  MSH2004SimpleBasicBehaviors(const BehaviorControlInterfaces& interfaces,
                       Xabsl2ErrorHandler& errorHandler)
    : BehaviorControlInterfaces(interfaces),
      errorHandler(errorHandler),
      basicBehaviorAvoidObstacles(interfaces,errorHandler),
      basicBehaviorGetBehindBall(interfaces,errorHandler),
      basicBehaviorGoToBall(interfaces,errorHandler),
      basicBehaviorGoToBallWithDirection(interfaces,errorHandler),
      basicBehaviorGoToPoint(interfaces,errorHandler),
      basicBehaviorGoToPointPrecisely(interfaces,errorHandler),
      basicBehaviorDribbleBall(interfaces,errorHandler),
      basicBehaviorGoToPointAndAvoidObstacles(interfaces,errorHandler),
      basicBehaviorMoveTheBridge(interfaces,errorHandler),
      basicBehaviorTurnAroundPoint(interfaces,errorHandler)
  {}

  /** Registers basic behaviors at the engine */
  void registerBasicBehaviors(Xabsl2Engine& engine);

private:
  /** Is invoked when errors occurs */
  Xabsl2ErrorHandler& errorHandler;

  //!@name Basic Behaviors
  //!@{
  MSH2004BasicBehaviorAvoidObstacles  basicBehaviorAvoidObstacles;  
  MSH2004BasicBehaviorGetBehindBall  basicBehaviorGetBehindBall;
  MSH2004BasicBehaviorGoToBall  basicBehaviorGoToBall;
  MSH2004BasicBehaviorGoToBallWithDirection  basicBehaviorGoToBallWithDirection;
  MSH2004BasicBehaviorGoToPoint  basicBehaviorGoToPoint;  
  MSH2004BasicBehaviorGoToPointPrecisely  basicBehaviorGoToPointPrecisely;  
  MSH2004BasicBehaviorDribbleBall  basicBehaviorDribbleBall;  
  MSH2004BasicBehaviorGoToPointAndAvoidObstacles  basicBehaviorGoToPointAndAvoidObstacles;
  MSH2004BasicBehaviorTurnAroundPoint basicBehaviorTurnAroundPoint;  
  MSH2004BasicBehaviorMoveTheBridge basicBehaviorMoveTheBridge;
  //!@}
};

#endif // __MSH2004SimpleBasicBehaviors_h_

/*
* Change Log
* 
* $Log: MSH2004SimpleBasicBehaviors.h,v $
* Revision 1.11  2004/06/23 03:07:01  kerdels
* adjusted some positions, smoothed walking values, etc...
*
* Revision 1.10  2004/06/04 03:06:38  hamerla
* OpenChallenge
*
* Revision 1.9  2004/05/12 19:33:12  kerdels
* merged the behavior changes during australian, american and japan open
*
* Revision 1.9  2004/04/30 18:55:38  pg_joke
* merged the berlin turn-around-point behavior
*
* Revision 1.8  2004/03/08 01:07:01  roefer
* Interfaces should be const
*
* Revision 1.7  2003/12/08 13:46:43  cesarz
* no message
*
* Revision 1.6  2003/12/02 13:46:39  cesarz
* added basic behavior dribble-ball
*
* Revision 1.5  2003/10/31 08:32:49  dueffert
* doxygen bugs fixed
*
* Revision 1.4  2003/10/28 14:04:50  dueffert
* unused evolution removed
*
* Revision 1.3  2003/10/28 13:57:46  dueffert
* unused evolution removed
*
* Revision 1.2  2003/10/28 13:25:52  dueffert
* spelling improved
*
* Revision 1.1  2003/10/26 22:49:40  loetzsch
* created ATH2004BehaviorControl from GT2003BehaviorControl
*  - strongly simplified option graph
*  - moved some symbols from GT2003 to CommonXabsl2Symbols
*  - moved some basic behaviors from GT2003 to CommonXabsl2BasicBehaviors
*
* cloned ATH2004 three times (BB2004, DDD2004, MSH2004)
*
*/

