/** 
* @file MSH2004SimpleBasicBehaviors.cpp
*
* Implementation of basic behaviors defined in simple-basic-behaviors.xml.
*
* @author Uwe Dffert
* @author Jan Hoffmann
* @author Matthias Jngel
* @author Martin Ltzsch
* @author Max Risler
*/

#include "Tools/Math/PIDsmoothedValue.h"
#include "MSH2004SimpleBasicBehaviors.h"
#include "Tools/Math/Geometry.h"
#include "Tools/Math/Common.h"

#include "Tools/Debugging/Debugging.h"
#include "Tools/Debugging/DebugDrawings.h"

void MSH2004SimpleBasicBehaviors::registerBasicBehaviors(Xabsl2Engine& engine)
{
  engine.registerBasicBehavior(basicBehaviorAvoidObstacles);
  engine.registerBasicBehavior(basicBehaviorGetBehindBall);
  engine.registerBasicBehavior(basicBehaviorGoToBall);
  engine.registerBasicBehavior(basicBehaviorGoToBallWithDirection);
  engine.registerBasicBehavior(basicBehaviorDribbleBall);
  engine.registerBasicBehavior(basicBehaviorGoToPoint);
  engine.registerBasicBehavior(basicBehaviorGoToPointPrecisely);
  engine.registerBasicBehavior(basicBehaviorGoToPointAndAvoidObstacles);
  engine.registerBasicBehavior(basicBehaviorTurnAroundPoint);
  engine.registerBasicBehavior(basicBehaviorMoveTheBridge);
}

void MSH2004BasicBehaviorGoToBallWithDirection::execute()
{
  //accelerationRestrictor.saveLastWalkParameters();
  
  if (maxSpeed == 0)
  {
    maxSpeed = 200;
  }
  
  Vector2<double> ball = ballPosition.getKnownPosition(
    BallModel::behaviorControlTimeAfterWhichCommunicatedBallsAreAccepted);
  
  Vector2<double> pointToReach(ball.x-cos(direction)*distanceX-sin(direction)*distanceY,
    ball.y-sin(direction)*distanceX-cos(direction)*distanceY);
  

  double destinationDistance = Geometry::distanceTo(robotPose.getPose(),pointToReach);
  double destinationX = destinationDistance * cos(Geometry::angleTo(robotPose.getPose(),pointToReach));
  double destinationY = destinationDistance * sin(Geometry::angleTo(robotPose.getPose(),pointToReach));

/*
  Vector2<double> pointToReachMirrored(ball.x-cos(direction)*distanceX+sin(direction)*distanceY,
    ball.y-sin(direction)*distanceX+cos(direction)*distanceY);

  double destinationDistanceMirrored = Geometry::distanceTo(robotPose.getPose(),pointToReach);
	double destinationYMirrored = destinationDistanceMirrored * sin(Geometry::angleTo(robotPose.getPose(),pointToReachMirrored));
 
	if (fabs(destinationYMirrored) < fabs(destinationY))
		{
		destinationY = destinationYMirrored;
		}
*/

  if(destinationDistance < 1)
  {
    destinationDistance = 1;
  }
  
  double minimalTime=destinationDistance/maxSpeed;
  
  double angleDiff=direction-robotPose.rotation;
  if (angleDiff<-pi)
  {
    angleDiff += pi2;
  }
  else if (angleDiff >=pi)
  {
    angleDiff -= pi2;
  }
  
  double maxTurnSpeed = fromDegrees(60);
  double turnTime = angleDiff/maxTurnSpeed;
  if (turnTime>minimalTime)
  {
    minimalTime=turnTime;
  }
  minimalTime += 0.2; //get slower near to destination
  
  motionRequest.motionType = MotionRequest::walk;
  motionRequest.walkType = MotionRequest::normal;
  
  motionRequest.walkParams.translation.x = destinationX / minimalTime;
  motionRequest.walkParams.translation.y = destinationY / minimalTime;
  
  motionRequest.walkParams.rotation = 0.5*angleDiff;
  
  //~ if (motionRequest.walkParams.rotation > maxTurnSpeed)
    //~ motionRequest.walkParams.rotation = maxTurnSpeed;
  //~ if (motionRequest.walkParams.rotation < -maxTurnSpeed) 
    //~ motionRequest.walkParams.rotation = -maxTurnSpeed;
  
  if ((fabs(toDegrees(angleDiff))<10)&&(destinationDistance<10))
  {
    motionRequest.motionType = MotionRequest::stand;
  }
}

void MSH2004BasicBehaviorMoveTheBridge::execute()
{
  outgoingBehaviorTeamMessage.walkRequest.type = type;
  sax.add((int)(x*1000));
  sax.trigger();
  outgoingBehaviorTeamMessage.walkRequest.x = (double)(sax.getAverage()) / 1000.0;
  if (outgoingBehaviorTeamMessage.walkRequest.x > 50)
    outgoingBehaviorTeamMessage.walkRequest.x = 50;
  if (outgoingBehaviorTeamMessage.walkRequest.x < -50)
    outgoingBehaviorTeamMessage.walkRequest.x = -50;

  say.add((int)(y*1000));
  say.trigger();
  outgoingBehaviorTeamMessage.walkRequest.y = (double)(say.getAverage()) / 1000.0;
  if (outgoingBehaviorTeamMessage.walkRequest.y > 50)
    outgoingBehaviorTeamMessage.walkRequest.y = 50;
  if (outgoingBehaviorTeamMessage.walkRequest.y < -50)
    outgoingBehaviorTeamMessage.walkRequest.y = -50;

  sarotation.add((int)(rotation*1000));
  sarotation.trigger();
  outgoingBehaviorTeamMessage.walkRequest.rotation = (double)(sarotation.getAverage()) / 1000.0;
  if (outgoingBehaviorTeamMessage.walkRequest.rotation > 2)
    outgoingBehaviorTeamMessage.walkRequest.rotation = 2;
  if (outgoingBehaviorTeamMessage.walkRequest.rotation < -2)
    outgoingBehaviorTeamMessage.walkRequest.rotation = -2;

  // stand and send
  motionRequest.motionType = MotionRequest::stand;
}

void MSH2004BasicBehaviorGoToBall::execute()
{
  //accelerationRestrictor.saveLastWalkParameters();

  if (maxSpeed == 0) maxSpeed = 200;

  double maxSpeedX = maxSpeed;
  double maxSpeedY = maxSpeed;

  double maxTurnSpeed = fromDegrees(180);

  double timeToReachX, timeToReachY, timeToTurnToAngle, estimatedTimeToReachDestination;

  Vector2<double> ball = ballPosition.getKnownPosition(
    BallModel::behaviorControlTimeAfterWhichCommunicatedBallsAreAccepted);
  double ballDistance = Geometry::distanceTo(robotPose.getPose(),ball);
  double ballX = ballDistance * cos(Geometry::angleTo(robotPose.getPose(),ball)) + 50;
  double ballY = ballDistance * sin(Geometry::angleTo(robotPose.getPose(),ball));

  if(ballDistance == 0)
    ballDistance = 1;
  double ballAngle = Geometry::angleTo(robotPose.getPose(),ball);

  double diffX = distanceAtEnd / ballDistance * ballX;
  double diffY = distanceAtEnd / ballDistance * ballY;

  Vector2<double> destination(ballX - diffX, ballY - diffY);
  double distanceToDestination;
  distanceToDestination = fabs(ballDistance - distanceAtEnd);

  timeToReachX = fabs(destination.x) / maxSpeedX;
  timeToReachY = fabs(destination.y) / maxSpeedY;
  timeToTurnToAngle = ballAngle / maxTurnSpeed;
  estimatedTimeToReachDestination = 
    (timeToReachX < timeToReachY) ? timeToReachY : timeToReachX;
  estimatedTimeToReachDestination = 
    (estimatedTimeToReachDestination < timeToTurnToAngle) ? timeToTurnToAngle : estimatedTimeToReachDestination;

  motionRequest.motionType = MotionRequest::walk;
  motionRequest.walkType = MotionRequest::normal;

  motionRequest.walkParams.translation.x = 
    destination.x / estimatedTimeToReachDestination;
  motionRequest.walkParams.translation.y = 
    destination.y / estimatedTimeToReachDestination;

  if (distanceToDestination<135) // 200 //150
  {
  motionRequest.walkParams.translation.x *= 0.5; //0.6
  motionRequest.walkParams.translation.y *= 0.4; // 0.8
  }
  else 
  {
  motionRequest.walkParams.translation.y *= 0.4; // 0.8
  }

  motionRequest.walkParams.rotation = ballAngle*0.9;// *0.5;

  //~ if (motionRequest.walkParams.rotation > maxTurnSpeed)
    //~ motionRequest.walkParams.rotation = maxTurnSpeed;
  //~ if (motionRequest.walkParams.rotation < -maxTurnSpeed) 
    //~ motionRequest.walkParams.rotation = -maxTurnSpeed;

/*
  if ((fabs(toDegrees(ballAngle))<10)&&(distanceToDestination<20))
  {
    motionRequest.motionType = MotionRequest::stand;
  }
  else
  {
//    accelerationRestrictor.restrictAccelerations(250,250,180);
  }
*/

}

void MSH2004BasicBehaviorGetBehindBall::execute()
{
  // This was adapted from the XabslSkillGetBehindBall written by Max Risler.

  accelerationRestrictor.saveLastWalkParameters();

  double dir = fromDegrees(angle); 

  Vector2<double> ball = ballPosition.getKnownPosition(
    BallModel::behaviorControlTimeAfterWhichCommunicatedBallsAreAccepted);

  double destX = ball.x;
  double destY = ball.y;

  double posX = robotPose.getPose().translation.x;
  double posY = robotPose.getPose().translation.y;
  double posDir = robotPose.getPose().getAngle();

  Vector2<double> destPosDiff = ball - robotPose.getPose().translation;

  double absDestDir = atan2(destPosDiff.y, destPosDiff.x);
  double relDestDir = normalize(absDestDir - posDir);
  double destDist = destPosDiff.abs();

  if (destDist > 1000.0)
  {
    gotoPoint(destX,destY,maxSpeed);
  }
  else if (destDist >distance* 1.2)
  {
    // calculate destination point
    double px,py;
  
    px = destX -distance* cos(dir);
    py = destY -distance* sin(dir);
  
    double distToP = sqrt((posX-px)*(posX-px) + (posY-py)*(posY-py));
  
    if (destDist < distToP)
    {
      // dest is closer than destination point
      // walk to tangent
      double a = acos(distance/destDist);
    
      if ((leftRight == 0 && normalize(dir - absDestDir) > 0) ||
        leftRight > 0)
      {
        px = destX -distance* cos(absDestDir + a);
        py = destY -distance* sin(absDestDir + a);
      } else {
        px = destX -distance* cos(absDestDir - a);
        py = destY -distance* sin(absDestDir - a);
      }
      // distance is distance to tangent plus distance on perimeter
      distToP = sqrt((posX-px)*(posX-px) + (posY-py)*(posY-py));
      distToP +=distance* normalize(fabs((dir - absDestDir)));
    }
    gotoPoint(px,py,maxSpeed);
  }
  else
  {
    // circle around dest
    double diffAngle = normalize(posDir-dir);
    if (diffAngle > pi_2 && leftRight > 0) diffAngle -= pi2;
    if (diffAngle < -pi_2 && leftRight < 0) diffAngle += pi2;
  
    motionRequest.walkParams.translation.x = destDist - distance;
    motionRequest.walkParams.translation.y = diffAngle * distance;
    motionRequest.walkParams.rotation = relDestDir*0.5;
  }

    motionRequest.motionType = MotionRequest::walk;
    motionRequest.walkType = MotionRequest::normal;

  accelerationRestrictor.restrictAccelerations(300,300,180);
}

void MSH2004BasicBehaviorGetBehindBall::gotoPoint(double x, double y, double maxSpeed)
{
  //~ double maxTurnSpeed = fromDegrees(60);

  double timeToReachX, timeToReachY, estimatedTimeToReachDestination;
  double distance = Geometry::distanceTo(robotPose.getPose(),Vector2<double>(x,y));
  double distanceX = distance
    * cos(Geometry::angleTo(robotPose.getPose(),Vector2<double>(x,y)));
  double distanceY = distance
    * sin(Geometry::angleTo(robotPose.getPose(),Vector2<double>(x,y)));

  if(distance == 0)
    distance = 1;
  double angle = Geometry::angleTo(robotPose.getPose(),ballPosition.getKnownPosition(
    BallModel::behaviorControlTimeAfterWhichCommunicatedBallsAreAccepted));

  Vector2<double> destination(distanceX, distanceY);

  timeToReachX = fabs(destination.x) / maxSpeed;
  timeToReachY = fabs(destination.y) / maxSpeed;

  estimatedTimeToReachDestination = 
    (timeToReachX < timeToReachY) ? timeToReachY : timeToReachX;

  motionRequest.motionType = MotionRequest::walk;
  motionRequest.walkType = MotionRequest::normal;

  motionRequest.walkParams.translation.x = 
    destination.x / estimatedTimeToReachDestination;
  motionRequest.walkParams.translation.y = 
    destination.y / estimatedTimeToReachDestination;

  motionRequest.walkParams.translation.y *= 0.6; 

  motionRequest.walkParams.rotation = 0.5*angle;

  //~ if (motionRequest.walkParams.rotation > maxTurnSpeed)
    //~ motionRequest.walkParams.rotation = maxTurnSpeed;
  //~ if (motionRequest.walkParams.rotation < -maxTurnSpeed) 
    //~ motionRequest.walkParams.rotation = -maxTurnSpeed;
}



void MSH2004BasicBehaviorGoToPointAndAvoidObstacles::execute()
{
  double turnDistance = 400,
    angleToDestination = Geometry::angleTo(robotPose.getPose(),Vector2<double>(x,y)),
		distanceToDestination = Geometry::distanceTo(robotPose.getPose(),Vector2<double>(x,y));
	double targetX, targetAngle;//, obstacleAvoidAngle;

  if (maxSpeed == 0) maxSpeed = 200;
	
  targetX = obstaclesModel.getDistanceInCorridor(0.0, 300);
	targetX -= 300; 
	if (targetX > maxSpeed)
		targetX = maxSpeed;
	if (targetX < -50)
		targetX = -50;
	if (obstaclesModel.getDistanceInCorridor(pi, 300) < 300)
		targetX = 0;
	
	speedPhi.setWeightP(0.7);
	speedX.setWeightP(0.7);
	targetAngle = 0.0;

  
	if (targetX > .4 * maxSpeed)
  {

		double leftForward = obstaclesModel.getTotalFreeSpaceInSector(.4, .79, turnDistance);
		double rightForward = obstaclesModel.getTotalFreeSpaceInSector(-.4, .79, turnDistance);

		// go in the direction of the next free sector ...
		targetAngle = obstaclesModel.getAngleOfNextFreeSector(pi/4, angleToDestination, (int )turnDistance);
		// ... but check if there's enough space for walking
		if (obstaclesModel.getDistanceInCorridor(targetAngle, 300) < turnDistance)
			targetAngle =	2*(leftForward - rightForward) / (leftForward + rightForward);

		if (fabs(targetAngle) < .2)
		{
			// only do this if no obstacle avoidance was necessary in the 
			// past 1 sec
			if (goToDestCounter > 10)
			{
				targetAngle = angleToDestination;
				//OUTPUT(idText, text, "DESTINATION: " << targetAngle);
			}
			// destination within reach
			else if ((obstaclesModel.getDistanceInCorridor(angleToDestination, 300) > distanceToDestination-100) &&
				(goToDestCounter > 5))
			{
				targetAngle = angleToDestination;
				//OUTPUT(idText, text, "GO FOR THE DESTINATION");
			}
			goToDestCounter++;
			targetAngle = (double)sgn(targetAngle) * .5;
		}
		else 
		{
			if (fabs(angleToDestination) < pi/1.8)
			{
				goToDestCounter = 0;
				//OUTPUT(idText, text, "AVOID: " << targetAngle);
			}
			if (fabs(targetAngle) > .5)
				targetAngle = (double)sgn(targetAngle) * .5;
		}
	}
  else 
  {
		if (stickToBackOffDirection > 15)
		{
			if (backOffAngle < 0)
				backOffAngle = -pi;
			else
				backOffAngle = pi;
			stickToBackOffDirection = 0;
		}
		targetAngle = backOffAngle;
		goToDestCounter = -10;

//		OUTPUT(idText, text, "BACKING OFF " << targetAngle);
  }
	stickToBackOffDirection++;
			
	//OUTPUT(idText, text, "... " << targetAngle);
	backOffAngle = targetAngle;

	if(obstaclesModel.getDistanceInCorridor((double)sgn(targetAngle) * pi/2, 300) < 300)
		targetAngle = 0;  
	
	motionRequest.motionType = MotionRequest::walk;
  motionRequest.walkType = MotionRequest::normal;
  motionRequest.walkParams.translation.x = speedX.approximateVal(targetX);
  motionRequest.walkParams.translation.y = speedY.getVal();
  motionRequest.walkParams.rotation = speedPhi.approximateVal(targetAngle)*0.5;
}








void MSH2004BasicBehaviorAvoidObstacles::execute()
{
  if(turnDistance == 0) turnDistance = 300;
  if(walkSpeed == 0) walkSpeed = 250;
  if(turnSpeed == 0) turnSpeed = 1.2;

  double distanceToObstacle = 200;
  if(stopDistance) distanceToObstacle = stopDistance;

  static PIDsmoothedValue speedPhi(0, .1, 0, 0, -pi/4, pi/4, pi/10);
  static PIDsmoothedValue speedX(0, .1, 0, 0, -walkSpeed, walkSpeed, 100);
  static PIDsmoothedValue speedY(0, .1, 0, 0, -walkSpeed/2, walkSpeed/2, 100);
  static double turnDirection = 0;
  speedX.setMax(walkSpeed);

  speedX.approximateVal(obstaclesModel.corridorInFront - distanceToObstacle);

/*    
  // if something is in the way but far away
  // or it is close and turning was not set...
  if(((obstaclesModel.distance[ObstaclesModel::front] < turnDistance) && 
    (obstaclesModel.distance[ObstaclesModel::front] >= distanceToObstacle)) 
    ||
    ((obstaclesModel.distance[ObstaclesModel::front] < turnDistance) && 
    (turnDirection == 0)))
  {
    if(obstaclesModel.distance[ObstaclesModel::frontLeft] < obstaclesModel.distance[ObstaclesModel::frontRight])
      turnDirection = -turnSpeed;
    else 
      turnDirection = turnSpeed;
  }
  // if nothing is in the way, walk forward
  else if (obstaclesModel.distance[ObstaclesModel::front] > turnDistance) 
  {
  }
*/  
  turnDirection = 0;
  if(obstaclesModel.corridorInFront < turnDistance) 
  {
    if(obstaclesModel.getDistanceInMajorDirection(ObstaclesModel::frontLeft) < obstaclesModel.getDistanceInMajorDirection(ObstaclesModel::frontRight))
      speedY.approximateVal(-walkSpeed);
    else if (obstaclesModel.getDistanceInMajorDirection(ObstaclesModel::frontLeft) > obstaclesModel.getDistanceInMajorDirection(ObstaclesModel::frontLeft))
      speedY.approximateVal(walkSpeed);
    else if (obstaclesModel.getDistanceInMajorDirection(ObstaclesModel::frontLeft) == obstaclesModel.getDistanceInMajorDirection(ObstaclesModel::frontRight))
      speedY.approximateVal(0);
  }
  else  
    speedY.approximateVal(0);

  motionRequest.motionType = MotionRequest::walk;
  motionRequest.walkType = MotionRequest::normal;
  motionRequest.walkParams.translation.x = speedX.getVal();
  motionRequest.walkParams.translation.y = speedY.getVal();
  motionRequest.walkParams.rotation = speedPhi.approximateVal(turnDirection)*0.5;
}


void MSH2004BasicBehaviorGoToPoint::execute()
{
  accelerationRestrictor.saveLastWalkParameters();

  Vector2<double> destination(x,y);
  const Vector2<double>& self = robotPose.translation;

  if (maxSpeed==0) maxSpeed = 120;

  //~ double maxTurnSpeed = fromDegrees(40);

  double distanceToDestination = Geometry::distanceTo(self,destination);

  double angleDifference = normalize(fromDegrees(destinationAngle) - robotPose.rotation);

  double angleToDestination = Geometry::angleTo(robotPose,destination);


  motionRequest.motionType = MotionRequest::walk;
  motionRequest.walkType = MotionRequest::normal;

  //this time does not necessarily include time for turning!:
  double estimatedTimeToReachDestination;
  if (distanceToDestination > 200)
  {
    estimatedTimeToReachDestination = (distanceToDestination+200)/ maxSpeed;
  
    motionRequest.walkParams.translation.x = cos(angleToDestination) * maxSpeed;
    motionRequest.walkParams.translation.y = sin(angleToDestination) * maxSpeed;
  }
  else
  {
    estimatedTimeToReachDestination = 2*distanceToDestination / maxSpeed;
    if (distanceToDestination > 30)
    {
      motionRequest.walkParams.translation.x = 
        cos(angleToDestination) * maxSpeed*distanceToDestination/200;
      motionRequest.walkParams.translation.y = 
        sin(angleToDestination) * maxSpeed*distanceToDestination/200;
    }
    else
    {
      motionRequest.walkParams.translation.x = 0;
      motionRequest.walkParams.translation.y = 0;
    }
  }
  if (estimatedTimeToReachDestination==0)
  {
    estimatedTimeToReachDestination = 0.001;
  }

  //~ if (fabs(toDegrees(angleDifference)) > 20)
  //~ {
    //~ motionRequest.walkParams.rotation = 
      //~ 0.5 * angleDifference / estimatedTimeToReachDestination;
    //~ if (motionRequest.walkParams.rotation > maxTurnSpeed)
    //~ {
      //~ motionRequest.walkParams.rotation = maxTurnSpeed;
    //~ }
    //~ if (motionRequest.walkParams.rotation < -maxTurnSpeed) 
    //~ {
      //~ motionRequest.walkParams.rotation = -maxTurnSpeed;
    //~ }
  //~ }
  //~ else
  //~ {
    motionRequest.walkParams.rotation = 0.5*angleDifference;
  //~ }



  if ((fabs(toDegrees(angleDifference))<4)&&(distanceToDestination<25))
  {
    motionRequest.walkParams.translation.x = 0;
    motionRequest.walkParams.translation.y = 0;
    motionRequest.walkParams.rotation = 0;
  }

//  accelerationRestrictor.restrictAccelerations(250,250,200); //!!!!!!!!!!!

  if (fabs(motionRequest.walkParams.translation.x) < 10
    && fabs(motionRequest.walkParams.translation.y) < 10
    && fabs(motionRequest.walkParams.rotation) < fromDegrees(10)
    && fabs(distanceToDestination) < 40
    && fabs(angleToDestination) < fromDegrees(5))
  {
    motionRequest.motionType = MotionRequest::stand;
  }
}

void MSH2004BasicBehaviorGoToPointPrecisely::execute()
{
  Pose2D pose = robotPose + Pose2D(Vector2<double>(-105,0));

  accelerationRestrictor.saveLastWalkParameters();

  Vector2<double> destination = (Pose2D(fromDegrees(destinationAngle), Vector2<double>(x,y)) + Pose2D(Vector2<double>(-105,0))).translation;
  const Vector2<double>& self = pose.translation;

  if (maxSpeed==0) maxSpeed = 120;

  //~ double maxTurnSpeed = fromDegrees(40);

  double distanceToDestination = Geometry::distanceTo(self,destination);

  double angleDifference = normalize(fromDegrees(destinationAngle) - pose.rotation);

  double angleToDestination = Geometry::angleTo(pose,destination);


  motionRequest.motionType = MotionRequest::walk;
  motionRequest.walkType = MotionRequest::normal;

  //this time does not necessarily include time for turning!:
  double estimatedTimeToReachDestination;
  if (distanceToDestination > 200)
  {
    estimatedTimeToReachDestination = (distanceToDestination+200)/ maxSpeed;
  
    motionRequest.walkParams.translation.x = cos(angleToDestination) * maxSpeed;
    motionRequest.walkParams.translation.y = sin(angleToDestination) * maxSpeed;
  }
  else
  {
    estimatedTimeToReachDestination = 2*distanceToDestination / maxSpeed;
    motionRequest.walkParams.translation.x = 
      cos(angleToDestination) * maxSpeed*distanceToDestination/200;
    motionRequest.walkParams.translation.y = 
      sin(angleToDestination) * maxSpeed*distanceToDestination/200;
    const double minSpeed = 30;
    if(fabs(motionRequest.walkParams.translation.x) > fabs(motionRequest.walkParams.translation.y) &&
       fabs(motionRequest.walkParams.translation.x) < minSpeed)
    {
      motionRequest.walkParams.translation.x = motionRequest.walkParams.translation.x > 0 ? minSpeed : -minSpeed;
      motionRequest.walkParams.translation.x *= minSpeed / fabs(motionRequest.walkParams.translation.x);
    }
    else if(fabs(motionRequest.walkParams.translation.x) <= fabs(motionRequest.walkParams.translation.y) &&
            fabs(motionRequest.walkParams.translation.y) < minSpeed)
    {
      motionRequest.walkParams.translation.y = motionRequest.walkParams.translation.y > 0 ? minSpeed : -minSpeed;
      motionRequest.walkParams.translation.x *= minSpeed / fabs(motionRequest.walkParams.translation.y);
    }

  }
  if (estimatedTimeToReachDestination==0)
  {
    estimatedTimeToReachDestination = 0.001;
  }

  //~ if (fabs(toDegrees(angleDifference)) > 20)
  //~ {
    //~ motionRequest.walkParams.rotation = 
      //~ angleDifference / estimatedTimeToReachDestination;
    //~ if (motionRequest.walkParams.rotation > maxTurnSpeed)
    //~ {
      //~ motionRequest.walkParams.rotation = maxTurnSpeed;
    //~ }
    //~ if (motionRequest.walkParams.rotation < -maxTurnSpeed) 
    //~ {
      //~ motionRequest.walkParams.rotation = -maxTurnSpeed;
    //~ }
  //~ }
  //~ else
  //~ {
    //~ motionRequest.walkParams.rotation = 2*angleDifference;
  //~ }
  motionRequest.walkParams.rotation = 0.5*angleDifference;
  
  accelerationRestrictor.restrictAccelerations(150,150,100);

  if (fabs(motionRequest.walkParams.translation.x) < 10
    && fabs(motionRequest.walkParams.translation.y) < 10
    && fabs(motionRequest.walkParams.rotation) < fromDegrees(10)
    && fabs(distanceToDestination) < 40
    && fabs(angleToDestination) < fromDegrees(5))
  {
    motionRequest.motionType = MotionRequest::stand;
  }
}

void MSH2004BasicBehaviorDribbleBall::calcTangent( Vector2<double> &tangentPoint,
                                                 const Vector2<double> &circlePos, 
                                                 const double &circleRadius,
                                                 const double &leftRight )
{
  double angleDiff = Geometry::angleTo(robotPose.getPose().translation, circlePos);

  if (robotDistanceToCenterOfCircle > circleRadius) // robot outside the circle
  {
    double tangentAngle = leftRight * acos(circleRadius/robotDistanceToCenterOfCircle); 
    tangentPoint.x = circleRadius * cos( -pi + tangentAngle + angleDiff );
    tangentPoint.y = circleRadius * sin( -pi + tangentAngle + angleDiff );
    tangentPoint += circlePos;
  }
  else // calculate secant
  {
    double secantAngle = pi + angleDiff;
    tangentPoint.x = circleRadius * cos( secantAngle );
    tangentPoint.y = circleRadius * sin( secantAngle );
    tangentPoint += circlePos;
  }
}

void MSH2004BasicBehaviorDribbleBall::setWalkingParameters()
{
  double angleDiff =  Geometry::angleTo(robotPose.getPose(), walkingDirection);
  Vector2<double> walkTranslation;
  double walkRotation;

//  accelerationRestrictor.saveLastWalkParameters();

  walkTranslation.x = cos(angleDiff);
  walkTranslation.y = sin(angleDiff);
  
  walkTranslation.normalize(); 
  walkTranslation *= maxSpeed;

  if ( (targetPos - robotPose.translation).abs() < 2*avoidanceRadius )
  {
    walkRotation = 2.5 * toDegrees(Geometry::angleTo(robotPose.getPose(), mainCirclePos));
    walkTranslation *= 150;
    walkTranslation.y = walkTranslation.y > 100 ? 100 : (walkTranslation.y < -100 ? -100 : walkTranslation.y);
  }
  else
  {
    walkRotation = 1. * toDegrees(Geometry::angleTo(robotPose.getPose(), mainCirclePos));
    walkTranslation *= maxSpeed;
  }
  if ( (targetPos - robotPose.translation).abs() < 130.0 )
    walkTranslation.x = maxSpeed;

  motionRequest.motionType             = MotionRequest::walk;
  motionRequest.walkType               = MotionRequest::normal;
  motionRequest.walkParams.translation = walkTranslation;
  motionRequest.walkParams.rotation    = fromDegrees(walkRotation);
  //motionRequest.walkParams.translation.x = 0;

//  accelerationRestrictor.restrictAccelerations(170,90,200);
}


void MSH2004BasicBehaviorDribbleBall::execute()
{
  double actRadius, leftRight;
  const Vector2<double> &self = robotPose.translation;
  Vector2<double> dist;

  double destAngleRad = fromDegrees(destinationAngle);

  // destAngleRad = robotPose.rotation;

  // target point 40 mm in front of the ball ...
  mainCirclePos.x = targetPos.x + 20 * cos(destAngleRad);
  mainCirclePos.y = targetPos.y + 20 * sin(destAngleRad);
  

  CIRCLE(dribbleBall, mainCirclePos.x, mainCirclePos.y, avoidanceRadius, 1, Drawings::ps_solid, Drawings::gray );
  CIRCLE(dribbleBall, mainCirclePos.x, mainCirclePos.y, 20, 1, Drawings::ps_solid, Drawings::white );

  dist = mainCirclePos - self;

  relXDistToTarget = dist.abs() * -sin(destAngleRad + atan2(dist.x, dist.y));
  relYDistToTarget = dist.abs() * cos(destAngleRad + atan2(dist.x, dist.y));

  // ... and 75 mm besides the ball for dribbling
/*  if (relYDistToTarget > 0)
  {
    mainCirclePos.x = mainCirclePos.x - 75 * -sin(destAngleRad);
    mainCirclePos.y = mainCirclePos.y - 75 * cos(destAngleRad);
  }
  else
  {
    mainCirclePos.x = mainCirclePos.x + 75 * -sin(destAngleRad);
    mainCirclePos.y = mainCirclePos.y + 75 * cos(destAngleRad);
  }
*/
  // calculate circle and walkdirection for avoiding target point
  if( (relXDistToTarget>0) && (relYDistToTarget>0) && (relYDistToTarget<avoidanceRadius))
    {
      actCirclePos = mainCirclePos;
      actRadius    = avoidanceRadius;
      leftRight    = -1;
    }
  else if((relXDistToTarget>0) && (relYDistToTarget<0) && (relYDistToTarget>-avoidanceRadius))
    {
      actCirclePos = mainCirclePos;
      actRadius    = avoidanceRadius;
      leftRight    = 1;
    }
  else if( (relYDistToTarget>0) )
    {
      rightCirclePos.x = mainCirclePos.x - avoidanceRadius/2 * sin(-destAngleRad);
      rightCirclePos.y = mainCirclePos.y - avoidanceRadius/2 * cos(-destAngleRad);
      actCirclePos = rightCirclePos;
      actRadius    = avoidanceRadius/2;
      leftRight    = -1;
    }
  else
    {
      leftCirclePos.x = mainCirclePos.x + avoidanceRadius/2 * sin(-destAngleRad);
      leftCirclePos.y = mainCirclePos.y + avoidanceRadius/2 * cos(-destAngleRad);
      actCirclePos = leftCirclePos;
      actRadius    = avoidanceRadius/2;
      leftRight    = 1;
    }
    rightCirclePos.x = mainCirclePos.x - avoidanceRadius/2 * sin(-destAngleRad);
    rightCirclePos.y = mainCirclePos.y - avoidanceRadius/2 * cos(-destAngleRad);
    leftCirclePos.x = mainCirclePos.x + avoidanceRadius/2 * sin(-destAngleRad);
    leftCirclePos.y = mainCirclePos.y + avoidanceRadius/2 * cos(-destAngleRad);
    LINE(dribbleBall, rightCirclePos.x, rightCirclePos.y,  
       leftCirclePos.x, leftCirclePos.y, 1, Drawings::ps_solid, Drawings::white );
    CIRCLE(dribbleBall, rightCirclePos.x, rightCirclePos.y, avoidanceRadius/2, 1, Drawings::ps_solid, Drawings::red );
    CIRCLE(dribbleBall, leftCirclePos.x, leftCirclePos.y, avoidanceRadius/2, 1, Drawings::ps_solid, Drawings::red );
    LARGE_DOT( dribbleBall, mainCirclePos.x, 
       mainCirclePos.y, 2, Drawings::red );


  LARGE_DOT( dribbleBall, actCirclePos.x, 
       actCirclePos.y, 1, Drawings::white );

  robotDistanceToCenterOfCircle = Geometry::distance(robotPose.getPose().translation, actCirclePos);

  calcTangent(tangentPoint, actCirclePos, actRadius, leftRight);

  // if ball to near, dribble forward
/*  if (Geometry::distance(robotPose.getPose().translation, targetPos)<130)
  {
    //walk straight forward
    walkingDirection.x = targetPos.x + 200 * cos(Geometry::angleTo(robotPose.getPose(),targetPos));
    walkingDirection.y = targetPos.y + 200 * sin(Geometry::angleTo(robotPose.getPose(),targetPos));
    //ARROW(goToPoint2Path, self.x, self.y,  
    //    walkingDirection.x, walkingDirection.y, 1, Drawings::ps_solid, Drawings::red );
    maxSpeed = 250;
   //walkingDirection = mainCirclePos;
  }
  // calculation of direction if robot is inside the circle
  else*/ if (robotDistanceToCenterOfCircle < actRadius)
  {
    ARROW(dribbleBall, self.x, self.y,  
        tangentPoint.x, tangentPoint.y, 1, Drawings::ps_solid, Drawings::gray );
    walkingDirection = tangentPoint - self;
    (leftRight == 1) ? walkingDirection.rotateLeft() : walkingDirection.rotateRight();
    walkingDirection.normalize();
    walkingDirection *= robotDistanceToCenterOfCircle;
    walkingDirection += self;
    walkingDirection = (walkingDirection - self) * 0.2 + tangentPoint - self;
    walkingDirection += self;
  }
  else
  {
    walkingDirection = tangentPoint;
  }
  LINE(dribbleBall, self.x, self.y,  
       walkingDirection.x, walkingDirection.y, 2, Drawings::ps_solid, Drawings::white );

  ARROW(dribbleBall, mainCirclePos.x, mainCirclePos.y,  //target angle
     mainCirclePos.x + 150 * cos(destAngleRad), 
     mainCirclePos.y + 150 * sin(destAngleRad),  
     1, Drawings::ps_solid, Drawings::white );

  DEBUG_DRAWING_FINISHED(dribbleBall);

  setWalkingParameters();
}

void MSH2004BasicBehaviorTurnAroundPoint::execute()
{
  double maxTurnSpeed = fromDegrees(180);
  if (forwardComponent == 0) forwardComponent = 200.0;
  
  Vector2<double> destinationInWorldCoord(x,y);
  double angleToDestination = Geometry::angleTo(robotPose.getPose(),destinationInWorldCoord);
  double distance = Geometry::distanceTo(robotPose.getPose(),destinationInWorldCoord);

  // destination = ball position in robot 
  // coordintes plus 100mm in x direction
  Vector2<double> destination(
      distance * cos(angleToDestination), 
      distance * sin(angleToDestination));
  
  double factor = (distance - radius) / 800; 
  if (factor > 1) factor = 1;
  if (factor < 0) factor = 0;

  destination *= factor;
  destination.y += (leftRight > 0 ? 1 : -1)*(200 - forwardComponent/5)*(1-factor);

  if (destination.x < forwardComponent)   
    destination.x = forwardComponent;

  motionRequest.motionType = MotionRequest::walk;
  motionRequest.walkType = MotionRequest::normal;  
  motionRequest.walkParams.translation.x = destination.x;
  motionRequest.walkParams.translation.y = destination.y;
  motionRequest.walkParams.rotation = angleToDestination * 2;
  
  // clip rotation speed
  motionRequest.walkParams.rotation = min(motionRequest.walkParams.rotation, maxTurnSpeed);
  motionRequest.walkParams.rotation = max(motionRequest.walkParams.rotation, -maxTurnSpeed); 
}


/*
* Change Log
* 
* $Log: MSH2004SimpleBasicBehaviors.cpp,v $
* Revision 1.15  2004/06/23 03:07:01  kerdels
* adjusted some positions, smoothed walking values, etc...
*
* Revision 1.14  2004/06/22 18:35:13  hamerla
* OC version test di abend
*
* Revision 1.13  2004/06/04 03:06:38  hamerla
* OpenChallenge
*
* Revision 1.12  2004/05/12 19:33:12  kerdels
* merged the behavior changes during australian, american and japan open
*
* Revision 1.11  2004/05/04 10:48:59  loetzsch
* replaced all enums
* xxxBehaviorControlTimeAfterWhichCommunicatedBallsAreAccepted
* by
* behaviorControlTimeAfterWhichCommunicatedBallsAreAccepted
* (this mechanism was neither fully implemented nor used)
*
* Revision 1.10  2004/04/08 15:33:01  wachter
* GT04 checkin of Microsoft-Hellounds
*
* Revision 1.17  2004/04/04 07:01:05  nistico
* Smoothed movements gotoball and gotopoint
*
* Revision 1.16  2004/04/04 04:55:48  lohmann
* approaching and grabbing
*
* Revision 1.15  2004/04/03 11:03:55  lohmann
* Grab check
* approaching
* and anti-deadlock for kicking
*
* Revision 1.14  2004/03/31 16:36:12  pg_tilo
* Walters approaching
* kick distances fixed for new approaching
*
* Revision 1.13  2004/03/30 15:13:34  pg_arce
* changed dribble ball
*
* Revision 1.12  2004/03/30 09:52:38  pg_arce
* some changes in dribble ball basic behavior
*
* Revision 1.11  2004/03/26 11:42:23  pg_arce
* improved catch and turn with ball
*
* Revision 1.10  2004/03/22 10:42:49  pg_mane
* - removed some speed limits
*
* Revision 1.9  2004/03/16 14:08:18  nistico
* Replaced unnecessary strong non linearities with proper gain
* Restored working version of approach-close-ball
* Note: old code not stripped yet, extensive testing pending
*
* Revision 1.8  2004/03/15 13:44:29  cesarz
* Changed turning sensivity factor for goToPoint.
*
* Revision 1.7  2004/02/03 13:20:48  spranger
* renamed all references to  class BallPosition to BallModel (possibly changed include files)
*
* Revision 1.6  2003/12/09 16:29:06  jhoffman
* removed OUTPUTs
*
* Revision 1.5  2003/12/08 13:46:43  cesarz
* no message
*
* Revision 1.4  2003/12/02 13:46:39  cesarz
* added basic behavior dribble-ball
*
* Revision 1.3  2003/10/28 13:57:46  dueffert
* unused evolution removed
*
* Revision 1.2  2003/10/28 13:25:52  dueffert
* spelling improved
*
* Revision 1.1  2003/10/26 22:49:40  loetzsch
* created ATH2004BehaviorControl from GT2003BehaviorControl
*  - strongly simplified option graph
*  - moved some symbols from GT2003 to CommonXabsl2Symbols
*  - moved some basic behaviors from GT2003 to CommonXabsl2BasicBehaviors
*
* cloned ATH2004 three times (BB2004, DDD2004, MSH2004)
*
*/

