/** 
* @file MSH2004EvoBasicBehaviors.h
*
* Declaration of basic behaviors defined in evo-basic-behaviors.xml.
*
* @author Arthur Cesarz
* @author Matthias Hebbel
*/

#ifndef __MSH2004EvoBasicBehaviors_h_
#define __MSH2004EvoBasicBehaviors_h_

#include "../../BehaviorControl.h"
#include "Tools/Xabsl2/Xabsl2Engine/Xabsl2Engine.h"
#include "Tools/Xabsl2/Xabsl2Engine/Xabsl2BasicBehavior.h"
#include "Modules/WalkingEngine/InvKinWalkingEngine.h"
#include "Modules/WalkingEngine/InvKinWalkingParameterSets.h"

#define maxNumToShowIndividual 2
#define fitnessFactor 0.3


/**
* Basic Behaviors for evolving walking parameter sets
* @author Arthur Cesarz
* @author Matthias Hebbel
*/
class MSH2004AutonomousEvolution : public Xabsl2BasicBehavior, public BehaviorControlInterfaces
{
public:
/*
* Constructor.
* @param errorHandler Is invoked when errors occur
* @param interfaces The paramters of the BehaviorControl module.
  */
  MSH2004AutonomousEvolution(const BehaviorControlInterfaces& interfaces,
    Xabsl2ErrorHandler& errorHandler)
    : Xabsl2BasicBehavior("evo-go-to-point", errorHandler),
    BehaviorControlInterfaces(interfaces),
    numberOfMutations(0),
    numberOfSuccessfulMutations(0),
    numberOfGenerations(0),
    numberOfEvolveValues(0),
    tempNumOfIndividual(0),
    mutationStrength(0),
    toMutate(true),
    toCalculateFitness(false)
  {
    registerParameter("evo-go-to-point.state", state);
    ReadConfigFile();
    ReadLogFile();
    Init();
  }
  
  /**  Executes the basic behavior. */
  virtual void execute();
      
private:
  /** The parameters "evo-go-to-point.x" and "evo-go-to-point.y".*/
  Vector2<double> evoPoint[2];
  
  /** The parameter "evo-go-to-point.angle"*/
  double evoWalkAngle;

  /** The parameter "evo-go-to-point.duration"*/
  double evoWalkDuration;

  /** The parameter "evo-go-to-point.state" for handling the states in execute().*/
  double state;

  /** Specifies, if the walk direction is corrected or not.*/
  int evoWalkMode;
  
  /** A buffer for the last 8 robot poses.*/
  RingBuffer <Pose2D, 8> robotPoses;

  /** The robots average pose at the start and the target point calculated using a ringbuffer.*/
  Pose2D averageStartPose, averageTargetPose;

  /** The parent and the offspring individual.*/
  InvKinWalkingParameters parentParam, offspringParam;
 
  /** The actual mutation strength.*/
  double mutationStrength;

  /** Some counters for mutation.*/
  int numberOfMutations;
  int numberOfSuccessfulMutations;
  int numberOfGenerations;
  int numberOfEvolveValues;

  /** An Array for all unique fitness values of one individual.*/
  double uniqueFitness[maxNumToShowIndividual];

  /** Every individual is shown several times to verify fitness.*/
  int tempNumOfIndividual;

  /** Values are read from "evo_walk.cfg" and indicate, if a parameter shall be mutated or not.*/
  bool valueToEvolve[34];
  
  /** Some flags to ensure that certain parts of a state in execute() are only executed ones.*/
  bool toMutate, toCalculateFitness, beforeFirstRun;
  
  /** After startup or a break the parent individual is shown once.*/
  bool showParentAfterInit;

  /* Initializes the basic behavior.*/
  void Init();

  /** 
  * Reads all needed values from "evo_walk.cfg".
  * If the file doesnt exist, default values are used.
  */
  void ReadConfigFile();

  /** 
  * Reads the last parent and offspring from "evo_walk.log" to continue evolution.
  * If the file doesnt exist, a new one with default values is created.
  */
  void ReadLogFile();

  /**
  * The function writes (appends) all evolved individuals to "evo_walk.log".
  * @param paramsToWrite The individual that is used to be written into the file.
  * @param isParent Indicates, if the individual is a parent individual.
  */
  void WriteLogFile(InvKinWalkingParameters & paramsToWrite, int status);
  
  /**
  * The function creates a default logfile "evo_walk.log".
  */
  void CreateDefLogFile();

  /** The function calculates the last individuals fitness.*/
  void CalculateFitness();

  /** The function calculates the last individuals average fitness.*/
  double CalculateAverageFitness();

  /**
  * The function lets the robot walk to a specified point.
  * @param pointNumber The number of the point.
  */
  void WalkTowardsPoint(int pointNumber);

  /** 
  * The function determines the robots start position and corrects its angle to the target point.
  * @param pointNumber The number of the point the robot rotates to.
  */
  void DetermineStartPos(int pointNumber);

  /** The function determines the robots target position.*/
  void DetermineTargetPos(int pointNumber);

  /** A function, that automatically adapts the mutation strength to find the (local) maximum.*/
  void AdaptMutationStrength();

  /** Ensures that only chosen parameters are evolved depending on "valueToEvolve[34]".*/
  void CheckValuesToEvolve();
};


/**
* @class MSH2004EvoBasicBehaviors
*
* Creates and registers evolution basic behaviors
*/
class MSH2004EvoBasicBehaviors : public BehaviorControlInterfaces
{
public:
  /**
  * Constructor
  */
  MSH2004EvoBasicBehaviors(const BehaviorControlInterfaces& interfaces,
                       Xabsl2ErrorHandler& errorHandler)
    : BehaviorControlInterfaces(interfaces),
      errorHandler(errorHandler),
      autonomousEvolution(interfaces,errorHandler)
  {}

  /** Registers basic behaviors at the engine */
  void registerBasicBehaviors(Xabsl2Engine& engine);

private:
  /** Is invoked when errors occurs */
  Xabsl2ErrorHandler& errorHandler;

  //!@name Basic Behaviors
  //!@{
  MSH2004AutonomousEvolution autonomousEvolution;  
  //!@}
};


#endif // __MSH2004EvoBasicBehaviors_h_

/*
* Change Log
* 
* $Log: MSH2004EvoBasicBehaviors.h,v $
* Revision 1.9  2004/04/08 15:33:01  wachter
* GT04 checkin of Microsoft-Hellounds
*
* Revision 1.11  2004/04/01 18:43:07  cesarz
* robot turns to target point after walking
*
* Revision 1.10  2004/03/31 14:29:04  pg_arce
* fixed a bug
*
* Revision 1.9  2004/03/30 21:17:10  pg_thki
* - EvoWalker now writes .iwp files to the stick.
*
* Revision 1.8  2004/03/12 21:26:20  cesarz
* Evolution of walks in any walking direction is possible now.
*
* Revision 1.7  2004/03/11 10:19:40  cesarz
* parents fitness in logfile is being determined, if it equals 0.
*
* Revision 1.6  2004/03/08 01:07:01  roefer
* Interfaces should be const
*
* Revision 1.5  2004/03/07 22:51:54  cesarz
* some improvements
*
* Revision 1.4  2004/02/28 11:50:21  cesarz
* - not direction corrected evolving mode added
* - some cosmetic changes
*
* Revision 1.3  2004/02/27 12:10:50  cesarz
* - successful mutation is being repeated 4 times to verify fitness
* - increased max speed
*
* Revision 1.2  2004/02/26 23:41:58  cesarz
* first shown individual after startup or break is the parent
*
* Revision 1.1  2004/02/26 18:07:28  cesarz
* first version of evolution behavior
*
*/

