/** 
* @file MSH2004EvoBasicBehaviors.cpp
*
* Implementation of basic behaviors defined in evo-basic-behaviors.xml.
*
* @author Arthur Cesarz
* @author Matthias Hebbel
*/

#include "MSH2004EvoBasicBehaviors.h"

#include "Tools/Math/PIDsmoothedValue.h"
#include "Tools/Math/Geometry.h"
#include "Tools/Math/Common.h"
#include "Tools/Location.h"
#include "Tools/Streams/InStreams.h"
#include "Tools/Streams/OutStreams.h"
#include "Tools/RingBuffer.h"

#include "Tools/Debugging/Debugging.h"

void MSH2004EvoBasicBehaviors::registerBasicBehaviors(Xabsl2Engine& engine)
{
  engine.registerBasicBehavior(autonomousEvolution);
}

void MSH2004AutonomousEvolution::execute()
{
  switch (static_cast<int>(state))
  {
  // init, if evolution was started or paused
  case -1: 
    Init();
    break;

  // calculate last fitness and localize at start pos (point 0)
  case 0: 
    if (toCalculateFitness && !showParentAfterInit && !beforeFirstRun)
      CalculateFitness();
    DetermineStartPos(0);
    break;

  // walk towards point 1
  case 1:
    WalkTowardsPoint(1);
    break;

  // localize at target pos (near point 1)
  case 2: 
    beforeFirstRun = false;
    DetermineTargetPos(1);
    toCalculateFitness = true;
    break;

  // calculate last fitness and localize at start pos (point 1)
  case 3: 
    if (toCalculateFitness && !showParentAfterInit)
      CalculateFitness();
    DetermineStartPos(1);
    break;
  
  // walk towards point 0
  case 4: 
    showParentAfterInit = false;
    WalkTowardsPoint(0);
    break;
  
  // localize at target pos (near point 0)
  case 5:
    DetermineTargetPos(0);
    toCalculateFitness = true;
    break;

  default:
    break;
  }
}

void MSH2004AutonomousEvolution::Init()
{
  toMutate = false;
  //if fitness in logfile == 0 then determine parent fitness
  if( parentParam.fitness == 0.0 )
  {
    toCalculateFitness = true;
    showParentAfterInit = false;
  }
  else
  {
    toCalculateFitness = false;
    showParentAfterInit = true;
  }
  // the first time show parent individual
  offspringParam = parentParam;
  tempNumOfIndividual = 0;
  beforeFirstRun = true;
  int i;
  for( i=0; i < maxNumToShowIndividual; i++)
    uniqueFitness[i] = 0.0;
}

void MSH2004AutonomousEvolution::CalculateFitness()
{ 
  toCalculateFitness = false;
  if (tempNumOfIndividual < maxNumToShowIndividual)
  {
    if (evoWalkMode == 1)
    {
      uniqueFitness[tempNumOfIndividual] = 1000.0 * (fabs(Geometry::vectorTo(averageStartPose, averageTargetPose.translation).x) -
                               fitnessFactor * fabs(Geometry::vectorTo(averageStartPose, averageTargetPose.translation).y)) / 
                               evoWalkDuration;
      double speedX = 1000.0 * (averageTargetPose.translation-averageStartPose.translation).abs() /
                               evoWalkDuration;
      OUTPUT(idText, text, "Rob " << getPlayer().getPlayerNumber() << 
                           " SpeedX " << speedX << " mm/s, Fit " << tempNumOfIndividual << ": " << 
                           uniqueFitness[tempNumOfIndividual]); 
    }
    else
    {
      uniqueFitness[tempNumOfIndividual]  = 1000.0 * (averageTargetPose.translation-averageStartPose.translation).abs() /
                                         evoWalkDuration;
      OUTPUT(idText, text, "Rob " << getPlayer().getPlayerNumber() << 
                           " Gen " << numberOfGenerations << 
                           " Mut " << numberOfMutations << 
                           " (" << mutationStrength << ")" <<
                           " Fit " << tempNumOfIndividual << ": " << uniqueFitness[tempNumOfIndividual] << " mm/s");
    }
    offspringParam.fitness = uniqueFitness[tempNumOfIndividual];
    WriteLogFile(offspringParam, 1);
    if(offspringParam.fitness > parentParam.fitness)
      soundRequest.soundID = SoundRequest::impressive;
    else
      soundRequest.soundID = SoundRequest::rob002;
  }
   
  if (++tempNumOfIndividual == maxNumToShowIndividual)
  {
    tempNumOfIndividual = 0;
    offspringParam.fitness = CalculateAverageFitness();
 
    if (offspringParam.fitness > parentParam.fitness)
    {
      numberOfSuccessfulMutations++;
      numberOfGenerations++;
      parentParam = offspringParam;
      OUTPUT(idText, text, "Rob " << getPlayer().getPlayerNumber() << 
                           " Gen " << numberOfGenerations << 
                           " Mut " << numberOfMutations << 
                           " (" << mutationStrength << ")" <<
                           " Fit +: " << offspringParam.fitness << " mm/s");
      WriteLogFile(offspringParam, 2);
      soundRequest.soundID = SoundRequest::excellent;
    }
    else
    { 
      OUTPUT(idText, text, "Rob " << getPlayer().getPlayerNumber() << 
                           " Gen " << numberOfGenerations << 
                           " Mut " << numberOfMutations << 
                           " (" << mutationStrength << ")" <<
                           " Fit -: " << offspringParam.fitness << " mm/s");
      WriteLogFile(offspringParam, 0);
      soundRequest.soundID = SoundRequest::howl;
    }
  }
}

double MSH2004AutonomousEvolution::CalculateAverageFitness()
{ 
  int minIndex = 0, maxIndex = 0;
  int i, counter = 0;
  double averageFitness = 0.0;

  if( maxNumToShowIndividual > 2 )
  {
    for (i=0; i<maxNumToShowIndividual; i++)
    {
      if( uniqueFitness[i] <= uniqueFitness[minIndex] )
        minIndex = i;
      if( uniqueFitness[i] >= uniqueFitness[maxIndex] )
        maxIndex = i;
    }
    // take only medial values
    for (i=0; i<maxNumToShowIndividual; i++)
    {
      if( (i!=minIndex) && (i!=maxIndex) )
      {
        averageFitness += uniqueFitness[i];
        counter++;
      }
    }
    return (averageFitness /= (double)counter);
  }
  else
  {
    for (i=0; i<maxNumToShowIndividual; i++)
    {
      averageFitness += uniqueFitness[i];
      counter++;
    }
  }
  return (averageFitness /= (double)counter);
}  

void MSH2004AutonomousEvolution::WalkTowardsPoint(int pointNumber)
{
  if(toMutate && !showParentAfterInit && (tempNumOfIndividual == 0) && !beforeFirstRun)
  {
    toMutate = false;
    AdaptMutationStrength();
    offspringParam.mutationOf(&parentParam, 1, mutationStrength/100.0);
    numberOfMutations++;
    CheckValuesToEvolve();
  }
  
  invKinWalkingParameters = offspringParam;
  walkParameterTimeStamp  = SystemCall::getCurrentSystemTime();

  getDebugOut().bin << offspringParam;
  getDebugOut().finishMessage(idInvKinWalkingParameters);
  
  // set parameters and walk
  // 2do: adjust walk orientation for evoWalkMode 1
  motionRequest.motionType = MotionRequest::walk;
  motionRequest.walkType   = MotionRequest::normal;
  motionRequest.walkParams = Pose2D(0.0, walkMaxForwardSpeed, 0.0);
  if (evoWalkMode == 0)
  {
    //correct walking direction
    double tempAngle;
    tempAngle = (pointNumber==0) ? evoWalkAngle : -evoWalkAngle;
    motionRequest.walkParams.translation.x =  
      Geometry::distanceTo(robotPose.getPose(), evoPoint[pointNumber])*cos(tempAngle);
    motionRequest.walkParams.translation.y =  
      Geometry::distanceTo(robotPose.getPose(), evoPoint[pointNumber])*sin(tempAngle);
    motionRequest.walkParams.translation.normalize();
    motionRequest.walkParams.translation *= 400.0;
    motionRequest.walkParams.rotation = 
      0.7 * normalize(Geometry::angleTo(robotPose.getPose(), evoPoint[pointNumber]) 
      - tempAngle);
  }
}

void MSH2004AutonomousEvolution::DetermineStartPos(int pointNumber)
{   
  int i;

  if (evoWalkMode == 1)
  {
    // correct start position
    motionRequest.motionType = MotionRequest::walk;
    motionRequest.walkType   = MotionRequest::normal;
    motionRequest.walkParams = Pose2D(0.8 * Geometry::angleTo(robotPose.getPose(), evoPoint[pointNumber]), 0.1, 0.0);
    Quick2UNSWWalkingParameters defaultParam;
    invKinWalkingParameters  = defaultParam;
    walkParameterTimeStamp   = SystemCall::getCurrentSystemTime();
  }

  robotPoses.add(robotPose);
  averageStartPose = Pose2D(robotPose.rotation, 0.0, 0.0);
  // calculate average pose
  for (i=0; i<8; i++)
    averageStartPose.translation += robotPoses[i].translation;
  averageStartPose.translation /= 8.0;
  toMutate = true;
}

void MSH2004AutonomousEvolution::DetermineTargetPos(int pointNumber)
{
  int i;

  // turn a little bit around to improve localization
  motionRequest.motionType = MotionRequest::walk;
  motionRequest.walkType   = MotionRequest::normal;
  // turn towards next goal to see more landmarks
  // double turnDirection = -0.25 * sgn(robotPose.translation.x * robotPose.translation.y);
  // motionRequest.walkParams = Pose2D(turnDirection, -0.01, 0.0);
  
  // Turn to target point
  motionRequest.walkParams = Pose2D(0.8 * Geometry::angleTo(robotPose.getPose(), evoPoint[pointNumber]), 0.1, 0.0);

  Quick2UNSWWalkingParameters defaultParam;
  invKinWalkingParameters = defaultParam;
  walkParameterTimeStamp  = SystemCall::getCurrentSystemTime();

  getDebugOut().bin << defaultParam;
  getDebugOut().finishMessage(idInvKinWalkingParameters);

  robotPoses.add(robotPose);
  averageTargetPose = Pose2D(robotPose.rotation, 0.0, 0.0 );
  // calculate average pose
  for (i=0; i<8; i++)
  {
    averageTargetPose.translation += robotPoses[i].translation;
  }
  averageTargetPose.translation /= 8.0;
}

void MSH2004AutonomousEvolution::ReadConfigFile()
{
  int i;
  double temp;
  InConfigFile evoConfigFile(getLocation().getFilename("evo_walk.cfg"));
  if (evoConfigFile.exists())
  { 
    evoConfigFile >> evoPoint[0].x;
    evoConfigFile >> evoPoint[0].y;
    evoConfigFile >> evoPoint[1].x;
    evoConfigFile >> evoPoint[1].y;
    evoConfigFile >> evoWalkAngle;
    evoConfigFile >> evoWalkDuration;
    evoConfigFile >> evoWalkMode;
    for (i = 0; i < 34; i++)
    {
      evoConfigFile >> temp;
      valueToEvolve[i] = temp == 0 ? false : true; // to avoid cast warning
      if (valueToEvolve[i])
        numberOfEvolveValues++;
    }
    if (evoWalkDuration == 0.0)
      evoWalkDuration = 0.1;
    evoWalkAngle = fromDegrees(evoWalkAngle);
  }
  else
  {
    evoPoint[0].x = 0.0;
    evoPoint[0].y = 0.0;
    evoPoint[1].x = 0.0;
    evoPoint[1].y = 0.0;
    evoWalkAngle = 0.0;
    evoWalkDuration = 0.1;
    evoWalkMode = 0;
  } 
}

void MSH2004AutonomousEvolution::ReadLogFile()
{
  numberOfMutations = -1;
  int  numLines = 0;
  { //<-dont know, if file will be closed, so leave the brackets
    InTextFile evologfile(getLocation().getFilename("evo_walk.log"));
  
    while(evologfile.exists() && !evologfile.eof())
    {
      char temp [2];
      evologfile >> temp;
      // is the logfile invalid? (very simple test)
      if((temp[0] == '+') || (temp[0] == '-') || (temp[0] == '~'))
      {
        if( temp[0]=='+' || temp[0]=='-' )
          numberOfMutations++;
        bool isParent = (temp[0] == '+' ? true : false);
        evologfile >> numberOfGenerations;
        isParent ? evologfile >> parentParam.fitness : evologfile >> offspringParam.fitness;
        evologfile >> mutationStrength;
        isParent ? evologfile >> parentParam : evologfile >> offspringParam;
        numLines++;
      } 
      else
      {
        OUTPUT(idText, text, "(MSH2004EvoBasicBehaviors): Robot " << getPlayer().getPlayerNumber() << 
                             ": Invalid line in evo_walk.log!");
      }
    }
  }
  if (numLines == 0)
    CreateDefLogFile();
}

void MSH2004AutonomousEvolution::WriteLogFile(InvKinWalkingParameters & paramsToWrite, int status)
{
  OutTextFile evoLogFile(getLocation().getFilename("evo_walk.log"), true);
  switch (status)
  {
   case 0:  evoLogFile << "-"; break;
   case 1:  evoLogFile << "~"; break;
   default: evoLogFile << "+"; break;
  }
  evoLogFile 
    << numberOfGenerations
            << paramsToWrite.fitness
            << mutationStrength 
            << paramsToWrite 
            << endl;

  bool saved = false;
  int  index = 0;
  while (!saved && index < 10) {
    char filename[32];
    sprintf(filename, "%d_%d_%d.iwp", 
      numberOfGenerations, 
      (int)paramsToWrite.fitness,
      index++
    );
  
    InBinaryFile  iwpFile(getLocation().getFilename(filename));
    if (!iwpFile.exists())
    {
      OutBinaryFile  iwpFile(getLocation().getFilename(filename), true);
      iwpFile << paramsToWrite;
      break;
    }
  }

  //OutTextFile battLogFile(getLocation().getFilename("evo_batt.log"), true);
  //battLogFile << SystemCall::getCurrentSystemTime() << " " << SystemCall::getRemainingPower() << endl;
}

void MSH2004AutonomousEvolution::CreateDefLogFile()
{
  OUTPUT(idText, text, "(MSH2004EvoBasicBehaviors): Robot " << getPlayer().getPlayerNumber() << 
                             " created default logfile");
  // set default parameters
  Quick2UNSWWalkingParameters defaultParam;
  parentParam = defaultParam;
  offspringParam = defaultParam;
  parentParam.fitness = 0.0;
  mutationStrength = 1.0;
  WriteLogFile(parentParam, 2);
}

void MSH2004AutonomousEvolution::AdaptMutationStrength()
{
  if((numberOfMutations >= numberOfEvolveValues) && (numberOfMutations > 0))
  {
    if(((double)numberOfSuccessfulMutations/(double)numberOfMutations) < 0.2)
    {
      mutationStrength *= 0.8;
    }
    else mutationStrength /= 0.8;
  numberOfMutations = 0;
  numberOfSuccessfulMutations = 0;
  }
}
 
void MSH2004AutonomousEvolution::CheckValuesToEvolve()
{
  if(!valueToEvolve[0])  offspringParam.footMode          = parentParam.footMode;
  if(!valueToEvolve[1])  offspringParam.foreHeight        = parentParam.foreHeight;
  if(!valueToEvolve[2])  offspringParam.foreWidth         = parentParam.foreWidth;
  if(!valueToEvolve[3])  offspringParam.foreCenterX       = parentParam.foreCenterX;
  if(!valueToEvolve[4])  offspringParam.hindHeight        = parentParam.hindHeight;
  if(!valueToEvolve[5])  offspringParam.hindWidth         = parentParam.hindWidth;
  if(!valueToEvolve[6])  offspringParam.hindCenterX       = parentParam.hindCenterX;
  if(!valueToEvolve[7])  offspringParam.foreFootLift      = parentParam.foreFootLift;
  if(!valueToEvolve[8])  offspringParam.hindFootLift      = parentParam.hindFootLift;
  if(!valueToEvolve[9])  offspringParam.foreFootTilt      = parentParam.foreFootTilt;
  if(!valueToEvolve[10]) offspringParam.hindFootTilt      = parentParam.hindFootTilt;
  if(!valueToEvolve[11]) offspringParam.legSpeedFactorX   = parentParam.legSpeedFactorX;
  if(!valueToEvolve[12]) offspringParam.legSpeedFactorY   = parentParam.legSpeedFactorY;
  if(!valueToEvolve[13]) offspringParam.legSpeedFactorR   = parentParam.legSpeedFactorR;
  if(!valueToEvolve[14]) offspringParam.maxStepSizeX      = parentParam.maxStepSizeX;
  if(!valueToEvolve[15]) offspringParam.maxStepSizeY      = parentParam.maxStepSizeY;
  if(!valueToEvolve[16]) offspringParam.maxSpeedXChange   = parentParam.maxSpeedXChange;
  if(!valueToEvolve[17]) offspringParam.maxSpeedYChange   = parentParam.maxSpeedYChange;
  if(!valueToEvolve[18]) offspringParam.maxRotationChange = parentParam.maxRotationChange;
  if(!valueToEvolve[19]) offspringParam.counterRotation   = parentParam.counterRotation;
  if(!valueToEvolve[20]) offspringParam.stepLen           = parentParam.stepLen;
  if(!valueToEvolve[21]) offspringParam.groundPhase[0]    = parentParam.groundPhase[0];
  if(!valueToEvolve[22]) offspringParam.liftPhase[0]      = parentParam.liftPhase[0];
  if(!valueToEvolve[23]) offspringParam.loweringPhase[0]  = parentParam.loweringPhase[0];
  if(!valueToEvolve[24]) offspringParam.groundPhase[1]    = parentParam.groundPhase[1];
  if(!valueToEvolve[25]) offspringParam.liftPhase[1]      = parentParam.liftPhase[1];
  if(!valueToEvolve[26]) offspringParam.loweringPhase[1]  = parentParam.loweringPhase[1];
  if(!valueToEvolve[27]) offspringParam.legPhase[0]       = parentParam.legPhase[0];
  if(!valueToEvolve[28]) offspringParam.legPhase[1]       = parentParam.legPhase[1];
  if(!valueToEvolve[29]) offspringParam.legPhase[2]       = parentParam.legPhase[2];
  if(!valueToEvolve[30]) offspringParam.legPhase[3]       = parentParam.legPhase[3];
  if(!valueToEvolve[31]) offspringParam.bodyShiftX        = parentParam.bodyShiftX;
  if(!valueToEvolve[32]) offspringParam.bodyShiftY        = parentParam.bodyShiftY;
  if(!valueToEvolve[33]) offspringParam.bodyShiftOffset   = parentParam.bodyShiftOffset;
}
/*
* Change Log
* 
* $Log: MSH2004EvoBasicBehaviors.cpp,v $
* Revision 1.9  2004/04/08 15:33:01  wachter
* GT04 checkin of Microsoft-Hellounds
*
* Revision 1.11  2004/04/01 18:43:07  cesarz
* robot turns to target point after walking
*
* Revision 1.10  2004/03/31 14:29:04  pg_arce
* fixed a bug
*
* Revision 1.9  2004/03/30 21:17:10  pg_thki
* - EvoWalker now writes .iwp files to the stick.
*
* Revision 1.8  2004/03/12 21:26:20  cesarz
* Evolution of walks in any walking direction is possible now.
*
* Revision 1.7  2004/03/11 10:19:40  cesarz
* parents fitness in logfile is being determined, if it equals 0.
*
* Revision 1.6  2004/03/09 22:33:38  cesarz
* evolution behavior should work now
*
* Revision 1.5  2004/03/07 22:51:53  cesarz
* some improvements
*
* Revision 1.4  2004/02/28 11:50:21  cesarz
* - not direction corrected evolving mode added
* - some cosmetic changes
*
* Revision 1.3  2004/02/27 12:10:49  cesarz
* - successful mutation is being repeated 4 times to verify fitness
* - increased max speed
*
* Revision 1.2  2004/02/26 23:41:57  cesarz
* first shown individual after startup or break is the parent
*
* Revision 1.1  2004/02/26 18:07:28  cesarz
* first version of evolution behavior
*
*
*/

