/**
* @file MSH2004ContinuousRules/GoToPoint.h
*
* @author Dirk Thomas
*/

#ifndef __MSH2004GoToPoint_h_
#define __MSH2004GoToPoint_h_

#include "Tools/ContinuousBasicBehaviors/ContinuousRule.h"
#include "Tools/FieldDimensions.h"

namespace MSH2004ContinuousRules
{

/**@class GoToPoint
 */
class GoToPoint: public ContinuousRule{
private:
  /** reference to the point to go to */
  const double &pointX;
  const double &pointY;

  /** the point is in relative coordinates */
  bool relative;

  /** if the rule not constructed via a reference to the point
   * these variables are used to store the coordinates */
  double localPointX;
  double localPointY;

	/** the distance above which the result vector is maxForce in mm*/
  double maxForceLength;
  /** the distance below which the result vector is 0 in mm*/
  double minDistance;
  /** the maximum attracting force */
  double maxForce;


public:

  /** constructor
   */
  GoToPoint(
    const BehaviorControlInterfaces& interfaces,
    const Vector2<double>& point,
    bool relative = false,
    double maxForce = 1.0,
    double minDistance = 0.0,
    double maxForceLength = 350.0)
    :
    ContinuousRule(interfaces,"GoToPoint"),
    pointX(point.x),
    pointY(point.y),
    relative(relative),
    maxForce(maxForce),
    minDistance(minDistance),
    maxForceLength(maxForceLength)
  {};

  /** constructor
   */
  GoToPoint(
    const BehaviorControlInterfaces& interfaces,
    const double& pointX,
    const double& pointY,
    bool relative = false,
    double maxForce = 1.0,
    double minDistance = 0.0,
    double maxForceLength = 350.0)
    :
    ContinuousRule(interfaces, "GoToPoint"),
    pointX(pointX),
    pointY(pointY),
    relative(relative),
    maxForce(maxForce),
    minDistance(minDistance),
    maxForceLength(maxForceLength)
  {};

  /** constructor
   */
  GoToPoint(
    const BehaviorControlInterfaces& interfaces,
    const Vector2<int> point,
    bool relative = false,
    double maxForce = 1.0,
    double minDistance = 0.0,
    double maxForceLength = 350.0)
    :
    ContinuousRule(interfaces,"GoToPoint"),
    pointX(localPointX),
    pointY(localPointY),
    relative(relative),
    maxForce(maxForce),
    minDistance(minDistance),
    maxForceLength(maxForceLength)
  {
      localPointX = (double)point.x;
      localPointY = (double)point.y;
  };

  /**
   * Returns whether this rule generates absolute or robot relative coordinates.
   */
  virtual bool isRelative() {return relative;}

 	/** executes the basic behavior rule
   * @param robotPose the current robots pose at which the rule is to be evaluated
   * @param walk the direction and speed of the suggested motion coded as an vector (output)
	 * @param ra the rotation angle, the direction the robot should be directed (output)
	 * @param rweight the rotation weight the weight of the suggested rotation (output)
	 */
  virtual void execute(const RobotPose& robotPose,
                       Vector2<double>& walk,
                       double& ra, double& rweight)
  {
    walk = Vector2<double>(pointX, pointY);

    if (!relative) 
      walk -= robotPose.translation;

    double distance = walk.abs();
    walk.normalize();

    walk *= getAttractingForce(distance, maxForce, minDistance, maxForceLength);

    rweight=0;
  }
};

}


#endif //__GoToPoint_h_


/*
* Change log:
*
* $Log: GoToPoint.h,v $
* Revision 1.3  2004/03/08 01:07:05  roefer
* Interfaces should be const
*
* Revision 1.2  2003/10/31 08:32:48  dueffert
* doxygen bugs fixed
*
* Revision 1.1  2003/10/26 22:49:40  loetzsch
* created ATH2004BehaviorControl from GT2003BehaviorControl
*  - strongly simplified option graph
*  - moved some symbols from GT2003 to CommonXabsl2Symbols
*  - moved some basic behaviors from GT2003 to CommonXabsl2BasicBehaviors
*
* cloned ATH2004 three times (BB2004, DDD2004, MSH2004)
*
*/
