/** 
* @file MSH2004ContinuousRules/AvoidOwnPenaltyArea.h
*
* 
* @author Martin Kallnik
*/

#ifndef __MSH2004AvoidOwnPenaltyArea_h_
#define __MSH2004AvoidOwnPenaltyArea_h_

#include "Tools/ContinuousBasicBehaviors/ContinuousRule.h"
#include "Tools/FieldDimensions.h"

namespace MSH2004ContinuousRules
{

/**@class AvoidOwnPenaltyArea
 * rule which avoids the own penalty area
 */
class AvoidOwnPenaltyArea: public ContinuousRule{
private:
	/** the maximum length of the result vector*/
  double maxForce;
	/** the distance below which the result vector length is maxForce (in mm)*/
  double maxForceLength;
	/** the distance above which the result vector length is 0 (in mm)*/
  double influenceLength;

public:

  /** constructor
   */
  AvoidOwnPenaltyArea(const BehaviorControlInterfaces& interfaces,
                 double maxForce = 1.7, 
                 double maxForceLength = 50.0, 
                 double influenceLength = 150.0)
      :ContinuousRule(interfaces, "AvoidOwnPenaltyArea"),
       maxForce(maxForce),
       maxForceLength(maxForceLength),
       influenceLength(influenceLength)
  {};
 
 	/** executes the basic behavior rule
   * @param robotPose the current robots pose at which the rule is to be evaluated
	 * @param walk the direction and speed of the suggested motion coded as an vector (output)
	 * @param ra the rotation angle, the direction the robot should be directed (output)
	 * @param rweight the rotation weight the weight of the suggested rotation (output)
	 */
  virtual void execute(const RobotPose& robotPose,
                       Vector2<double>& walk,
                       double& ra, double& rweight)
  {

    if (getPlayer().getPlayerNumber() != Player::one)
    {
    
      double distance = min (
        FieldDimensions::distanceToOwnPenaltyArea ( robotPose.translation ),
        FieldDimensions::distanceToOwnPenaltyArea ( robotPose*Vector2<double>(-bodyLength,0) ) );
    
      walk = robotPose.translation;
      walk.x -= (xPosOwnGroundline - 300) ;
      walk.normalize();
    
      walk *= getRepellingForce(distance*1.5, maxForce, maxForceLength, influenceLength);

	/*   Vector2<double> point2;
	   point2 = (walk * 100) + robotPose.translation;
      
		  LINE(teamBallLocatorField,robotPose.translation.x,robotPose.translation.y,point2.x,point2.y,1, Drawings::ps_solid, Drawings::skyblue);
	 
	  DEBUG_DRAWING_FINISHED(teamBallLocatorField);
    */
    } else {
      // no avoid own penalty area for goalie
      walk.x = walk.y = 0;
    }
    
    ra=0;
    rweight=0;
  };
};

}


#endif 


/*
* Change log:
*
* $Log: AvoidOwnPenaltyArea.h,v $
* Revision 1.5  2004/05/12 19:33:12  kerdels
* merged the behavior changes during australian, american and japan open
*
* Revision 1.4  2004/03/08 01:07:04  roefer
* Interfaces should be const
*
* Revision 1.3  2003/12/06 17:45:33  loetzsch
* replaced Player::playerRole (goalie, defender, striker1, striker2)
* by Player::playerNumber (one, two, three, four)
*
* Revision 1.2  2003/10/28 15:04:09  loetzsch
* fixed doxygen bugs
*
* Revision 1.1  2003/10/26 22:49:40  loetzsch
* created ATH2004BehaviorControl from GT2003BehaviorControl
*  - strongly simplified option graph
*  - moved some symbols from GT2003 to CommonXabsl2Symbols
*  - moved some basic behaviors from GT2003 to CommonXabsl2BasicBehaviors
*
* cloned ATH2004 three times (BB2004, DDD2004, MSH2004)
*
*/
