/** 
* @file MSH2004ContinuousRules/AvoidObstacles.h
*
* @author Max Risler
*/

#ifndef __MSH2004AvoidObstacles_h_
#define __MSH2004AvoidObstacles_h_

#include "Tools/ContinuousBasicBehaviors/ContinuousRule.h"

namespace MSH2004ContinuousRules
{

/**@class AvoidObstacles
 * rule which avoids collision with the obstacles
 */
class AvoidObstacles: public ContinuousRule{
public:

  /** constructor
   */
  AvoidObstacles(const BehaviorControlInterfaces& interfaces,
                 double maxForce = 1.0, 
                 double maxForceLength = 50.0, 
                 double influenceLength = 300.0)
      :ContinuousRule(interfaces, "AvoidObstacles"),
       maxForce(maxForce),
       maxForceLength(maxForceLength),
       influenceLength(influenceLength),
       numOfSectors(20)
  {};

  /**
   * Returns whether this rule generates absolute or robot relative coordinates.
   * This rule is robot relative.
   */
  virtual bool isRelative() {return true;}

  /**
   * Returns whether this rule affect the walk speed of the basic behavior.
   * This rule does not affect speed.
   */
  virtual bool affectSpeed() {return false;}

 	/** executes the basic behavior rule
   * @param robotPose the current robots pose at which the rule is to be evaluated
   * @param walk the direction and speed of the suggested motion coded as an vector (output)
	 * @param ra the rotation angle, the direction the robot should be directed (output)
	 * @param rweight the rotation weight the weight of the suggested rotation (output)
	 */
  virtual void execute(const RobotPose& robotPose,
                       Vector2<double>& walk,
                       double& ra, double& rweight)
  {

    Vector2<double> fromObstacle;
    double angle;

    walk.x=walk.y=0;
    rweight = 0;
/*
    for (int i=0;i<ObstaclesModel::numOfSectors;i++)
      if (obstaclesModel.distance[i] < ObstaclesModel::maxDistance)
      {
        angle = normalize(ObstaclesModel::getAngleOfSector(i));

        fromObstacle.x = -cos(angle);
        fromObstacle.y = -sin(angle);

        walk += fromObstacle * 
                getRepellingForce(obstaclesModel.distance[i], maxForce, maxForceLength, influenceLength);
      }
*/
    for (angle = -pi_2; angle < pi_2; angle += pi / (numOfSectors-1))
    {
      double distance = obstaclesModel.getDistanceInDirection(angle, pi_2 / (numOfSectors-1));
      if (distance < ObstaclesModel::maxDistance)
      {
        fromObstacle.x = -cos(angle);
        fromObstacle.y = -sin(angle);

        walk += fromObstacle * 
                getRepellingForce(distance, maxForce, maxForceLength, influenceLength);
      }
    }
  };

protected:
	/** the maximum length of the result vector*/
  double maxForce;
	/** the distance below which the result vector length is maxForce (in mm)*/
  double maxForceLength;
	/** the distance above which the result vector length is 0 (in mm)*/
  double influenceLength;

  /** the number of sectors to reduce the obstacles model to */
  double numOfSectors;
};

}


#endif 


/*
* Change log:
*
* $Log: AvoidObstacles.h,v $
* Revision 1.3  2004/03/08 01:07:03  roefer
* Interfaces should be const
*
* Revision 1.2  2003/10/28 15:04:09  loetzsch
* fixed doxygen bugs
*
* Revision 1.1  2003/10/26 22:49:40  loetzsch
* created ATH2004BehaviorControl from GT2003BehaviorControl
*  - strongly simplified option graph
*  - moved some symbols from GT2003 to CommonXabsl2Symbols
*  - moved some basic behaviors from GT2003 to CommonXabsl2BasicBehaviors
*
* cloned ATH2004 three times (BB2004, DDD2004, MSH2004)
*
*/
