/** 
* @file StrategySymbols.h
*
* Declaration of class StrategySymbols.
*
* @author Martin Ltzsch
*/

#ifndef __StrategySymbols_h_
#define __StrategySymbols_h_

#include "../../BehaviorControl.h"
#include "Tools/Xabsl2/Xabsl2Engine/Xabsl2Engine.h"
#include "Tools/Math/Vector2.h"

/**
* The Xabsl2 symbols that are defined in "strategy-symbols.xml"
*
* @author Martin Ltzsch
*/ 
class StrategySymbols : public Xabsl2FunctionProvider, public BehaviorControlInterfaces
{
public:
/*
* Constructor.
* @param interfaces The paramters of the BehaviorControl module.
  */
  StrategySymbols(const BehaviorControlInterfaces& interfaces);
  
  /** registers the symbols at an engine */
  void registerSymbols(Xabsl2Engine& engine);
  
  /** updates the symbols */
  void update();
  
  /** current estimated time to reach the ball */
  double estimatedTimeToReachBall;
  
  /** last time the robot pose validity was above 0.5 */
  unsigned long timeOfLastGoodSelfLocalization;
  
  /** computes the estimated time to reach the ball */
  void estimateTimeToReachBall();
  
  /** current role */
  BehaviorTeamMessage::DynamicRole role;
  
  /** computes the current role */
  void computeRole();
  
  //!@name Input symbols
  //!@{
  /* 
  * A function for the symbol "robot-number". 
  * Returns the robots back number as read from player.cfg 
  */
  double getRobotNumber() { return (double)(getPlayer().getPlayerNumber() + 1); };
  
  /** A function for the symbol "another-player-is-in-ready-state". */
  bool getAnotherPlayerIsInReadyState();
  
  /** A function for the symbol "another-teammate-is-preparing-a-kick" */
  bool getAnotherTeammateIsPreparingAKick();
  
  /** A function for the symbol "another-teammate-is-performing-a-kick" */
  bool getAnotherTeammateIsPerformingAKick();
  
  /** A function for the symbol "another-teammate-just-performed-a-kick" */
  bool getAnotherTeammateJustPerformedAKick();
  
  /** A function for the symbol "the-striker-is-playing-near-the-opponent-goal" */
  bool getTheStrikerIsPlayingNearTheOpponentGoal();
  
  /** A function for the symbol "the-striker-is-not-playing-near-the-opponent-goal" */
  bool getTheStrikerIsNotPlayingNearTheOpponentGoal();
  
  /** A function for the symbol "the-striker-is-playing-near-the-own-goal" */
  bool getTheStrikerIsPlayingNearTheOwnGoal();
  
  /** A function for the symbol "the-striker-is-not-playing-near-the-own-goal" */
  bool getTheStrikerIsNotPlayingNearTheOwnGoal();
  
  /** A funtion for the symbol "catch-ball" */
  bool getCaught(); 
  
  /** A function for the symbol "catch-ball-time" */
  double getCatchTime(); 
  
  /** A function for the symbol "search-ball.x" */
  double getSearchBallX();
  
  //!@}
  
  /** The time when the ball was started to catch */
  unsigned long timeWhenBallWasStartedToCatch;
  
  /** The time until the ball was caught */
  unsigned long timeUntilBallWasCaught;
  
  /** Indicates if the robot is stuck (based on obstacles model)*/
  bool robotIsStuck;
  
  /** Indicates that there are close obstacles */
  bool obstaclesAreClose;
};


#endif // __StrategySymbols_h_

/*
* Change Log
* 
* $Log: StrategySymbols.h,v $
* Revision 1.8  2004/05/27 13:31:27  dueffert
* walking evolution stuff separated
*
* Revision 1.7  2004/04/16 14:56:37  dueffert
* cleanup for Martins data flow graphics
*
* Revision 1.6  2004/03/09 08:45:17  dueffert
* second measurement behavior added; calculation of tsart position improved
*
* Revision 1.5  2004/03/08 01:06:59  roefer
* Interfaces should be const
*
* Revision 1.4  2004/02/18 14:49:20  dueffert
* behavior control can now change walking parameters
*
* Revision 1.3  2004/02/10 11:13:18  dueffert
* symbol for evolution added
*
* Revision 1.2  2003/12/06 17:45:33  loetzsch
* replaced Player::playerRole (goalie, defender, striker1, striker2)
* by Player::playerNumber (one, two, three, four)
*
* Revision 1.1  2003/10/06 13:39:29  cvsadm
* Created GT2004 (M.J.)
*
* Revision 1.4  2003/07/07 21:15:39  risler
* search-ball.x added
* search-for-ball works without wlan
*
* Revision 1.3  2003/07/03 11:04:42  juengel
* obstaclesAreClose is calculated
*
* Revision 1.2  2003/07/03 10:15:22  loetzsch
* added symbol "not-stuck-but..."
*
* Revision 1.1.1.1  2003/07/02 09:40:23  cvsadm
* created new repository for the competitions in Padova from the 
* tamara CVS (Tuesday 2:00 pm)
*
* removed unused solutions
*
* Revision 1.12  2003/06/19 21:31:45  juengel
* Added robot-is-stuck.
*
* Revision 1.11  2003/06/19 10:17:17  risler
* symbol another-teammate-is-preparing-a-kick added
*
* Revision 1.10  2003/06/17 19:54:59  risler
* ball caught symbols moved to strategy
*
* Revision 1.9  2003/06/02 14:10:20  loetzsch
* added some hysteresises for the supporter position options
*
* Revision 1.8  2003/05/31 14:38:48  loetzsch
* changed the symbols for the intercept
*
* Revision 1.7  2003/05/28 17:48:35  loetzsch
* better positioning for the supporters
*
* Revision 1.6  2003/05/27 16:52:53  loetzsch
* first passing experiments
*
* Revision 1.5  2003/05/25 22:37:02  loetzsch
* finished the game state options of GT2003
*
* Revision 1.4  2003/05/08 14:25:30  risler
* some bugfixes
*
* Revision 1.3  2003/05/08 13:20:37  loetzsch
* some cleanup
*
* Revision 1.2  2003/05/08 00:22:47  risler
* added strategy symbols
*
* Revision 1.1  2003/05/06 16:28:19  loetzsch
* added class StrategySymbols
*
*/

