/** 
* @file StrategySymbols.cpp
*
* Implementation of class StrategySymbols.
*
* @author Martin Ltzsch
*/

#include "StrategySymbols.h"
#include "Tools/FieldDimensions.h"
#include "Tools/Math/Pose2D.h"

StrategySymbols::StrategySymbols(const BehaviorControlInterfaces& interfaces):
BehaviorControlInterfaces(interfaces),
timeWhenBallWasStartedToCatch(0), timeUntilBallWasCaught(0),
role(BehaviorTeamMessage::goalie),
estimatedTimeToReachBall(0.0),
timeOfLastGoodSelfLocalization(SystemCall::getCurrentSystemTime())
{
}


void StrategySymbols::registerSymbols(Xabsl2Engine& engine)
{
  // "robot-number"
  engine.registerDecimalInputSymbol("robot-number", this,
    (double(Xabsl2FunctionProvider::*)())&StrategySymbols::getRobotNumber);
  
  // "role"
  engine.registerEnumeratedInputSymbol("role", (int *)&role);
  engine.registerEnumeratedInputSymbolEnumElement("role", "goalie", BehaviorTeamMessage::goalie);
  engine.registerEnumeratedInputSymbolEnumElement("role", "striker", BehaviorTeamMessage::striker);
  engine.registerEnumeratedInputSymbolEnumElement("role", "defensive-supporter", BehaviorTeamMessage::defensiveSupporter);
  engine.registerEnumeratedInputSymbolEnumElement("role", "offensive-supporter", BehaviorTeamMessage::offensiveSupporter);
  
  // "estimated-time-to-reach-ball"
  engine.registerDecimalInputSymbol("estimated-time-to-reach-ball", (double *)&estimatedTimeToReachBall);
  
  // "sent-game-state"
  engine.registerEnumeratedOutputSymbol("sent-game-state",(int*)&outgoingBehaviorTeamMessage.gameState);
  engine.registerEnumeratedOutputSymbolEnumElement("sent-game-state","sent-game-state.sleep",BehaviorTeamMessage::initial);
  engine.registerEnumeratedOutputSymbolEnumElement("sent-game-state","sent-game-state.initial",BehaviorTeamMessage::initial);
  engine.registerEnumeratedOutputSymbolEnumElement("sent-game-state","sent-game-state.ready",BehaviorTeamMessage::ready);
  engine.registerEnumeratedOutputSymbolEnumElement("sent-game-state","sent-game-state.playing",BehaviorTeamMessage::playing);
  engine.registerEnumeratedOutputSymbolEnumElement("sent-game-state","sent-game-state.penalized",BehaviorTeamMessage::penalized);
  engine.registerEnumeratedOutputSymbolEnumElement("sent-game-state","sent-game-state.finished",BehaviorTeamMessage::finished);
  engine.registerEnumeratedOutputSymbolEnumElement("sent-game-state","sent-game-state.set",BehaviorTeamMessage::set);
  
  // "another-player-is-in-ready-state"
  engine.registerBooleanInputSymbol("another-player-is-in-ready-state",
    this,(bool(Xabsl2FunctionProvider::*)())&StrategySymbols::getAnotherPlayerIsInReadyState);
  
  // "team-message"
  engine.registerEnumeratedOutputSymbol("team-message",(int*)&outgoingBehaviorTeamMessage.message);
  engine.registerEnumeratedOutputSymbolEnumElement("team-message","team-message.none", BehaviorTeamMessage::none);
  engine.registerEnumeratedOutputSymbolEnumElement("team-message","team-message.just-performed-a-kick",BehaviorTeamMessage::justPerformedAKick);
  engine.registerEnumeratedOutputSymbolEnumElement("team-message","team-message.performing-a-kick",BehaviorTeamMessage::performingAKick);
  engine.registerEnumeratedOutputSymbolEnumElement("team-message","team-message.preparing-a-kick",BehaviorTeamMessage::preparingAKick);
  
  // "another-teammate-is-preparing-a-kick"
  engine.registerBooleanInputSymbol("another-teammate-is-preparing-a-kick",
    this,(bool(Xabsl2FunctionProvider::*)())&StrategySymbols::getAnotherTeammateIsPreparingAKick);
  
  // "another-teammate-is-performing-a-kick"
  engine.registerBooleanInputSymbol("another-teammate-is-performing-a-kick",
    this,(bool(Xabsl2FunctionProvider::*)())&StrategySymbols::getAnotherTeammateIsPerformingAKick);
  
  // "another-teammate-just-performed-a-kick"
  engine.registerBooleanInputSymbol("another-teammate-just-performed-a-kick",
    this,(bool(Xabsl2FunctionProvider::*)())&StrategySymbols::getAnotherTeammateJustPerformedAKick);
  
  // "the-striker-is-playing-near-the-opponent-goal"
  engine.registerBooleanInputSymbol("the-striker-is-playing-near-the-opponent-goal",
    this,(bool(Xabsl2FunctionProvider::*)())&StrategySymbols::getTheStrikerIsPlayingNearTheOpponentGoal);
  
  // "the-striker-is-not-playing-near-the-opponent-goal"
  engine.registerBooleanInputSymbol("the-striker-is-not-playing-near-the-opponent-goal",
    this,(bool(Xabsl2FunctionProvider::*)())&StrategySymbols::getTheStrikerIsNotPlayingNearTheOpponentGoal);
  
  // "the-striker-is-playing-near-the-own-goal"
  engine.registerBooleanInputSymbol("the-striker-is-playing-near-the-own-goal",
    this,(bool(Xabsl2FunctionProvider::*)())&StrategySymbols::getTheStrikerIsPlayingNearTheOwnGoal);
  
  // "the-striker-is-not-playing-near-the-own-goal"
  engine.registerBooleanInputSymbol("the-striker-is-not-playing-near-the-own-goal",
    this,(bool(Xabsl2FunctionProvider::*)())&StrategySymbols::getTheStrikerIsNotPlayingNearTheOwnGoal);
  
  // "catch-ball"
  engine.registerBooleanInputSymbol("catch-ball", this,
    (bool (Xabsl2FunctionProvider::*)())&StrategySymbols::getCaught);
  
  // "catch-ball-time"
  engine.registerDecimalInputSymbol("catch-ball-time",this,
    (double (Xabsl2FunctionProvider::*)())&StrategySymbols::getCatchTime);
  
  // "robot-is-stuck"
  engine.registerBooleanInputSymbol("robot-is-stuck", &robotIsStuck);
  
  // "obstacles-are-close"
  engine.registerBooleanInputSymbol("obstacles-are-close", &obstaclesAreClose);
  
  engine.registerDecimalInputSymbol("search-ball.x", this,
    (double (Xabsl2FunctionProvider::*)())&StrategySymbols::getSearchBallX);
}

void StrategySymbols::update()
{
  estimateTimeToReachBall();
  computeRole();
  
  outgoingBehaviorTeamMessage.estimatedTimeToReachBall = estimatedTimeToReachBall;
  outgoingBehaviorTeamMessage.dynamicRole = role;
  
  if (((executedMotionRequest.motionType == MotionRequest::specialAction
    && 
    (executedMotionRequest.specialActionType == MotionRequest::catchBall2
    || executedMotionRequest.specialActionType == MotionRequest::lookForLandmarksWithBallCaught)
    )
    || 
    (executedMotionRequest.motionType == MotionRequest::walk 
    && executedMotionRequest.walkType == MotionRequest::turnWithBall)
    )
    || headControlMode.headControlMode == HeadControlMode::catchBall)
  {
    if (SystemCall::getTimeSince(timeUntilBallWasCaught) > 1000)
    {
      timeWhenBallWasStartedToCatch = SystemCall::getCurrentSystemTime();
    }
    
    timeUntilBallWasCaught = SystemCall::getCurrentSystemTime();
  }
  
  // robot is stuck
  if(obstaclesModel.getPercentageOfLowDistanceObstaclesInRange(0, pi_2, 300) > 0.2)
  {
    robotIsStuck = true;
    obstaclesAreClose = true;
  }
  else
  {
    robotIsStuck = false;
    if(obstaclesModel.getPercentageOfLowDistanceObstaclesInRange(0, pi_2, 500) > 0.15)
      obstaclesAreClose = true;
    else
      obstaclesAreClose = false;
  }
}

void StrategySymbols::computeRole()
{
  int i;
  if (getPlayer().getPlayerNumber() == Player::one)
  {
    // no role changes for the goalie
    role = BehaviorTeamMessage::goalie;
    return;
  }
  
  // fall-back if no wlan
  if (SystemCall::getTimeSince(teamMessageCollection[0].lastReceivedTimeStamp) > 5000 
    && SystemCall::getTimeSince(teamMessageCollection[1].lastReceivedTimeStamp) > 5000
    && SystemCall::getTimeSince(teamMessageCollection[2].lastReceivedTimeStamp) > 5000)
  {
    switch(getPlayer().getPlayerNumber())
    {
    case Player::two:
      if (SystemCall::getTimeSince (ballPosition.seen.timeWhenLastSeen) < 5000 &&
        ballPosition.seen.x < 75)
        role = BehaviorTeamMessage::striker;
      else
        role = BehaviorTeamMessage::defensiveSupporter;
      break;
    case Player::three:
      if (SystemCall::getTimeSince (ballPosition.seen.timeWhenLastSeen) < 5000 &&
        ballPosition.seen.x > 500)
        role = BehaviorTeamMessage::striker;
      else
        role = BehaviorTeamMessage::offensiveSupporter;
      break;
    case Player::four:
    default:
      if (SystemCall::getTimeSince (ballPosition.seen.timeWhenLastSeen) < 5000 &&
        ballPosition.seen.x > -75)
        role = BehaviorTeamMessage::striker;
      else
        role = BehaviorTeamMessage::offensiveSupporter;
      break;
    }
    return;
  }
  
  // with wlan
  
  // estimate the closest other teammate to the ball
  double minTeammateTime = 100000.0;
  for (i = 0; i < teamMessageCollection.numberOfTeamMessages; i++ ) 
  {
    if ( teamMessageCollection[i].isActual() )
    {
      if ( teamMessageCollection[i].behaviorTeamMessage.dynamicRole != BehaviorTeamMessage::goalie 
        && teamMessageCollection[i].behaviorTeamMessage.gameState == BehaviorTeamMessage::playing) 
      {
        double time = 
          teamMessageCollection[i].behaviorTeamMessage.estimatedTimeToReachBall;
        
        // bonus for current striker
        if ( teamMessageCollection[i].behaviorTeamMessage.dynamicRole == BehaviorTeamMessage::striker ) 
          time -= 500.0;
        
        if (time < minTeammateTime ) 
        {
          minTeammateTime = time;
        }
      }
    }
  }
  
  // bonus for current striker
  if (role == BehaviorTeamMessage::striker)
    minTeammateTime += 1000.0;
  
  // assign striker-role if nearest to the ball
  if ( estimatedTimeToReachBall < minTeammateTime )
  {
    role = BehaviorTeamMessage::striker;
    return;
  }
  
  // assign supporting roles
  double maxTeammateX = (double)xPosOwnGroundline;
  for (i = 0; i < teamMessageCollection.numberOfTeamMessages; i++ ) 
  {
    if ( teamMessageCollection[i].isActual() )
    {
      if ( teamMessageCollection[i].behaviorTeamMessage.dynamicRole != BehaviorTeamMessage::striker 
        && teamMessageCollection[i].behaviorTeamMessage.dynamicRole != BehaviorTeamMessage::goalie 
        && teamMessageCollection[i].behaviorTeamMessage.gameState == BehaviorTeamMessage::playing)
      {
        double teammateX = 
          teamMessageCollection[i].robotPose.translation.x;
        
        // bonus for current offensive supporter
        if ( teamMessageCollection[i].behaviorTeamMessage.dynamicRole == BehaviorTeamMessage::offensiveSupporter ) 
          teammateX += 300.0;
        
        if (teammateX > maxTeammateX ) 
        {
          maxTeammateX = teammateX;
        }
      }
    }
  }
  
  // bonus for current offensive supporter
  if ( role == BehaviorTeamMessage::offensiveSupporter ) 
    maxTeammateX -= 300.0;
  
  if ( robotPose.translation.x >= maxTeammateX ) 
    role = BehaviorTeamMessage::offensiveSupporter;
  else 
    role = BehaviorTeamMessage::defensiveSupporter;
}

void StrategySymbols::estimateTimeToReachBall()
{
  // stay striker if ball is caught
  if (getCaught())
  {
    estimatedTimeToReachBall = 0.0;
    return;
  }
  
  // account for distance to ball
  estimatedTimeToReachBall = Geometry::distanceTo(robotPose.translation, ballPosition.seen) / 0.2;
  
  // account if the robot is between the ball and the opponent goal
  
  // the position of the robot
  Vector2<double> robotPosition = robotPose.translation;
  double angleToLeftOpponentGoalPost = Geometry::angleTo(robotPosition, Vector2<double>(xPosOpponentGroundline,yPosLeftGoal));
  double angleToRightOpponentGoalPost = Geometry::angleTo(robotPosition, Vector2<double>(xPosOpponentGroundline,yPosRightGoal));
  if(angleToLeftOpponentGoalPost < angleToRightOpponentGoalPost)
  {
    angleToLeftOpponentGoalPost += pi2;
  }
  double angleToOpponentGoal= (angleToLeftOpponentGoalPost + angleToRightOpponentGoalPost) / 2.0;
  double absoluteAngleToBall = Geometry::angleTo(robotPosition,ballPosition.seen);
  double angleBetweenBallAndOpponentGoal = normalize(angleToOpponentGoal - absoluteAngleToBall);
  
  estimatedTimeToReachBall += 400.0 * fabs(angleBetweenBallAndOpponentGoal);
  
  // longer if ball not seen
  estimatedTimeToReachBall += 2.0 * SystemCall::getTimeSince(ballPosition.seen.timeWhenLastSeen);
  
  // test for obstacles
  /*
  if ( isOpponentBetweenRobotAndBall(robotPose) || 
  isOwnBetweenRobotAndBall (robotPose) )
  time += 5000.0;
  */
  
}

bool StrategySymbols::getAnotherPlayerIsInReadyState()
{
  for (int i=0; i<3;i++)
  {
    if (teamMessageCollection[i].isActual())
    {
      if (teamMessageCollection[i].behaviorTeamMessage.gameState 
        == BehaviorTeamMessage::ready)
      {
        return true;
      }
    }
  }
  return false;
}

bool StrategySymbols::getAnotherTeammateIsPreparingAKick()
{
  for (int i=0; i<3;i++)
  {
    if (teamMessageCollection[i].isActual())
    {
      if (teamMessageCollection[i].behaviorTeamMessage.message
        == BehaviorTeamMessage::preparingAKick)
      {
        return true;
      }
    }
  }
  return false;
}

bool StrategySymbols::getAnotherTeammateIsPerformingAKick()
{
  for (int i=0; i<3;i++)
  {
    if (teamMessageCollection[i].isActual())
    {
      if (teamMessageCollection[i].behaviorTeamMessage.message
        == BehaviorTeamMessage::performingAKick)
      {
        return true;
      }
    }
  }
  return false;
}

bool StrategySymbols::getAnotherTeammateJustPerformedAKick()
{
  for (int i=0; i<3;i++)
  {
    if (teamMessageCollection[i].isActual())
    {
      if (teamMessageCollection[i].behaviorTeamMessage.message
        == BehaviorTeamMessage::justPerformedAKick)
      {
        return true;
      }
    }
  }
  return false;
}

bool StrategySymbols::getTheStrikerIsPlayingNearTheOpponentGoal()
{
  for (int i=0; i<3;i++)
  {
    if (teamMessageCollection[i].isActual())
    {
      if (teamMessageCollection[i].behaviorTeamMessage.dynamicRole == BehaviorTeamMessage::striker
        && teamMessageCollection[i].robotPose.translation.x > 1400)
      {
        return true;
      }
    }
  }
  return false;
}

bool StrategySymbols::getTheStrikerIsNotPlayingNearTheOpponentGoal()
{
  for (int i=0; i<3;i++)
  {
    if (teamMessageCollection[i].isActual())
    {
      if (teamMessageCollection[i].behaviorTeamMessage.dynamicRole == BehaviorTeamMessage::striker
        && teamMessageCollection[i].robotPose.translation.x < 1200)
      {
        return true;
      }
    }
  }
  return false;
}

bool StrategySymbols::getTheStrikerIsPlayingNearTheOwnGoal()
{
  for (int i=0; i<3;i++)
  {
    if (teamMessageCollection[i].isActual())
    {
      if (teamMessageCollection[i].behaviorTeamMessage.dynamicRole == BehaviorTeamMessage::striker
        && teamMessageCollection[i].robotPose.translation.x < -1200)
      {
        return true;
      }
    }
  }
  return false;
}

bool StrategySymbols::getTheStrikerIsNotPlayingNearTheOwnGoal()
{
  for (int i=0; i<3;i++)
  {
    if (teamMessageCollection[i].isActual())
    {
      if (teamMessageCollection[i].behaviorTeamMessage.dynamicRole == BehaviorTeamMessage::striker
        && teamMessageCollection[i].robotPose.translation.x > -1000)
      {
        return true;
      }
    }
  }
  return false;
}

bool StrategySymbols::getCaught()
{
  return (SystemCall::getTimeSince(timeUntilBallWasCaught) < 500);  
}

double StrategySymbols::getCatchTime()
{
  return (SystemCall::getTimeSince(timeUntilBallWasCaught) < 500?
    timeUntilBallWasCaught - timeWhenBallWasStartedToCatch : 0);
}

double StrategySymbols::getSearchBallX()
{
  // fall-back if no wlan
  if (SystemCall::getTimeSince(teamMessageCollection[0].lastReceivedTimeStamp) > 5000 
    && SystemCall::getTimeSince(teamMessageCollection[1].lastReceivedTimeStamp) > 5000
    && SystemCall::getTimeSince(teamMessageCollection[2].lastReceivedTimeStamp) > 5000)
  {
    switch(getPlayer().getPlayerNumber())
    {
    case Player::two:
      return -1000;
    case Player::three:
      return 0;
    case Player::four:
    default:
      return 1000;
    }
  }
  else
  {
    switch (role)
    {
    case BehaviorTeamMessage::defensiveSupporter:
      return -1000;
    case BehaviorTeamMessage::offensiveSupporter:
      return 1000;
    case BehaviorTeamMessage::striker:
    default:
      return 0;
    }
  }
}

/*
* Change Log
* 
* $Log: StrategySymbols.cpp,v $
* Revision 1.23  2004/05/27 13:31:27  dueffert
* walking evolution stuff separated
*
* Revision 1.22  2004/05/23 13:30:31  dueffert
* some start positions corrected
*
* Revision 1.21  2004/05/20 17:18:51  dueffert
* automatic measurement (hopefully) finalized
*
* Revision 1.20  2004/05/19 08:00:27  dueffert
* automatic walk speed measurement significantly improved
*
* Revision 1.19  2004/05/17 11:16:51  dueffert
* blind measurement improved
*
* Revision 1.18  2004/04/19 10:34:55  dueffert
* parameters tuned to allow measurement oft *fast* walking
*
* Revision 1.17  2004/04/16 14:56:37  dueffert
* cleanup for Martins data flow graphics
*
* Revision 1.16  2004/03/25 17:40:15  loetzsch
* adaptations to changes in the game controller and the led request
*
* Revision 1.15  2004/03/19 09:21:23  dueffert
* ability to measure freely choosen motion request added
*
* Revision 1.14  2004/03/15 12:37:54  dueffert
* start position improved
*
* Revision 1.13  2004/03/10 15:00:00  dueffert
* copy'n'past bug fixed; starting position improved
*
* Revision 1.12  2004/03/10 10:00:52  dueffert
* copy'n'paste bug fixed; start position tweaking
*
* Revision 1.11  2004/03/09 08:45:16  dueffert
* second measurement behavior added; calculation of tsart position improved
*
* Revision 1.10  2004/03/08 01:06:59  roefer
* Interfaces should be const
*
* Revision 1.9  2004/03/03 08:37:51  dueffert
* measurement start position is now calculated instead of predefined; file beautified
*
* Revision 1.8  2004/03/01 15:00:47  dueffert
* small start position improvements
*
* Revision 1.7  2004/02/29 13:42:58  dueffert
* UDParametersSet symmetries handled
*
* Revision 1.6  2004/02/27 16:43:21  dueffert
* small improvements
*
* Revision 1.5  2004/02/23 16:48:11  dueffert
* several improvements for measurement of walking
*
* Revision 1.4  2004/02/18 14:49:20  dueffert
* behavior control can now change walking parameters
*
* Revision 1.3  2004/02/10 11:13:18  dueffert
* symbol for evolution added
*
* Revision 1.2  2003/12/06 17:45:33  loetzsch
* replaced Player::playerRole (goalie, defender, striker1, striker2)
* by Player::playerNumber (one, two, three, four)
*
* Revision 1.1  2003/10/06 13:39:29  cvsadm
* Created GT2004 (M.J.)
*
* Revision 1.11  2003/07/07 21:15:39  risler
* search-ball.x added
* search-for-ball works without wlan
*
* Revision 1.10  2003/07/07 15:05:57  risler
* no wlan fallback :
* 5 seconds after wlan failure
* striker1 goes slightly into own half as well
*
* Revision 1.9  2003/07/06 21:58:25  risler
* role change : orientation influence reduced
*
* Revision 1.8  2003/07/05 21:27:19  risler
* bugfix: no wlan fallback, time for out of role striker increased
*
* Revision 1.7  2003/07/05 10:15:58  risler
* no wlan fallback for role change improved
*
* Revision 1.6  2003/07/04 16:30:34  loetzsch
* bonus for the current Striker in computeRole()
*
* Revision 1.5  2003/07/03 16:08:17  loetzsch
* renamed "robot-is-not-stuck-but-obstacles-are-close" to "obstacles-are-close"
*
* Revision 1.4  2003/07/03 11:04:42  juengel
* obstaclesAreClose is calculated
*
* Revision 1.3  2003/07/03 10:15:22  loetzsch
* added symbol "not-stuck-but..."
*
* Revision 1.2  2003/07/02 19:15:44  loetzsch
* ::computeRole()  does not use BehaviorTeamMessage::whoIsOnline() anymore
*
* Revision 1.1.1.1  2003/07/02 09:40:23  cvsadm
* created new repository for the competitions in Padova from the 
* tamara CVS (Tuesday 2:00 pm)
*
* removed unused solutions
*
* Revision 1.22  2003/06/26 16:20:29  loetzsch
* some role assignment experiments
*
* Revision 1.21  2003/06/23 19:13:21  loetzsch
* shifted the position of  "the-striker-is-playing-near-the-opponent-goal" for 20cm to the opponent goal
*
* Revision 1.20  2003/06/22 11:48:54  juengel
* new robot is stuck condition: > 20 percent of range is closer than 300mm
*
* Revision 1.19  2003/06/21 15:26:46  juengel
* obstaclesModel.getPercentageOfLowDistanceObstaclesInRange is used to calculate robot-is-stuck
*
* Revision 1.18  2003/06/20 20:15:24  juengel
* Renamed some methods in ObstaclesModel.
*
* Revision 1.17  2003/06/20 13:58:31  juengel
* Added calculation of robot is stuck.
*
* Revision 1.16  2003/06/19 21:31:45  juengel
* Added robot-is-stuck.
*
* Revision 1.15  2003/06/19 10:17:17  risler
* symbol another-teammate-is-preparing-a-kick added
*
* Revision 1.14  2003/06/17 19:54:59  risler
* ball caught symbols moved to strategy
*
* Revision 1.13  2003/06/02 14:10:20  loetzsch
* added some hysteresises for the supporter position options
*
* Revision 1.12  2003/05/31 14:38:48  loetzsch
* changed the symbols for the intercept
*
* Revision 1.11  2003/05/30 11:14:49  dueffert
* temporary bugfix
*
* Revision 1.10  2003/05/28 17:48:35  loetzsch
* better positioning for the supporters
*
* Revision 1.9  2003/05/27 16:52:53  loetzsch
* first passing experiments
*
* Revision 1.8  2003/05/26 13:45:02  loetzsch
* some improvements
*
* Revision 1.7  2003/05/25 22:37:02  loetzsch
* finished the game state options of GT2003
*
* Revision 1.6  2003/05/14 09:24:49  risler
* current role bonus hyteresis no longer in estimateTimeToReachBall
* added current role bonus for offensive supporter
*
* Revision 1.5  2003/05/08 14:25:30  risler
* some bugfixes
*
* Revision 1.4  2003/05/08 13:20:37  loetzsch
* some cleanup
*
* Revision 1.3  2003/05/08 10:20:35  dueffert
* bugs fixed
*
* Revision 1.2  2003/05/08 00:22:46  risler
* added strategy symbols
*
* Revision 1.1  2003/05/06 16:28:19  loetzsch
* added class StrategySymbols
*
*/

