/** 
* @file ConfigurationSymbols.cpp
*
* Implementation of class ConfigurationSymbols.
*
* @author Max Risler
*/

#include "ConfigurationSymbols.h"

ConfigurationSymbols::ConfigurationSymbols(
                           const BehaviorControlInterfaces& interfaces,
                           GT2003BehaviorConfiguration& configuration)
: BehaviorControlInterfaces(interfaces),
  configuration(configuration)
{
  currentKickoff = 0;
}

void ConfigurationSymbols::registerSymbols(Xabsl2Engine& engine)
{
  engine.registerDecimalInputSymbol("current-kickoff-variant",this,
    (double (Xabsl2FunctionProvider::*)())&ConfigurationSymbols::getCurrentKickoff);

  engine.registerDecimalInputSymbol("go-to-kickoff-position.max-speed",&(configuration.goToKickoffMaxSpeed));

  engine.registerDecimalInputSymbol("own-kickoff.angle",this,
    (double (Xabsl2FunctionProvider::*)())&ConfigurationSymbols::getOwnKickoffAngle);
  engine.registerDecimalInputSymbol("own-kickoff.x",this,
    (double (Xabsl2FunctionProvider::*)())&ConfigurationSymbols::getOwnKickoffX);
  engine.registerDecimalInputSymbol("own-kickoff.y",this,
    (double (Xabsl2FunctionProvider::*)())&ConfigurationSymbols::getOwnKickoffY);

  engine.registerDecimalInputSymbol("opponent-kickoff.angle",this,
    (double (Xabsl2FunctionProvider::*)())&ConfigurationSymbols::getOpponentKickoffAngle);
  engine.registerDecimalInputSymbol("opponent-kickoff.x",this,
    (double (Xabsl2FunctionProvider::*)())&ConfigurationSymbols::getOpponentKickoffX);
  engine.registerDecimalInputSymbol("opponent-kickoff.y",this,
    (double (Xabsl2FunctionProvider::*)())&ConfigurationSymbols::getOpponentKickoffY);

  engine.registerDecimalInputSymbol("ball-near-distance",&(configuration.ballNearDistance));
  engine.registerDecimalInputSymbol("ball-not-near-distance",&(configuration.ballNotNearDistance));
  engine.registerDecimalInputSymbol("at-ball-distance",&(configuration.atBallDistance));
  engine.registerDecimalInputSymbol("not-at-ball-distance",&(configuration.notAtBallDistance));

  // "dribble.speed-y"
  engine.registerDecimalInputSymbol("dribble.speed-y",this,
    (double (Xabsl2FunctionProvider::*)())&ConfigurationSymbols::getDribbleSpeedY);
}

void ConfigurationSymbols::update()
{
  currentKickoff = gameControlData.data.opponentScore % GT2003BehaviorConfiguration::numOfKickoff;
}

double ConfigurationSymbols::getCurrentKickoff()
{
  return currentKickoff;
}

double ConfigurationSymbols::getOwnKickoffAngle()
{
  switch (getPlayer().getPlayerNumber())
  {
    case Player::one: // robot number 1
    case Player::two: // robot number 2
    default:
      return 0;
    case Player::three: // robot number 3
      return configuration.offensiveSupporterOwnKickoffAngle[currentKickoff];
    case Player::four: // robot number 4
      return configuration.strikerOwnKickoffAngle[currentKickoff];
  }
}

double ConfigurationSymbols::getOwnKickoffX()
{
  switch (getPlayer().getPlayerNumber())
  {
    case Player::one: // robot number 1
    default:
      return configuration.goalie.x;
    case Player::two: // robot number 2
      return configuration.defensiveSupporter.x;
    case Player::three: // robot number 3
      return configuration.offensiveSupporterOwnKickoff[currentKickoff].x;
    case Player::four: // robot number 4
      return configuration.strikerOwnKickoff[currentKickoff].x;
  }
}

double ConfigurationSymbols::getOwnKickoffY()
{
  switch (getPlayer().getPlayerNumber())
  {
    case Player::one: // robot number 1
    default:
      return configuration.goalie.y;
    case Player::two: // robot number 2
      return configuration.defensiveSupporter.y;
    case Player::three: // robot number 3
      return configuration.offensiveSupporterOwnKickoff[currentKickoff].y;
    case Player::four: // robot number 4
      return configuration.strikerOwnKickoff[currentKickoff].y;
  }
}

double ConfigurationSymbols::getOpponentKickoffAngle()
{
  switch (getPlayer().getPlayerNumber())
  {
    case Player::one: // robot number 1
    case Player::two: // robot number 2
    case Player::three: // robot number 3
    case Player::four: // robot number 4
    default:
      return 0;
  }
}

double ConfigurationSymbols::getOpponentKickoffX()
{
  switch (getPlayer().getPlayerNumber())
  {
    case Player::one: // robot number 1
    default:
      return configuration.goalie.x;
    case Player::two: // robot number 2
      return configuration.defensiveSupporter.x;
    case Player::three: // robot number 3
      return configuration.offensiveSupporterOpponentKickoff.x;
    case Player::four: // robot number 4
      return configuration.strikerOpponentKickoff.x;
  }
}

double ConfigurationSymbols::getOpponentKickoffY()
{
  switch (getPlayer().getPlayerNumber())
  {
    case Player::one: // robot number 1
    default:
      return configuration.goalie.y;
    case Player::two: // robot number 2
      return configuration.defensiveSupporter.y;
    case Player::three: // robot number 3
      return configuration.offensiveSupporterOpponentKickoff.y;
    case Player::four: // robot number 4
      return configuration.strikerOpponentKickoff.y;
  }
}

double ConfigurationSymbols::getDribbleSpeedY()
{
  double distanceY = Geometry::distanceTo(robotPose.getPose(),ballPosition.seen) * 
    sin(Geometry::angleTo(robotPose.getPose(),ballPosition.seen));

  if (distanceY < -175)
    return -200;
  else if (distanceY < 0)
    return (150 + 2 * distanceY); 
  else if (distanceY < 175)
    return (-150 + 2 * distanceY);
  else return 200;
}


/*
* Change Log
* 
* $Log: ConfigurationSymbols.cpp,v $
* Revision 1.4  2004/03/08 01:06:58  roefer
* Interfaces should be const
*
* Revision 1.3  2003/12/06 17:45:33  loetzsch
* replaced Player::playerRole (goalie, defender, striker1, striker2)
* by Player::playerNumber (one, two, three, four)
*
* Revision 1.2  2003/10/22 22:18:47  loetzsch
* prepared the cloning of the GT2003BehaviorControl
*
* Revision 1.1  2003/10/06 13:39:28  cvsadm
* Created GT2004 (M.J.)
*
* Revision 1.1.1.1  2003/07/02 09:40:23  cvsadm
* created new repository for the competitions in Padova from the 
* tamara CVS (Tuesday 2:00 pm)
*
* removed unused solutions
*
* Revision 1.3  2003/06/21 15:01:13  risler
* kickoff with more than one player possible
* different kickoff variants added
*
* Revision 1.2  2003/05/26 09:51:19  risler
* configuration has to be reference
*
* Revision 1.1  2003/05/25 14:41:40  risler
* added GT2003BehaviorConfiguration
*
*/

