/** 
* @file GT2003SimpleBasicBehaviors.cpp
*
* Implementation of basic behaviors defined in simple-basic-behaviors.xml.
*
* @author Uwe Dffert
* @author Jan Hoffmann
* @author Matthias Jngel
* @author Martin Ltzsch
* @author Max Risler
*/

#include "Tools/Math/PIDsmoothedValue.h"
#include "GT2003SimpleBasicBehaviors.h"
#include "Tools/Math/Geometry.h"
#include "Tools/Math/Common.h"
#include "Tools/Streams/OutStreams.h"

#include "Tools/Debugging/Debugging.h"
#include "Platform/GTAssert.h"

void GT2003SimpleBasicBehaviors::registerBasicBehaviors(Xabsl2Engine& engine)
{
  engine.registerBasicBehavior(basicBehaviorAvoidObstacles);
  engine.registerBasicBehavior(basicBehaviorGetBehindBall);
  engine.registerBasicBehavior(basicBehaviorGoToBall);
  engine.registerBasicBehavior(basicBehaviorGoToBallWithDirection);
  engine.registerBasicBehavior(basicBehaviorGoToPoint);
  engine.registerBasicBehavior(basicBehaviorGoToPointPrecisely);
  engine.registerBasicBehavior(basicBehaviorGoToPointAndAvoidObstacles);
}

void GT2003BasicBehaviorGoToBallWithDirection::execute()
{
  //accelerationRestrictor.saveLastWalkParameters();
  
  if (maxSpeed == 0)
  {
    maxSpeed = 200;
  }
  
  Vector2<double> ball = ballPosition.getKnownPosition(
    BallModel::behaviorControlTimeAfterWhichCommunicatedBallsAreAccepted);
  
  Vector2<double> pointToReach(ball.x-cos(direction)*distanceX-sin(direction)*distanceY,
    ball.y-sin(direction)*distanceX-cos(direction)*distanceY);
  
  
  double destinationDistance = Geometry::distanceTo(robotPose.getPose(),pointToReach);
  double destinationX = destinationDistance * cos(Geometry::angleTo(robotPose.getPose(),pointToReach));
  double destinationY = destinationDistance * sin(Geometry::angleTo(robotPose.getPose(),pointToReach));
  
  /*
  Vector2<double> pointToReachMirrored(ball.x-cos(direction)*distanceX+sin(direction)*distanceY,
  ball.y-sin(direction)*distanceX+cos(direction)*distanceY);
  
  double destinationDistanceMirrored = Geometry::distanceTo(robotPose.getPose(),pointToReach);
  double destinationYMirrored = destinationDistanceMirrored * sin(Geometry::angleTo(robotPose.getPose(),pointToReachMirrored));
  
  if (fabs(destinationYMirrored) < fabs(destinationY))
  {
  destinationY = destinationYMirrored;
  }
  */
  
  if(destinationDistance < 1)
  {
    destinationDistance = 1;
  }
  
  double minimalTime=destinationDistance/maxSpeed;
  
  double angleDiff=direction-robotPose.rotation;
  if (angleDiff<-pi)
  {
    angleDiff += pi2;
  }
  else if (angleDiff >=pi)
  {
    angleDiff -= pi2;
  }
  
  double maxTurnSpeed = fromDegrees(60);
  double turnTime = angleDiff/maxTurnSpeed;
  if (turnTime>minimalTime)
  {
    minimalTime=turnTime;
  }
  minimalTime += 0.2; //get slower near to destination
  
  motionRequest.motionType = MotionRequest::walk;
  motionRequest.walkType = MotionRequest::normal;
  
  motionRequest.walkParams.translation.x = destinationX / minimalTime;
  motionRequest.walkParams.translation.y = destinationY / minimalTime;
  
  motionRequest.walkParams.rotation = angleDiff;
  
  if (motionRequest.walkParams.rotation > maxTurnSpeed)
    motionRequest.walkParams.rotation = maxTurnSpeed;
  if (motionRequest.walkParams.rotation < -maxTurnSpeed) 
    motionRequest.walkParams.rotation = -maxTurnSpeed;
  
  if ((fabs(toDegrees(angleDiff))<10)&&(destinationDistance<10))
  {
    motionRequest.motionType = MotionRequest::stand;
  }
}

void GT2003BasicBehaviorGoToBall::execute()
{
  //accelerationRestrictor.saveLastWalkParameters();
  
  if (maxSpeed == 0) maxSpeed = 200;
  
  double maxSpeedX = maxSpeed;
  double maxSpeedY = maxSpeed;
  
  double maxTurnSpeed = fromDegrees(60);
  
  double timeToReachX, timeToReachY, timeToTurnToAngle, estimatedTimeToReachDestination;
  
  Vector2<double> ball = ballPosition.getKnownPosition(
    BallModel::behaviorControlTimeAfterWhichCommunicatedBallsAreAccepted);
  double ballDistance = Geometry::distanceTo(robotPose.getPose(),ball);
  double ballX = ballDistance * cos(Geometry::angleTo(robotPose.getPose(),ball)) + 50;
  double ballY = ballDistance * sin(Geometry::angleTo(robotPose.getPose(),ball));
  
  if(ballDistance == 0)
    ballDistance = 1;
  double ballAngle = Geometry::angleTo(robotPose.getPose(),ball);
  
  double diffX = distanceAtEnd / ballDistance * ballX;
  double diffY = distanceAtEnd / ballDistance * ballY;
  
  Vector2<double> destination(ballX - diffX, ballY - diffY);
  double distanceToDestination;
  distanceToDestination = fabs(ballDistance - distanceAtEnd);
  
  timeToReachX = fabs(destination.x) / maxSpeedX;
  timeToReachY = fabs(destination.y) / maxSpeedY;
  timeToTurnToAngle = ballAngle / maxTurnSpeed;
  estimatedTimeToReachDestination = 
    (timeToReachX < timeToReachY) ? timeToReachY : timeToReachX;
  estimatedTimeToReachDestination = 
    (estimatedTimeToReachDestination < timeToTurnToAngle) ? timeToTurnToAngle : estimatedTimeToReachDestination;
  
  motionRequest.motionType = MotionRequest::walk;
  motionRequest.walkType = MotionRequest::normal;
  
  motionRequest.walkParams.translation.x = 
    destination.x / estimatedTimeToReachDestination;
  motionRequest.walkParams.translation.y = 
    destination.y / estimatedTimeToReachDestination;
  
  
  motionRequest.walkParams.rotation = ballAngle;
  
  if (motionRequest.walkParams.rotation > maxTurnSpeed)
    motionRequest.walkParams.rotation = maxTurnSpeed;
  if (motionRequest.walkParams.rotation < -maxTurnSpeed) 
    motionRequest.walkParams.rotation = -maxTurnSpeed;
  
  
  if ((fabs(toDegrees(ballAngle))<10)&&(distanceToDestination<20))
  {
    motionRequest.motionType = MotionRequest::stand;
  }
  else
  {
    //    accelerationRestrictor.restrictAccelerations(250,250,180);
  }
  
}

void GT2003BasicBehaviorGetBehindBall::execute()
{
  // This was adapted from the XabslSkillGetBehindBall written by Max Risler.
  
  accelerationRestrictor.saveLastWalkParameters();
  
  double dir = fromDegrees(angle); 
  
  Vector2<double> ball = ballPosition.getKnownPosition(
    BallModel::behaviorControlTimeAfterWhichCommunicatedBallsAreAccepted);

  double destX = ball.x;
  double destY = ball.y;
  
  double posX = robotPose.getPose().translation.x;
  double posY = robotPose.getPose().translation.y;
  double posDir = robotPose.getPose().getAngle();
  
  Vector2<double> destPosDiff = ball - robotPose.getPose().translation;
  
  double absDestDir = atan2(destPosDiff.y, destPosDiff.x);
  double relDestDir = normalize(absDestDir - posDir);
  double destDist = destPosDiff.abs();
  
  if (destDist > 1000.0)
  {
    gotoPoint(destX,destY,maxSpeed);
  }
  else if (destDist >distance* 1.2)
  {
    // calculate destination point
    double px,py;
    
    px = destX -distance* cos(dir);
    py = destY -distance* sin(dir);
    
    double distToP = sqrt((posX-px)*(posX-px) + (posY-py)*(posY-py));
    
    if (destDist < distToP)
    {
      // dest is closer than destination point
      // walk to tangent
      double a = acos(distance/destDist);
      
      if ((leftRight == 0 && normalize(dir - absDestDir) > 0) ||
        leftRight > 0)
      {
        px = destX -distance* cos(absDestDir + a);
        py = destY -distance* sin(absDestDir + a);
      } else {
        px = destX -distance* cos(absDestDir - a);
        py = destY -distance* sin(absDestDir - a);
      }
      // distance is distance to tangent plus distance on perimeter
      distToP = sqrt((posX-px)*(posX-px) + (posY-py)*(posY-py));
      distToP +=distance* normalize(fabs((dir - absDestDir)));
    }
    gotoPoint(px,py,maxSpeed);
  }
  else
  {
    // circle around dest
    double diffAngle = normalize(posDir-dir);
    if (diffAngle > pi_2 && leftRight > 0) diffAngle -= pi2;
    if (diffAngle < -pi_2 && leftRight < 0) diffAngle += pi2;
    
    motionRequest.walkParams.translation.x = destDist - distance;
    motionRequest.walkParams.translation.y = diffAngle * distance;
    motionRequest.walkParams.rotation = relDestDir;
  }
  
  motionRequest.motionType = MotionRequest::walk;
  motionRequest.walkType = MotionRequest::normal;
  
  accelerationRestrictor.restrictAccelerations(300,300,180);
}

void GT2003BasicBehaviorGetBehindBall::gotoPoint(double x, double y, double maxSpeed)
{
  double maxTurnSpeed = fromDegrees(60);
  
  double timeToReachX, timeToReachY, estimatedTimeToReachDestination;
  double distance = Geometry::distanceTo(robotPose.getPose(),Vector2<double>(x,y));
  double distanceX = distance
    * cos(Geometry::angleTo(robotPose.getPose(),Vector2<double>(x,y)));
  double distanceY = distance
    * sin(Geometry::angleTo(robotPose.getPose(),Vector2<double>(x,y)));
  
  if(distance == 0)
    distance = 1;
  double angle = Geometry::angleTo(robotPose.getPose(),ballPosition.getKnownPosition(
    BallModel::behaviorControlTimeAfterWhichCommunicatedBallsAreAccepted));

  Vector2<double> destination(distanceX, distanceY);
  
  timeToReachX = fabs(destination.x) / maxSpeed;
  timeToReachY = fabs(destination.y) / maxSpeed;
  
  estimatedTimeToReachDestination = 
    (timeToReachX < timeToReachY) ? timeToReachY : timeToReachX;
  
  motionRequest.motionType = MotionRequest::walk;
  motionRequest.walkType = MotionRequest::normal;
  
  motionRequest.walkParams.translation.x = 
    destination.x / estimatedTimeToReachDestination;
  motionRequest.walkParams.translation.y = 
    destination.y / estimatedTimeToReachDestination;
  
  motionRequest.walkParams.rotation = angle;
  
  if (motionRequest.walkParams.rotation > maxTurnSpeed)
    motionRequest.walkParams.rotation = maxTurnSpeed;
  if (motionRequest.walkParams.rotation < -maxTurnSpeed) 
    motionRequest.walkParams.rotation = -maxTurnSpeed;
}



void GT2003BasicBehaviorGoToPointAndAvoidObstacles::execute()
{
  double turnDistance = 400,
    angleToDestination = Geometry::angleTo(robotPose.getPose(),Vector2<double>(x,y)),
    distanceToDestination = Geometry::distanceTo(robotPose.getPose(),Vector2<double>(x,y));
  double targetX, targetAngle;//, obstacleAvoidAngle;
  
  if (maxSpeed == 0) maxSpeed = 200;
  
  targetX = obstaclesModel.getDistanceInCorridor(0.0, 300);
  targetX -= 300; 
  if (targetX > maxSpeed)
    targetX = maxSpeed;
  if (targetX < -50)
    targetX = -50;
  if (obstaclesModel.getDistanceInCorridor(pi, 300) < 300)
    targetX = 0;
  
  speedPhi.setWeightP(0.7);
  speedX.setWeightP(0.7);
  targetAngle = 0.0;
  
  
  if (targetX > .4 * maxSpeed)
  {
    
    double leftForward = obstaclesModel.getTotalFreeSpaceInSector(.4, .79, turnDistance);
    double rightForward = obstaclesModel.getTotalFreeSpaceInSector(-.4, .79, turnDistance);
    
    // go in the direction of the next free sector ...
    targetAngle = obstaclesModel.getAngleOfNextFreeSector(pi/4, angleToDestination, (int )turnDistance);
    // ... but check if there's enough space for walking
    if (obstaclesModel.getDistanceInCorridor(targetAngle, 300) < turnDistance)
      targetAngle =	2*(leftForward - rightForward) / (leftForward + rightForward);
    
    if (fabs(targetAngle) < .2)
    {
      // only do this if no obstacle avoidance was necessary in the 
      // past 1 sec
      if (goToDestCounter > 10)
      {
        targetAngle = angleToDestination;
        //OUTPUT(idText, text, "DESTINATION: " << targetAngle);
      }
      // destination within reach
      else if ((obstaclesModel.getDistanceInCorridor(angleToDestination, 300) > distanceToDestination-100) &&
        (goToDestCounter > 5))
      {
        targetAngle = angleToDestination;
        //OUTPUT(idText, text, "GO FOR THE DESTINATION");
      }
      goToDestCounter++;
      targetAngle = (double)sgn(targetAngle) * .5;
    }
    else 
    {
      if (fabs(angleToDestination) < pi/1.8)
      {
        goToDestCounter = 0;
        //OUTPUT(idText, text, "AVOID: " << targetAngle);
      }
      if (fabs(targetAngle) > .5)
        targetAngle = (double)sgn(targetAngle) * .5;
    }
  }
  else 
  {
    if (stickToBackOffDirection > 15)
    {
      if (backOffAngle < 0)
        backOffAngle = -pi;
      else
        backOffAngle = pi;
      stickToBackOffDirection = 0;
    }
    targetAngle = backOffAngle;
    goToDestCounter = -10;
    
    //		OUTPUT(idText, text, "BACKING OFF " << targetAngle);
  }
  stickToBackOffDirection++;
  
  //OUTPUT(idText, text, "... " << targetAngle);
  backOffAngle = targetAngle;
  
  if(obstaclesModel.getDistanceInCorridor((double)sgn(targetAngle) * pi/2, 300) < 300)
    targetAngle = 0;  
  
  motionRequest.motionType = MotionRequest::walk;
  motionRequest.walkType = MotionRequest::normal;
  motionRequest.walkParams.translation.x = speedX.approximateVal(targetX);
  motionRequest.walkParams.translation.y = speedY.getVal();
  motionRequest.walkParams.rotation = speedPhi.approximateVal(targetAngle);
}








void GT2003BasicBehaviorAvoidObstacles::execute()
{
  if(turnDistance == 0) turnDistance = 300;
  if(walkSpeed == 0) walkSpeed = 250;
  if(turnSpeed == 0) turnSpeed = 1.2;
  
  double distanceToObstacle = 200;
  if(stopDistance) distanceToObstacle = stopDistance;
  
  static PIDsmoothedValue speedPhi(0, .1, 0, 0, -pi/4, pi/4, pi/10);
  static PIDsmoothedValue speedX(0, .1, 0, 0, -walkSpeed, walkSpeed, 100);
  static PIDsmoothedValue speedY(0, .1, 0, 0, -walkSpeed/2, walkSpeed/2, 100);
  static double turnDirection = 0;
  speedX.setMax(walkSpeed);
  
  speedX.approximateVal(obstaclesModel.corridorInFront - distanceToObstacle);
  
  /*    
  // if something is in the way but far away
  // or it is close and turning was not set...
  if(((obstaclesModel.distance[ObstaclesModel::front] < turnDistance) && 
  (obstaclesModel.distance[ObstaclesModel::front] >= distanceToObstacle)) 
  ||
  ((obstaclesModel.distance[ObstaclesModel::front] < turnDistance) && 
  (turnDirection == 0)))
  {
  if(obstaclesModel.distance[ObstaclesModel::frontLeft] < obstaclesModel.distance[ObstaclesModel::frontRight])
  turnDirection = -turnSpeed;
  else 
  turnDirection = turnSpeed;
  }
  // if nothing is in the way, walk forward
  else if (obstaclesModel.distance[ObstaclesModel::front] > turnDistance) 
  {
  }
  */  
  turnDirection = 0;
  if(obstaclesModel.corridorInFront < turnDistance) 
  {
    if(obstaclesModel.getDistanceInMajorDirection(ObstaclesModel::frontLeft) < obstaclesModel.getDistanceInMajorDirection(ObstaclesModel::frontRight))
      speedY.approximateVal(-walkSpeed);
    else if (obstaclesModel.getDistanceInMajorDirection(ObstaclesModel::frontLeft) > obstaclesModel.getDistanceInMajorDirection(ObstaclesModel::frontLeft))
      speedY.approximateVal(walkSpeed);
    else if (obstaclesModel.getDistanceInMajorDirection(ObstaclesModel::frontLeft) == obstaclesModel.getDistanceInMajorDirection(ObstaclesModel::frontRight))
      speedY.approximateVal(0);
  }
  else  
    speedY.approximateVal(0);
  
  motionRequest.motionType = MotionRequest::walk;
  motionRequest.walkType = MotionRequest::normal;
  motionRequest.walkParams.translation.x = speedX.getVal();
  motionRequest.walkParams.translation.y = speedY.getVal();
  motionRequest.walkParams.rotation = speedPhi.approximateVal(turnDirection);
}


void GT2003BasicBehaviorGoToPoint::execute()
{
  accelerationRestrictor.saveLastWalkParameters();
  
  Vector2<double> destination(x,y);
  const Vector2<double>& self = robotPose.translation;
  
  if (maxSpeed==0) maxSpeed = 120;
  
  double maxTurnSpeed = fromDegrees(40);
  
  double distanceToDestination = Geometry::distanceTo(self,destination);
  
  double angleDifference = normalize(fromDegrees(destinationAngle) - robotPose.rotation);
  
  double angleToDestination = Geometry::angleTo(robotPose,destination);
  
  
  motionRequest.motionType = MotionRequest::walk;
  motionRequest.walkType = MotionRequest::normal;
  
  //this time does not necessarily include time for turning!:
  double estimatedTimeToReachDestination;
  if (distanceToDestination > 200)
  {
    estimatedTimeToReachDestination = (distanceToDestination+200)/ maxSpeed;
    
    motionRequest.walkParams.translation.x = cos(angleToDestination) * maxSpeed;
    motionRequest.walkParams.translation.y = sin(angleToDestination) * maxSpeed;
  }
  else
  {
    estimatedTimeToReachDestination = 2*distanceToDestination / maxSpeed;
    if (distanceToDestination > 30)
    {
      motionRequest.walkParams.translation.x = 
        cos(angleToDestination) * maxSpeed*distanceToDestination/200;
      motionRequest.walkParams.translation.y = 
        sin(angleToDestination) * maxSpeed*distanceToDestination/200;
    }
    else
    {
      motionRequest.walkParams.translation.x = 0;
      motionRequest.walkParams.translation.y = 0;
    }
  }
  if (estimatedTimeToReachDestination==0)
  {
    estimatedTimeToReachDestination = 0.001;
  }
  
  if (fabs(toDegrees(angleDifference)) > 20)
  {
    motionRequest.walkParams.rotation = 
      angleDifference / estimatedTimeToReachDestination;
    if (motionRequest.walkParams.rotation > maxTurnSpeed)
    {
      motionRequest.walkParams.rotation = maxTurnSpeed;
    }
    if (motionRequest.walkParams.rotation < -maxTurnSpeed) 
    {
      motionRequest.walkParams.rotation = -maxTurnSpeed;
    }
  }
  else
  {
    motionRequest.walkParams.rotation = 2*angleDifference;
  }
  
  
  
  if ((fabs(toDegrees(angleDifference))<4)&&(distanceToDestination<25))
  {
    motionRequest.walkParams.translation.x = 0;
    motionRequest.walkParams.translation.y = 0;
    motionRequest.walkParams.rotation = 0;
  }
  
  accelerationRestrictor.restrictAccelerations(150,150,100);
  
  if (fabs(motionRequest.walkParams.translation.x) < 10
    && fabs(motionRequest.walkParams.translation.y) < 10
    && fabs(motionRequest.walkParams.rotation) < fromDegrees(10)
    && fabs(distanceToDestination) < 40
    && fabs(angleToDestination) < fromDegrees(5))
  {
    motionRequest.motionType = MotionRequest::stand;
  }
}

void GT2003BasicBehaviorGoToPointPrecisely::execute()
{
  Pose2D pose = robotPose + Pose2D(Vector2<double>(-105,0));
  
  accelerationRestrictor.saveLastWalkParameters();
  
  Vector2<double> destination = (Pose2D(fromDegrees(destinationAngle), Vector2<double>(x,y)) + Pose2D(Vector2<double>(-105,0))).translation;
  const Vector2<double>& self = pose.translation;
  
  if (maxSpeed==0) maxSpeed = 120;
  
  double maxTurnSpeed = fromDegrees(40);
  
  double distanceToDestination = Geometry::distanceTo(self,destination);
  
  double angleDifference = normalize(fromDegrees(destinationAngle) - pose.rotation);
  
  double angleToDestination = Geometry::angleTo(pose,destination);
  
  
  motionRequest.motionType = MotionRequest::walk;
  motionRequest.walkType = MotionRequest::normal;
  
  //this time does not necessarily include time for turning!:
  double estimatedTimeToReachDestination;
  if (distanceToDestination > 200)
  {
    estimatedTimeToReachDestination = (distanceToDestination+200)/ maxSpeed;
    
    motionRequest.walkParams.translation.x = cos(angleToDestination) * maxSpeed;
    motionRequest.walkParams.translation.y = sin(angleToDestination) * maxSpeed;
  }
  else
  {
    estimatedTimeToReachDestination = 2*distanceToDestination / maxSpeed;
    motionRequest.walkParams.translation.x = 
      cos(angleToDestination) * maxSpeed*distanceToDestination/200;
    motionRequest.walkParams.translation.y = 
      sin(angleToDestination) * maxSpeed*distanceToDestination/200;
    const double minSpeed = 30;
    if(fabs(motionRequest.walkParams.translation.x) > fabs(motionRequest.walkParams.translation.y) &&
      fabs(motionRequest.walkParams.translation.x) < minSpeed)
    {
      motionRequest.walkParams.translation.x = motionRequest.walkParams.translation.x > 0 ? minSpeed : -minSpeed;
      motionRequest.walkParams.translation.x *= minSpeed / fabs(motionRequest.walkParams.translation.x);
    }
    else if(fabs(motionRequest.walkParams.translation.x) <= fabs(motionRequest.walkParams.translation.y) &&
      fabs(motionRequest.walkParams.translation.y) < minSpeed)
    {
      motionRequest.walkParams.translation.y = motionRequest.walkParams.translation.y > 0 ? minSpeed : -minSpeed;
      motionRequest.walkParams.translation.x *= minSpeed / fabs(motionRequest.walkParams.translation.y);
    }
    
  }
  if (estimatedTimeToReachDestination==0)
  {
    estimatedTimeToReachDestination = 0.001;
  }
  
  if (fabs(toDegrees(angleDifference)) > 20)
  {
    motionRequest.walkParams.rotation = 
      angleDifference / estimatedTimeToReachDestination;
    if (motionRequest.walkParams.rotation > maxTurnSpeed)
    {
      motionRequest.walkParams.rotation = maxTurnSpeed;
    }
    if (motionRequest.walkParams.rotation < -maxTurnSpeed) 
    {
      motionRequest.walkParams.rotation = -maxTurnSpeed;
    }
  }
  else
  {
    motionRequest.walkParams.rotation = 2*angleDifference;
  }
  
  accelerationRestrictor.restrictAccelerations(150,150,100);
  
  if (fabs(motionRequest.walkParams.translation.x) < 10
    && fabs(motionRequest.walkParams.translation.y) < 10
    && fabs(motionRequest.walkParams.rotation) < fromDegrees(10)
    && fabs(distanceToDestination) < 40
    && fabs(angleToDestination) < fromDegrees(5))
  {
    motionRequest.motionType = MotionRequest::stand;
  }
}

/*
* Change Log
* 
* $Log: GT2003SimpleBasicBehaviors.cpp,v $
* Revision 1.44  2004/05/27 13:31:26  dueffert
* walking evolution stuff separated
*
* Revision 1.43  2004/05/24 12:48:26  dueffert
* outputs corrected
*
* Revision 1.42  2004/05/23 13:31:10  dueffert
* output improved; mirroring corrected
*
* Revision 1.41  2004/05/20 17:18:35  dueffert
* automatic measurement (hopefully) finalized
*
* Revision 1.40  2004/05/19 07:59:44  dueffert
* automatic walk speed measurement significantly improved
*
* Revision 1.39  2004/05/17 11:16:51  dueffert
* blind measurement improved
*
* Revision 1.38  2004/05/14 22:53:07  roefer
* Warnings removed
*
* Revision 1.37  2004/05/14 10:15:49  dueffert
* exact measurement behavior with logging implemented
*
* Revision 1.36  2004/05/12 14:23:31  dueffert
* measurement basic behavior added
*
* Revision 1.35  2004/05/11 16:33:36  dueffert
* send current parameters even if they were not auto evolved/changed
*
* Revision 1.34  2004/05/04 10:48:58  loetzsch
* replaced all enums
* xxxBehaviorControlTimeAfterWhichCommunicatedBallsAreAccepted
* by
* behaviorControlTimeAfterWhichCommunicatedBallsAreAccepted
* (this mechanism was neither fully implemented nor used)
*
* Revision 1.33  2004/05/03 18:52:38  dueffert
* comments corrected; max speed / rest distinction added
*
* Revision 1.32  2004/04/29 15:15:00  dueffert
* measurement is now frame instead of time based
*
* Revision 1.31  2004/04/19 10:34:55  dueffert
* parameters tuned to allow measurement oft *fast* walking
*
* Revision 1.30  2004/04/07 17:00:19  dueffert
* idea of srand() added
*
* Revision 1.29  2004/03/30 11:54:47  dueffert
* first useful evolutionMode implementation (0=none, 1=auto)
*
* Revision 1.28  2004/03/29 15:27:35  dueffert
* output improved, evolution mode idea added
*
* Revision 1.27  2004/03/26 09:24:17  dueffert
* missing conditions and one UDParameters for everything support added
*
* Revision 1.26  2004/03/24 15:09:39  dueffert
* do not let Motion calculate RobotPose updates during evolution runs
*
* Revision 1.25  2004/03/24 13:45:25  dueffert
* support for uniform noise mutation readded; evolution logic improved
*
* Revision 1.24  2004/03/19 16:37:27  dueffert
* blind measurement works for the first time
*
* Revision 1.23  2004/03/16 10:25:24  dueffert
* headControl for blind measurement changed
*
* Revision 1.22  2004/03/15 12:40:17  dueffert
* measurement of freely choosen MotionRequest allowed; output improved
*
* Revision 1.21  2004/03/12 17:23:36  dueffert
* own bug fixed
*
* Revision 1.20  2004/03/12 17:11:38  dueffert
* omni-evo uses parameters in package now; blind speed calculation improved
*
* Revision 1.19  2004/03/10 15:01:07  dueffert
* first simple fast turn (blind) measurement implemented
*
* Revision 1.18  2004/03/10 10:02:22  dueffert
* population size tweaked; statistics output added
*
* Revision 1.17  2004/03/09 11:40:55  wachter
* included Platform/GTAssert.h
*
* Revision 1.16  2004/03/09 08:46:30  dueffert
* second measurement behavior added
*
* Revision 1.15  2004/03/03 08:34:16  dueffert
* output improved
*
* Revision 1.14  2004/03/01 15:10:04  dueffert
* measurement order defined, measurement output beautified
*
* Revision 1.13  2004/02/29 17:28:22  dueffert
* UDParameters have char* names now
*
* Revision 1.12  2004/02/29 13:42:58  dueffert
* UDParametersSet symmetries handled
*
* Revision 1.11  2004/02/27 16:43:03  dueffert
* UDEvolutionRequest introduced
*
* Revision 1.10  2004/02/23 16:48:11  dueffert
* several improvements for measurement of walking
*
* Revision 1.9  2004/02/18 14:49:20  dueffert
* behavior control can now change walking parameters
*
* Revision 1.8  2004/02/10 11:56:53  dueffert
* typo corrected
*
* Revision 1.7  2004/02/10 11:19:19  dueffert
* simple basic behavior for evolution added, another improved, beautified
*
* Revision 1.6  2004/02/03 13:20:48  spranger
* renamed all references to  class BallPosition to BallModel (possibly changed include files)
*
* Revision 1.5  2003/12/09 16:29:06  jhoffman
* removed OUTPUTs
*
* Revision 1.4  2003/12/09 15:15:20  dueffert
* WalkAccelerationRestrictor replaced by PIDsmoothedValue for evolving
*
* Revision 1.3  2003/11/19 13:49:39  dueffert
* better positioning
*
* Revision 1.2  2003/10/28 13:27:22  dueffert
* spelling and evolution logic improved
*
* Revision 1.1  2003/10/22 22:18:45  loetzsch
* prepared the cloning of the GT2003BehaviorControl
*
* Revision 1.1  2003/10/06 13:39:31  cvsadm
* Created GT2004 (M.J.)
*
* Revision 1.18  2003/09/25 10:14:26  juengel
* Removed some walk-types.
*
* Revision 1.17  2003/09/24 15:29:20  dueffert
* first somehow working automatic walk parameter evolution
*
* Revision 1.16  2003/09/11 15:21:31  dueffert
* starting of evolution added
*
* Revision 1.15  2003/08/08 14:30:04  dueffert
* some evolution implementation added
*
* Revision 1.14  2003/08/04 07:46:39  roefer
* Challenge 2 modifications
*
* Revision 1.13  2003/07/30 14:52:03  dueffert
* walk evolution agent added
*
* Revision 1.12  2003/07/09 11:44:26  jhoffman
* obstacle avoidance simple behavior mode as used in challenge 3
* known bug: sometimes stops in front of obstacle (but eventually the situation is resolved)
*
* Revision 1.11  2003/07/08 06:11:56  dueffert
* further debug output removed
*
* Revision 1.10  2003/07/08 01:27:53  dueffert
* another debug output suppressed
*
* Revision 1.9  2003/07/08 01:21:00  dueffert
* debug output suppressed
*
* Revision 1.8  2003/07/07 07:15:36  jhoffman
* commented out "auto" mode for dribbling
*
* Revision 1.7  2003/07/07 07:08:19  jhoffman
* dribble as a basic behavior added to kicktest
*
* Revision 1.6  2003/07/06 20:34:30  dueffert
* GT2003BasicBehaviorGoToBallWithDirection added
*
* Revision 1.5  2003/07/05 13:35:49  jhoffman
* another bug fixed
*
* Revision 1.3  2003/07/05 13:32:46  jhoffman
* obstacle avoider challenge additions
* plus: kick-off behavior uses "simple-avoidance"
*
* Revision 1.2  2003/07/04 16:24:24  jhoffman
* updated obstacle avoider (simple behavior) to be used in challenge
*
* Revision 1.1.1.1  2003/07/02 09:40:23  cvsadm
* created new repository for the competitions in Padova from the 
* tamara CVS (Tuesday 2:00 pm)
*
* removed unused solutions
*
* Revision 1.9  2003/06/04 00:43:45  loetzsch
* put the GT2003WalkAccelerationRestrictor into two extra files in /Src/Tools
*
* Revision 1.8  2003/06/02 04:45:51  loetzsch
* "get-to-position-and-avoid-obstacles" now faster turns
*
* Revision 1.7  2003/05/14 13:08:38  risler
* removed DefaultObstaclesLocator
* renamed MicroSectorsObstaclesLocator to DefaultObstaclesLocator
* ObstaclesModel contains increased number of sectors
* DefaultObstaclesLocator clean up
*
* Revision 1.6  2003/05/09 15:31:18  loetzsch
* added time-after-which-communicated-ball-are-accepted
*
* Revision 1.5  2003/05/07 13:02:34  risler
* added first continuous basic behaviors to GT2003BehaviorControl
*
* Revision 1.4  2003/05/06 11:18:33  goehring
* Cast bug removed
*
* Revision 1.3  2003/05/06 07:59:19  dueffert
* generating XML symbols in C
*
* Revision 1.2  2003/05/05 19:22:03  risler
* added classes Simple/ContinuousBasicBehaviors for registering basic behaviors
*
* Revision 1.1  2003/05/03 15:14:02  loetzsch
* GT2003BehaviorControl first draft
*
*/
