/**
* @file GT2003ContinuousRules/GoBetweenBallAndOwnGoalClear.h
*
* @author Dirk Thomas
*/

#ifndef __GT2003GoBetweenBallAndOwnGoalClear_h_
#define __GT2003GoBetweenBallAndOwnGoalClear_h_

#include "Tools/ContinuousBasicBehaviors/ContinuousRule.h"
#include "Tools/FieldDimensions.h"

namespace GT2003ContinuousRules
{

/**@class GoBetweenBallAndOwnGoal
 */
class GoBetweenBallAndOwnGoalClear: public ContinuousRule{
private:
	/** the distance above which the result vector is maxForce in mm*/
  double maxForceLength;
  /** the distance below which the result vector is 0 in mm*/
  double minDistance;
  /** the maximum attracting force */
  double maxForce;

  /** the distance from the ball */
  double distanceToBall;

public:

  /** constructor
   */
  GoBetweenBallAndOwnGoalClear(
    const BehaviorControlInterfaces& interfaces,
    double maxForce = 1.0,
    double minDistance = 0.0,
    double maxForceLength = 350.0,
    double distanceToBall = 100.0
		)
    :
    ContinuousRule(interfaces,"GoBetweenBallAndOwnGoalClear"),
    maxForce(maxForce),
    minDistance(minDistance),
    maxForceLength(maxForceLength),
    distanceToBall(distanceToBall)
  {};

	/** executes the basic behavior rule
   * @param robotPose the current robots pose at which the rule is to be evaluated
   * @param walk the direction and speed of the suggested motion coded as an vector (output)
	 * @param ra the rotation angle, the direction the robot should be directed (output)
	 * @param rweight the rotation weight the weight of the suggested rotation (output)
	 */
  virtual void execute(const RobotPose& robotPose,
                       Vector2<double>& walk,
                       double& ra, double& rweight)
  {
    
    walk = ballPosition.seen - robotPose.translation;

    Vector2<double> ballToGoal(xPosOwnGroundline+50, yPosCenterGoal);
    ballToGoal -= ballPosition.seen;
    ballToGoal.normalize();

    // position between ball and own goal with a given distance to the ball
    ballToGoal *= distanceToBall;
		walk += ballToGoal;

    double distance = walk.abs();
    walk.normalize();

    walk *= getAttractingForce(distance, maxForce, minDistance, maxForceLength);

    ra = 0;
    rweight = 0;

  }
};

}


#endif //__GT2003GoBetweenBallAndOwnGoal_h_


/*
* Change log:
*
* $Log: GoBetweenBallAndOwnGoalClear.h,v $
* Revision 1.3  2004/03/08 01:06:54  roefer
* Interfaces should be const
*
* Revision 1.2  2003/10/31 08:32:48  dueffert
* doxygen bugs fixed
*
* Revision 1.1  2003/10/22 22:18:46  loetzsch
* prepared the cloning of the GT2003BehaviorControl
*
* Revision 1.1  2003/10/06 13:39:31  cvsadm
* Created GT2004 (M.J.)
*
* Revision 1.1  2003/07/07 09:28:03  thomas
* modified: goalie based on cont-goalie-clear and angle limit in guard-mode
*
* Revision 1.1.1.1  2003/07/02 09:40:23  cvsadm
* created new repository for the competitions in Padova from the 
* tamara CVS (Tuesday 2:00 pm)
*
* removed unused solutions
*
* Revision 1.4  2003/06/19 14:55:46  risler
* goalie ellipse y radius increased
*
* Revision 1.3  2003/06/17 20:39:10  thomas
* modified: goalie position etc.
* added: parameter for max-speed etc.
*
* Revision 1.2  2003/06/17 18:28:45  thomas
* added: goalie cont-behaviours, return-state, etc.
*
* Revision 1.1  2003/06/04 11:15:59  risler
* added Rule GoBetweenBallAndOwnGoal
*
* Revision 1.9  2003/05/22 06:48:03  dueffert
* extra ; warning removed
*
* Revision 1.8  2003/05/21 11:43:31  risler
* added namespaces to continuous rules to avoid problems with ambiguous class names
*
* Revision 1.7  2003/05/12 09:50:33  risler
* doxygen comments
*
* Revision 1.6  2003/05/09 15:25:08  risler
* ContinuousBasicBehavior uses Xabsl2Array,
* ContinuousRules inherit from Xabsl2NamedItem
*
* Revision 1.5  2003/05/07 13:02:33  risler
* added first continuous basic behaviors to GT2003BehaviorControl
*
* Revision 1.4  2003/05/07 11:05:55  risler
* ContinuousBasicBehaviors:
* removed classes Attracting/RepellingFieldRule, replaced with getAttracting/RepellingForce functions
* some clean up
*
* Revision 1.3  2003/05/07 10:46:38  kallnik
* warnings removed
*
* Revision 1.2  2003/05/07 10:02:08  kallnik
* ContinuousBasicBehavior:
* -DebugDrawings fixed
* -copyfiles: do no copy Config/ContBeh (files deleted)
*
* Revision 1.1  2003/05/06 17:03:43  kallnik
* ContinuousBasicBehaviors code review
*
* Revision 1.3  2003/05/01 17:09:07  loetzsch
* Redesign of ball modeling:
* - Modularized class BallPosition
* - splitted up module "BallLocator" into "BallLocator" for modeling of percepts
*   and "TeamBallLocator" for modelling communicated positions
* - Removed solution JumpingBallLocator
* - Splitted Solution DefaultBallLocator into DefaultBallLocator and DefaultTeamBallLocator
* - Renamed SensorFusionBallLocator to GaussBellTeamBallLocator
*
* Revision 1.2  2003/04/25 12:57:13  kallnik
* new version of ContinuousBasicBehavior released
*
* Revision 1.1  2003/04/15 15:52:11  risler
* DDD GO 2003 code integrated
*
* Revision 1.7  2003/04/08 17:04:23  dthomas
* modified: raised parabolic_length to slow down at target position
*
* Revision 1.6  2003/04/07 18:27:27  max
* turn ahead when ball not known
*
* Revision 1.5  2003/04/07 13:03:22  max
* double constant
*
* Revision 1.4  2003/04/03 14:43:44  max
* added border distance function to fielddimensions
* changed goto cont rules to use parabolic function
*
* Revision 1.3  2003/04/02 15:19:02  dthomas
* modified: continuous behaviour of goalie
*
* Revision 1.2  2003/04/02 12:11:12  max
* some rules added, modified
*
* Revision 1.1  2003/04/01 17:47:05  dthomas
* initial commit
*
*
*/
