/** 
* @file GT2003ContinuousRules/AvoidTeammates.h
*
* 
* @author Max Risler
*/

#ifndef __GT2003AvoidTeammates_h_
#define __GT2003AvoidTeammates_h_

#include "Tools/ContinuousBasicBehaviors/ContinuousRule.h"

namespace GT2003ContinuousRules
{

/**@class AvoidTeammates
 * rule which avoids collision with the teammates
 */
class AvoidTeammates: public ContinuousRule{
public:

  /** constructor
   */
  AvoidTeammates(const BehaviorControlInterfaces& interfaces,
                 double maxForce = 3.0, 
                 double maxForceLength = 150.0, 
                 double influenceLength = 250.0)
      :ContinuousRule(interfaces, "AvoidTeammates"),
       maxForce(maxForce),
       maxForceLength(maxForceLength),
       influenceLength(influenceLength)
  {};
 
	/** executes the basic behavior rule
   * @param robotPose the current robots pose at which the rule is to be evaluated
	 * @param walk the direction and speed of the suggested motion coded as an vector (output)
	 * @param ra the rotation angle, the direction the robot should be directed (output)
	 * @param rweight the rotation weight the weight of the suggested rotation (output)
	 */
  virtual void execute(const RobotPose& robotPose,
                       Vector2<double>& walk,
                       double& ra, double& rweight)
  {

    Vector2<double> fromTeammate;
    double distance;

    walk.x=walk.y=0;

    Geometry::Line robot(robotPose.translation,
                         robotPose*Vector2<double>(-bodyLength,0)-robotPose.translation);

    for (int i=0;i<playerPoseCollection.numberOfOwnPlayers;i++)
    {
      distance = Geometry::getDistanceToEdge(robot,playerPoseCollection.getOwnPlayerPose(i).getPose().translation);

      fromTeammate = robotPose.translation - playerPoseCollection.getOwnPlayerPose(i).getPose().translation;
      fromTeammate.normalize();

      walk += fromTeammate * 
              getRepellingForce(distance, maxForce, maxForceLength, influenceLength) * 
              playerPoseCollection.getOwnPlayerPose(i).getValidity();
    }

    ra=0;
    rweight=0;
  };

protected:
	/** the maximum length of the result vector*/
  double maxForce;
	/** the distance below which the result vector length is max_force (in mm)*/
  double maxForceLength;
	/** the distance above which the result vector length is 0 (in mm)*/
  double influenceLength;
};

}

#endif 


/*
* Change log:
*
* $Log: AvoidTeammates.h,v $
* Revision 1.3  2004/03/08 01:06:53  roefer
* Interfaces should be const
*
* Revision 1.2  2003/10/31 08:32:48  dueffert
* doxygen bugs fixed
*
* Revision 1.1  2003/10/22 22:18:46  loetzsch
* prepared the cloning of the GT2003BehaviorControl
*
* Revision 1.1  2003/10/06 13:39:31  cvsadm
* Created GT2004 (M.J.)
*
* Revision 1.1.1.1  2003/07/02 09:40:23  cvsadm
* created new repository for the competitions in Padova from the 
* tamara CVS (Tuesday 2:00 pm)
*
* removed unused solutions
*
* Revision 1.6  2003/05/22 06:48:02  dueffert
* extra ; warning removed
*
* Revision 1.5  2003/05/21 11:43:31  risler
* added namespaces to continuous rules to avoid problems with ambiguous class names
*
* Revision 1.4  2003/05/12 09:50:33  risler
* doxygen comments
*
* Revision 1.3  2003/05/12 00:03:29  dueffert
* doxygen bugs fixed
*
* Revision 1.2  2003/05/09 15:25:08  risler
* ContinuousBasicBehavior uses Xabsl2Array,
* ContinuousRules inherit from Xabsl2NamedItem
*
* Revision 1.1  2003/05/07 13:02:33  risler
* added first continuous basic behaviors to GT2003BehaviorControl
*
* Revision 1.4  2003/05/07 11:05:55  risler
* ContinuousBasicBehaviors:
* removed classes Attracting/RepellingFieldRule, replaced with getAttracting/RepellingForce functions
* some clean up
*
* Revision 1.3  2003/05/07 10:46:38  kallnik
* warnings removed
*
* Revision 1.2  2003/05/07 10:02:08  kallnik
* ContinuousBasicBehavior:
* -DebugDrawings fixed
* -copyfiles: do no copy Config/ContBeh (files deleted)
*
* Revision 1.1  2003/05/06 17:03:43  kallnik
* ContinuousBasicBehaviors code review
*
* Revision 1.4  2003/04/25 12:57:13  kallnik
* new version of ContinuousBasicBehavior released
*
* Revision 1.3  2003/04/15 15:52:11  risler
* DDD GO 2003 code integrated
*
* Revision 1.9  2003/04/10 16:13:37  max
* increased radius
*
* Revision 1.8  2003/04/09 13:32:40  max
* increased maxForce
*
* Revision 1.7  2003/04/09 10:26:16  dthomas
* raised influence length
*
* Revision 1.6  2003/04/07 22:04:20  max
* update rules with linear potential function
* rotation speed limit corrected
*
* Revision 1.5  2003/04/07 01:25:54  dthomas
* modified: one converted to double in distance
* modified: changed factor and influence in avoid teammate/opponents
* other rebalancing
*
* Revision 1.4  2003/03/27 14:33:22  max
* some rules added
*
* Revision 1.3  2003/03/26 17:47:25  max
* rules improved
* robot now is "edge" instead of "point"
*
* Revision 1.2  2003/03/25 19:29:28  max
* tamara update
* doxygen bugs fixed
*
* Revision 1.2  2003/03/25 17:04:56  dueffert
* doxygen bugs fixed
*
* Revision 1.1  2003/03/24 19:21:42  risler
* added some continuous behavior rules
*
*/
