/** 
* @file GT2003ContinuousRules/AvoidObstaclesInDirectionToPointTurning.h
*
* @author Matthias Jngel
*/

#ifndef __AvoidObstaclesInDirectionToPointTurning_h_
#define __AvoidObstaclesInDirectionToPointTurning_h_

#include "Tools/ContinuousBasicBehaviors/ContinuousRule.h"
#include "Tools/Debugging/DebugDrawings.h"

namespace GT2003ContinuousRules
{
  
/**@class AvoidObstaclesInDirectionToPointTurning
* rule which avoids collision with the obstacles
*/
  class AvoidObstaclesInDirectionToPointTurning: public ContinuousRule{
  private:
    /** reference to the point to turn to */
    const double& pointX;
    const double& pointY;
    const double& obstacleInfluenceLength;
    
    /** the point is in relative coordinates */
    bool relative;
  public:
    
  /** constructor
    */
    AvoidObstaclesInDirectionToPointTurning(const BehaviorControlInterfaces& interfaces,
      const double& pointX,
      const double& pointY,
      const double& obstacleInfluenceLength,
      bool relative = false,
      double maxForce = 1.5, 
      double maxForceLength = 100.0
      )
      :ContinuousRule(interfaces, "AvoidObstaclesInDirectionToPointTurning"),
      maxForce(maxForce),
      maxForceLength(maxForceLength),
      pointX(pointX),
      pointY(pointY),
      obstacleInfluenceLength(obstacleInfluenceLength)
    {};
    
    /**
    * Returns whether this rule generates absolute or robot relative coordinates.
    * This rule is robot relative.
    */
    virtual bool isRelative() {return true;}
    
    /** executes the basic behavior rule
    * @param robotPose the current robots pose at which the rule is to be evaluated
    * @param walk the direction and speed of the suggested motion coded as an vector (output)
    * @param ra the rotation angle, the direction the robot should be directed (output)
    * @param rweight the rotation weight the weight of the suggested rotation (output)
    */
    virtual void execute(const RobotPose& robotPose,
      Vector2<double>& walk,
      double& ra, double& rweight)
    {
      double angleToPoint = 
        Geometry::angleTo(robotPose, Vector2<double>(pointX, pointY));

      double angleToTurn = obstaclesModel.getAngleOfNextFreeSector(pi/3.0, angleToPoint, (int)obstacleInfluenceLength);
      
      LINE(models_corridorsRadar, cos(angleToPoint) * 1000, sin(angleToPoint) * 1000, 0 , 0, 100, Drawings::ps_solid, Drawings::red);
      LINE(models_corridorsRadar, cos(angleToTurn) * 1000, sin(angleToTurn) * 1000, 0 , 0, 100, Drawings::ps_solid, Drawings::gray);
      CIRCLE(models_corridorsRadar, 0, 0, obstacleInfluenceLength , 10, Drawings::ps_solid, Drawings::gray);
      DEBUG_DRAWING_FINISHED(models_corridorsRadar);
//      OUTPUT(idText, text, toDegrees(angleToPoint) << " - " << toDegrees(angleToTurn));

//      if(angleToTurn < fromDegrees(-10)) ra = -pi_2;
//      if(angleToTurn > fromDegrees(10)) ra = pi_2;
      ra = angleToTurn - angleToPoint / 3;

      walk.x = -getRepellingForce(obstaclesModel.corridorInFront, maxForce, maxForceLength, obstacleInfluenceLength / 2.0);
      rweight = 1.0;
      /*
      if(obstaclesModel.corridorInFront < obstacleInfluenceLength) 
      {
        if(obstaclesModel.getAngleOfNextFreeSector(pi/4, 0, (int)obstacleInfluenceLength) < 0)
          ra = -pi_2;
        else
          ra = pi_2;
        
        walk.x = -getRepellingForce(obstaclesModel.corridorInFront, maxForce, maxForceLength, obstacleInfluenceLength);
        rweight = 1.0;
      }
      DEBUG_DRAWING_FINISHED(models_corridorsRadar);
*/
    /* ??????????????????????????????????
      rweight = 1.0;
      walk.x = 0;
      walk.y = 0;
      double angle = 
        Geometry::angleTo(robotPose, Vector2<double>(pointX, pointY));

      ra = obstaclesModel.getAngleOfNextFreeSector(pi/4, angle, (int)influenceLength);

      if(obstaclesModel.getDistanceInCorridor(0, 280) < influenceLength) 
      {
        walk.x = -getRepellingForce(obstaclesModel.corridorInFront, maxForce, maxForceLength, influenceLength);
      }
      DEBUG_DRAWING_FINISHED(models_corridorsRadar);
      */
/*
      ra = 0;
      rweight = 0.0;
      walk.x = 0;
      walk.y = 0;
      double angle = 
        Geometry::angleTo(robotPose, Vector2<double>(pointX, pointY));

//      if(obstaclesModel.getDistanceInCorridor(0, 280) < influenceLength) 
      {
//        if(obstaclesModel.getAngleOfNextFreeSector(pi/4, 0, 850) < 0)
        double nextAngle = normalize(obstaclesModel.getAngleOfNextFreeSector(pi/4, angle, 2*(int)obstacleInfluenceLength));
        if(nextAngle < 0)
          ra = nextAngle;//-pi_2;
        else
          ra = nextAngle;//pi_2;
        
        walk.x = -getRepellingForce(obstaclesModel.corridorInFront, maxForce, maxForceLength, obstacleInfluenceLength);
        rweight = 1.0;
      }
      DEBUG_DRAWING_FINISHED(models_corridorsRadar);
*/  
    };
  protected:
    /** the maximum length of the result vector*/
    double maxForce;
    /** the distance below which the result vector length is maxForce (in mm)*/
    double maxForceLength;
  };
  
}


#endif 


/*
* Change log:
*
* $Log: AvoidObstaclesInDirectionToPointTurning.h,v $
* Revision 1.3  2004/03/08 01:06:52  roefer
* Interfaces should be const
*
* Revision 1.2  2003/10/31 08:32:48  dueffert
* doxygen bugs fixed
*
* Revision 1.1  2003/10/22 22:18:45  loetzsch
* prepared the cloning of the GT2003BehaviorControl
*
* Revision 1.1  2003/10/06 13:39:31  cvsadm
* Created GT2004 (M.J.)
*
* Revision 1.3  2003/09/01 10:20:11  juengel
* DebugDrawings clean-up 2
* DebugImages clean-up
* MessageIDs clean-up
* Stopwatch clean-up
*
* Revision 1.2  2003/07/03 10:50:42  juengel
* improved...
*
* Revision 1.1.1.1  2003/07/02 09:40:23  cvsadm
* created new repository for the competitions in Padova from the 
* tamara CVS (Tuesday 2:00 pm)
*
* removed unused solutions
*
* Revision 1.1  2003/06/17 19:54:40  juengel
* Added AvoidObstaclesInDirectionToPointTurning
*
*
*/
