/** 
* @file GT2003ContinuousRules/AvoidObstacles.h
*
* @author Max Risler
*/

#ifndef __GT2003AvoidObstacles_h_
#define __GT2003AvoidObstacles_h_

#include "Tools/ContinuousBasicBehaviors/ContinuousRule.h"

namespace GT2003ContinuousRules
{

/**@class AvoidObstacles
 * rule which avoids collision with the obstacles
 */
class AvoidObstacles: public ContinuousRule{
public:

  /** constructor
   */
  AvoidObstacles(const BehaviorControlInterfaces& interfaces,
                 double maxForce = 1.0, 
                 double maxForceLength = 50.0, 
                 double influenceLength = 300.0)
      :ContinuousRule(interfaces, "AvoidObstacles"),
       maxForce(maxForce),
       maxForceLength(maxForceLength),
       influenceLength(influenceLength),
       numOfSectors(20)
  {};

  /**
   * Returns whether this rule generates absolute or robot relative coordinates.
   * This rule is robot relative.
   */
  virtual bool isRelative() {return true;}

  /**
   * Returns whether this rule affect the walk speed of the basic behavior.
   * This rule does not affect speed.
   */
  virtual bool affectSpeed() {return false;}

 	/** executes the basic behavior rule
   * @param robotPose the current robots pose at which the rule is to be evaluated
   * @param walk the direction and speed of the suggested motion coded as an vector (output)
	 * @param ra the rotation angle, the direction the robot should be directed (output)
	 * @param rweight the rotation weight the weight of the suggested rotation (output)
	 */
  virtual void execute(const RobotPose& robotPose,
                       Vector2<double>& walk,
                       double& ra, double& rweight)
  {

    Vector2<double> fromObstacle;
    double angle;

    walk.x=walk.y=0;
    rweight = 0;
/*
    for (int i=0;i<ObstaclesModel::numOfSectors;i++)
      if (obstaclesModel.distance[i] < ObstaclesModel::maxDistance)
      {
        angle = normalize(ObstaclesModel::getAngleOfSector(i));

        fromObstacle.x = -cos(angle);
        fromObstacle.y = -sin(angle);

        walk += fromObstacle * 
                getRepellingForce(obstaclesModel.distance[i], maxForce, maxForceLength, influenceLength);
      }
*/
    for (angle = -pi_2; angle < pi_2; angle += pi / (numOfSectors-1))
    {
      double distance = obstaclesModel.getDistanceInDirection(angle, pi_2 / (numOfSectors-1));
      if (distance < ObstaclesModel::maxDistance)
      {
        fromObstacle.x = -cos(angle);
        fromObstacle.y = -sin(angle);

        walk += fromObstacle * 
                getRepellingForce(distance, maxForce, maxForceLength, influenceLength);
      }
    }
  };

protected:
	/** the maximum length of the result vector*/
  double maxForce;
	/** the distance below which the result vector length is maxForce (in mm)*/
  double maxForceLength;
	/** the distance above which the result vector length is 0 (in mm)*/
  double influenceLength;

  /** the number of sectors to reduce the obstacles model to */
  double numOfSectors;
};

}


#endif 


/*
* Change log:
*
* $Log: AvoidObstacles.h,v $
* Revision 1.3  2004/03/08 01:06:51  roefer
* Interfaces should be const
*
* Revision 1.2  2003/10/31 08:32:48  dueffert
* doxygen bugs fixed
*
* Revision 1.1  2003/10/22 22:18:45  loetzsch
* prepared the cloning of the GT2003BehaviorControl
*
* Revision 1.1  2003/10/06 13:39:31  cvsadm
* Created GT2004 (M.J.)
*
* Revision 1.1.1.1  2003/07/02 09:40:23  cvsadm
* created new repository for the competitions in Padova from the 
* tamara CVS (Tuesday 2:00 pm)
*
* removed unused solutions
*
* Revision 1.7  2003/06/03 15:54:14  risler
* avoidObstacles does not affect speed
* obstaclesModel reduced to fewer sectors for more stability and efficiency
*
* Revision 1.6  2003/06/03 10:51:24  risler
* ContinuousRules can be in relative coordinates
* no rotation rule active => no rotation
*
* Revision 1.5  2003/06/02 17:44:38  juengel
* no message
*
* Revision 1.4  2003/05/22 06:48:02  dueffert
* extra ; warning removed
*
* Revision 1.3  2003/05/21 11:43:31  risler
* added namespaces to continuous rules to avoid problems with ambiguous class names
*
* Revision 1.2  2003/05/15 10:26:57  risler
* added continuous rule Walk
* added continuous basic behavior walkAvoidingObstacles
*
* Revision 1.1  2003/05/15 00:56:47  risler
* added functions for sector to angle mapping to ObstaclesModel
* sector to angle mapping unified
* added rule AvoidObstacles
*
*/
