/** 
* @file GT2003ContinuousBasicBehaviors.cpp
*
* Implementation of basic behaviors defined in continuous-basic-behaviors.xml.
*
* @author Martin Kallnik
* @author Max Risler
*/

#include "GT2003ContinuousRules/AvoidBorder.h"
#include "GT2003ContinuousRules/AvoidBorderTurning.h"
#include "GT2003ContinuousRules/AvoidObstacles.h"
#include "GT2003ContinuousRules/AvoidObstaclesAheadGoingSideways.h"
#include "GT2003ContinuousRules/AvoidObstaclesAheadTurning.h"
#include "GT2003ContinuousRules/AvoidObstaclesInDirectionToPointTurning.h"
#include "GT2003ContinuousRules/AvoidOpponents.h"
#include "GT2003ContinuousRules/AvoidPoint.h"
#include "GT2003ContinuousRules/AvoidTeammates.h"
#include "GT2003ContinuousRules/AvoidOwnPenaltyArea.h"
#include "GT2003ContinuousRules/GoBetweenBallAndOwnGoal.h"
#include "GT2003ContinuousRules/GoBetweenBallAndOwnGoalClear.h"
#include "GT2003ContinuousRules/GoToPoint.h"
#include "GT2003ContinuousRules/TurnToAngle.h"
#include "GT2003ContinuousRules/TurnToPoint.h"
#include "GT2003ContinuousRules/TurnToPointGoalie.h"
#include "GT2003ContinuousRules/Walk.h"
#include "GT2003ContinuousRules/WalkX.h"
#include "GT2003ContinuousRules/WalkY.h"

#include "GT2003ContinuousBasicBehaviors.h"

using namespace GT2003ContinuousRules;

GT2003ContinuousBasicBehaviors::GT2003ContinuousBasicBehaviors(
                       const BehaviorControlInterfaces& interfaces,
                       GT2003BehaviorConfiguration& configuration,
                       Xabsl2ErrorHandler& errorHandler) : 
  BehaviorControlInterfaces(interfaces),
  configuration(configuration),
  errorHandler(errorHandler),
  goToPoint(interfaces,errorHandler,"cont-go-to-point"),
  goForwardToPoint(interfaces,errorHandler,"cont-go-forward-to-point"),
  goForwardToPointTurning(interfaces,errorHandler,"cont-go-forward-to-point-turning"),
  goToBall(interfaces,errorHandler,"cont-go-to-ball"),
  dribble(interfaces,errorHandler,"cont-dribble"),
  turnWithBall(interfaces,errorHandler,"cont-turn-with-ball"),
  walkAvoidingObstacles(interfaces,errorHandler,"cont-walk-avoiding-obstacles"),
  walkAroundAvoidingObstacles(interfaces,errorHandler,"cont-walk-around-avoiding-obstacles"),
  turnToBall(interfaces,errorHandler,"cont-turn-to-ball"),
  turn(interfaces,errorHandler,"cont-turn"),
  goalieClear(interfaces,errorHandler,"cont-goalie-clear"),
  goalieGuard(interfaces,errorHandler,"cont-goalie-guard"),
  goaliePosition(interfaces,errorHandler,"cont-goalie-position"),
  goalieReturn(interfaces,errorHandler,"cont-goalie-return"),
  supporterPosition(interfaces,errorHandler,"cont-supporter-position")
{
  goToPoint.setMaxRotationSpeed(fromDegrees(120.0));
  goToPoint.addRule(new GoToPoint(interfaces,goToPointX,goToPointY));
  goToPoint.addRule(new TurnToAngle(interfaces,goToPointAngle));
  goToPoint.addRule(new AvoidBorder(interfaces));
  goToPoint.addRule(new AvoidObstacles(interfaces));
  goToPoint.addRule(new AvoidOwnPenaltyArea(interfaces));

  goToPoint.registerParameter("cont-go-to-point.x",goToPointX);
  goToPoint.registerParameter("cont-go-to-point.y",goToPointY);
  goToPoint.registerParameter("cont-go-to-point.angle",goToPointAngle);
  goToPoint.registerParameter("cont-go-to-point.max-speed",goToPointMaxSpeed);

  goForwardToPoint.setMaxRotationSpeed(fromDegrees(120.0));
  goForwardToPoint.addRule(new GoToPoint(interfaces,goForwardToPointX,goForwardToPointY,false,1.0,100.0));
  goForwardToPoint.addRule(new TurnToPoint(interfaces,goForwardToPointX,goForwardToPointY));
  goForwardToPoint.addRule(new AvoidBorder(interfaces));
  goForwardToPoint.addRule(new AvoidObstaclesAheadGoingSideways(interfaces));
  goForwardToPoint.addRule(new AvoidOwnPenaltyArea(interfaces));

  goForwardToPoint.registerParameter("cont-go-forward-to-point.x",goForwardToPointX);
  goForwardToPoint.registerParameter("cont-go-forward-to-point.y",goForwardToPointY);
  goForwardToPoint.registerParameter("cont-go-forward-to-point.max-speed",goForwardToPointMaxSpeed);

  goForwardToPointTurning.addRule(new WalkX(interfaces,goForwardToPointTurningMaxSpeed));
  goForwardToPointTurning.addRule(new AvoidObstaclesInDirectionToPointTurning(interfaces,goForwardToPointTurningX,goForwardToPointTurningY,goForwardToPointTurningObstacleInfluenceLength));

  goForwardToPointTurning.registerParameter("cont-go-forward-to-point-turning.x",goForwardToPointTurningX);
  goForwardToPointTurning.registerParameter("cont-go-forward-to-point-turning.y",goForwardToPointTurningY);
  goForwardToPointTurning.registerParameter("cont-go-forward-to-point-turning.max-speed",goForwardToPointTurningMaxSpeed);
  goForwardToPointTurning.registerParameter("cont-go-forward-to-point-turning.obstacle-influence-length", goForwardToPointTurningObstacleInfluenceLength);

  goToBall.setMaxRotationSpeed(fromDegrees(120.0));
  goToBall.addRule(new GoToPoint(interfaces,knownBallPosRobotRelative,true,1.0,100.0));
  goToBall.addRule(new TurnToPoint(interfaces,knownBallPosRobotRelative,true));
  goToBall.addRule(new AvoidBorder(interfaces));
  goToBall.addRule(new AvoidOwnPenaltyArea(interfaces));

  dribble.setWalkType(MotionRequest::normal);
  dribble.setMaxRotationSpeed(fromDegrees(50.0));
  dribble.enableBackwards(false);
  dribble.setAccelerations(80.0,50.0,50.0);
  dribble.addRule(new GoToPoint(interfaces,dribbleX,dribbleY,false,1.0,150.0,400.0));
  dribble.addRule(new TurnToPoint(interfaces,dribbleX,dribbleY));
  dribble.addRule(new AvoidBorder(interfaces));
  dribble.addRule(new AvoidOwnPenaltyArea(interfaces));
  dribble.addRule(new AvoidTeammates(interfaces));

  dribble.registerParameter("cont-dribble.x", dribbleX);
  dribble.registerParameter("cont-dribble.y", dribbleY);

  turnWithBall.setWalkType(MotionRequest::normal);
  turnWithBall.enableBackwards(false);
  turnWithBall.setAccelerations(80.0,50.0,50.0);
  turnWithBall.addRule(new TurnToAngle(interfaces,turnWithBallAngle,true));
  turnWithBall.addRule(new WalkY(interfaces,turnWithBallSpeedY));
  turnWithBall.addRule(new AvoidBorder(interfaces));
  turnWithBall.addRule(new AvoidOwnPenaltyArea(interfaces));
  turnWithBall.addRule(new AvoidTeammates(interfaces));

  turnWithBall.registerParameter("cont-turn-with-ball.angle", turnWithBallAngle);
  turnWithBall.registerParameter("cont-turn-with-ball.max-rotation-speed", turnWithBallMaxRotationSpeed);
  turnWithBall.registerParameter("cont-turn-with-ball.rotation-center", turnWithBallRotationCenter);

  walkAvoidingObstacles.addRule(new AvoidObstacles(interfaces));
  walkAvoidingObstacles.addRule(new Walk(interfaces,walkAvoidingObstaclesX,walkAvoidingObstaclesY));
  walkAvoidingObstacles.addRule(new TurnToAngle(interfaces,walkAvoidingObstaclesRot,true));

  walkAvoidingObstacles.registerParameter("cont-walk-avoiding-obstacles.type",walkAvoidingObstaclesType);
  walkAvoidingObstacles.registerParameter("cont-walk-avoiding-obstacles.speed-x",walkAvoidingObstaclesX);
  walkAvoidingObstacles.registerParameter("cont-walk-avoiding-obstacles.speed-y",walkAvoidingObstaclesY);
  walkAvoidingObstacles.registerParameter("cont-walk-avoiding-obstacles.rotation-speed",walkAvoidingObstaclesRot);

  walkAroundAvoidingObstacles.addRule(new AvoidObstaclesAheadTurning(interfaces));
  walkAroundAvoidingObstacles.addRule(new AvoidBorderTurning(interfaces));
  walkAroundAvoidingObstacles.addRule(new Walk(interfaces, walkAroundAvoidingObstaclesX, walkAroundAvoidingObstaclesY));
  walkAroundAvoidingObstacles.registerParameter("cont-walk-around-avoiding-obstacles.speed-x",walkAroundAvoidingObstaclesX);
  walkAroundAvoidingObstacles.registerParameter("cont-walk-around-avoiding-obstacles.speed-y",walkAroundAvoidingObstaclesY);

  turnToBall.addRule(new TurnToPoint(interfaces,knownBallPosRobotRelative,true));
  turnToBall.addRule(new AvoidBorder(interfaces));
  turnToBall.addRule(new AvoidOwnPenaltyArea(interfaces));
  turnToBall.addRule(new AvoidTeammates(interfaces));
  turnToBall.addRule(new AvoidObstacles(interfaces));

  turnToBall.registerParameter("cont-turn-to-ball.max-rotation-speed",turnToBallMaxRotationSpeed);

  turn.addRule(new TurnToAngle(interfaces,turnRotationSpeed,true));
  turn.addRule(new AvoidBorder(interfaces));
  turn.addRule(new AvoidOwnPenaltyArea(interfaces));
  turn.addRule(new AvoidTeammates(interfaces));
  turn.addRule(new AvoidObstacles(interfaces));

  turn.registerParameter("cont-turn.rotation-speed",turnRotationSpeed);

  goalieClear.setMaxRotationSpeed(fromDegrees(120.0));
  goalieClear.setMaxSpeed(150.0);
  goalieClear.addRule(new GoBetweenBallAndOwnGoalClear(interfaces));
  goalieClear.addRule(new TurnToPoint(interfaces,knownBallPosRobotRelative,true));

  goalieClear.registerParameter("cont-goalie-clear.max-speed",goalieClearMaxSpeed);

  goalieGuard.setMaxRotationSpeed(fromDegrees(120.0));
  goalieGuard.setMaxSpeed(150.0);
  goalieGuard.addRule(new GoBetweenBallAndOwnGoal(interfaces));
  goalieGuard.addRule(new TurnToPointGoalie(interfaces,knownBallPosRobotRelative,true));
  goalieGuard.addRule(new AvoidBorder(interfaces));

  goalieGuard.registerParameter("cont-goalie-guard.max-speed",goalieGuardMaxSpeed);

  goaliePosition.setMaxRotationSpeed(fromDegrees(120.0));
  goaliePosition.addRule(new GoToPoint(interfaces,goaliePositionX,goaliePositionY, false));
  goaliePosition.addRule(new TurnToAngle(interfaces,goaliePositionAngle,false));
  goaliePosition.addRule(new AvoidPoint(interfaces,knownBallPosRobotRelative,true));
  goaliePosition.addRule(new AvoidBorder(interfaces));

  goaliePosition.registerParameter("cont-goalie-position.x",goaliePositionX);
  goaliePosition.registerParameter("cont-goalie-position.y",goaliePositionY);
  goaliePosition.registerParameter("cont-goalie-position.max-speed",goaliePositionMaxSpeed);
  goaliePosition.registerParameter("cont-goalie-position.angle",goaliePositionAngle);

  goalieReturn.setMaxRotationSpeed(fromDegrees(120.0));
  goalieReturn.addRule(new GoToPoint(interfaces,ownGoalRearPanelMiddle, false));
  goalieReturn.addRule(new TurnToPoint(interfaces,knownBallPosRobotRelative,true));
  goalieReturn.addRule(new AvoidPoint(interfaces,knownBallPosRobotRelative,true));
  goalieReturn.addRule(new AvoidBorder(interfaces));
  goalieReturn.addRule(new AvoidObstacles(interfaces));

  supporterPosition.setMaxRotationSpeed(fromDegrees(120.0));
  // go to destination but no need to reach it exactly
  supporterPosition.addRule(new GoToPoint(interfaces,supporterPositionX,supporterPositionY,false,1.0,0.0,500.0));
  // turn to ball
  supporterPosition.addRule(new TurnToPoint(interfaces,knownBallPosRobotRelative,true));
  // avoid running into opponent goal area
  supporterPosition.addRule(new AvoidPoint(interfaces,opponentGoalRearPanelMiddle,false,3.0,600.0,850.0));
  // avoid running over ball
  supporterPosition.addRule(new AvoidPoint(interfaces,knownBallPosRobotRelative,true));
  supporterPosition.addRule(new AvoidBorder(interfaces));
  supporterPosition.addRule(new AvoidOwnPenaltyArea(interfaces));
  supporterPosition.addRule(new AvoidTeammates(interfaces));
  supporterPosition.addRule(new AvoidObstacles(interfaces));

  supporterPosition.registerParameter("cont-supporter-position.x",supporterPositionX);
  supporterPosition.registerParameter("cont-supporter-position.y",supporterPositionY);
  supporterPosition.registerParameter("cont-supporter-position.max-speed",supporterPositionMaxSpeed);
}


void GT2003ContinuousBasicBehaviors::registerBasicBehaviors(Xabsl2Engine& engine)
{
  engine.registerBasicBehavior(goToPoint);
  engine.registerBasicBehavior(goForwardToPoint);
  engine.registerBasicBehavior(goForwardToPointTurning);
  engine.registerBasicBehavior(goToBall);
  engine.registerBasicBehavior(dribble);
  engine.registerBasicBehavior(turnWithBall);
  engine.registerBasicBehavior(walkAvoidingObstacles);
  engine.registerBasicBehavior(walkAroundAvoidingObstacles);
  engine.registerBasicBehavior(turnToBall);
  engine.registerBasicBehavior(turn);
  engine.registerBasicBehavior(goalieClear);
  engine.registerBasicBehavior(goalieGuard);
  engine.registerBasicBehavior(goaliePosition);
  engine.registerBasicBehavior(goalieReturn);
  engine.registerBasicBehavior(supporterPosition);
}

void GT2003ContinuousBasicBehaviors::update()
{
  knownBallPos = ballPosition.getKnownPosition(
    BallModel::behaviorControlTimeAfterWhichCommunicatedBallsAreAccepted);

  knownBallPosRobotRelative = Geometry::vectorTo(robotPose, knownBallPos);


  walkAvoidingObstacles.setWalkType(MotionRequest::normal);
  switch((int)walkAvoidingObstaclesType)
  {
    case 0: walkAvoidingObstacles.setWalkType(MotionRequest::normal); break;
    case 1: walkAvoidingObstacles.setWalkType(MotionRequest::upsideDown); break;
    case 2: walkAvoidingObstacles.setWalkType(MotionRequest::turnWithBall); break;
    case 3: walkAvoidingObstacles.setWalkType(MotionRequest::turnWithBall2); break;
    case 4: walkAvoidingObstacles.setWalkType(MotionRequest::turnWithBall3); break;
    case 5: walkAvoidingObstacles.setWalkType(MotionRequest::turnWithBall4); break;
    case 6: walkAvoidingObstacles.setWalkType(MotionRequest::turnWithBall5); break;
    case 7: walkAvoidingObstacles.setWalkType(MotionRequest::dash); break;
  }

  goToPoint.setMaxSpeed(goToPointMaxSpeed);
  goForwardToPoint.setMaxSpeed(goForwardToPointMaxSpeed);
  goForwardToPointTurning.setMaxSpeed(goForwardToPointTurningMaxSpeed);
  turnWithBall.setMaxRotationSpeed(fromDegrees(turnWithBallMaxRotationSpeed));
  turnWithBall.setMaxSpeed(fromDegrees(turnWithBallMaxRotationSpeed)*turnWithBallRotationCenter);
  turnToBall.setMaxRotationSpeed(turnToBallMaxRotationSpeed);
  goalieClear.setMaxSpeed(goalieClearMaxSpeed);
  goalieGuard.setMaxSpeed(goalieGuardMaxSpeed);
  goaliePosition.setMaxSpeed(goaliePositionMaxSpeed);
  supporterPosition.setMaxSpeed(supporterPositionMaxSpeed);

  turnWithBallSpeedY = turnWithBallRotationCenter*fromDegrees(turnWithBallAngle);

  // set the evaluation mode
  // this is only done for localization dependent behaviors, others keep discrete mode
  goToPoint.setEvaluationMode(configuration.contEvaluationMode);
  goForwardToPoint.setEvaluationMode(configuration.contEvaluationMode);
  goForwardToPointTurning.setEvaluationMode(configuration.contEvaluationMode);
  goToBall.setEvaluationMode(configuration.contEvaluationMode);
  dribble.setEvaluationMode(configuration.contEvaluationMode);
  turnWithBall.setEvaluationMode(configuration.contEvaluationMode);
  turnToBall.setEvaluationMode(configuration.contEvaluationMode);
  turn.setEvaluationMode(configuration.contEvaluationMode);
  goalieClear.setEvaluationMode(configuration.contEvaluationMode);
  goalieGuard.setEvaluationMode(configuration.contEvaluationMode);
  goaliePosition.setEvaluationMode(configuration.contEvaluationMode);
  goalieReturn.setEvaluationMode(configuration.contEvaluationMode);
  supporterPosition.setEvaluationMode(configuration.contEvaluationMode);
}

/*
* Change Log
* 
* $Log: GT2003ContinuousBasicBehaviors.cpp,v $
* Revision 1.5  2004/05/04 10:48:58  loetzsch
* replaced all enums
* xxxBehaviorControlTimeAfterWhichCommunicatedBallsAreAccepted
* by
* behaviorControlTimeAfterWhichCommunicatedBallsAreAccepted
* (this mechanism was neither fully implemented nor used)
*
* Revision 1.4  2004/04/07 13:44:33  risler
* ddd checkin after go04 - second part fixed walktype
*
* Revision 1.3  2004/03/08 01:06:49  roefer
* Interfaces should be const
*
* Revision 1.2  2004/02/03 13:20:48  spranger
* renamed all references to  class BallPosition to BallModel (possibly changed include files)
*
* Revision 1.1  2003/10/22 22:18:45  loetzsch
* prepared the cloning of the GT2003BehaviorControl
*
* Revision 1.1  2003/10/06 13:39:31  cvsadm
* Created GT2004 (M.J.)
*
* Revision 1.5  2003/09/25 10:14:26  juengel
* Removed some walk-types.
*
* Revision 1.4  2003/07/07 09:28:03  thomas
* modified: goalie based on cont-goalie-clear and angle limit in guard-mode
*
* Revision 1.3  2003/07/05 20:56:59  thomas
* bug fix: goalie was only running in release, wrong constructor-call for turn-to-angle in contGoaliePosition
*
* Revision 1.2  2003/07/03 10:51:32  juengel
* Added CBB goForwardToPointTurning
*
* Revision 1.1.1.1  2003/07/02 09:40:23  cvsadm
* created new repository for the competitions in Padova from the 
* tamara CVS (Tuesday 2:00 pm)
*
* removed unused solutions
*
* Revision 1.42  2003/06/21 12:48:11  juengel
* Removed avoid AvoidObstacles from dribble and turnWithBall.
*
* Revision 1.41  2003/06/18 15:36:49  risler
* added parameters max-rotation-speed, rotation-center to turn-with-ball
*
* Revision 1.40  2003/06/18 10:49:11  risler
* added cont-turn-with-ball
*
* Revision 1.39  2003/06/17 20:39:10  thomas
* modified: goalie position etc.
* added: parameter for max-speed etc.
*
* Revision 1.38  2003/06/17 19:59:46  juengel
* Added second version of goForwardToPoint.
*
* Revision 1.37  2003/06/17 18:28:45  thomas
* added: goalie cont-behaviours, return-state, etc.
*
* Revision 1.36  2003/06/17 13:04:16  risler
* added avoidOwnPentaltyArea to goToPoint
*
* Revision 1.35  2003/06/10 18:09:26  juengel
* Added rule AvoidBorderTurning to walkAroundAvoidingObstacles.
*
* Revision 1.34  2003/06/07 16:58:30  risler
* added cont-turn
*
* Revision 1.33  2003/06/06 15:55:08  risler
* added parameter supporter-position.max-speed
* supporter-position avoids ball
*
* Revision 1.32  2003/06/06 14:41:28  risler
* removed unused rules
*
* Revision 1.31  2003/06/06 13:55:12  risler
* supporterPosition avoid opponent goal radius increased
*
* Revision 1.30  2003/06/05 17:30:47  risler
* added supporterPosition
* removed some unused behaviors
* dribble now has destination as parameter
*
* Revision 1.29  2003/06/05 14:37:48  risler
* dribble speed reduced
*
* Revision 1.28  2003/06/05 08:09:28  juengel
* New version of goForwardToPoint
*
* Revision 1.27  2003/06/04 11:20:06  risler
* added cont-goalie-position, cont-position-on-line
* walk rule does not set rotation
* dribble uses stricter acceleration restriction
*
* Revision 1.26  2003/06/03 13:07:07  juengel
* Added cont-walk-around-avoiding-obstacles.
*
* Revision 1.25  2003/06/03 12:40:40  risler
* always use relative ball position as with new relative rule mode it is no longer less efficient
* turnToBall: reduced avoidTeammates radius
*
* Revision 1.24  2003/06/03 10:51:24  risler
* ContinuousRules can be in relative coordinates
* no rotation rule active => no rotation
*
* Revision 1.23  2003/05/30 14:22:11  risler
* cont-forward-to-point should no longer turn around uncontrolled when point is reached
*
* Revision 1.22  2003/05/29 12:32:31  risler
* added cont-turn-to-ball
*
* Revision 1.21  2003/05/26 09:51:48  risler
* configuration has to be reference
*
* Revision 1.20  2003/05/25 14:41:40  risler
* added GT2003BehaviorConfiguration
*
* Revision 1.19  2003/05/25 11:35:39  risler
* maxSpeed added to cont-go-to-point,cont-go-forward-to-point
*
* Revision 1.18  2003/05/21 11:43:31  risler
* added namespaces to continuous rules to avoid problems with ambiguous class names
*
* Revision 1.17  2003/05/19 10:26:19  risler
* added max speeds to continuousbasicbehaviors
*
* Revision 1.16  2003/05/15 11:54:57  brunn
* added sample mode (currently commented out)
*
* Revision 1.15  2003/05/15 10:26:57  risler
* added continuous rule Walk
* added continuous basic behavior walkAvoidingObstacles
*
* Revision 1.14  2003/05/15 01:22:23  risler
* added continuous basic behavior goToBall
*
* Revision 1.13  2003/05/15 00:56:47  risler
* added functions for sector to angle mapping to ObstaclesModel
* sector to angle mapping unified
* added rule AvoidObstacles
*
* Revision 1.12  2003/05/14 09:38:58  risler
* clean up
*
* Revision 1.11  2003/05/14 09:05:53  risler
* added relative position to GoToPoint/TurnToPoint rules
*
* Revision 1.10  2003/05/13 16:23:47  risler
* added basic behavior go-forward-to-point
*
* Revision 1.9  2003/05/13 15:39:56  risler
* goToPoint rule has no rotation
* added rules turnToPoint, turnToAngle
* added basic behavior go-forward-to-point
* added parameter angle to basic behavior go-to-point
*
* Revision 1.8  2003/05/09 15:38:43  loetzsch
* bug fix
*
* Revision 1.7  2003/05/09 15:31:18  loetzsch
* added time-after-which-communicated-ball-are-accepted
*
* Revision 1.6  2003/05/09 15:23:08  risler
* renamed ballPos to knownBallPos
*
* Revision 1.5  2003/05/08 01:10:26  risler
* no message
*
* Revision 1.4  2003/05/08 01:09:00  risler
* bugfix: gotopoint target point was no reference
*
* Revision 1.3  2003/05/07 13:02:34  risler
* added first continuous basic behaviors to GT2003BehaviorControl
*
* Revision 1.2  2003/05/05 19:22:03  risler
* added classes Simple/ContinuousBasicBehaviors for registering basic behaviors
*
* Revision 1.1  2003/05/03 15:14:02  loetzsch
* GT2003BehaviorControl first draft
*
*/

