/** 
* @file DDD2004SimpleBasicBehaviors.h
*
* Declaration of basic behaviors defined in simple-basic-behaviors.xml.
*
* @author Uwe Dffert
* @author Jan Hoffmann
* @author Matthias Jngel
* @author Martin Ltzsch
* @author Max Risler
*/

#ifndef __DDD2004SimpleBasicBehaviors_h_
#define __DDD2004SimpleBasicBehaviors_h_

#include "../../BehaviorControl.h"
#include "Tools/Xabsl2/Xabsl2Engine/Xabsl2Engine.h"
#include "Tools/Xabsl2/Xabsl2Engine/Xabsl2BasicBehavior.h"
#include "Tools/Actorics/WalkAccelerationRestrictor.h"
#include "Tools/Evolution/Population.h"
#include "Tools/Math/PIDsmoothedValue.h"
#include "Modules/WalkingEngine/InvKinWalkingEngine.h"
#include "Modules/WalkingEngine/InvKinWalkingParameterSets.h"



/**
* A Basic Behavior for going to the ball
* @author Matthias Jngel
* @author Martin Ltzsch
*/
class DDD2004BasicBehaviorGoToBall : public Xabsl2BasicBehavior, public BehaviorControlInterfaces
{
public:
/*
* Constructor.
* @param errorHandler Is invoked when errors occur
* @param interfaces The paramters of the BehaviorControl module.
  */
  DDD2004BasicBehaviorGoToBall(const BehaviorControlInterfaces& interfaces,
    Xabsl2ErrorHandler& errorHandler)
    : Xabsl2BasicBehavior("go-to-ball", errorHandler),
    BehaviorControlInterfaces(interfaces),
    accelerationRestrictor(motionRequest)
  {
    registerParameter("go-to-ball.distance",distanceAtEnd);
    registerParameter("go-to-ball.max-speed", maxSpeed);
  }
  
  /**  Executes the basic behavior. */
  virtual void execute();
  
private:
  /** parameter "go-to-ball.distance" */
  double distanceAtEnd;
  
  /** parameter "go-to-ball.max-speed" */
  double maxSpeed;
  
  /** Restricts the walk acceleration to maximum values */
  WalkAccelerationRestrictor accelerationRestrictor;
};

/**
* A Basic Behavior for going to the ball and reaching it with certain distance (x and y relative to robot) and direction
* @author Uwe Dffert
* @author Matthias Jngel
* @author Martin Ltzsch
*/
class DDD2004BasicBehaviorGoToBallWithDirection : public Xabsl2BasicBehavior, public BehaviorControlInterfaces
{
public:
/*
* Constructor.
* @param errorHandler Is invoked when errors occur
* @param interfaces The paramters of the BehaviorControl module.
  */
  DDD2004BasicBehaviorGoToBallWithDirection(const BehaviorControlInterfaces& interfaces,
    Xabsl2ErrorHandler& errorHandler)
    : Xabsl2BasicBehavior("go-to-ball-with-direction", errorHandler),
    BehaviorControlInterfaces(interfaces),
    accelerationRestrictor(motionRequest)
  {
    registerParameter("go-to-ball-with-direction.distance.x",distanceX);
    registerParameter("go-to-ball-with-direction.distance.y",distanceY);
    registerParameter("go-to-ball-with-direction.max-speed", maxSpeed);
    registerParameter("go-to-ball-with-direction.direction", direction);
  }
  
  /**  Executes the basic behavior. */
  virtual void execute();
  
private:
  /** parameter "go-to-ball-with-direction.distance.x" */
  double distanceX;
  /** parameter "go-to-ball-with-direction.distance.y" */
  double distanceY;
  /** parameter "go-to-ball-with-direction.max-speed" */
  double maxSpeed;
  /** parameter "go-to-ball-with-direction.direction" */
  double direction;
  
  /** Restricts the walk acceleration to maximum values */
  WalkAccelerationRestrictor accelerationRestrictor;
};


/**
* A Basic Behavior for avoiding obstacles
* @author Matthias Jngel
*/
class DDD2004BasicBehaviorAvoidObstacles : public Xabsl2BasicBehavior, public BehaviorControlInterfaces
{
public:
/*
* Constructor.
* @param errorHandler Is invoked when errors occur
* @param interfaces The paramters of the BehaviorControl module.
  */
  DDD2004BasicBehaviorAvoidObstacles(const BehaviorControlInterfaces& interfaces,
    Xabsl2ErrorHandler& errorHandler)
    : Xabsl2BasicBehavior("avoid-obstacles", errorHandler),
    BehaviorControlInterfaces(interfaces)
  {
    registerParameter("avoid-obstacles.turn-distance",turnDistance);
    registerParameter("avoid-obstacles.stop-distance",stopDistance);
    registerParameter("avoid-obstacles.walk-speed", walkSpeed);
    registerParameter("avoid-obstacles.turn-speed", turnSpeed);
  }
  
  /**  Executes the basic behavior. */
  virtual void execute();
  
private:
  /** parameter "avoid-obstacles.turn-distance" */
  double turnDistance;
  
  /** parameter "avoid-obstacles.stop-distance" */
  double stopDistance;
  
  /** parameter "avoid-obstacles.walk-speed" */
  double walkSpeed;
  
  /** parameter "avoid-obstacles.turn-speed" */
  double turnSpeed;
};

/**
* A Basic Behavior for going to a point on the field
* @author Uwe Dffert
* @author Martin Ltzsch
*/
class DDD2004BasicBehaviorGoToPoint : public Xabsl2BasicBehavior, public BehaviorControlInterfaces
{
public:
/*
* Constructor.
* @param errorHandler Is invoked when errors occur
* @param interfaces The paramters of the BehaviorControl module.
  */
  DDD2004BasicBehaviorGoToPoint(const BehaviorControlInterfaces& interfaces,
    Xabsl2ErrorHandler& errorHandler)
    : Xabsl2BasicBehavior("go-to-point", errorHandler),
    BehaviorControlInterfaces(interfaces),
    accelerationRestrictor(motionRequest)
  {
    registerParameter("go-to-point.x", x);
    registerParameter("go-to-point.y", y);
    registerParameter("go-to-point.destination-angle", destinationAngle);
    registerParameter("go-to-point.max-speed", maxSpeed);
    registerParameter("go-to-point.walk-type", walkType);
  }
  
  /**  Executes the basic behavior. */
  virtual void execute();
  
protected:
  /** parameter "go-to-point.x" */
  double x;
  
  /** parameter "go-to-point.y" */
  double y;
  
  /** parameter "go-to-point.destination-angle" */
  double destinationAngle;
  
  /** parameter "go-to-point.max-speed" */
  double maxSpeed;
  
  /** parameter "go-to-point.walk-type" */
  double walkType;

  /** Restricts the walk acceleration to maximum values */
  WalkAccelerationRestrictor accelerationRestrictor;

  friend class DDD2004BasicBehaviorGetBehindBall;
};

/**
* A Basic Behavior for precisely going to a point on the field
* @author Uwe Dffert
* @author Martin Ltzsch
* @author Thomas Rfer
*/
class DDD2004BasicBehaviorGoToPointPrecisely : public Xabsl2BasicBehavior, public BehaviorControlInterfaces
{
public:
/*
* Constructor.
* @param errorHandler Is invoked when errors occur
* @param interfaces The paramters of the BehaviorControl module.
  */
  DDD2004BasicBehaviorGoToPointPrecisely(const BehaviorControlInterfaces& interfaces,
    Xabsl2ErrorHandler& errorHandler)
    : Xabsl2BasicBehavior("go-to-point-precisely", errorHandler),
    BehaviorControlInterfaces(interfaces),
    accelerationRestrictor(motionRequest)
  {
    registerParameter("go-to-point-precisely.x", x);
    registerParameter("go-to-point-precisely.y", y);
    registerParameter("go-to-point-precisely.destination-angle", destinationAngle);
    registerParameter("go-to-point-precisely.max-speed", maxSpeed);
  }
  
  /**  Executes the basic behavior. */
  virtual void execute();
  
protected:
  /** parameter "go-to-point-precisely.x" */
  double x;
  
  /** parameter "go-to-point-precisely.y" */
  double y;
  
  /** parameter "go-to-point-precisely.destination-angle" */
  double destinationAngle;
  
  /** parameter "go-to-point-precisely.max-speed" */
  double maxSpeed;
  
  /** Restricts the walk acceleration to maximum values */
  WalkAccelerationRestrictor accelerationRestrictor;
};

/**
* A Basic Behavior for going to a point on the field and avoiding obstacles on the way there.
* @author Jan Hoffmann
*/
class DDD2004BasicBehaviorGoToPointAndAvoidObstacles : public Xabsl2BasicBehavior, public BehaviorControlInterfaces
{
public:
/*
* Constructor.
* @param errorHandler Is invoked when errors occur
* @param interfaces The paramters of the BehaviorControl module.
  */
  DDD2004BasicBehaviorGoToPointAndAvoidObstacles(const BehaviorControlInterfaces& interfaces,
    Xabsl2ErrorHandler& errorHandler)
    : Xabsl2BasicBehavior("go-to-point-and-avoid-obstacles", errorHandler),
    BehaviorControlInterfaces(interfaces),
	  speedPhi(0, .1, 0, 0, -pi_4, pi_4, pi/10),
	  speedX(0, .6, 0, 0, -80, 200, 100),
	  speedY(0, .1, 0, 0, -100, 100, 100),
		goToDestCounter(0),
		stickToBackOffDirection(0)
  {
    registerParameter("go-to-point-and-avoid-obstacles.x", x);
    registerParameter("go-to-point-and-avoid-obstacles.y", y);
    registerParameter("go-to-point-and-avoid-obstacles.max-speed", maxSpeed);
  }
  
  /**  Executes the basic behavior. */
  virtual void execute();
  
private:
  /** parameter "go-to-point-and-avoid-obstacles.x" */
  double x;
  
  /** parameter "go-to-point-and-avoid-obstacles.y" */
  double y;
  
  /** parameter "go-to-point-and-avoid-obstacles.max-speed" */
  double maxSpeed;
    
  /** smoothes the rotation */  
  PIDsmoothedValue speedPhi;

  /** smoothes the speed in forward direction */  
  PIDsmoothedValue speedX;

  /** smoothes the speed in sideways direction */  
  PIDsmoothedValue speedY;

  /** history that holds information in which direction the robot is/was turning */
  double turnDirection;

  /** history that holds information in which direction (left/right) the robot is/was moving  */
  double sideways;

	/** parameter "goToDestCounter" used for hysteresis*/
  long goToDestCounter;

	/** parameter "go-to-point-avoiding" */
  double backOffAngle;

	/** */
	long stickToBackOffDirection;
};


/**
* A Basic Behavior for getting behind the ball
* @author Max Risler
* @author Martin Ltzsch
*/
class DDD2004BasicBehaviorGetBehindBall : public Xabsl2BasicBehavior, public BehaviorControlInterfaces
{
public:
/*
* Constructor.
* @param errorHandler Is invoked when errors occur
* @param interfaces The paramters of the BehaviorControl module.
  */
  DDD2004BasicBehaviorGetBehindBall(const BehaviorControlInterfaces& interfaces,
    Xabsl2ErrorHandler& errorHandler)
    : Xabsl2BasicBehavior("get-behind-ball", errorHandler),
    BehaviorControlInterfaces(interfaces),
    accelerationRestrictor(motionRequest)
  {
    registerParameter("get-behind-ball.distance",distance);
    registerParameter("get-behind-ball.left-right", leftRight);
    registerParameter("get-behind-ball.max-speed", maxSpeed);
    registerParameter("get-behind-ball.angle", angle);
  }
  
  /**  Executes the basic behavior. */
  virtual void execute();
  
private:
  /** parameter "get-behind-ball.distance" */
  double distance;
  
  /** parameter "get-behind-ball.left-right" */
  double leftRight;
  
  /** parameter "get-behind-ball.max-speed" */
  double maxSpeed;
  
  /** parameter "get-behind-ball.angle" */
  double angle;
  
  /** Restricts the walk acceleration to maximum values */
  WalkAccelerationRestrictor accelerationRestrictor;
  
  /** Walks to a point */
  void gotoPoint(double x, double y, double maxSpeed);
};

/**
* @class DDD2004SimpleBasicBehaviors
*
* Creates and registers simple basic behaviors
*/
class DDD2004SimpleBasicBehaviors : public BehaviorControlInterfaces
{
public:
  /**
  * Constructor
  */
  DDD2004SimpleBasicBehaviors(const BehaviorControlInterfaces& interfaces,
                       Xabsl2ErrorHandler& errorHandler)
    : BehaviorControlInterfaces(interfaces),
      errorHandler(errorHandler),
      basicBehaviorAvoidObstacles(interfaces,errorHandler),
      basicBehaviorGetBehindBall(interfaces,errorHandler),
      basicBehaviorGoToBall(interfaces,errorHandler),
      basicBehaviorGoToBallWithDirection(interfaces,errorHandler),
      basicBehaviorGoToPoint(interfaces,errorHandler),
      basicBehaviorGoToPointPrecisely(interfaces,errorHandler),
      basicBehaviorGoToPointAndAvoidObstacles(interfaces,errorHandler)
  {}

  /** Registers basic behaviors at the engine */
  void registerBasicBehaviors(Xabsl2Engine& engine);

private:
  /** Is invoked when errors occurs */
  Xabsl2ErrorHandler& errorHandler;

  //!@name Basic Behaviors
  //!@{
  DDD2004BasicBehaviorAvoidObstacles  basicBehaviorAvoidObstacles;  
  DDD2004BasicBehaviorGetBehindBall  basicBehaviorGetBehindBall;
  DDD2004BasicBehaviorGoToBall  basicBehaviorGoToBall;
  DDD2004BasicBehaviorGoToBallWithDirection  basicBehaviorGoToBallWithDirection;
  DDD2004BasicBehaviorGoToPoint  basicBehaviorGoToPoint;  
  DDD2004BasicBehaviorGoToPointPrecisely  basicBehaviorGoToPointPrecisely;  
  DDD2004BasicBehaviorGoToPointAndAvoidObstacles  basicBehaviorGoToPointAndAvoidObstacles;
  //!@}
};

#endif // __DDD2004SimpleBasicBehaviors_h_

/*
* Change Log
* 
* $Log: DDD2004SimpleBasicBehaviors.h,v $
* Revision 1.7  2004/04/07 12:28:57  risler
* ddd checkin after go04 - first part
*
* Revision 1.2  2004/04/01 22:39:29  risler
* added go-to-point.walk-type
*
* Revision 1.1.1.1  2004/03/29 08:28:49  Administrator
* initial transfer from tamara
*
* Revision 1.6  2004/03/08 00:59:00  roefer
* Interfaces should be const
*
* Revision 1.5  2003/10/31 08:32:49  dueffert
* doxygen bugs fixed
*
* Revision 1.4  2003/10/28 14:04:50  dueffert
* unused evolution removed
*
* Revision 1.3  2003/10/28 13:57:45  dueffert
* unused evolution removed
*
* Revision 1.2  2003/10/28 13:25:52  dueffert
* spelling improved
*
* Revision 1.1  2003/10/26 22:49:38  loetzsch
* created ATH2004BehaviorControl from GT2003BehaviorControl
*  - strongly simplified option graph
*  - moved some symbols from GT2003 to CommonXabsl2Symbols
*  - moved some basic behaviors from GT2003 to CommonXabsl2BasicBehaviors
*
* cloned ATH2004 three times (BB2004, DDD2004, MSH2004)
*
*/

