/** 
* @file DDD2004ContinuousRules/AvoidObstaclesInDirectionToPointTurning.h
*
* @author Matthias Jngel
*/

#ifndef __AvoidObstaclesInDirectionToPointTurning_h_
#define __AvoidObstaclesInDirectionToPointTurning_h_

#include "Tools/ContinuousBasicBehaviors/ContinuousRule.h"
#include "Tools/Debugging/DebugDrawings.h"

namespace DDD2004ContinuousRules
{
  
/**@class AvoidObstaclesInDirectionToPointTurning
* rule which avoids collision with the obstacles
*/
  class AvoidObstaclesInDirectionToPointTurning: public ContinuousRule{
  private:
    /** reference to the point to turn to */
    const double& pointX;
    const double& pointY;
    const double& obstacleInfluenceLength;
    
    /** the point is in relative coordinates */
    bool relative;
  public:
    
  /** constructor
    */
    AvoidObstaclesInDirectionToPointTurning(const BehaviorControlInterfaces& interfaces,
      const double& pointX,
      const double& pointY,
      const double& obstacleInfluenceLength,
      bool relative = false,
      double maxForce = 1.5, 
      double maxForceLength = 100.0
      )
      :ContinuousRule(interfaces, "AvoidObstaclesInDirectionToPointTurning"),
      maxForce(maxForce),
      maxForceLength(maxForceLength),
      pointX(pointX),
      pointY(pointY),
      obstacleInfluenceLength(obstacleInfluenceLength)
    {};
    
    /**
    * Returns whether this rule generates absolute or robot relative coordinates.
    * This rule is robot relative.
    */
    virtual bool isRelative() {return true;}
    
    /** executes the basic behavior rule
    * @param robotPose the current robots pose at which the rule is to be evaluated
    * @param walk the direction and speed of the suggested motion coded as an vector (output)
    * @param ra the rotation angle, the direction the robot should be directed (output)
    * @param rweight the rotation weight the weight of the suggested rotation (output)
    */
    virtual void execute(const RobotPose& robotPose,
      Vector2<double>& walk,
      double& ra, double& rweight)
    {
      double angleToPoint = 
        Geometry::angleTo(robotPose, Vector2<double>(pointX, pointY));

      double angleToTurn = obstaclesModel.getAngleOfNextFreeSector(pi/3.0, angleToPoint, (int)obstacleInfluenceLength);
      
      LINE(models_corridorsRadar, cos(angleToPoint) * 1000, sin(angleToPoint) * 1000, 0 , 0, 100, Drawings::ps_solid, Drawings::red);
      LINE(models_corridorsRadar, cos(angleToTurn) * 1000, sin(angleToTurn) * 1000, 0 , 0, 100, Drawings::ps_solid, Drawings::gray);
      CIRCLE(models_corridorsRadar, 0, 0, obstacleInfluenceLength , 10, Drawings::ps_solid, Drawings::gray);
      DEBUG_DRAWING_FINISHED(models_corridorsRadar);
//      OUTPUT(idText, text, toDegrees(angleToPoint) << " - " << toDegrees(angleToTurn));

//      if(angleToTurn < fromDegrees(-10)) ra = -pi_2;
//      if(angleToTurn > fromDegrees(10)) ra = pi_2;
      ra = angleToTurn - angleToPoint / 3;

      walk.x = -getRepellingForce(obstaclesModel.corridorInFront, maxForce, maxForceLength, obstacleInfluenceLength / 2.0);
      rweight = 1.0;
      /*
      if(obstaclesModel.corridorInFront < obstacleInfluenceLength) 
      {
        if(obstaclesModel.getAngleOfNextFreeSector(pi/4, 0, (int)obstacleInfluenceLength) < 0)
          ra = -pi_2;
        else
          ra = pi_2;
        
        walk.x = -getRepellingForce(obstaclesModel.corridorInFront, maxForce, maxForceLength, obstacleInfluenceLength);
        rweight = 1.0;
      }
      DEBUG_DRAWING_FINISHED(models_corridorsRadar);
*/
    /* ??????????????????????????????????
      rweight = 1.0;
      walk.x = 0;
      walk.y = 0;
      double angle = 
        Geometry::angleTo(robotPose, Vector2<double>(pointX, pointY));

      ra = obstaclesModel.getAngleOfNextFreeSector(pi/4, angle, (int)influenceLength);

      if(obstaclesModel.getDistanceInCorridor(0, 280) < influenceLength) 
      {
        walk.x = -getRepellingForce(obstaclesModel.corridorInFront, maxForce, maxForceLength, influenceLength);
      }
      DEBUG_DRAWING_FINISHED(models_corridorsRadar);
      */
/*
      ra = 0;
      rweight = 0.0;
      walk.x = 0;
      walk.y = 0;
      double angle = 
        Geometry::angleTo(robotPose, Vector2<double>(pointX, pointY));

//      if(obstaclesModel.getDistanceInCorridor(0, 280) < influenceLength) 
      {
//        if(obstaclesModel.getAngleOfNextFreeSector(pi/4, 0, 850) < 0)
        double nextAngle = normalize(obstaclesModel.getAngleOfNextFreeSector(pi/4, angle, 2*(int)obstacleInfluenceLength));
        if(nextAngle < 0)
          ra = nextAngle;//-pi_2;
        else
          ra = nextAngle;//pi_2;
        
        walk.x = -getRepellingForce(obstaclesModel.corridorInFront, maxForce, maxForceLength, obstacleInfluenceLength);
        rweight = 1.0;
      }
      DEBUG_DRAWING_FINISHED(models_corridorsRadar);
*/  
    };
  protected:
    /** the maximum length of the result vector*/
    double maxForce;
    /** the distance below which the result vector length is maxForce (in mm)*/
    double maxForceLength;
  };
  
}


#endif 


/*
* Change log:
*
* $Log: AvoidObstaclesInDirectionToPointTurning.h,v $
* Revision 1.3  2004/03/08 00:59:01  roefer
* Interfaces should be const
*
* Revision 1.2  2003/10/28 15:04:09  loetzsch
* fixed doxygen bugs
*
* Revision 1.1  2003/10/26 22:49:38  loetzsch
* created ATH2004BehaviorControl from GT2003BehaviorControl
*  - strongly simplified option graph
*  - moved some symbols from GT2003 to CommonXabsl2Symbols
*  - moved some basic behaviors from GT2003 to CommonXabsl2BasicBehaviors
*
* cloned ATH2004 three times (BB2004, DDD2004, MSH2004)
*
*/
