/** 
* @file DDD2004ContinuousRules/AvoidLineBetweenBallAndOpponentGoal.h
*
* @author Dirk Thomas
*/

#ifndef __DDD2004AvoidLineBetweenBallAndOpponentGoal_h_
#define __DDD2004AvoidLineBetweenBallAndOpponentGoal_h_

#include "Tools/ContinuousBasicBehaviors/ContinuousRule.h"
#include "Tools/FieldDimensions.h"

namespace DDD2004ContinuousRules
{

/**@class AvoidLineBetweenBallAndOpponentGoal
 * rule which avoids the line between the ball and the opponent goal
 */
class AvoidLineBetweenBallAndOpponentGoal: public ContinuousRule{
public:

  /** constructor
   */
  AvoidLineBetweenBallAndOpponentGoal(const BehaviorControlInterfaces& interfaces,
              const Vector2<double>& relativeBall,
              double maxForce = 1.0, 
              double maxForceLength = 0.0, 
              double influenceLength = 500.0)
      :ContinuousRule(interfaces, "AvoidLineBetweenBallAndOpponentGoal"), 
       relativeBall(relativeBall), 
       maxForce(maxForce), 
       maxForceLength(maxForceLength),
       influenceLength(influenceLength) 
  {
  };

 	/** executes the basic behavior rule
   * @param robotPose the current robots pose at which the rule is to be evaluated
	 * @param walk the direction and speed of the suggested motion coded as an vector (output)
	 * @param ra the rotation angle, the direction the robot should be directed (output)
	 * @param rweight the rotation weight the weight of the suggested rotation (output)
	 */
  virtual void execute(const RobotPose& robotPose,
                       Vector2<double>& walk,
                       double& ra,
                       double& rweight)
  {
    Vector2<double> ball = robotPose*relativeBall;
    Geometry::Line line(ball.x,ball.y,xPosOpponentGroundline,0);
    double distance = Geometry::getDistanceToEdge(line,robotPose.translation);
    double dist = Geometry::getDistanceToLine(line,robotPose.translation);

    if(dist<0)
    {
      walk = line.direction.rotateLeft();
    } else {
      walk = line.direction.rotateRight();
    }
    walk.normalize();

    walk *= getRepellingForce(distance, maxForce, maxForceLength, influenceLength);

    ra=0;
    rweight=0;
  }
  
protected:
	/** the relative vector of the ball*/
  const Vector2<double>& relativeBall;
	/** the maximum length of the result vector*/
  double maxForce;
	/** the distance below which the result vector length is maxForce (in mm)*/
  double maxForceLength;
	/** the distance above which the result vector length is 0 (in mm)*/
  double influenceLength;

};

}

#endif 


/*
* Change log:
*
* $Log: AvoidLineBetweenBallAndOpponentGoal.h,v $
* Revision 1.1  2004/04/29 15:17:04  thomas
* added rules for supports to avoid line between ball and opp goal and better border-distances
*
*
*/
