/** 
* @file CommonBasicBehaviors.h
*
* Declaration of basic behaviors defined in common-basic-behaviors.xml.
*
* @author Martin Ltzsch
* @author Uwe Dffert
* @author Matthias Jngel
* @author Max Risler
*/

#ifndef __CommonBasicBehaviors_h_
#define __CommonBasicBehaviors_h_

#include "../BehaviorControl.h"
#include "Tools/Xabsl2/Xabsl2Engine/Xabsl2Engine.h"
#include "Tools/Actorics/WalkAccelerationRestrictor.h"

/**
* A Basic Behavior for shutdown. This is e.g. usefull after special action pack.
* @author Uwe Dffert
*/
class BasicBehaviorShutdown : public Xabsl2BasicBehavior, public BehaviorControlInterfaces
{
public:
/*
* Constructor.
* @param errorHandler Is invoked when errors occur
* @param interfaces The paramters of the BehaviorControl module.
  */
  BasicBehaviorShutdown(const BehaviorControlInterfaces& interfaces,
    Xabsl2ErrorHandler& errorHandler)
    : Xabsl2BasicBehavior("shutdown", errorHandler),
    BehaviorControlInterfaces(interfaces)
  {}
  
  /**  Executes the basic behavior. */
  virtual void execute();
};


/**
* A Basic Behavior for doing nothing. The motion request is not changed.
* @author Martin Ltzsch
*/
class BasicBehaviorDoNothing : public Xabsl2BasicBehavior, public BehaviorControlInterfaces
{
public:
/*
* Constructor.
* @param errorHandler Is invoked when errors occur
* @param interfaces The paramters of the BehaviorControl module.
  */
  BasicBehaviorDoNothing(const BehaviorControlInterfaces& interfaces,
    Xabsl2ErrorHandler& errorHandler)
    : Xabsl2BasicBehavior("do-nothing", errorHandler),
    BehaviorControlInterfaces(interfaces)
  {}
  
  /**  Executes the basic behavior. */
  virtual void execute();
};

/**
* A Basic Behavior for walking
* @author Matthias Jngel
* @author Martin Ltzsch
* @author Max Risler
*/
class BasicBehaviorWalk : public Xabsl2BasicBehavior, public BehaviorControlInterfaces
{
public:
/*
* Constructor.
* @param errorHandler Is invoked when errors occur
* @param interfaces The paramters of the BehaviorControl module.
  */
  BasicBehaviorWalk(const BehaviorControlInterfaces& interfaces,
    Xabsl2ErrorHandler& errorHandler)
    : Xabsl2BasicBehavior("walk", errorHandler),
    BehaviorControlInterfaces(interfaces)
  {
    registerParameter("walk.type", type);
    registerParameter("walk.speed-x", speedX);
    registerParameter("walk.speed-y", speedY);
    registerParameter("walk.rotation-speed", rotationSpeed);
  }
  
  /**  Executes the basic behavior. */
  virtual void execute();

private:
  /** Parameter "walk.type" */
  double type;

  /** Parameter "walk.speed-x" */
  double speedX;

  /** Parameter "walk.speed-y" */
  double speedY;

  /** Parameter "walk.rotation-speed" */
  double rotationSpeed;
};

/**
* A Basic Behavior for special actions like kicks.
* @author Matthias Jngel
* @author Martin Ltzsch
* @author Max Risler
*/
class BasicBehaviorSpecialAction : public Xabsl2BasicBehavior, public BehaviorControlInterfaces
{
public:
/*
* Constructor.
* @param errorHandler Is invoked when errors occur
* @param interfaces The paramters of the BehaviorControl module.
  */
  BasicBehaviorSpecialAction(const BehaviorControlInterfaces& interfaces,
    Xabsl2ErrorHandler& errorHandler)
    : Xabsl2BasicBehavior("special-action", errorHandler),
    BehaviorControlInterfaces(interfaces)
  {
    registerParameter("special-action-id", specialActionID);
  }
  
  /**  Executes the basic behavior. */
  virtual void execute();

private:
  /** Parameter "special-action-id" */
  double specialActionID;
};


/**
* A Basic Behavior for standing around. 
* If the robot previously walked, it is slowed down first.
* @author Martin Ltzsch
*/
class BasicBehaviorStand : public Xabsl2BasicBehavior, public BehaviorControlInterfaces
{
public:
/*
* Constructor.
* @param errorHandler Is invoked when errors occur
* @param interfaces The paramters of the BehaviorControl module.
  */
  BasicBehaviorStand(const BehaviorControlInterfaces& interfaces,
    Xabsl2ErrorHandler& errorHandler)
    : Xabsl2BasicBehavior("stand", errorHandler),
    BehaviorControlInterfaces(interfaces),
    accelerationRestrictor(motionRequest)
  {
  }
  
  /**  Executes the basic behavior. */
  virtual void execute();
  
private:
  /** Restricts the walk acceleration to maximum values */
  WalkAccelerationRestrictor accelerationRestrictor;
};



/**
* @class CommonBasicBehaviors
*
* Creates and registers simple basic behaviors
*/
class CommonBasicBehaviors : public BehaviorControlInterfaces
{
public:
  /**
  * Constructor
  */
  CommonBasicBehaviors(const BehaviorControlInterfaces& interfaces,
                       Xabsl2ErrorHandler& errorHandler)
    : BehaviorControlInterfaces(interfaces),
      errorHandler(errorHandler),
      basicBehaviorDoNothing(interfaces,errorHandler),
      basicBehaviorShutdown(interfaces,errorHandler),
      basicBehaviorSpecialAction(interfaces,errorHandler),
      basicBehaviorStand(interfaces,errorHandler),
      basicBehaviorWalk(interfaces,errorHandler)
  {}

  /** Registers basic behaviors at the engine */
  void registerBasicBehaviors(Xabsl2Engine& engine);

private:
  /** Is invoked when errors occurs */
  Xabsl2ErrorHandler& errorHandler;

  //!@name Basic Behaviors
  //!@{
  BasicBehaviorDoNothing  basicBehaviorDoNothing;
  BasicBehaviorShutdown  basicBehaviorShutdown;
  BasicBehaviorSpecialAction  basicBehaviorSpecialAction;
  BasicBehaviorStand  basicBehaviorStand;
  BasicBehaviorWalk  basicBehaviorWalk;
  //!@}
};

#endif // __CommonBasicBehaviors_h_

/*
* Change Log
* 
* $Log: CommonBasicBehaviors.h,v $
* Revision 1.3  2004/03/08 00:58:53  roefer
* Interfaces should be const
*
* Revision 1.2  2003/10/31 08:32:48  dueffert
* doxygen bugs fixed
*
* Revision 1.1  2003/10/22 22:18:44  loetzsch
* prepared the cloning of the GT2003BehaviorControl
*
*/

