/**
* @file BBPotentialfields.h
* 
* This file contains a class which wraps the GT2004-WorldModel
* to the representation of the PotentialfieldComposition.
*
* @author <a href="mailto:timlaue@informatik.uni-bremen.de">Tim Laue</a>
*/

#ifndef BB_POTENTIALFIELDS_H_
#define BB_POTENTIALFIELDS_H_


#include "Tools/PotentialFields/PotentialfieldComposition.h"

class BallModel;
class RobotPose;
class PlayerPoseCollection;
class ObstaclesModel;
class TeamMessageCollection;
class Pose2D;
class string;


/**
* @class BBPotentialfields
*
* A class to use the PotentialfieldComposition inside the GT2004 architecture
*/
class BBPotentialfields
{
public:
  /** 
  * Constructor.
  * @param fileName The name of a file containing a configuration for a PotentialFieldComposition.
  */
  BBPotentialfields(const char* fileName);
  
  /** Constructor */
  BBPotentialfields();

  /** Loads a configuration file
  * @param filename The name of the file
  */
  void load(const char* filename);

  /** Calls the potential fields with data from the world model and returns an action 
  * @param robotPose The robot's pose
  * @param ballPosition The position of the ball
  * @param playerPoseCollection The positions of all other robots
  * @param obstaclesModel Other obstacles
  * @param result Returns the result
  */
  void execute(const RobotPose& robotPose,
               const BallModel& ballPosition,
               const PlayerPoseCollection& playerPoseCollection,
               const ObstaclesModel& obstaclesModel,
               const TeamMessageCollection& teamMessageCollection,
               PotentialfieldResult& result);

  /** Calls the potential fields with data from the world model and returns an action 
  * @param robotPose The robot's pose
  * @param ballPosition The position of the ball
  * @param playerPoseCollection The positions of all other robots
  * @param obstaclesModel Other obstacles
  * @param specialDestination A destination, e.g. for go-to-point skills
  * @param useSpecialDestination Flag: set to false if specialDestination should not be used
  * @param result Returns the result
  */
  void execute(const RobotPose& robotPose,
               const BallModel& ballPosition,
               const PlayerPoseCollection& playerPoseCollection,
               const ObstaclesModel& obstaclesModel,
               const TeamMessageCollection& teamMessageCollection,
               const Pose2D& specialDestination,
               const bool& useSpecialDestination,
               PotentialfieldResult& result);

  /** Computes an array of field values, used by visualization
  * @param robotPose The pose of the robot
  * @param ballPosition The position of the ball
  * @param playerPoseCollection The positions of all other robots
  * @param obstaclesModel The obstacles
  * @param fieldname The name of the potential field
  * @param x1 Minimum x-coordinate to compute
  * @param y1 Minimum y-coordinate to compute
  * @param x2 Maximum x-coordinate to compute
  * @param y2 Maximum y-coordinate to compute
  * @param xSteps The computation steps in x-direction
  * @param ySteps The computation steps in y-direction
  * @param value An array containing all computed values, memory has to be allocated BEFORE calling the function
  * @param max Returns the maximum value
  */
  void getValueArray(const RobotPose& robotPose,
                     const BallModel& ballPosition,
                     const PlayerPoseCollection& playerPoseCollection, 
                     const ObstaclesModel& obstaclesModel,
                     const TeamMessageCollection& teamMessageCollection,
                     const std::string& fieldname, double x1, double y1,
                     double x2, double y2, int xSteps, int ySteps, double value[], double& max);

  /** Computes an array of gradient directions, used by visualization
  * @param robotPose The pose of the robot
  * @param ballPosition The position of the ball
  * @param playerPoseCollection The positions of all other robots
  * @param obstaclesModel The obstacles
  * @param fieldname The name of the potential field
  * @param x1 Minimum x-coordinate to compute
  * @param y1 Minimum y-coordinate to compute
  * @param x2 Maximum x-coordinate to compute
  * @param y2 Maximum y-coordinate to compute
  * @param xSteps The computation steps in x-direction
  * @param ySteps The computation steps in y-direction
  * @param directions An array containing all computed gradients, memory has to be allocated BEFORE calling the function
  */
  void getDirectionArray(const RobotPose& robotPose,
                         const BallModel& ballPosition,
                         const PlayerPoseCollection& playerPoseCollection,
                         const ObstaclesModel& obstaclesModel,
                         const TeamMessageCollection& teamMessageCollection,
                         const std::string& fieldname, double x1, double y1,
                         double x2, double y2, int xSteps, int ySteps, PfVec directions[]);


  /** Returns the names of all fields within the potential field composition
  * @param fieldNames The names
  */
  void getFieldNames(std::vector<std::string>& fieldNames);

private:
  /** An instance of the potential field implementation*/
  PotentialfieldComposition potentialfields;
  /** Information about the ball*/
  ObjectStateDescription seenBallDesc, ballDesc;
  /** Information about the robot, needed by some behaviors*/
  ObjectStateDescription selfDesc;
  /** Information about the own team*/
  ObjectStateDescription goalieDesc, player2Desc, player3Desc, player4Desc;
  ObjectStateDescription unknownTeammate1Desc, unknownTeammate2Desc, 
                         unknownTeammate3Desc, unknownTeammate4Desc;
  /** Information about the other team*/
  ObjectStateDescription opponent1Desc, opponent2Desc, opponent3Desc, 
                         opponent4Desc, opponent5Desc, opponent6Desc;
  /** Information about a special destination*/
  ObjectStateDescription destinationDesc;
  /** Information about obstacles*/
  ObjectStateDescription obstacleFrontDesc, obstacleLeftDesc, obstacleRightDesc, 
                         obstacleFrontLeftDesc, obstacleFrontRightDesc;
  /** Information about the free part of the goal*/
  ObjectStateDescription freeGoalDesc;
  /** Position to look be looked at by goalie*/
  ObjectStateDescription goalieLookingPositionDesc;
  /** The goalie position*/
  ObjectStateDescription goaliePositionDesc;
  /** Angles to obstacles*/
  double angleToFrontObstacle, angleToLeftObstacle, angleToRightObstacle,
         angleToFrontLeftObstacle, angleToFrontRightObstacle;
  /** Opening angle of an obstacle sector*/
  double openingAngle;

  /** Updates the potentialfields with data from the world model
  * @param robotPose The robot's pose
  * @param ballPosition The position of the ball
  * @param playerPoseCollection The positions of all other robots
  * @param obstaclesModel The obstacles
  */
  void setData(const RobotPose& robotPose,
               const BallModel& ballPosition,
               const PlayerPoseCollection& playerPoseCollection,
               const ObstaclesModel& obstaclesModel,
               const TeamMessageCollection& teamMessageCollection);
};


#endif  //BB_POTENTIALFIELDS_H_



/*
* $Log: BBPotentialfields.h,v $
* Revision 1.6  2004/04/09 14:06:44  tim
* integrated changes from GO2004
*
* Revision 1.9  2004/04/02 21:11:18  tim
* changed parameters
*
* Revision 1.8  2004/04/01 22:16:11  tim
* Changed behavior
*
* Revision 1.7  2004/04/01 13:15:13  tim
* "improved" behavior
*
* Revision 1.6  2004/03/31 20:13:06  tim
* changed behavior
*
* Revision 1.5  2004/03/24 14:05:50  tim
* several small changes
*
* Revision 1.4  2004/02/03 13:19:49  spranger
* renamed all references to  class BallPosition to BallModel
*
* Revision 1.3  2004/01/23 15:22:34  tim
* bugs fixed
*
* Revision 1.2  2004/01/23 14:56:47  tim
* Added files for BremenByters potential field behavior
*
* Revision 1.1  2004/01/20 15:43:33  tim
* Added potential fields to Bremen Byters behavior
*
* Revision 1.7  2003/06/13 14:27:58  tim
* added random generator and tangential fields
*
* Revision 1.6  2003/06/09 20:00:04  tim
* Changed potentialfield architecture
*
* Revision 1.5  2003/05/06 22:19:11  dueffert
* Greenhills compilability restored
*
* Revision 1.4  2003/04/03 15:47:32  tim
* Added modelling for teammates
*
* Revision 1.3  2003/04/02 14:39:11  tim
* Changed Bremen Byters Behavior
*
* Revision 1.2  2003/03/30 15:32:09  tim
* several minor changes
*
* Revision 1.1  2003/03/23 17:51:27  tim
* Added potentialfields
*
*/
