/** 
* @file PotentialFieldBehaviorSymbolSet.cpp
*
* Implementation of class PotentialFieldBehaviorSymbolSet.
*
* @author <a href="mailto:timlaue@informatik.uni-bremen.de">Tim Laue</a>
*/

#include "PotentialFieldBehaviorSymbolSet.h"
#include <cassert>


PotentialFieldBehaviorSymbolSet::PotentialFieldBehaviorSymbolSet
(const BehaviorControlInterfaces& interfaces, const std::string& behaviorName, const std::string& pfcFileName):
BehaviorControlInterfaces(interfaces), pfields(pfcFileName.c_str()), lastComputationTimeStamp(0), name(behaviorName)
{
}


void PotentialFieldBehaviorSymbolSet::registerSymbols(Xabsl2Engine& engine)
{
  std::string xStr(name+".x");
  engine.registerDecimalInputSymbol(xStr.c_str(),this,
    (double (Xabsl2FunctionProvider::*)())&PotentialFieldBehaviorSymbolSet::getX);
  std::string yStr(name+".y");
  engine.registerDecimalInputSymbol(yStr.c_str(),this,
    (double (Xabsl2FunctionProvider::*)())&PotentialFieldBehaviorSymbolSet::getY);
  std::string rotationStr(name+".rotation");
  engine.registerDecimalInputSymbol(rotationStr.c_str(),this,
    (double (Xabsl2FunctionProvider::*)())&PotentialFieldBehaviorSymbolSet::getRotation);
  std::string actionStr(name+".action");
  engine.registerEnumeratedInputSymbol(actionStr.c_str(), this,
    (int (Xabsl2FunctionProvider::*)())&PotentialFieldBehaviorSymbolSet::getAction);
  if(name == "play-ball")
  {
    std::string gtb(actionStr+".run-to-ball");
    engine.registerEnumeratedInputSymbolEnumElement(actionStr.c_str(), gtb.c_str(), (int)runToBall);
    std::string obst(actionStr+".avoid-obstacles");
    engine.registerEnumeratedInputSymbolEnumElement(actionStr.c_str(), obst.c_str(), (int)avoidObstacles);
    std::string kickF(actionStr+".kick-forward");
    engine.registerEnumeratedInputSymbolEnumElement(actionStr.c_str(), kickF.c_str(), (int)kickForward);
    std::string kickR(actionStr+".kick-right");
    engine.registerEnumeratedInputSymbolEnumElement(actionStr.c_str(), kickR.c_str(), (int)kickRight);
    std::string kickL(actionStr+".kick-left");
    engine.registerEnumeratedInputSymbolEnumElement(actionStr.c_str(), kickL.c_str(), (int)kickLeft);
    std::string dribbleStr(actionStr+".dribble");
    engine.registerEnumeratedInputSymbolEnumElement(actionStr.c_str(), dribbleStr.c_str(), (int)dribble);
  }
  else if(name == "pfield-goalie")
  {
    std::string gtp(actionStr+".go-to-position");
    engine.registerEnumeratedInputSymbolEnumElement(actionStr.c_str(), gtp.c_str(), (int)goaliePosition);
    std::string gtb(actionStr+".go-to-ball");
    engine.registerEnumeratedInputSymbolEnumElement(actionStr.c_str(), gtb.c_str(), (int)goalieGotoBall);
    std::string kick(actionStr+".kick-forward");
    engine.registerEnumeratedInputSymbolEnumElement(actionStr.c_str(), kick.c_str(), (int)goalieKickForward);
    std::string kickR(actionStr+".kick-right");
    engine.registerEnumeratedInputSymbolEnumElement(actionStr.c_str(), kickR.c_str(), (int)goalieKickRight);
    std::string kickL(actionStr+".kick-left");
    engine.registerEnumeratedInputSymbolEnumElement(actionStr.c_str(), kickL.c_str(), (int)goalieKickLeft);
    std::string stand(actionStr+".stand");
    engine.registerEnumeratedInputSymbolEnumElement(actionStr.c_str(), stand.c_str(), (int)goalieStand);
  }
  else if(name == "standard-defender")
  {
    std::string gtdef(actionStr+".go-to-defense");
    engine.registerEnumeratedInputSymbolEnumElement(actionStr.c_str(), gtdef.c_str(), (int)gotoDefense);
  }
  else if(name == "standard-supporter")
  {
    std::string gtsup(actionStr+".go-to-support");
    engine.registerEnumeratedInputSymbolEnumElement(actionStr.c_str(), gtsup.c_str(), (int)gotoSupport);
  }
}


double PotentialFieldBehaviorSymbolSet::getX()
{
  update();
  return result.motion.pos.x;
}


double PotentialFieldBehaviorSymbolSet::getY()
{
  update();
  return result.motion.pos.y;
}


double PotentialFieldBehaviorSymbolSet::getRotation()
{
  update();
  double value(toDegrees(result.motion.rotation));
  if(fabs(value) < 5.0 && result.motion.pos.x == 0.0 && result.motion.pos.y == 0.0)
  {
    value = 0.0;
  }
  return value;
}


int PotentialFieldBehaviorSymbolSet::getAction()
{
  update();
  resolveActionName();
  return action;
}


void PotentialFieldBehaviorSymbolSet::update()
{
  if(SystemCall::getTimeSince(lastComputationTimeStamp) > 33)
  {
    lastComputationTimeStamp = SystemCall::getCurrentSystemTime();
    pfields.execute(robotPose, ballPosition, playerPoseCollection, obstaclesModel, teamMessageCollection, result);
    if(fabs(result.motion.speed) < 0.04)
    {
      result.motion.pos.x = 0.0;
      result.motion.pos.y = 0.0;
    }
    /*if(fabs(result.motion.rotation) < fromDegrees(3))
    {
      result.motion.rotation = 0;
    }*/
  }
}


void PotentialFieldBehaviorSymbolSet::resolveActionName()
{
  if(name == "play-ball")
  {
    if(result.action == "run-to-ball")
    {
      action = (int)runToBall;
    }
    else if(result.action == "avoid-obstacles")
    {
      action = (int)avoidObstacles;
    }
    else if(result.action == "dribble")
    {
      action = (int)dribble;
    }
    else if(result.action == "kick-right")
    {
      action = (int)kickRight;
    }
    else if(result.action == "kick-left")
    {
      action = (int)kickLeft;
    }
    else if(result.action == "kick-forward")
    {
      action = (int)kickForward;
    }
    else
    {
      //This should not happen!
      assert(false);
    }
  }
  else if(name == "pfield-goalie")
  {
    if(result.action == "go-to-ball")
    {
      action = (int)goalieGotoBall;
    }
    else if(result.action == "kick-right")
    {
      action = (int)goalieKickRight;
    }
    else if(result.action == "kick-left")
    {
      action = (int)goalieKickLeft;
    }
    else if(result.action == "kick-forward")
    {
      action = (int)goalieKickForward;
    }
    else if(result.action == "stand")
    {
      action = (int)goalieStand;
    }
    else
    {
      action = (int)goaliePosition;
    }
  }
}


/*
* $Log: PotentialFieldBehaviorSymbolSet.cpp,v $
* Revision 1.6  2004/04/09 14:06:44  tim
* integrated changes from GO2004
*
* Revision 1.14  2004/04/03 05:21:08  tim
* improved behavior
*
* Revision 1.13  2004/04/02 23:50:25  tim
* changed parameters
*
* Revision 1.12  2004/04/02 01:11:02  tim
* Changed behavior
*
* Revision 1.11  2004/04/01 23:36:27  tim
* Changed behavior
*
* Revision 1.10  2004/04/01 22:16:11  tim
* Changed behavior
*
* Revision 1.9  2004/03/31 21:55:53  tim
* changed behavior
*
* Revision 1.8  2004/03/31 21:22:35  tim
* changed behavior
*
* Revision 1.7  2004/03/31 20:13:06  tim
* changed behavior
*
* Revision 1.6  2004/03/30 22:45:16  tim
* changed behavior
*
* Revision 1.5  2004/03/27 09:34:04  tim
* changed BB2004 behavior
*
* Revision 1.4  2004/03/25 12:05:55  tim
* added new tasks
*
* Revision 1.3  2004/03/24 14:05:50  tim
* several small changes
*
* Revision 1.2  2004/03/19 15:38:01  tim
* changed BB2004BehaviorControl
*
* Revision 1.1  2004/03/15 12:37:24  tim
* BB2004 dynamic task selection classes added
*
*/

