/**
* @file BB2004BehaviorControl.cpp
* 
* Implementation of class BB2004BehaviorControl.
*
* @author Martin Ltzsch
* @author Tim Laue
*/

#include "BB2004BehaviorControl.h"
#include "Tools/Player.h"


BB2004BehaviorControl::BB2004BehaviorControl(const BehaviorControlInterfaces& interfaces)
: Xabsl2BehaviorControl(interfaces,SolutionRequest::bb2004),
ballSymbols(interfaces),
headAndTailSymbols(interfaces),
joystickSymbols(interfaces),
ledAndSoundSymbols(interfaces),
mathFunctions(interfaces),
motionRequestSymbols(interfaces),
roboCupGameManagerSymbols(interfaces),
robotPoseSymbols(interfaces),
robotStateSymbols(interfaces),
specialVisionSymbols(interfaces),
strategySymbols(interfaces),
taskSymbols(interfaces),
configurationSymbols(interfaces, behaviorConfiguration),
challengeSymbols(interfaces),
simpleBasicBehaviors(interfaces,errorHandler),
commonBasicBehaviors(interfaces,errorHandler),
lastSound(SoundRequest::none),
kickoffSwitchTime(0)
{
  Xabsl2FileInputSource file("Xabsl2/bb04-ic.dat");
  init(file);
}

BB2004BehaviorControl::~BB2004BehaviorControl()
{
}

void BB2004BehaviorControl::registerSymbolsAndBasicBehaviors()
{
  simpleBasicBehaviors.registerBasicBehaviors(*pEngine);
  commonBasicBehaviors.registerBasicBehaviors(*pEngine);
  
  ballSymbols.registerSymbols(*pEngine);
  headAndTailSymbols.registerSymbols(*pEngine);
  joystickSymbols.registerSymbols(*pEngine);
  ledAndSoundSymbols.registerSymbols(*pEngine);
  mathFunctions.registerSymbols(*pEngine);
  motionRequestSymbols.registerSymbols(*pEngine);
  roboCupGameManagerSymbols.registerSymbols(*pEngine);
  robotPoseSymbols.registerSymbols(*pEngine);
  robotStateSymbols.registerSymbols(*pEngine);
  specialVisionSymbols.registerSymbols(*pEngine);
  strategySymbols.registerSymbols(*pEngine);
  taskSymbols.registerSymbols(*pEngine);
  configurationSymbols.registerSymbols(*pEngine);
  challengeSymbols.registerSymbols(*pEngine);
}

void BB2004BehaviorControl::execute()
{
  ballSymbols.update();
  roboCupGameManagerSymbols.update();
  robotPoseSymbols.update();
  strategySymbols.update();
  configurationSymbols.update();
  RState gameState = gameControlData.data.state;

  // set the outgoint behavior team message to none
  outgoingBehaviorTeamMessage.message = BehaviorTeamMessage::none;
  
  // set the upper leds depending on the dynamic role.
  switch (strategySymbols.role)
  {
  case BehaviorTeamMessage::striker:
    ledRequest.redTopLEDs = LEDRequest::bothOn;
    break;
  case BehaviorTeamMessage::offensiveSupporter:
    ledRequest.redTopLEDs = LEDRequest::bothFastBlink;
    break;
  case BehaviorTeamMessage::defensiveSupporter:
  default:
    ledRequest.redTopLEDs = LEDRequest::bothOff;
  }

  // Set team color and kickoff state 
  if(gameState == ROBOCUP_STATE_INITIAL)
  {
    RobotState::Switches currentSwitch(robotState.getSwitches());
    long timeSincePressed(SystemCall::getTimeSince(robotState.getSwitchTimestamp()));
    if(timeSincePressed > 300)
    {
      currentSwitch = RobotState::nothingPressed;
    }
    if(currentSwitch == RobotState::backBackPressed)
    {
      gameControlData.data.teamColor = ROBOCUP_TEAMCOLOR_RED;
    }
    else if(currentSwitch == RobotState::backFrontPressed)
    {
      gameControlData.data.teamColor = ROBOCUP_TEAMCOLOR_BLUE;
    }
	  else if((currentSwitch == RobotState::backPressed) &&
	          (SystemCall::getTimeSince(kickoffSwitchTime) > 700))
    {
      kickoffSwitchTime = SystemCall::getCurrentSystemTime();
      gameControlData.data.kickoff = (gameControlData.data.kickoff == ROBOCUP_KICKOFF_OWN ?
                                      ROBOCUP_KICKOFF_OPPONENT : ROBOCUP_KICKOFF_OWN);
	  }     
    // Transfer team color information
    getPlayer().setTeamColor(gameControlData.data.teamColor == ROBOCUP_TEAMCOLOR_BLUE ?
                             Player::blue : Player::red);
  }

  // Display setup information
  ledRequest.backFrontBlueLED = getPlayer().getTeamColor()== Player::blue ? LEDRequest::llll : LEDRequest::oooo;
  ledRequest.backRearRedLED = getPlayer().getTeamColor()== Player::red ? LEDRequest::llll : LEDRequest::oooo;
  ledRequest.backMiddleOrangeLED = gameControlData.data.kickoff == ROBOCUP_KICKOFF_OWN ? LEDRequest::llll : LEDRequest::oooo;
  
  // Set the tail request depending on whether the ball was seen
  if (SystemCall::getTimeSince(ballPosition.seen.timeWhenLastSeen) < 120)
  {
    // ball seen now
    motionRequest.tailRequest = MotionRequest::wagHorizontalFast;
  }
  else if (SystemCall::getTimeSince(ballPosition.seen.timeWhenLastSeen) < 2000)
  {
    // ball seen
    motionRequest.tailRequest = MotionRequest::wagHorizontal;
  }
  else if (SystemCall::getTimeSince(ballPosition.seen.timeWhenLastSeen) 
    > BallModel::behaviorControlTimeAfterWhichCommunicatedBallsAreAccepted
    && SystemCall::getTimeSince(ballPosition.communicated.timeWhenLastObserved) < 3000)
  {
    // ball known
    motionRequest.tailRequest = MotionRequest::wagVertical;
  }
  else
  {
    // ball not known
    motionRequest.tailRequest = MotionRequest::noTailWag;
  }
  
  // set head control mode lookToStars to detect missing settings
  headControlMode.headControlMode = HeadControlMode::lookToStars;
  
  soundRequest.soundID = SoundRequest::none;

  // execute the engine
  executeEngine();

  if((robotState.getState() == RobotState::crashed)||
     (robotState.getState() == RobotState::rollLeft)||
     (robotState.getState() == RobotState::rollRight))
  {
    if ((motionRequest.motionType == MotionRequest::walk &&
      motionRequest.walkType != MotionRequest::upsideDown)||
      (motionRequest.motionType == MotionRequest::stand))
    {
      motionRequest.motionType = MotionRequest::getup;
      return;
    }
  }

  if (headControlMode.headControlMode == HeadControlMode::lookToStars)
  {
    soundRequest.soundID = SoundRequest::rob101;
  }
  
  switch(gameState)
  {
    case ROBOCUP_STATE_INITIAL: outgoingBehaviorTeamMessage.gameState = BehaviorTeamMessage::initial; break;
    case ROBOCUP_STATE_READY: outgoingBehaviorTeamMessage.gameState = BehaviorTeamMessage::ready; break;
    case ROBOCUP_STATE_SET: outgoingBehaviorTeamMessage.gameState = BehaviorTeamMessage::set; break;
    case ROBOCUP_STATE_PLAYING: outgoingBehaviorTeamMessage.gameState = BehaviorTeamMessage::playing; break;
    case ROBOCUP_STATE_PENALIZED: outgoingBehaviorTeamMessage.gameState = BehaviorTeamMessage::penalized; break;
    default: outgoingBehaviorTeamMessage.gameState = BehaviorTeamMessage::finished; break;
  }
  outgoingBehaviorTeamMessage.estimatedTimeToReachBall = strategySymbols.estimatedTimeToReachBall;
}

bool BB2004BehaviorControl::handleMessage(InMessage& message)
{
  return (behaviorConfiguration.handleMessage(message) ||
    Xabsl2BehaviorControl::handleMessage(message));
}


/*
* Change log :
* 
* $Log: BB2004BehaviorControl.cpp,v $
* Revision 1.15  2004/05/14 11:37:08  loetzsch
* support for multiple xabsl2engines in different modules
* preliminary GT2004HeadControl (does not work at all)
*
* Revision 1.14  2004/05/04 10:48:58  loetzsch
* replaced all enums
* xxxBehaviorControlTimeAfterWhichCommunicatedBallsAreAccepted
* by
* behaviorControlTimeAfterWhichCommunicatedBallsAreAccepted
* (this mechanism was neither fully implemented nor used)
*
* Revision 1.13  2004/04/08 11:21:55  loetzsch
* removed function getCurrentState from RoboCupGameManagerSymbols
*
* Revision 1.12  2004/03/28 12:53:21  loetzsch
* improved led setting
*
* Revision 1.11  2004/03/27 09:34:04  tim
* changed BB2004 behavior
*
* Revision 1.10  2004/03/25 19:56:28  loetzsch
* again changed the led request
*
* Revision 1.9  2004/03/25 17:40:15  loetzsch
* adaptations to changes in the game controller and the led request
*
* Revision 1.8  2004/03/25 15:49:47  tim
* changed handling of current state
*
* Revision 1.7  2004/03/25 12:05:55  tim
* added new tasks
*
* Revision 1.6  2004/03/24 14:05:49  tim
* several small changes
*
* Revision 1.5  2004/03/19 15:38:00  tim
* changed BB2004BehaviorControl
*
* Revision 1.4  2004/03/15 12:50:30  tim
* Adaptions to new GameController
*
* Revision 1.3  2004/03/08 00:58:49  roefer
* Interfaces should be const
*
* Revision 1.2  2004/02/03 13:20:47  spranger
* renamed all references to  class BallPosition to BallModel (possibly changed include files)
*
* Revision 1.1  2003/10/26 22:49:36  loetzsch
* created ATH2004BehaviorControl from GT2003BehaviorControl
*  - strongly simplified option graph
*  - moved some symbols from GT2003 to CommonXabsl2Symbols
*  - moved some basic behaviors from GT2003 to CommonXabsl2BasicBehaviors
*
* cloned ATH2004 three times (BB2004, DDD2004, MSH2004)
*
*/

