/**
* @file ATH2004ERS7Xabsl2Profiler.h
* 
* Definition and Implementation of classes ATH2004ERS7Xabsl2Profiler, ATH2004ERS7Xabsl2ProfilerNameTable
*
* @author Michael Spranger
*/
#ifndef __ATH2004ERS7Xabsl2Profiler_h_
#define __ATH2004ERS7Xabsl2Profiler_h_

#ifdef WIN32
// disable some annoying warnings
//   C4503: decorated name length exceeded, name was truncated.
//   C4786: identifier was truncated to '255' characters in the browser information.
//
// # pragma warning(disable:4503)
# pragma warning(disable:4786)
#endif

#include <string>
#include <vector>
#include <deque>

#include "Platform/GTAssert.h"

#include "Tools/Streams/OutStreams.h"
#include "Platform/SystemCall.h"

#include "Tools/Xabsl2/Xabsl2Engine/Xabsl2Array.h"
#include "Tools/Xabsl2/Xabsl2Engine/Xabsl2Engine.h"
#include "Tools/Debugging/Debugging.h"

#include "Tools/Streams/InStreams.h"


/* Nr of Logelements written out per call to writeLog */
#define NROFELEMW 2
class ATH2004ERS7Xabsl2ProfilerNameTableEntry{
  public:
    std::string optionName;
    std::vector<std::string> states;
    std::vector<std::string> parameters;
    int maxdepth;
    ATH2004ERS7Xabsl2ProfilerNameTableEntry(){};
    ATH2004ERS7Xabsl2ProfilerNameTableEntry(const std::string option,std::vector<std::string> state, std::vector<std::string> params, const int depth)
      : optionName(option), states(state), parameters(params), maxdepth(depth){}
};
/*
* @class ATH2004ERS7Xabsl2ProfilerNameTable
* Dynamic Array for keeping a correlation of Xabsl2Options and States to Numbers for profiling purposes (remembering activation paths).
* @author Michael Spranger
*/
class ATH2004ERS7Xabsl2ProfilerNameTable
{
  /** The array */
  ATH2004ERS7Xabsl2ProfilerNameTableEntry* data;
  
  
  /** The number of elements in the array */
  int usedSize, length;
  public:
    /** Constructor */
    ATH2004ERS7Xabsl2ProfilerNameTable():usedSize(0), length(0), data(NULL)
    {}
    
    /** Destructor */
    ~ATH2004ERS7Xabsl2ProfilerNameTable() 
    { 
      if(data)
        delete[] data;
    }
    /**
    * Initializes the Array to a given length
    * @param length_ The length of the initialised array
    */
    void init(int length_){
      if(data){
        delete[] data; data = NULL;
      }
      usedSize = 0;
      length = length_;
      data = new ATH2004ERS7Xabsl2ProfilerNameTableEntry[length];
    }
    
    /** Clears the array */
    void clear()
    { 
      if(data){
        delete[] data;
        usedSize = 0;
        data = new ATH2004ERS7Xabsl2ProfilerNameTableEntry[length];
      }
      
    }
    
    /** 
    * Returns the Position for a given optionname and statename. 
    * Note that the function crashes if the element does not exist.
    */
    int getOptionPosition(const std::string optionName) const
    {
      return findOption(optionName);
    }
    int getStatePosition(const std::string optionName, const std::string state){
      return findState(optionName, state);
    }
    // int getParameterPosition(const std::string optionName, const std::string state, const std::string);
    
    /**
    * The function appends a new element to the array.
    * @param optionName A string label for the option.
    * @param stateName A string label for the state.
    */  
    void append(const std::string optionName, const std::vector<std::string> states, const std::vector<std::string> params, int depth)
    {
      if(usedSize == length)
      {
        length += length > 1 ? length / 2 : 2; 
        
        ATH2004ERS7Xabsl2ProfilerNameTableEntry* temp = new ATH2004ERS7Xabsl2ProfilerNameTableEntry[length];
        if(data){
          for(int i = 0; i < getSize(); ++i)
            temp[i] = data[i];
          delete[] data;
        }
        data = temp;
      }
      data[usedSize++]=ATH2004ERS7Xabsl2ProfilerNameTableEntry(optionName, states, params, depth);
    }
    
    
    /**
    * The function sets the value of an element in the array.
    * Note that the function crashes if the element does not exist.
    * @param pos The position of the element in the array.
    * @param value The new element.
    */
    void setElement(int pos, ATH2004ERS7Xabsl2ProfilerNameTableEntry value)
    {
      data[pos] = value;
    }
    
    /**
    * The function returns the number of elements in the array.
    * @return The length of the list.
    */
    int getSize() const {return usedSize;}
    
    /** 
    * Returns the value for a given array position.
    * Note that the function crashes if the required position is bigger than the 
    * size of the array.
    */
    ATH2004ERS7Xabsl2ProfilerNameTableEntry& operator[](int pos) const
    {
      return data[pos];
    }
    
    /** Returns whether an element for the given name exists */
    bool exists(const std::string optionName, const std::string stateName) const
    {
      return find(optionName, stateName) >= 0;
    }
    bool existsOption(const std::string option){
      return findOption(option) >=0;
    }
    
   protected:
   /** 
   * Returns the index of an element with the given names.
   * @return The index of the element of -1 if the name does not exist.
     */
     int find(const std::string optionName, const std::string state) const
     {
       for(int i = 0; i < getSize(); ++i){
         
         if(data[i].optionName  == optionName){
           for(std::vector<std::string>::iterator j = data[i].states.begin(); j != data[i].states.end(); ++j){
             if((*j) == state)
               return i;
           }
           
         }
       }
       return -1;
     }
     int findState(const std::string optionName, const std::string state) const
     {
       for(int i = 0; i < getSize(); ++i){
        
         if(data[i].optionName  == optionName){
           for(int j = 0; j < (int) data[i].states.size(); ++j){
             if(state == data[i].states[j])
               return j;
           }
           
         }
       }
       return -1;
     }
     int findOption(const std::string optionName) const
     {
       for(int i = 0; i < getSize(); ++i)
         if(data[i].optionName  == optionName) 
           return i;
         return -1;
     }
};

class ATH2004ERS7Xabsl2ActiveOption{
public:
  int optionNumber;
  int stateNumber;
  std::vector<double> parameters;
  ATH2004ERS7Xabsl2ActiveOption(): optionNumber(-1), stateNumber(-1){}
  ATH2004ERS7Xabsl2ActiveOption(int o, int st, std::vector<double> p): optionNumber(o), stateNumber(st), parameters(p){}
  
  bool operator ==(const ATH2004ERS7Xabsl2ActiveOption other)const
  {
    if(this->optionNumber == other.optionNumber && this->stateNumber == other.stateNumber
      && this->parameters == other.parameters)
      return true;
    return false;
  }
};

/**
* @class ATH2004ERS7Xabsl2LogEntry
* Class for keeping numerical activation paths and framenumbers
* @author Michael Spranger
*/

class ATH2004ERS7Xabsl2LogEntry{
public:
  unsigned long framenumber;
  std::vector<ATH2004ERS7Xabsl2ActiveOption> activeOptions;
  ATH2004ERS7Xabsl2LogEntry(unsigned long fn, std::vector<ATH2004ERS7Xabsl2ActiveOption> aO):framenumber(fn), activeOptions(aO){}
  ATH2004ERS7Xabsl2LogEntry(unsigned long fn):framenumber(fn){}
  // ATH2004ERS7Xabsl2LogEntry():framenumber(0){}
  
};

/**
* @class ATH2004ERS7Xabsl2Profiler
* Class for profiling and analysis of Xabsl-Activation-Paths
* @author Michael Spranger
*/
class ATH2004ERS7Xabsl2Profiler{
  ATH2004ERS7Xabsl2ProfilerNameTable nameTable;
  std::deque<ATH2004ERS7Xabsl2LogEntry> log;
  std::string outfilename;
  int maxdepth;

  
  void registerEngine(Xabsl2Engine& pEngine);
  void registerOptions(const Xabsl2Option*, int);
  void doDepthCount(const Xabsl2Option*, int);
  void writeLogToStream(Out&);
  void writeCompleteLogToStream(Out&);
  void writeNameTableToStream(Out&);

public: 
  ATH2004ERS7Xabsl2Profiler();
  ATH2004ERS7Xabsl2Profiler(char*);
  void setOutputFile(char* s){outfilename = s;};
  void init(Xabsl2Engine&, int = 1);
  void doProfiling(Xabsl2Engine& pEngine, unsigned long framenumber);
  void recordCollectedLogs();
  void recordCompleteLog();
  void debugLogToStream(Out&);
  
  /*
   void writeNameTableRToStream(Out&);
   void writeLogRToStream(Out&);	
   void exportReadableLogFile(char*);  
   void writeCompleteLogToStream(Out&);
   */
  
  void writeXMLtoStream(Out&);
  void exportXMLFile(const char*);
  bool importLogFile(const char*);

  std::vector<ATH2004ERS7Xabsl2ActiveOption> getActiveOptionsAtFrame(int time);
  std::string getOptionName(int);
  std::string getStateName(int, int);
  int getDepth(char* optionname);

  int getDepth(int);
  int getMaxDepth(){ return maxdepth; }
  int getBeginningFramenumber(){ return log.size()?log.front().framenumber:0; }
  int getLastFramenumber(){ return log.size()?log.back().framenumber:0;}
  int getNumberofLogEntries();
  std::vector<ATH2004ERS7Xabsl2ActiveOption> getActiveOptionsAtNumber(int);
  ATH2004ERS7Xabsl2LogEntry getLogEntryAtIndex(int);
  int getFollowingFramenumber(int frame);
  int getFramenumberAtEntryNumber(int);
  int getIndex(int);

  ATH2004ERS7Xabsl2LogEntry& operator[](int i);
  int size();
  ATH2004ERS7Xabsl2ProfilerNameTableEntry& getNameTableEntry(int index);
  bool getActiveOption(int index, int maxdepth, ATH2004ERS7Xabsl2ActiveOption& retour);
};

#endif// __ATH2004ERS7Xabsl2Profiler_h_


/*
* Change Log:
*
* $Log: ATH2004ERS7Xabsl2Profiler.h,v $
* Revision 1.10  2004/05/22 16:58:04  spranger
* changed dialog appearance, added features (including extended profiler-interface)
*
* Revision 1.9  2004/05/04 15:35:02  spranger
* extended interface
*
* Revision 1.8  2004/05/03 13:21:57  spranger
* extended interface
*
* Revision 1.7  2004/04/30 09:26:23  spranger
* redone interface, added xml-export
*
* Revision 1.6  2004/04/28 17:31:40  spranger
* redesign of nametable, added import facilities
*
* Revision 1.5  2004/04/22 12:21:33  spranger
* removed compiler warnings
*
* Revision 1.4  2004/04/22 11:33:19  spranger
* different log handling (queue), added writingoglogs which does not write out whole log
*
* Revision 1.3  2004/04/20 18:17:01  spranger
* added nametable-structure, added log functionality
*
* Revision 1.2  2004/04/14 13:33:54  loetzsch
* added change log
*
*/
