/** 
* @file ATH2004ERS7StrategySymbols.cpp
*
* Implementation of class ATH2004ERS7StrategySymbols.
*
* @author Martin Ltzsch
*/

#include "ATH2004ERS7StrategySymbols.h"
#include "Tools/FieldDimensions.h"


ATH2004ERS7StrategySymbols::ATH2004ERS7StrategySymbols(BehaviorControlInterfaces& interfaces)
: BehaviorControlInterfaces(interfaces),
timeWhenBallWasStartedToCatch(0), timeUntilBallWasCaught(0),
doProfile(ATH2004ERS7StrategySymbols::dontDoProfiling), writeProfile(ATH2004ERS7StrategySymbols::dontWriteProfiles)
{
  role = BehaviorTeamMessage::goalie;
  estimatedTimeToReachBall = 0.0;
}

void ATH2004ERS7StrategySymbols::registerSymbols(Xabsl2Engine& engine)
{
  // "robot-number"
  engine.registerDecimalInputSymbol("robot-number", this,
    (double(Xabsl2FunctionProvider::*)())&ATH2004ERS7StrategySymbols::getRobotNumber);
  
  // "role"
  engine.registerEnumeratedInputSymbol("role", (int *)&role);
  engine.registerEnumeratedInputSymbolEnumElement("role", "goalie", BehaviorTeamMessage::goalie);
  engine.registerEnumeratedInputSymbolEnumElement("role", "striker", BehaviorTeamMessage::striker);
  engine.registerEnumeratedInputSymbolEnumElement("role", "defensive-supporter", BehaviorTeamMessage::defensiveSupporter);
  engine.registerEnumeratedInputSymbolEnumElement("role", "offensive-supporter", BehaviorTeamMessage::offensiveSupporter);
  
  // "estimated-time-to-reach-ball"
  engine.registerDecimalInputSymbol("estimated-time-to-reach-ball", (double *)&estimatedTimeToReachBall);
  
  // "sent-game-state"
  engine.registerEnumeratedOutputSymbol("sent-game-state",(int*)&outgoingBehaviorTeamMessage.gameState);
  engine.registerEnumeratedOutputSymbolEnumElement("sent-game-state","sent-game-state.sleep",BehaviorTeamMessage::sleep);
  engine.registerEnumeratedOutputSymbolEnumElement("sent-game-state","sent-game-state.initial",BehaviorTeamMessage::initial);
  engine.registerEnumeratedOutputSymbolEnumElement("sent-game-state","sent-game-state.ready",BehaviorTeamMessage::ready);
  engine.registerEnumeratedOutputSymbolEnumElement("sent-game-state","sent-game-state.playing",BehaviorTeamMessage::playing);
  engine.registerEnumeratedOutputSymbolEnumElement("sent-game-state","sent-game-state.penalized",BehaviorTeamMessage::penalized);
  engine.registerEnumeratedOutputSymbolEnumElement("sent-game-state","sent-game-state.finished",BehaviorTeamMessage::finished);
  engine.registerEnumeratedOutputSymbolEnumElement("sent-game-state","sent-game-state.set",BehaviorTeamMessage::set);
  
  
  // "another-player-is-in-ready-state"
  engine.registerBooleanInputSymbol("another-player-is-in-ready-state",
    this,(bool(Xabsl2FunctionProvider::*)())&ATH2004ERS7StrategySymbols::getAnotherPlayerIsInReadyState);
  
  // "another-player-is-in-initial-state"
  engine.registerBooleanInputSymbol("another-player-is-in-initial-state",
    this,(bool(Xabsl2FunctionProvider::*)())&ATH2004ERS7StrategySymbols::getAnotherPlayerIsInInitialState);
  
  // "team-message"
  engine.registerEnumeratedOutputSymbol("team-message",(int*)&outgoingBehaviorTeamMessage.message);
  engine.registerEnumeratedOutputSymbolEnumElement("team-message","team-message.none", BehaviorTeamMessage::none);
  engine.registerEnumeratedOutputSymbolEnumElement("team-message","team-message.just-performed-a-kick",BehaviorTeamMessage::justPerformedAKick);
  engine.registerEnumeratedOutputSymbolEnumElement("team-message","team-message.performing-a-kick",BehaviorTeamMessage::performingAKick);
  engine.registerEnumeratedOutputSymbolEnumElement("team-message","team-message.preparing-a-kick",BehaviorTeamMessage::preparingAKick);
  
  // "another-teammate-is-preparing-a-kick"
  engine.registerBooleanInputSymbol("another-teammate-is-preparing-a-kick",
    this,(bool(Xabsl2FunctionProvider::*)())&ATH2004ERS7StrategySymbols::getAnotherTeammateIsPreparingAKick);
  
  // "another-teammate-is-performing-a-kick"
  engine.registerBooleanInputSymbol("another-teammate-is-performing-a-kick",
    this,(bool(Xabsl2FunctionProvider::*)())&ATH2004ERS7StrategySymbols::getAnotherTeammateIsPerformingAKick);
  
  // "another-teammate-just-performed-a-kick"
  engine.registerBooleanInputSymbol("another-teammate-just-performed-a-kick",
    this,(bool(Xabsl2FunctionProvider::*)())&ATH2004ERS7StrategySymbols::getAnotherTeammateJustPerformedAKick);
  
  // "the-striker-is-playing-near-the-opponent-goal"
  engine.registerBooleanInputSymbol("the-striker-is-playing-near-the-opponent-goal",
    this,(bool(Xabsl2FunctionProvider::*)())&ATH2004ERS7StrategySymbols::getTheStrikerIsPlayingNearTheOpponentGoal);
  
  // "the-striker-is-not-playing-near-the-opponent-goal"
  engine.registerBooleanInputSymbol("the-striker-is-not-playing-near-the-opponent-goal",
    this,(bool(Xabsl2FunctionProvider::*)())&ATH2004ERS7StrategySymbols::getTheStrikerIsNotPlayingNearTheOpponentGoal);
  
  // "the-striker-is-playing-near-the-own-goal"
  engine.registerBooleanInputSymbol("the-striker-is-playing-near-the-own-goal",
    this,(bool(Xabsl2FunctionProvider::*)())&ATH2004ERS7StrategySymbols::getTheStrikerIsPlayingNearTheOwnGoal);
  
  // "the-striker-is-not-playing-near-the-own-goal"
  engine.registerBooleanInputSymbol("the-striker-is-not-playing-near-the-own-goal",
    this,(bool(Xabsl2FunctionProvider::*)())&ATH2004ERS7StrategySymbols::getTheStrikerIsNotPlayingNearTheOwnGoal);
  
  // "catch-ball"
  engine.registerBooleanInputSymbol("catch-ball", this,
    (bool (Xabsl2FunctionProvider::*)())&ATH2004ERS7StrategySymbols::getCaught);
  
  // "goalie-max-position-speed"
  engine.registerDecimalInputSymbol("goalie-max-position-speed", this,
    (double (Xabsl2FunctionProvider::*)())&ATH2004ERS7StrategySymbols::getGoalieMaxPositionSpeed);
  
  // "catch-ball-time"
  engine.registerDecimalInputSymbol("catch-ball-time",this,
    (double (Xabsl2FunctionProvider::*)())&ATH2004ERS7StrategySymbols::getCatchTime);
  
  // "robot-is-stuck"
  engine.registerBooleanInputSymbol("robot-is-stuck", &robotIsStuck);
  
  // "obstacles-are-close"
  engine.registerBooleanInputSymbol("obstacles-are-close", &obstaclesAreClose);
  
  engine.registerDecimalInputSymbol("search-ball.x", this,
    (double (Xabsl2FunctionProvider::*)())&ATH2004ERS7StrategySymbols::getSearchBallX);
  
  // "get-angle-to-teammate
  engine.registerDecimalInputFunction("get-angle-to-teammate",this,
    (double (Xabsl2FunctionProvider::*)())&ATH2004ERS7StrategySymbols::getAngleToTeammate);
  engine.registerDecimalInputFunctionParameter("get-angle-to-teammate","get-angle-to-teammate.index",ATH2004ERS7StrategySymbols::angleToTeammateIndex);

  engine.registerEnumeratedOutputSymbol("do-profiling",(int*)&doProfile);
  engine.registerEnumeratedOutputSymbolEnumElement("do-profiling","do-profiling.true", ATH2004ERS7StrategySymbols::doProfiling);
  engine.registerEnumeratedOutputSymbolEnumElement("do-profiling","do-profiling.false",ATH2004ERS7StrategySymbols::dontDoProfiling);

  engine.registerEnumeratedOutputSymbol("write-profile",(int*)&writeProfile);
  engine.registerEnumeratedOutputSymbolEnumElement("write-profile","write-profile.true", ATH2004ERS7StrategySymbols::writeProfiles  );
  engine.registerEnumeratedOutputSymbolEnumElement("write-profile","write-profile.complete",ATH2004ERS7StrategySymbols::writeCompleteProfiles);
  engine.registerEnumeratedOutputSymbolEnumElement("write-profile","write-profile.false",ATH2004ERS7StrategySymbols::dontWriteProfiles);
  
  
  
}

void ATH2004ERS7StrategySymbols::update()
{
  estimateTimeToReachBall();
  computeRole();
  
  outgoingBehaviorTeamMessage.estimatedTimeToReachBall = estimatedTimeToReachBall;
  outgoingBehaviorTeamMessage.dynamicRole = role;
  
  if ( 
    (executedMotionRequest.motionType == MotionRequest::walk 
    && (executedMotionRequest.walkType == MotionRequest::turnWithBall2
    || executedMotionRequest.walkType == MotionRequest::turnWithBall4)
    )
    
    || 
    
    headControlMode.headControlMode == HeadControlMode::catchBall
    
    )
  {
    if (SystemCall::getTimeSince(timeUntilBallWasCaught) > 500)
    {
      timeWhenBallWasStartedToCatch = SystemCall::getCurrentSystemTime();
    }
    
    timeUntilBallWasCaught = SystemCall::getCurrentSystemTime();
  }
  
  // robot is stuck
  if(obstaclesModel.getPercentageOfLowDistanceObstaclesInRange(0, pi_2, 300) > 0.2)
  {
    robotIsStuck = true;
    obstaclesAreClose = true;
  }
  else
  {
    robotIsStuck = false;
    if(obstaclesModel.getPercentageOfLowDistanceObstaclesInRange(0, pi_2, 500) > 0.15)
      obstaclesAreClose = true;
    else
      obstaclesAreClose = false;
  }
}

void ATH2004ERS7StrategySymbols::computeRole()
{
  int i;
  
  if (getPlayer().getPlayerNumber() == Player::one)
  {
    // no role changes for the goalie
    role = BehaviorTeamMessage::goalie;
    return;
  }

  
  // if in the goalzone and ball lies within angle infront, role = striker micha
  if(FieldDimensions::distanceToOpponentPenaltyArea(robotPose.translation) < 400
    && SystemCall::getTimeSince(ballPosition.seen.timeWhenLastSeen) < 1000)
  {
/*
version Paderborn 1
    if(
        fabs(Geometry::angleTo(robotPose, Vector2<double>(xPosOpponentGroundline, 0))) < pi/6
        && Geometry::angleTo(robotPose, Vector2<double>(xPosOpponentGroundline, -400)) 
           < 
           Geometry::angleTo(robotPose, ballPosition.seen)
        && Geometry::angleTo(robotPose, Vector2<double>(xPosOpponentGroundline, 400)) 
           >
           Geometry::angleTo(robotPose, ballPosition.seen)
       )
    {
        role = BehaviorTeamMessage::striker;
        return;
    }

*/
    if(ballPosition.seen.ballInFrontOfOpponentGoal)
    {
        role = BehaviorTeamMessage::striker;
        return;
    }


    /*else if( obstaclesModel.angleToFreePartOfGoalWasDetermined[ObstaclesModel::opponentGoal]
      && fabs(obstaclesModel.angleToFreePartOfGoal[ObstaclesModel::opponentGoal] - Geometry::angleTo(robotPose, ballPosition.seen)) < pi/4
          
            )
    { 
        role = BehaviorTeamMessage::striker;
        return;
    }else
    {
      role = BehaviorTeamMessage::offensiveSupporter;
      return;
    }
  }
    else
    {
      role = BehaviorTeamMessage::defensiveSupporter;
      return;
    }
    */
  }

  // fall-back if no wlan
  if (SystemCall::getTimeSince(teamMessageCollection[0].lastReceivedTimeStamp) > 5000 
    && SystemCall::getTimeSince(teamMessageCollection[1].lastReceivedTimeStamp) > 5000
    && SystemCall::getTimeSince(teamMessageCollection[2].lastReceivedTimeStamp) > 5000)
  {
    switch(getPlayer().getPlayerNumber())
    {
    case Player::two:
      if (SystemCall::getTimeSince (ballPosition.seen.timeWhenLastSeen) < 5000 &&
        ballPosition.seen.x < 400)
        role = BehaviorTeamMessage::striker;
      else
        role = BehaviorTeamMessage::defensiveSupporter;
      break;
    case Player::three:
      if (SystemCall::getTimeSince (ballPosition.seen.timeWhenLastSeen) < 5000 &&
        ballPosition.seen.x > -500)
        role = BehaviorTeamMessage::striker;
      else
        role = BehaviorTeamMessage::offensiveSupporter;
      break;
    case Player::four:
    default:
      if (SystemCall::getTimeSince (ballPosition.seen.timeWhenLastSeen) < 5000 &&
        ballPosition.seen.x > -200)
        role = BehaviorTeamMessage::striker;
      else
        role = BehaviorTeamMessage::offensiveSupporter;
      break;
    }
    return;
  }
  
  // with wlan
  
  // estimate the closest other teammate to the ball
  double minTeammateTime = 100000.0;
  
  for (i = 0; i < teamMessageCollection.numberOfTeamMessages; i++ ) 
  {
    if ( teamMessageCollection[i].isActual() ) // only new messages
    {
      if ( teamMessageCollection[i].behaviorTeamMessage.dynamicRole != BehaviorTeamMessage::goalie 
        && teamMessageCollection[i].behaviorTeamMessage.gameState == BehaviorTeamMessage::playing)
        // the robot is not interested in 
        // (1) how fast the goalie can approach the ball 
        // (2) how fast a penalized field player can approach the ball
      {
        double time = 
          teamMessageCollection[i].behaviorTeamMessage.estimatedTimeToReachBall;
        
        // bonus for current striker
        if ( teamMessageCollection[i].behaviorTeamMessage.dynamicRole == BehaviorTeamMessage::striker ) 
          time -= 500.0;
        
        if (time < minTeammateTime ) 
        {
          minTeammateTime = time;
        }
      }
    }
  }
  
  // bonus for current striker
  if (role == BehaviorTeamMessage::striker)
    minTeammateTime += 1000.0;
  
  // assign striker-role if nearest to the ball
  if ( estimatedTimeToReachBall < minTeammateTime )
  {
    role = BehaviorTeamMessage::striker;
    return;
  }
  
  // assign supporting roles
  double maxTeammateX = (double)xPosOwnGroundline;
  for (i = 0; i < teamMessageCollection.numberOfTeamMessages; i++ ) 
  {
    if ( teamMessageCollection[i].isActual() )
    {
      if ( teamMessageCollection[i].behaviorTeamMessage.dynamicRole != BehaviorTeamMessage::striker 
        && teamMessageCollection[i].behaviorTeamMessage.dynamicRole != BehaviorTeamMessage::goalie 
        && teamMessageCollection[i].behaviorTeamMessage.gameState == BehaviorTeamMessage::playing)
      {
        double teammateX = 
          teamMessageCollection[i].robotPose.translation.x;
        
        // bonus for current offensive supporter
        if ( teamMessageCollection[i].behaviorTeamMessage.dynamicRole == BehaviorTeamMessage::offensiveSupporter ) 
          teammateX += 300.0;
        
        if (teammateX > maxTeammateX ) 
        {
          maxTeammateX = teammateX;
        }
      }
    }
  }
  
  // bonus for current offensive supporter
  if ( role == BehaviorTeamMessage::offensiveSupporter ) 
    maxTeammateX -= 300.0;
  
  if ( robotPose.translation.x >= maxTeammateX ) 
    role = BehaviorTeamMessage::offensiveSupporter;
  else 
    role = BehaviorTeamMessage::defensiveSupporter;
   
}

double ATH2004ERS7StrategySymbols::getGoalieMaxPositionSpeed()
{
  double maxSpeed = 150;
  double minDistance = 300; 

  double minSpeed = 60;
  double maxDistance = 1500;

  double ballDistance = Geometry::distanceTo(robotPose.getPose(),ballPosition.seen);

  double speed = maxSpeed - ((ballDistance - minDistance) * maxSpeed) / (maxDistance ) ;

  speed = max(minSpeed, speed);
  speed = min(maxSpeed, speed);

  return speed;
}

void ATH2004ERS7StrategySymbols::estimateTimeToReachBall()
{
  // stay striker if ball is caught
  if (getCaught())
  {
    estimatedTimeToReachBall = 0.0;
    return;
  }
  
  // account for distance to ball
  estimatedTimeToReachBall = Geometry::distanceTo(robotPose.translation, ballPosition.seen) / 0.2;
  
  // account if the robot is between the ball and the opponent goal
  
  // the position of the robot
  Vector2<double> robotPosition = robotPose.translation;
  double angleToLeftOpponentGoalPost = Geometry::angleTo(robotPosition, Vector2<double>(xPosOpponentGroundline,yPosLeftGoal));
  double angleToRightOpponentGoalPost = Geometry::angleTo(robotPosition, Vector2<double>(xPosOpponentGroundline,yPosRightGoal));
  if(angleToLeftOpponentGoalPost < angleToRightOpponentGoalPost)
  {
    angleToLeftOpponentGoalPost += pi2;
  }
  double angleToOpponentGoal= (angleToLeftOpponentGoalPost + angleToRightOpponentGoalPost) / 2.0;
  double absoluteAngleToBall = Geometry::angleTo(robotPosition,ballPosition.seen);
  double angleBetweenBallAndOpponentGoal = normalize(angleToOpponentGoal - absoluteAngleToBall);
  
  estimatedTimeToReachBall += 400.0 * fabs(angleBetweenBallAndOpponentGoal);
  
  // longer if ball not seen
  estimatedTimeToReachBall += 2.0 * SystemCall::getTimeSince(ballPosition.seen.timeWhenLastSeen);
  
  // test for obstacles
  /*
  if ( isOpponentBetweenRobotAndBall(robotPose) || 
  isOwnBetweenRobotAndBall (robotPose) )
  time += 5000.0;
  */
  
}

double ATH2004ERS7StrategySymbols::getAngleToTeammate()
{
  
  //OUTPUT(idText, text,"::nrofownplayer:"<<playerPoseCollection.numberOfOwnPlayers);
  
  
  if(1 <= playerPoseCollection.numberOfOwnPlayers)
  {
    return toDegrees(Geometry::angleTo(robotPose, playerPoseCollection.getOwnPlayerPose((int)angleToTeammateIndex).getPose().translation));
  }
  else
  {
    return toDegrees(Geometry::angleTo(robotPose,
      Vector2<double>(xPosOpponentGroundline,yPosCenterGoal)));
  }
}

bool ATH2004ERS7StrategySymbols::getAnotherPlayerIsInReadyState()
{
  for (int i=0; i<3;i++)
  {
    if (teamMessageCollection[i].isActual())
    {
      if (teamMessageCollection[i].behaviorTeamMessage.gameState 
        == BehaviorTeamMessage::ready)
      {
        return true;
      }
    }
  }
  return false;
}

bool ATH2004ERS7StrategySymbols::getAnotherPlayerIsInInitialState()
{
  for (int i=0; i<3;i++)
  {
    if (teamMessageCollection[i].isActual())
    {
      if (teamMessageCollection[i].behaviorTeamMessage.gameState 
        == BehaviorTeamMessage::initial)
      {
        return true;
      }
    }
  }
  return false;
}

bool ATH2004ERS7StrategySymbols::getAnotherTeammateIsPreparingAKick()
{
  for (int i=0; i<3;i++)
  {
    if (teamMessageCollection[i].isActual())
    {
      if (teamMessageCollection[i].behaviorTeamMessage.message
        == BehaviorTeamMessage::preparingAKick)
      {
        return true;
      }
    }
  }
  return false;
}

bool ATH2004ERS7StrategySymbols::getAnotherTeammateIsPerformingAKick()
{
  for (int i=0; i<3;i++)
  {
    if (teamMessageCollection[i].isActual())
    {
      if (teamMessageCollection[i].behaviorTeamMessage.message
        == BehaviorTeamMessage::performingAKick)
      {
        return true;
      }
    }
  }
  return false;
}

bool ATH2004ERS7StrategySymbols::getAnotherTeammateJustPerformedAKick()
{
  for (int i=0; i<3;i++)
  {
    if (teamMessageCollection[i].isActual())
    {
      if (teamMessageCollection[i].behaviorTeamMessage.message
        == BehaviorTeamMessage::justPerformedAKick)
      {
        return true;
      }
    }
  }
  return false;
}

bool ATH2004ERS7StrategySymbols::getTheStrikerIsPlayingNearTheOpponentGoal()
{
  for (int i=0; i<3;i++)
  {
    if (teamMessageCollection[i].isActual())
    {
      if (teamMessageCollection[i].behaviorTeamMessage.dynamicRole == BehaviorTeamMessage::striker
        && teamMessageCollection[i].robotPose.translation.x > 1500)
      {
        return true;
      }
    }
  }
  return false;
}

bool ATH2004ERS7StrategySymbols::getTheStrikerIsNotPlayingNearTheOpponentGoal()
{
  for (int i=0; i<3;i++)
  {
    if (teamMessageCollection[i].isActual())
    {
      if (teamMessageCollection[i].behaviorTeamMessage.dynamicRole == BehaviorTeamMessage::striker
        && teamMessageCollection[i].robotPose.translation.x < 1300)
      {
        return true;
      }
    }
  }
  return false;
}

bool ATH2004ERS7StrategySymbols::getTheStrikerIsPlayingNearTheOwnGoal()
{
  for (int i=0; i<3;i++)
  {
    if (teamMessageCollection[i].isActual())
    {
      if (teamMessageCollection[i].behaviorTeamMessage.dynamicRole == BehaviorTeamMessage::striker
        && teamMessageCollection[i].robotPose.translation.x < -1200)
      {
        return true;
      }
    }
  }
  return false;
}

bool ATH2004ERS7StrategySymbols::getTheStrikerIsNotPlayingNearTheOwnGoal()
{
  for (int i=0; i<3;i++)
  {
    if (teamMessageCollection[i].isActual())
    {
      if (teamMessageCollection[i].behaviorTeamMessage.dynamicRole == BehaviorTeamMessage::striker
        && teamMessageCollection[i].robotPose.translation.x > -1000)
      {
        return true;
      }
    }
  }
  return false;
}

bool ATH2004ERS7StrategySymbols::getCaught()
{
  return (SystemCall::getTimeSince(timeUntilBallWasCaught) < 500);  
}

double ATH2004ERS7StrategySymbols::getCatchTime()
{
  return (SystemCall::getTimeSince(timeUntilBallWasCaught) < 500?
    timeUntilBallWasCaught - timeWhenBallWasStartedToCatch : 0);
}

double ATH2004ERS7StrategySymbols::getSearchBallX()
{
  // fall-back if no wlan
  if (SystemCall::getTimeSince(teamMessageCollection[0].lastReceivedTimeStamp) > 5000 
    && SystemCall::getTimeSince(teamMessageCollection[1].lastReceivedTimeStamp) > 5000
    && SystemCall::getTimeSince(teamMessageCollection[2].lastReceivedTimeStamp) > 5000)
  {
    switch(getPlayer().getPlayerNumber())
    {
    case Player::two:
      return -1000;
    case Player::three:
      return 0;
    case Player::four:
    default:
      return 1000;
    }
  }
  else
  {
    switch (role)
    {
    case BehaviorTeamMessage::defensiveSupporter:
      return -1000;
    case BehaviorTeamMessage::offensiveSupporter:
      return 1000;
    case BehaviorTeamMessage::striker:
    default:
      return 0;
    }
  }
}



/*
* Change Log
* 
* $Log: ATH2004ERS7StrategySymbols.cpp,v $
* Revision 1.11  2004/04/22 11:34:02  spranger
* extended logging symbols
*
* Revision 1.10  2004/04/20 18:17:39  spranger
* added profiling-symbols and profiler-handling
*
* Revision 1.9  2004/04/08 10:08:41  risler
* fixed and unified team color symbols
*
* Revision 1.8  2004/04/05 17:56:46  loetzsch
* merged the local German Open CVS of the aibo team humboldt with the tamara CVS
*
* Revision 1.6  2004/04/03 02:18:21  loetzsch
* added "max-goalie-position-speed"
*
* Revision 1.5  2004/04/02 15:31:32  juengel
* computeRole changed
*
* Revision 1.4  2004/04/02 11:05:16  loetzsch
* changed "the-striker-is-playing-near-the-opponent-goal"
*
* Revision 1.3  2004/04/01 22:55:05  jumped
* changed some stuff regarding computerole near opponentgoal
*
* Revision 1.2  2004/04/01 13:15:15  loetzsch
* improved "catch-ball"
*
* Revision 1.1.1.1  2004/03/31 11:16:40  loetzsch
* created ATH repository for german open 2004
*
* Revision 1.7  2004/03/26 15:48:46  spranger
* added striker role calculation in goal zone
*
* Revision 1.6  2004/03/26 14:34:08  loetzsch
* no message
*
* Revision 1.5  2004/03/25 19:57:04  loetzsch
* several improvements
*
* Revision 1.4  2004/03/25 17:40:15  loetzsch
* adaptations to changes in the game controller and the led request
*
* Revision 1.3  2004/03/18 13:47:03  loetzsch
* removed the golie stuff
*
* Revision 1.2  2004/03/18 13:33:06  loetzsch
* removed warnings
*
* Revision 1.1  2004/03/16 14:00:18  juengel
* Integrated Improvments from "Gnne"
* -ATH2004ERS7Behavior
* -ATHHeadControl
* -KickSelectionTable
* -KickEditor
*
* Revision 1.1  2004/03/06 12:52:12  loetzsch
* cloned ATH2004BehaviorControl into ATH2004ERS7BehaviorControl
*
* Revision 1.1.1.1  2004/03/05 10:10:11  loetzsch
* created local cvs for G?*
* Revision 1.20  2004/03/04 17:25:00  sombrutz
* added symbol goalie-clear-ball-behavior
*
* Revision 1.19  2004/03/04 15:08:16  sombrutz
* added some comments
* added getDecisionGoalieGoToBall(bool lastDecision)
*
* Revision 1.18  2004/03/03 15:10:43  loetzsch
* added some comments in computeRole()
*
* Revision 1.17  2004/03/02 00:26:20  roefer
* Warnings removed
*
* Revision 1.16  2004/02/27 20:33:19  brueckne
* improved penalty-shooter-hertha
* chooses nearly everywhere the right shot
*
* Revision 1.15  2004/02/27 18:21:02  brueckne
* continued with penalty-shooter hertha
*
* Revision 1.14  2004/02/27 15:29:04  juengel
* Added getTeamColor().
*
* Revision 1.13  2004/02/27 10:43:28  spranger
* added get-Angle-to-Teammate function and parameter
*
* Revision 1.12  2004/02/26 17:17:29  loetzsch
* bug fixes and renaming of "shoot" to "shot"
*
* Revision 1.11  2004/02/26 14:41:47  brueckne
* Continued penality shooter hertha
*
* Revision 1.10  2004/02/15 23:32:55  sombrutz
* new stuff for goalie-go-to-ball
*
* Revision 1.9  2004/02/13 17:34:49  sombrutz
* add code for goalieGoToBall
*
* Revision 1.8  2004/02/13 11:32:31  sombrutz
* add GoalieGoToBall ( used by the new goalie)
*
* Revision 1.7  2004/01/22 12:43:13  dueffert
* flush removed because it doesnt work properly on robots
*
* Revision 1.6  2004/01/20 17:53:01  brueckne
* try to save datas in shoots.log
*
* Revision 1.5  2004/01/13 18:06:30  brueckne
* One step closer to getGoodShoot
*
* Revision 1.4  2004/01/08 17:32:38  brueckne
* Still working on getGoodShoot
*
* Revision 1.3  2003/12/11 17:08:17  brueckne
* Continued penalty-shooter-hertha
*
* Revision 1.2  2003/12/06 17:45:33  loetzsch
* replaced Player::playerRole (goalie, defender, striker1, striker2)
* by Player::playerNumber (one, two, three, four)
*
* Revision 1.1  2003/10/26 22:49:35  loetzsch
* created ATH2004ERS7BehaviorControl from GT2003BehaviorControl
*  - strongly simplified option graph
*  - moved some symbols from GT2003 to CommonXabsl2Symbols
*  - moved some basic behaviors from GT2003 to CommonXabsl2BasicBehaviors
*
* cloned ATH2004ERS7 three times (BB2004, DDD2004, MSH2004)
*
*/

