/**
* @file ATH2004ShootLogger.cpp
* 
* Implementation of class ATH2004ERS7ShootLogger.
*
* @author Martin Ltzsch
*/

#include "ATH2004ERS7ShootLogger.h"
#include "Platform/SystemCall.h"
#include "Platform/GTAssert.h"
#include "Tools/Math/Geometry.h"

ATH2004ERS7ShootLogger::ATH2004ERS7ShootLogger(BehaviorControlInterfaces& interfaces)
: timeOfLastStateChange(SystemCall::getCurrentSystemTime()),
BehaviorControlInterfaces(interfaces), state(initial)
{
}

ATH2004ERS7ShootLogger::~ATH2004ERS7ShootLogger()
{
}

void ATH2004ERS7ShootLogger::execute()
{
  switch (state)
  {
  case initial:
    if (motionRequest.motionType == MotionRequest::specialAction
      && SystemCall::getTimeSince(ballPosition.seen.timeWhenLastSeen) < 500
      && Geometry::distanceTo(robotPose.getPose(),ballPosition.seen) < 350)
    {
      ballXR = getBallX(robotPose);
      ballYR = getBallY(robotPose);
      ballTR = SystemCall::getTimeSince(ballPosition.seen.timeWhenLastSeen);
      kick = motionRequest.specialActionType;
      changeState(kickRequested);
    }
    break;

  case kickRequested:
    if (executedMotionRequest.motionType == MotionRequest::specialAction)
    {
      ballXS = getBallX(robotPose);
      ballYS = getBallY(robotPose);
      ballTS = SystemCall::getTimeSince(ballPosition.seen.timeWhenLastSeen);
      odometryDataWhenKickWasStarted = odometryData;
      changeState(kickStarted);
    }
    break;

  case kickStarted:
    if (executedMotionRequest.motionType != MotionRequest::specialAction)
    {
      changeState(kickFinished);
    }
    break;

  case kickFinished:
    if (executedMotionRequest.motionType == MotionRequest::specialAction)
    {
      changeState(wait);
    }
    if (SystemCall::getTimeSince(timeOfLastStateChange) > 2000)
    {
      changeState(waitingForBall);
    }
    break;

  case waitingForBall:
    if (executedMotionRequest.motionType == MotionRequest::specialAction)
    {
      changeState(wait);
    }
    if (SystemCall::getTimeSince(ballPosition.seen.timeWhenLastSeen) < 100
      && Geometry::distanceTo(robotPose.getPose(),ballPosition.seen) > 700)
    {
      Pose2D odometryDiff = odometryData - odometryDataWhenKickWasStarted;
      Pose2D relativePoseWhenStarted = robotPose - odometryDiff;

      ballXA = getBallX(relativePoseWhenStarted);
      ballYA = getBallY(relativePoseWhenStarted);
      tA = SystemCall::getTimeSince(timeOfLastStateChange) + 2000;
      record();
      changeState(wait);
    }
    if (SystemCall::getTimeSince(timeOfLastStateChange) > 4000)
    {
      changeState(wait);
    }
    break;

  case wait:
    if (SystemCall::getTimeSince(timeOfLastStateChange) > 3000
      && (executedMotionRequest.motionType == MotionRequest::walk
      || executedMotionRequest.motionType == MotionRequest::stand))
    {
      changeState(initial);
    }
    break;

  default:
    ASSERT(false);
  }
}

double ATH2004ERS7ShootLogger::getBallX(Pose2D from)
{
  return Geometry::distanceTo(from,ballPosition.seen) 
    * cos(Geometry::angleTo(from,ballPosition.seen));
}

double ATH2004ERS7ShootLogger::getBallY(Pose2D from)
{
  return Geometry::distanceTo(from,ballPosition.seen) 
    * sin(Geometry::angleTo(from,ballPosition.seen));
}

void ATH2004ERS7ShootLogger::changeState(State newState)
{
  timeOfLastStateChange = SystemCall::getCurrentSystemTime();
  state = newState;
//  OUTPUT(idText,text,"ATH2004ERS7ShootLogger: changed state to " << state << ".");
}

void ATH2004ERS7ShootLogger::record()
{
  if (getDebugKeyTable().isActive(DebugKeyTable::sendShootRecords))
  {
    writeRecordToStream(getDebugOut().textRaw);
    getDebugOut().finishMessage(idText);
  }
  else
  {
    OutTextRawFile f("kicks.log",true);
    writeRecordToStream(f);
  }
}

void ATH2004ERS7ShootLogger::writeRecordToStream(Out& out)
{
  out << "<record";

  out << " kick=\"" << MotionRequest::getSpecialActionName(kick) << "\"";  
  out << " ballXR=\"" << ballXR << "\"";  
  out << " ballYR=\"" << ballYR << "\"";  
  out << " ballTR=\"" << ballTR << "\"";  
  out << " ballXS=\"" << ballXS << "\"";  
  out << " ballYS=\"" << ballYS << "\"";  
  out << " ballTS=\"" << ballTS << "\"";  
  out << " ballXA=\"" << ballXA << "\"";  
  out << " ballYA=\"" << ballYA << "\"";  
  out << " tA=\"" << tA << "\"";  

  out << "/>" << endl;
}

/*
 * Change log :
 * 
 * $Log: ATH2004ERS7ShootLogger.cpp,v $
 * Revision 1.3  2004/03/20 00:20:18  kerdels
 * removed some errors
 *
 * Revision 1.1  2004/03/16 14:00:17  juengel
 * Integrated Improvments from "Gnne"
 * -ATH2004ERS7Behavior
 * -ATHHeadControl
 * -KickSelectionTable
 * -KickEditor
 *
 * Revision 1.1  2004/03/06 12:52:11  loetzsch
 * cloned ATH2004BehaviorControl into ATH2004ERS7BehaviorControl
 *
 * Revision 1.1.1.1  2004/03/05 10:10:11  loetzsch
 * created local cvs for Gnne
 *
 * Revision 1.6  2004/03/04 18:30:53  loetzsch
 * bug fix
 *
 * Revision 1.5  2004/02/09 16:45:04  loetzsch
 * bug fix (missing break in switch statement)
 *
 * Revision 1.4  2004/02/04 14:09:56  loetzsch
 * improved
 *
 * Revision 1.3  2004/02/03 23:16:46  loetzsch
 * added debug key sendShootRecords
 *
 * Revision 1.2  2004/01/30 13:25:28  loetzsch
 * using odometry data in shoot logger,
 * first draft of visualization
 *
 * Revision 1.1  2004/01/28 21:16:56  loetzsch
 * added ATH2004ShootLogger
 *
 */
