/** 
* @file ATH2004ERS7SimpleBasicBehaviors.h
*
* Declaration of basic behaviors defined in simple-basic-behaviors.xml.
*
* @author Uwe Dffert
* @author Jan Hoffmann
* @author Matthias Jngel
* @author Martin Ltzsch
* @author Max Risler
*/

#ifndef __ATH2004ERS7SimpleBasicBehaviors_h_
#define __ATH2004ERS7SimpleBasicBehaviors_h_

#include "../../BehaviorControl.h"
#include "Tools/Xabsl2/Xabsl2Engine/Xabsl2Engine.h"
#include "Tools/Xabsl2/Xabsl2Engine/Xabsl2BasicBehavior.h"
#include "Tools/Actorics/WalkAccelerationRestrictor.h"
#include "Tools/Evolution/Population.h"
#include "Tools/Math/PIDsmoothedValue.h"
#include "Modules/WalkingEngine/InvKinWalkingEngine.h"
#include "Modules/WalkingEngine/InvKinWalkingParameterSets.h"



/**
* A Basic Behavior for going to the ball
* @author Matthias Jngel
* @author Martin Ltzsch
*/
class ATH2004ERS7BasicBehaviorGoToBall : public Xabsl2BasicBehavior, public BehaviorControlInterfaces
{
public:
/*
* Constructor.
* @param errorHandler Is invoked when errors occur
* @param interfaces The paramters of the BehaviorControl module.
  */
  ATH2004ERS7BasicBehaviorGoToBall(BehaviorControlInterfaces& interfaces,
    Xabsl2ErrorHandler& errorHandler)
    : Xabsl2BasicBehavior("go-to-ball", errorHandler),
    BehaviorControlInterfaces(interfaces),
    accelerationRestrictor(motionRequest)
  {
    registerParameter("go-to-ball.distance",distanceAtEnd);
    registerParameter("go-to-ball.max-speed", maxSpeed);
    registerParameter("go-to-ball.target-angle-to-ball", targetAngleToBall);
  }
  
  /**  Executes the basic behavior. */
  virtual void execute();
  
private:
  /** parameter "go-to-ball.distance" */
  double distanceAtEnd;
  
  /** parameter "go-to-ball.max-speed" */
  double maxSpeed;
  
  /** Restricts the walk acceleration to maximum values */
  WalkAccelerationRestrictor accelerationRestrictor;

  /** parameter "go-to-ball.target-angle-to-ball" */
  double targetAngleToBall;
};

/**
* A Basic Behavior for going to the ball
* @author Jan Hoffmann
*/
class ATH2004ERS7BasicBehaviorGoToBallOld : public Xabsl2BasicBehavior, public BehaviorControlInterfaces
{
public:
/*
* Constructor.
* @param errorHandler Is invoked when errors occur
* @param interfaces The paramters of the BehaviorControl module.
  */
  ATH2004ERS7BasicBehaviorGoToBallOld(BehaviorControlInterfaces& interfaces,
    Xabsl2ErrorHandler& errorHandler)
    : Xabsl2BasicBehavior("go-to-ball-old", errorHandler),
    BehaviorControlInterfaces(interfaces),
    accelerationRestrictor(motionRequest)
  {
    registerParameter("go-to-ball-old.distance",distanceAtEnd);
    registerParameter("go-to-ball-old.max-speed", maxSpeed);
  }
  
  /**  Executes the basic behavior. */
  virtual void execute();
  
private:
  /** parameter "go-to-ball-old.distance" */
  double distanceAtEnd;
  
  /** parameter "go-to-ball-old.max-speed" */
  double maxSpeed;
  
  /** Restricts the walk acceleration to maximum values */
  WalkAccelerationRestrictor accelerationRestrictor;
};

/**
* A Basic Behavior for going to a point on the field
* @author Uwe Dffert
* @author Martin Ltzsch
*/
class ATH2004ERS7BasicBehaviorGoToPoint : public Xabsl2BasicBehavior, public BehaviorControlInterfaces
{
public:
/*
* Constructor.
* @param errorHandler Is invoked when errors occur
* @param interfaces The paramters of the BehaviorControl module.
  */
  ATH2004ERS7BasicBehaviorGoToPoint(BehaviorControlInterfaces& interfaces,
    Xabsl2ErrorHandler& errorHandler)
    : Xabsl2BasicBehavior("go-to-point", errorHandler),
    BehaviorControlInterfaces(interfaces),
    accelerationRestrictor(motionRequest)
  {
    registerParameter("go-to-point.x", x);
    registerParameter("go-to-point.y", y);
    registerParameter("go-to-point.destination-angle", destinationAngle);
    registerParameter("go-to-point.max-speed", maxSpeed);
  }
  
  /**  Executes the basic behavior. */
  virtual void execute();
  
protected:
  /** parameter "go-to-point.x" */
  double x;
  
  /** parameter "go-to-point.y" */
  double y;
  
  /** parameter "go-to-point.destination-angle" */
  double destinationAngle;
  
  /** parameter "go-to-point.max-speed" */
  double maxSpeed;
  
  /** Restricts the walk acceleration to maximum values */
  WalkAccelerationRestrictor accelerationRestrictor;
};

/**
* A Basic Behavior for going turning around a point on the field
* @author Jan Hoffmann
*/
class ATH2004ERS7BasicBehaviorTurnAroundPoint : public Xabsl2BasicBehavior, public BehaviorControlInterfaces
{
public:
/*
* Constructor.
* @param errorHandler Is invoked when errors occur
* @param interfaces The paramters of the BehaviorControl module.
  */
  ATH2004ERS7BasicBehaviorTurnAroundPoint(BehaviorControlInterfaces& interfaces,
    Xabsl2ErrorHandler& errorHandler)
    : Xabsl2BasicBehavior("turn-around-point", errorHandler),
    BehaviorControlInterfaces(interfaces),
    accelerationRestrictor(motionRequest)
  {
    registerParameter("turn-around-point.x", x);
    registerParameter("turn-around-point.y", y);
    registerParameter("turn-around-point.radius", radius);
    registerParameter("turn-around-point.left-right", leftRight);
    registerParameter("turn-around-point.forward-component", forwardComponent);
  }
  
  /**  Executes the basic behavior. */
  virtual void execute();
  
protected:
  /** parameter "turn-around-point.x" */
  double x;
  
  /** parameter "turn-around-point.y" */
  double y;
  
  /** parameter "turn-around-point.radius" */
  double radius;
  
  /** parameter "go-to-point.left-right" */
  double leftRight;
  
  /** parameter "go-to-point.forward-component" */
  double forwardComponent;

  /** Restricts the walk acceleration to maximum values */
  WalkAccelerationRestrictor accelerationRestrictor;
};

/**
* A Basic Behavior for goalie positioning
* @author Jan Hoffmann
*/
class ATH2004ERS7BasicBehaviorGoaliePosition : public Xabsl2BasicBehavior, public BehaviorControlInterfaces
{
public:
/*
* Constructor.
* @param errorHandler Is invoked when errors occur
* @param interfaces The paramters of the BehaviorControl module.
  */
  ATH2004ERS7BasicBehaviorGoaliePosition(BehaviorControlInterfaces& interfaces,
    Xabsl2ErrorHandler& errorHandler)
    : Xabsl2BasicBehavior("goalie-position", errorHandler),
    BehaviorControlInterfaces(interfaces)
  {
    registerParameter("goalie-position.max-speed", maxSpeed);
  }
  
  /**  Executes the basic behavior. */
  virtual void execute();
  
protected:

  double maxSpeed;

};



/**
* A Basic Behavior for going to a point on the field facing forward
* @author Uwe Dffert
* @author Martin Ltzsch
*/
class ATH2004ERS7BasicBehaviorGoForwardToPoint : public Xabsl2BasicBehavior, public BehaviorControlInterfaces
{
public:
/*
* Constructor.
* @param errorHandler Is invoked when errors occur
* @param interfaces The paramters of the BehaviorControl module.
  */
  ATH2004ERS7BasicBehaviorGoForwardToPoint(BehaviorControlInterfaces& interfaces,
    Xabsl2ErrorHandler& errorHandler)
    : Xabsl2BasicBehavior("go-forward-to-point", errorHandler),
    BehaviorControlInterfaces(interfaces),
    accelerationRestrictor(motionRequest)
  {
    registerParameter("go-forward-to-point.x", x);
    registerParameter("go-forward-to-point.y", y);
    registerParameter("go-forward-to-point.max-speed", maxSpeed);
  }
  
  /**  Executes the basic behavior. */
  virtual void execute();
  
protected:
  /** parameter "go-forward-to-point.x" */
  double x;
  
  /** parameter "go-forward-to-point.y" */
  double y;
  
  /** parameter "go-forward-to-point.max-speed" */
  double maxSpeed;
  
  /** Restricts the walk acceleration to maximum values */
  WalkAccelerationRestrictor accelerationRestrictor;
};

/**
* A Basic Behavior for going to a point on the field and avoiding obstacles on the way there.
* @author Jan Hoffmann
*/
class ATH2004ERS7BasicBehaviorGoToPointAndAvoidObstacles : public Xabsl2BasicBehavior, public BehaviorControlInterfaces
{
public:
/*
* Constructor.
* @param errorHandler Is invoked when errors occur
* @param interfaces The paramters of the BehaviorControl module.
  */
  ATH2004ERS7BasicBehaviorGoToPointAndAvoidObstacles(BehaviorControlInterfaces& interfaces,
    Xabsl2ErrorHandler& errorHandler)
    : Xabsl2BasicBehavior("go-to-point-and-avoid-obstacles", errorHandler),
    BehaviorControlInterfaces(interfaces),
	  turnSpeed(0, .5, 0, 0, -pi, pi, pi/2),
	  speedX(0, .4, 0, 0, -80, 350, 100),
	  speedY(0, .4, 0, 0, -100, 100, 100)
  {
    registerParameter("go-to-point-and-avoid-obstacles.x", x);
    registerParameter("go-to-point-and-avoid-obstacles.y", y);
    registerParameter("go-to-point-and-avoid-obstacles.max-speed", maxSpeed);
    registerParameter("go-to-point-and-avoid-obstacles.avoidance-level", avoidanceLevel);
  }
  
  /**  Executes the basic behavior. */
  virtual void execute();
  
private:
  /** parameter "go-to-point-and-avoid-obstacles.x" */
  double x;
  
  /** parameter "go-to-point-and-avoid-obstacles.y" */
  double y;
  
  /** parameter "go-to-point-and-avoid-obstacles.max-speed" */
  double maxSpeed;
   
  /** parameter "go-to-point-and-avoid-obstacles.avoid-penalty-areas" */
  double avoidanceLevel;

  /** smoothes the rotation */  
  PIDsmoothedValue turnSpeed;

  /** smoothes the speed in forward direction */  
  PIDsmoothedValue speedX;

  /** smoothes the speed in sideways direction */  
  PIDsmoothedValue speedY;

  /** last time behavior was executed */
  long lastTimeExecuted;
};

/**
* A Basic Behavior for going to the ball
*/
class ATH2004ERS7BasicBehaviorDribbleAtBorder : public Xabsl2BasicBehavior, public BehaviorControlInterfaces
{
public:
/*
* Constructor.
* @param errorHandler Is invoked when errors occur
* @param interfaces The paramters of the BehaviorControl module.
  */
  ATH2004ERS7BasicBehaviorDribbleAtBorder(const BehaviorControlInterfaces& interfaces,
    Xabsl2ErrorHandler& errorHandler)
    : Xabsl2BasicBehavior("dribble-at-border", errorHandler),
    BehaviorControlInterfaces(interfaces)
  {
    registerParameter("dribble-at-border.max-speed", maxSpeed);
  }
  
  /**  Executes the basic behavior. */
  virtual void execute();
  
private:
  /** parameter "go-to-ball.distance" */
  double maxSpeed;
  
};

/**
* A Basic Behavior for dribbling in the center of the field
*/
class ATH2004ERS7BasicBehaviorDribbleInCenterOfField : public Xabsl2BasicBehavior, public BehaviorControlInterfaces
{
public:
/*
* Constructor.
* @param errorHandler Is invoked when errors occur
* @param interfaces The paramters of the BehaviorControl module.
  */
  ATH2004ERS7BasicBehaviorDribbleInCenterOfField(const BehaviorControlInterfaces& interfaces,
    Xabsl2ErrorHandler& errorHandler)
    : Xabsl2BasicBehavior("dribble-in-center-of-field", errorHandler),
    BehaviorControlInterfaces(interfaces)
  {
    registerParameter("dribble-in-center-of-field.speed", speed);
	registerParameter("dribble-in-center-of-field.modi", modi);
	direction = 361;
  }
  
  /**  Executes the basic behavior. */
  virtual void execute();
  
private:
  /** parameter "dribble-in-center-of-field.speed" */
  double speed;
  /** parameter "dribble-in-center-of-field.modi" */
  double modi;
  // dribble direction (angle)
  double direction;
  // dribble side
  int side;
  // rotation
  double rotationM;
  
};


/**
* @class SimpleBasicBehaviors
*
* Creates and registers simple basic behaviors
*/
class ATH2004ERS7SimpleBasicBehaviors : public BehaviorControlInterfaces
{
public:
  /**
  * Constructor
  */
  ATH2004ERS7SimpleBasicBehaviors(BehaviorControlInterfaces& interfaces,
                       Xabsl2ErrorHandler& errorHandler)
    : BehaviorControlInterfaces(interfaces),
      errorHandler(errorHandler),
      basicBehaviorGoToBall(interfaces,errorHandler),
      basicBehaviorGoToBallOld(interfaces,errorHandler),
      basicBehaviorGoToPoint(interfaces,errorHandler),
      basicBehaviorGoForwardToPoint(interfaces,errorHandler),
      basicBehaviorTurnAroundPoint(interfaces,errorHandler),
      basicBehaviorGoToPointAndAvoidObstacles(interfaces,errorHandler),
      basicBehaviorDribbleInCenterOfField(interfaces,errorHandler),
      basicBehaviorDribbleAtBorder(interfaces,errorHandler),
      basicBehaviorGoaliePosition(interfaces,errorHandler)
  {}

  /** Registers basic behaviors at the engine */
  void registerBasicBehaviors(Xabsl2Engine& engine)
  {
    engine.registerBasicBehavior(basicBehaviorGoToBall);
    engine.registerBasicBehavior(basicBehaviorGoToBallOld);
    engine.registerBasicBehavior(basicBehaviorGoToPoint);
    engine.registerBasicBehavior(basicBehaviorGoForwardToPoint);
    engine.registerBasicBehavior(basicBehaviorGoToPointAndAvoidObstacles);
    engine.registerBasicBehavior(basicBehaviorTurnAroundPoint);
    engine.registerBasicBehavior(basicBehaviorDribbleInCenterOfField);
    engine.registerBasicBehavior(basicBehaviorDribbleAtBorder);
    engine.registerBasicBehavior(basicBehaviorGoaliePosition);
  }

private:
  /** Is invoked when errors occurs */
  Xabsl2ErrorHandler& errorHandler;

  //!@name Basic Behaviors
  //!@{
  ATH2004ERS7BasicBehaviorGoToBallOld  basicBehaviorGoToBallOld;
  ATH2004ERS7BasicBehaviorGoToBall  basicBehaviorGoToBall;
  ATH2004ERS7BasicBehaviorGoToPoint  basicBehaviorGoToPoint;  
  ATH2004ERS7BasicBehaviorGoForwardToPoint  basicBehaviorGoForwardToPoint;  
  ATH2004ERS7BasicBehaviorTurnAroundPoint basicBehaviorTurnAroundPoint;  
  ATH2004ERS7BasicBehaviorGoToPointAndAvoidObstacles  basicBehaviorGoToPointAndAvoidObstacles;
  ATH2004ERS7BasicBehaviorDribbleInCenterOfField  basicBehaviorDribbleInCenterOfField;
  ATH2004ERS7BasicBehaviorDribbleAtBorder  basicBehaviorDribbleAtBorder;
  ATH2004ERS7BasicBehaviorGoaliePosition  basicBehaviorGoaliePosition;

  //!@}
};

#endif // __ATH2004ERS7SimpleBasicBehaviors_h_

/*
* Change Log
* 
* $Log: ATH2004ERS7SimpleBasicBehaviors.h,v $
* Revision 1.6  2004/04/05 17:56:46  loetzsch
* merged the local German Open CVS of the aibo team humboldt with the tamara CVS
*
* Revision 1.3  2004/04/03 19:41:40  hoffmann
* added parameter to turn to point
*
* Revision 1.2  2004/04/02 23:12:15  hoffmann
* added goalie-position basic behavior
*
* Revision 1.1.1.1  2004/03/31 11:16:39  loetzsch
* created ATH repository for german open 2004
*
* Revision 1.5  2004/03/29 13:15:19  jhoffman
* - new go-to-ball
* - old go-to-ball renamed to go-to-ball-old
* - new turn around point
*
* Revision 1.4  2004/03/28 14:08:05  jhoffman
* added go-to-ball-test
*
* Revision 1.3  2004/03/23 14:41:56  juengel
* Added dribble behaviors.
*
* Revision 1.2  2004/03/19 19:01:06  jhoffman
* added obstacle avoidance parameter "avoidance level" (0 - min, 1 - mid, 2 - max avoidance)
*
* Revision 1.1  2004/03/16 14:00:18  juengel
* Integrated Improvments from "Gnne"
* -ATH2004ERS7Behavior
* -ATHHeadControl
* -KickSelectionTable
* -KickEditor
*
* Revision 1.4  2004/03/15 17:11:39  hoffmann
* - added ATH2004HeadControl
* - added ATH2004LEDControl
* - headmotiontester shows "tilt2"
* - motion process updates odometry while no new robotPose is received, added to motion request
* - some ui adjustments
* - added member function to "field" to find out if robot is in own penalty area for use in the obstacles locator
*
* Revision 1.3  2004/03/11 17:33:36  loetzsch
* added basic behavior "turn-around-point"
*
* Revision 1.2  2004/03/08 19:01:32  loetzsch
* - removed option "simple-get-to-position-and-avoid-obstacles"
* - added parameter "avoid-penalty-areas" to basic behavior "get-to-point-and-avoid-obstacles"
*   and option "get-to-position-and-avoid-obstacles"
* - added basic behavior "go-forward-to-point"
*
* Revision 1.1  2004/03/06 12:52:11  loetzsch
* cloned ATH2004BehaviorControl into ATH2004ERS7BehaviorControl
*
* Revision 1.1.1.1  2004/03/05 10:10:11  loetzsch
* created local cvs for Gnne
*
* Revision 1.9  2004/03/02 10:10:11  mellmann
* 3 dribble-modes in DribbleInCenterOfField
*
* Revision 1.8  2004/01/08 13:51:48  loetzsch
* added go-to-point-fast
*
* Revision 1.7  2003/12/11 17:10:22  mellmann
* new Variable "side" in dribbleInCenterOfField
*
* Revision 1.6  2003/12/04 17:22:54  mellmann
* Added "GetBehindBall2" and "DribbleInCenterOfField".
*
* Revision 1.5  2003/11/25 19:53:36  heinze
* added basic behavior dribble-at-border
*
* Revision 1.4  2003/10/28 14:04:50  dueffert
* unused evolution removed
*
* Revision 1.3  2003/10/28 13:57:45  dueffert
* unused evolution removed
*
* Revision 1.2  2003/10/28 13:25:52  dueffert
* spelling improved
*
* Revision 1.1  2003/10/26 22:49:34  loetzsch
* created ATH2004ERS7BehaviorControl from GT2003BehaviorControl
*  - strongly simplified option graph
*  - moved some symbols from GT2003 to CommonXabsl2Symbols
*  - moved some basic behaviors from GT2003 to CommonXabsl2BasicBehaviors
*
* cloned ATH2004ERS7 three times (BB2004, DDD2004, MSH2004)
*
*/

