/**
* @file ATH2004ERS7ContinuousRules/TurnToPoint.h
*
* @author Max Risler
*/

#ifndef __ATH2004ERS7TurnToPoint_h_
#define __ATH2004ERS7TurnToPoint_h_

#include "Tools/ContinuousBasicBehaviors/ContinuousRule.h"

namespace ATH2004ERS7ContinuousRules
{

/**@class TurnToPoint
 */
class TurnToPoint: public ContinuousRule{
private:
  /** reference to the point to turn to */
  const double &pointX;
  const double &pointY;

  /** the point is in relative coordinates */
  bool relative;
  
public:

  /** constructor
   */
  TurnToPoint(
    BehaviorControlInterfaces& interfaces,
    const Vector2<double>& point,
    bool relative = false)
    :
    ContinuousRule(interfaces,"TurnToPoint"),
    pointX(point.x),
    pointY(point.y),
    relative(relative)
  {};

  /** constructor
   */
  TurnToPoint(
    BehaviorControlInterfaces& interfaces,
    const double& pointX,
    const double& pointY,
    bool relative = false)
    :
    ContinuousRule(interfaces, "TurnToPoint"),
    pointX(pointX),
    pointY(pointY),
    relative(relative)
  {};

  /**
   * Returns whether this rule generates absolute or robot relative coordinates.
   */
  virtual bool isRelative() {return relative;}

 	/** executes the basic behavior rule
   * @param robotPose the current robots pose at which the rule is to be evaluated
   * @param walk the direction and speed of the suggested motion coded as an vector (output)
	 * @param ra the rotation angle, the direction the robot should be directed (output)
	 * @param rweight the rotation weight the weight of the suggested rotation (output)
	 */
  virtual void execute(const RobotPose& robotPose,
                       Vector2<double>& walk,
                       double& ra, double& rweight)
  {

    walk.x = walk.y = 0;

    Vector2<double> destination(pointX, pointY);

    if (relative)
    {
      Vector2<double> destination(pointX, pointY);
      double distance = destination.abs();
      double factor = (distance - 100) / 500;
      if (factor > 1) factor = 1;
      if (factor < 0) factor = 0;
      factor *= 3;
      factor += 1;
      //factor = 1;
      ra = factor*destination.angle();
      if (ra > 3) ra = 3;
      else if (ra < -3) ra = -3;
    }
    else
    {
      Vector2<double> destination(pointX, pointY);
      double distance = Geometry::distanceTo(robotPose, destination);
      double factor = (distance - 100) / 500;
      if (factor > 1) factor = 1;
      if (factor < 0) factor = 0;
      factor *= 3;
      factor += 1;
      //factor = 1;
      ra = factor*destination.angle();
      if (ra > 3) ra = 3;
      else if (ra < -3) ra = -3;
    }

    rweight=1.0;
  }
};

}


#endif //__ATH2004ERS7TurnToBall_h_


/*
* Change log:
*
* $Log: TurnToPoint.h,v $
* Revision 1.2  2004/04/05 17:56:46  loetzsch
* merged the local German Open CVS of the aibo team humboldt with the tamara CVS
*
* Revision 1.2  2004/04/02 15:30:33  hoffmann
* speed up
*
* Revision 1.1  2004/03/16 14:00:18  juengel
* Integrated Improvments from "Gnne"
* -ATH2004ERS7Behavior
* -ATHHeadControl
* -KickSelectionTable
* -KickEditor
*
* Revision 1.1  2004/03/06 12:52:12  loetzsch
* cloned ATH2004BehaviorControl into ATH2004ERS7BehaviorControl
*
* Revision 1.1.1.1  2004/03/05 10:10:11  loetzsch
* created local cvs for Gnne
*
* Revision 1.2  2003/10/31 08:32:48  dueffert
* doxygen bugs fixed
*
* Revision 1.1  2003/10/26 22:49:35  loetzsch
* created ATH2004ERS7BehaviorControl from GT2003BehaviorControl
*  - strongly simplified option graph
*  - moved some symbols from GT2003 to CommonXabsl2Symbols
*  - moved some basic behaviors from GT2003 to CommonXabsl2BasicBehaviors
*
* cloned ATH2004ERS7 three times (BB2004, DDD2004, MSH2004)
*
*/
