/**
* @file ATH2004ERS7ContinuousRules/GoBetweenBallAndOwnGoal.h
*
* @author Dirk Thomas
*/

#ifndef __ATH2004ERS7GoBetweenBallAndOwnGoal_h_
#define __ATH2004ERS7GoBetweenBallAndOwnGoal_h_

#include "Tools/ContinuousBasicBehaviors/ContinuousRule.h"
#include "Tools/FieldDimensions.h"

namespace ATH2004ERS7ContinuousRules
{

/**@class GoBetweenBallAndOwnGoal
 */
class GoBetweenBallAndOwnGoal: public ContinuousRule{
private:
	/** the distance above which the result vector is maxForce in mm*/
  double maxForceLength;
  /** the distance below which the result vector is 0 in mm*/
  double minDistance;
  /** the maximum attracting force */
  double maxForce;

  /** the x distance from the goal */
  double xDistanceToGoal;
  /** the y distance from the goal */
  double yDistanceToGoal;

public:

  /** constructor
   */
  GoBetweenBallAndOwnGoal(
    BehaviorControlInterfaces& interfaces,
    double maxForce = 1.0,
    double minDistance = 0.0,
    double maxForceLength = 350.0,
    double xDistanceToGoal = 100.0,
    double yDistanceToGoal = 400.0)
    :
    ContinuousRule(interfaces,"GoBetweenBallAndOwnGoal"),
    maxForce(maxForce),
    minDistance(minDistance),
    maxForceLength(maxForceLength),
    xDistanceToGoal(xDistanceToGoal),
    yDistanceToGoal(yDistanceToGoal)
  {};

	/** executes the basic behavior rule
   * @param robotPose the current robots pose at which the rule is to be evaluated
   * @param walk the direction and speed of the suggested motion coded as an vector (output)
	 * @param ra the rotation angle, the direction the robot should be directed (output)
	 * @param rweight the rotation weight the weight of the suggested rotation (output)
	 */
  virtual void execute(const RobotPose& robotPose,
                       Vector2<double>& walk,
                       double& ra, double& rweight)
  {
    
    walk = -robotPose.translation;
    walk.x += xPosOwnGroundline + 50;

    Vector2<double> ballToGoal(xPosOwnGroundline+50, yPosCenterGoal);
    ballToGoal -= ballPosition.getKnownPosition(BallModel::behaviorControlTimeAfterWhichCommunicatedBallsAreAccepted);
    ballToGoal.normalize();

    // position on a half circle in front of the own goal
    walk.x -= ballToGoal.x * xDistanceToGoal;
    walk.y -= ballToGoal.y * yDistanceToGoal;

    double distance = walk.abs();
    walk.normalize();

    walk *= getAttractingForce(distance, maxForce, minDistance, maxForceLength);

    ra = 0;
    rweight = 0;

  }
};

}


#endif //__ATH2004ERS7GoBetweenBallAndOwnGoal_h_


/*
* Change log:
*
* $Log: GoBetweenBallAndOwnGoal.h,v $
* Revision 1.2  2004/05/04 10:48:58  loetzsch
* replaced all enums
* xxxBehaviorControlTimeAfterWhichCommunicatedBallsAreAccepted
* by
* behaviorControlTimeAfterWhichCommunicatedBallsAreAccepted
* (this mechanism was neither fully implemented nor used)
*
* Revision 1.1  2004/03/16 14:00:18  juengel
* Integrated Improvments from "Gnne"
* -ATH2004ERS7Behavior
* -ATHHeadControl
* -KickSelectionTable
* -KickEditor
*
* Revision 1.1  2004/03/06 12:52:12  loetzsch
* cloned ATH2004BehaviorControl into ATH2004ERS7BehaviorControl
*
* Revision 1.1.1.1  2004/03/05 10:10:11  loetzsch
* created local cvs for Gnne
*
* Revision 1.3  2004/02/03 13:19:49  spranger
* renamed all references to  class BallPosition to BallModel
*
* Revision 1.2  2003/10/28 15:04:09  loetzsch
* fixed doxygen bugs
*
* Revision 1.1  2003/10/26 22:49:34  loetzsch
* created ATH2004ERS7BehaviorControl from GT2003BehaviorControl
*  - strongly simplified option graph
*  - moved some symbols from GT2003 to CommonXabsl2Symbols
*  - moved some basic behaviors from GT2003 to CommonXabsl2BasicBehaviors
*
* cloned ATH2004ERS7 three times (BB2004, DDD2004, MSH2004)
*
*/
