/**
* @file ATH2004ERS7ContinuousRules/AvoidPoint.h
*
* @author Max Risler
*/

#ifndef __ATH2004ERS7AvoidPoint_h_
#define __ATH2004ERS7AvoidPoint_h_

#include "Tools/ContinuousBasicBehaviors/ContinuousRule.h"
#include "Tools/FieldDimensions.h"

namespace ATH2004ERS7ContinuousRules
{

/**@class AvoidPoint
 */
class AvoidPoint: public ContinuousRule{
private:
  /** reference to the point to go to */
  const double &pointX;
  const double &pointY;

  /** if the rule not constructed via a reference to the point
   * these variables are used to store the coordinates */
  double localPointX;
  double localPointY;

  /** the point is in relative coordinates */
  bool relative;

	/** the distance below which the result vector is maxForce in mm*/
  double maxForceLength;
  /** the distance above which the result vector is 0 in mm*/
  double influenceLength;
  /** the maximum repulsive force */
  double maxForce;


public:

  /** constructor
   */
  AvoidPoint(
    BehaviorControlInterfaces& interfaces,
    const Vector2<double>& point,
    bool relative = false,
    double maxForce = 3.0,
    double maxForceLength = 150.0,
    double influenceLength = 250.0)
    :
    ContinuousRule(interfaces,"AvoidPoint"),
    pointX(point.x),
    pointY(point.y),
    relative(relative),
    maxForce(maxForce),
    maxForceLength(maxForceLength),
    influenceLength(influenceLength)
  {};

  /** constructor
   */
  AvoidPoint(
    BehaviorControlInterfaces& interfaces,
    const Vector2<int> point,
    bool relative = false,
    double maxForce = 3.0,
    double maxForceLength = 150.0,
    double influenceLength = 250.0)
    :
    ContinuousRule(interfaces,"AvoidPoint"),
    pointX(localPointX),
    pointY(localPointY),
    relative(relative),
    maxForce(maxForce),
    maxForceLength(maxForceLength),
    influenceLength(influenceLength)
  {
      localPointX = (double)point.x;
      localPointY = (double)point.y;
  };

  /** constructor
   */
  AvoidPoint(
    BehaviorControlInterfaces& interfaces,
    const double& pointX,
    const double& pointY,
    bool relative = false,
    double maxForce = 3.0,
    double maxForceLength = 0.0,
    double influenceLength = 150.0)
    :
    ContinuousRule(interfaces, "AvoidPoint"),
    pointX(pointX),
    pointY(pointY),
    relative(relative),
    maxForce(maxForce),
    maxForceLength(maxForceLength),
    influenceLength(influenceLength)
  {};

  /**
   * Returns whether this rule generates absolute or robot relative coordinates.
   */
  virtual bool isRelative() {return relative;}

 	/** executes the basic behavior rule
   * @param robotPose the current robots pose at which the rule is to be evaluated
   * @param walk the direction and speed of the suggested motion coded as an vector (output)
	 * @param ra the rotation angle, the direction the robot should be directed (output)
	 * @param rweight the rotation weight the weight of the suggested rotation (output)
	 */
  virtual void execute(const RobotPose& robotPose,
                       Vector2<double>& walk,
                       double& ra, double& rweight)
  {
    walk = Vector2<double>(pointX, pointY);
    Pose2D start(0.0,0.0,0.0);

    if (!relative) 
      start = robotPose;

    double distance = min(
      Geometry::distance(start.translation, walk),
      Geometry::distance(start*Vector2<double>(-bodyLength,0), walk));

    if (!relative) 
      walk -= robotPose.translation;

    walk.normalize();

    walk *= -1.0 * getRepellingForce(distance, maxForce, maxForceLength, influenceLength);

    rweight=0;
  }
};

}


#endif //__AvoidPoint_h_


/*
* Change log:
*
* $Log: AvoidPoint.h,v $
* Revision 1.1  2004/03/16 14:00:18  juengel
* Integrated Improvments from "Gnne"
* -ATH2004ERS7Behavior
* -ATHHeadControl
* -KickSelectionTable
* -KickEditor
*
* Revision 1.1  2004/03/06 12:52:12  loetzsch
* cloned ATH2004BehaviorControl into ATH2004ERS7BehaviorControl
*
* Revision 1.1.1.1  2004/03/05 10:10:11  loetzsch
* created local cvs for Gnne
*
* Revision 1.2  2003/10/31 08:32:48  dueffert
* doxygen bugs fixed
*
* Revision 1.1  2003/10/26 22:49:34  loetzsch
* created ATH2004ERS7BehaviorControl from GT2003BehaviorControl
*  - strongly simplified option graph
*  - moved some symbols from GT2003 to CommonXabsl2Symbols
*  - moved some basic behaviors from GT2003 to CommonXabsl2BasicBehaviors
*
* cloned ATH2004ERS7 three times (BB2004, DDD2004, MSH2004)
*
*/
