/** 
* @file ATH2004ERS7ContinuousRules/AvoidObstaclesAheadGoingSideways.h
*
* @author Matthias Jngel
*/

#ifndef __ATH2004ERS7AvoidObstaclesAheadGoingSideways_h_
#define __ATH2004ERS7AvoidObstaclesAheadGoingSideways_h_

#include "Tools/ContinuousBasicBehaviors/ContinuousRule.h"
#include "Tools/Debugging/DebugDrawings.h"

namespace ATH2004ERS7ContinuousRules
{
  
/**@class AvoidObstaclesAheadGoingSideways
* rule which avoids collision with the obstacles
  */
  class AvoidObstaclesAheadGoingSideways: public ContinuousRule{
  public:
    
  /** constructor
    */
    AvoidObstaclesAheadGoingSideways(BehaviorControlInterfaces& interfaces,
      double maxForce = 1.5, 
      double maxForceLength = 100.0, 
      double influenceLength = 550.0)
      :ContinuousRule(interfaces, "AvoidObstaclesAheadGoingSideways"),
      maxForce(maxForce),
      maxForceLength(maxForceLength),
      influenceLength(influenceLength)
    {};
    
    /**
    * Returns whether this rule generates absolute or robot relative coordinates.
    * This rule is robot relative.
    */
    virtual bool isRelative() {return true;}
    
    /** executes the basic behavior rule
    * @param robotPose the current robots pose at which the rule is to be evaluated
    * @param walk the direction and speed of the suggested motion coded as an vector (output)
    * @param ra the rotation angle, the direction the robot should be directed (output)
    * @param rweight the rotation weight the weight of the suggested rotation (output)
    */
    virtual void execute(const RobotPose& robotPose,
      Vector2<double>& walk,
      double& ra, double& rweight)
    {
      //    if(obstaclesModel.corridorInFront < influenceLength)
      {
        double sideways;
        double forward = 0;
        double force = getRepellingForce(
          obstaclesModel.getDistanceInCorridor(0, 280), 
          maxForce, maxForceLength, influenceLength);
        if(obstaclesModel.getAngleOfNextFreeSector(pi/4, 0, (int)influenceLength) < 0)
          sideways = -force;
        else
          sideways = force;
        
        forward = -force;
        walk = Vector2<double>(forward, sideways);
      }
    };
    
  protected:
    /** the maximum length of the result vector*/
    double maxForce;
    /** the distance below which the result vector length is maxForce (in mm)*/
    double maxForceLength;
    /** the distance above which the result vector length is 0 (in mm)*/
    double influenceLength;
  };
  
}

#endif 

/*
* Change log:
*
* $Log: AvoidObstaclesAheadGoingSideways.h,v $
* Revision 1.1  2004/03/16 14:00:18  juengel
* Integrated Improvments from "Gnne"
* -ATH2004ERS7Behavior
* -ATHHeadControl
* -KickSelectionTable
* -KickEditor
*
* Revision 1.1  2004/03/06 12:52:11  loetzsch
* cloned ATH2004BehaviorControl into ATH2004ERS7BehaviorControl
*
* Revision 1.1.1.1  2004/03/05 10:10:11  loetzsch
* created local cvs for Gnne
*
* Revision 1.2  2003/10/28 15:04:09  loetzsch
* fixed doxygen bugs
*
* Revision 1.1  2003/10/26 22:49:34  loetzsch
* created ATH2004ERS7BehaviorControl from GT2003BehaviorControl
*  - strongly simplified option graph
*  - moved some symbols from GT2003 to CommonXabsl2Symbols
*  - moved some basic behaviors from GT2003 to CommonXabsl2BasicBehaviors
*
* cloned ATH2004ERS7 three times (BB2004, DDD2004, MSH2004)
*
*/
