/** 
* @file ATH2004ERS7ContinuousRules/AvoidBorder.h
*
* @author Max Risler
* @author Ronnie Brunn
*/

#ifndef __ATH2004ERS7AvoidBorder_h_
#define __ATH2004ERS7AvoidBorder_h_

#include "Tools/ContinuousBasicBehaviors/ContinuousRule.h"
#include "Tools/FieldDimensions.h"

namespace ATH2004ERS7ContinuousRules
{

/**@class AvoidBorder
 * rule which avoids collision with field borders
 */
class AvoidBorder: public ContinuousRule{
public:

  /** constructor
   */
  AvoidBorder(BehaviorControlInterfaces& interfaces,
              double maxForce = 1.5, 
              double maxForceLength = 0.0, 
              double influenceLength = 150.0)
      :ContinuousRule(interfaces, "AvoidBorder"), 
       maxForce(maxForce), 
       maxForceLength(maxForceLength),
       influenceLength(influenceLength) 
  {
    xPosInfluenceArea=((xPosOpponentGroundline+xPosOpponentSideCorner)/2.0)-sqrt((influenceLength*influenceLength)/2.0);
    yPosInfluenceArea=((yPosLeftGroundline+yPosLeftSideline)/2.0)-sqrt((influenceLength*influenceLength)/2.0);
  };

 	/** executes the basic behavior rule
   * @param robotPose the current robots pose at which the rule is to be evaluated
	 * @param walk the direction and speed of the suggested motion coded as an vector (output)
	 * @param ra the rotation angle, the direction the robot should be directed (output)
	 * @param rweight the rotation weight the weight of the suggested rotation (output)
	 */
  virtual void execute(const RobotPose& robotPose,
                       Vector2<double>& walk,
                       double& ra,
                       double& rweight)
  {
    if ((fabs(robotPose.translation.x) <= xPosInfluenceArea) &&
        (fabs(robotPose.translation.y) <= yPosInfluenceArea))
    { // no influence 
      walk.x=walk.y=ra=rweight=0;
      return;
    }

    double distance = min (
      FieldDimensions::distanceToBorder ( robotPose.translation ),
      FieldDimensions::distanceToBorder ( robotPose*Vector2<double>(-bodyLength,0) ) );

    walk = -robotPose.translation;
    walk.normalize();

    walk *= getRepellingForce(distance, maxForce, maxForceLength, influenceLength);

    ra=0;
    rweight=0;
  }
  
protected:
	/** the maximum length of the result vector*/
  double maxForce;
	/** the distance below which the result vector length is maxForce (in mm)*/
  double maxForceLength;
	/** the distance above which the result vector length is 0 (in mm)*/
  double influenceLength;
  /** the maximum absolute x-Position up to which there is no influence by groundline, goal and corner */
  double xPosInfluenceArea;
  /** the maximum absolute y-Position up to which there is no influence by sideline and corner */
  double yPosInfluenceArea;

};

}

#endif 


/*
* Change log:
*
* $Log: AvoidBorder.h,v $
* Revision 1.1  2004/03/16 14:00:18  juengel
* Integrated Improvments from "Gnne"
* -ATH2004ERS7Behavior
* -ATHHeadControl
* -KickSelectionTable
* -KickEditor
*
* Revision 1.1  2004/03/06 12:52:11  loetzsch
* cloned ATH2004BehaviorControl into ATH2004ERS7BehaviorControl
*
* Revision 1.1.1.1  2004/03/05 10:10:11  loetzsch
* created local cvs for Gnne
*
* Revision 1.2  2003/10/28 15:04:08  loetzsch
* fixed doxygen bugs
*
* Revision 1.1  2003/10/26 22:49:34  loetzsch
* created ATH2004ERS7BehaviorControl from GT2003BehaviorControl
*  - strongly simplified option graph
*  - moved some symbols from GT2003 to CommonXabsl2Symbols
*  - moved some basic behaviors from GT2003 to CommonXabsl2BasicBehaviors
*
* cloned ATH2004ERS7 three times (BB2004, DDD2004, MSH2004)
*
*/
