/** 
* @file ATH2004ERS7ContinuousBasicBehaviors.cpp
*
* Implementation of basic behaviors defined in continuous-basic-behaviors.xml.
*
* @author Martin Kallnik
* @author Max Risler
*/

#include "ATH2004ERS7ContinuousRules/AvoidBorder.h"
#include "ATH2004ERS7ContinuousRules/AvoidBorderTurning.h"
#include "ATH2004ERS7ContinuousRules/AvoidObstacles.h"
#include "ATH2004ERS7ContinuousRules/AvoidObstaclesAheadGoingSideways.h"
#include "ATH2004ERS7ContinuousRules/AvoidObstaclesAheadTurning.h"
#include "ATH2004ERS7ContinuousRules/AvoidObstaclesInDirectionToPointTurning.h"
#include "ATH2004ERS7ContinuousRules/AvoidOpponents.h"
#include "ATH2004ERS7ContinuousRules/AvoidPoint.h"
#include "ATH2004ERS7ContinuousRules/AvoidTeammates.h"
#include "ATH2004ERS7ContinuousRules/AvoidOwnPenaltyArea.h"
#include "ATH2004ERS7ContinuousRules/GoBetweenBallAndOwnGoal.h"
#include "ATH2004ERS7ContinuousRules/GoBetweenBallAndOwnGoalClear.h"
#include "ATH2004ERS7ContinuousRules/GoToPoint.h"
#include "ATH2004ERS7ContinuousRules/TurnToAngle.h"
#include "ATH2004ERS7ContinuousRules/TurnToPoint.h"
#include "ATH2004ERS7ContinuousRules/TurnToPointGoalie.h"
#include "ATH2004ERS7ContinuousRules/Walk.h"
#include "ATH2004ERS7ContinuousRules/WalkX.h"
#include "ATH2004ERS7ContinuousRules/WalkY.h"

#include "ATH2004ERS7ContinuousBasicBehaviors.h"

using namespace ATH2004ERS7ContinuousRules;

ATH2004ERS7ContinuousBasicBehaviors::ATH2004ERS7ContinuousBasicBehaviors(
                       BehaviorControlInterfaces& interfaces,
                       Xabsl2ErrorHandler& errorHandler) : 
  BehaviorControlInterfaces(interfaces),
  errorHandler(errorHandler),
  goToPoint(interfaces,errorHandler,"cont-go-to-point"),
  goForwardToPoint(interfaces,errorHandler,"cont-go-forward-to-point"),
  goForwardToPointTurning(interfaces,errorHandler,"cont-go-forward-to-point-turning"),
  goToBall(interfaces,errorHandler,"cont-go-to-ball"),
  dribble(interfaces,errorHandler,"cont-dribble"),
  turnWithBall(interfaces,errorHandler,"cont-turn-with-ball"),
  walkAvoidingObstacles(interfaces,errorHandler,"cont-walk-avoiding-obstacles"),
  walkAroundAvoidingObstacles(interfaces,errorHandler,"cont-walk-around-avoiding-obstacles"),
  turnToBall(interfaces,errorHandler,"cont-turn-to-ball"),
  //turnToPosBetweenBallAndGoal(interfaces,errorHandler,"cont-turn-to-pos-between-ball-and-goal"),
  turn(interfaces,errorHandler,"cont-turn"),
  goalieClear(interfaces,errorHandler,"cont-goalie-clear"),
  goalieGuard(interfaces,errorHandler,"cont-goalie-guard"),
  goaliePosition(interfaces,errorHandler,"cont-goalie-position"),
  goalieReturn(interfaces,errorHandler,"cont-goalie-return"),
  supporterPosition(interfaces,errorHandler,"cont-supporter-position"),
  offensiveSupporterPosition(interfaces, errorHandler, "cont-offensive-supporter-position")
{
  goToPoint.setMaxRotationSpeed(fromDegrees(120.0));
  goToPoint.addRule(new GoToPoint(interfaces,goToPointX,goToPointY));
  goToPoint.addRule(new TurnToAngle(interfaces,goToPointAngle));
  goToPoint.addRule(new AvoidBorder(interfaces));
  goToPoint.addRule(new AvoidObstacles(interfaces));
  goToPoint.addRule(new AvoidOwnPenaltyArea(interfaces));

  goToPoint.registerParameter("cont-go-to-point.x",goToPointX);
  goToPoint.registerParameter("cont-go-to-point.y",goToPointY);
  goToPoint.registerParameter("cont-go-to-point.angle",goToPointAngle);
  goToPoint.registerParameter("cont-go-to-point.max-speed",goToPointMaxSpeed);

  goForwardToPoint.setMaxRotationSpeed(fromDegrees(120.0));
  goForwardToPoint.addRule(new GoToPoint(interfaces,goForwardToPointX,goForwardToPointY,false,1.0,100.0));
  goForwardToPoint.addRule(new TurnToPoint(interfaces,goForwardToPointX,goForwardToPointY));
  goForwardToPoint.addRule(new AvoidBorder(interfaces));
  goForwardToPoint.addRule(new AvoidObstaclesAheadGoingSideways(interfaces));
  goForwardToPoint.addRule(new AvoidOwnPenaltyArea(interfaces));

  goForwardToPoint.registerParameter("cont-go-forward-to-point.x",goForwardToPointX);
  goForwardToPoint.registerParameter("cont-go-forward-to-point.y",goForwardToPointY);
  goForwardToPoint.registerParameter("cont-go-forward-to-point.max-speed",goForwardToPointMaxSpeed);

  goForwardToPointTurning.addRule(new WalkX(interfaces,goForwardToPointTurningMaxSpeed));
  goForwardToPointTurning.addRule(new AvoidObstaclesInDirectionToPointTurning(interfaces,goForwardToPointTurningX,goForwardToPointTurningY,goForwardToPointTurningObstacleInfluenceLength));

  goForwardToPointTurning.registerParameter("cont-go-forward-to-point-turning.x",goForwardToPointTurningX);
  goForwardToPointTurning.registerParameter("cont-go-forward-to-point-turning.y",goForwardToPointTurningY);
  goForwardToPointTurning.registerParameter("cont-go-forward-to-point-turning.max-speed",goForwardToPointTurningMaxSpeed);
  goForwardToPointTurning.registerParameter("cont-go-forward-to-point-turning.obstacle-influence-length", goForwardToPointTurningObstacleInfluenceLength);

  goToBall.setMaxRotationSpeed(fromDegrees(120.0));
  goToBall.addRule(new GoToPoint(interfaces,knownBallPosRobotRelative,true,1.0,100.0));
  goToBall.addRule(new TurnToPoint(interfaces,knownBallPosRobotRelative,true));
  goToBall.addRule(new AvoidBorder(interfaces));
  goToBall.addRule(new AvoidOwnPenaltyArea(interfaces));

  dribble.setWalkType(MotionRequest::normal);
  dribble.setMaxRotationSpeed(fromDegrees(50.0));
  dribble.enableBackwards(false);
  dribble.setAccelerations(80.0,50.0,50.0);
  dribble.addRule(new GoToPoint(interfaces,dribbleX,dribbleY,false,1.0,150.0,400.0));
  dribble.addRule(new TurnToPoint(interfaces,dribbleX,dribbleY));
  dribble.addRule(new AvoidBorder(interfaces));
  dribble.addRule(new AvoidOwnPenaltyArea(interfaces));
  dribble.addRule(new AvoidTeammates(interfaces));

  dribble.registerParameter("cont-dribble.x", dribbleX);
  dribble.registerParameter("cont-dribble.y", dribbleY);

  turnWithBall.setWalkType(MotionRequest::turnWithBall2);
  turnWithBall.enableBackwards(false);
  turnWithBall.setAccelerations(80.0,50.0,50.0);
  turnWithBall.addRule(new TurnToAngle(interfaces,turnWithBallAngle,true));
  turnWithBall.addRule(new WalkY(interfaces,turnWithBallSpeedY));
//  turnWithBall.addRule(new AvoidBorder(interfaces));
//  turnWithBall.addRule(new AvoidOwnPenaltyArea(interfaces));
//  turnWithBall.addRule(new AvoidTeammates(interfaces));

  turnWithBall.registerParameter("cont-turn-with-ball.angle", turnWithBallAngle);
  turnWithBall.registerParameter("cont-turn-with-ball.max-rotation-speed", turnWithBallMaxRotationSpeed);
  turnWithBall.registerParameter("cont-turn-with-ball.rotation-center", turnWithBallRotationCenter);

  walkAvoidingObstacles.addRule(new AvoidObstacles(interfaces));
  walkAvoidingObstacles.addRule(new Walk(interfaces,walkAvoidingObstaclesX,walkAvoidingObstaclesY));
  walkAvoidingObstacles.addRule(new TurnToAngle(interfaces,walkAvoidingObstaclesRot,true));

  walkAvoidingObstacles.registerParameter("cont-walk-avoiding-obstacles.type",walkAvoidingObstaclesType);
  walkAvoidingObstacles.registerParameter("cont-walk-avoiding-obstacles.speed-x",walkAvoidingObstaclesX);
  walkAvoidingObstacles.registerParameter("cont-walk-avoiding-obstacles.speed-y",walkAvoidingObstaclesY);
  walkAvoidingObstacles.registerParameter("cont-walk-avoiding-obstacles.rotation-speed",walkAvoidingObstaclesRot);

  walkAroundAvoidingObstacles.addRule(new AvoidObstaclesAheadTurning(interfaces));
  walkAroundAvoidingObstacles.addRule(new AvoidBorderTurning(interfaces));
  walkAroundAvoidingObstacles.addRule(new Walk(interfaces, walkAroundAvoidingObstaclesX, walkAroundAvoidingObstaclesY));
  walkAroundAvoidingObstacles.registerParameter("cont-walk-around-avoiding-obstacles.speed-x",walkAroundAvoidingObstaclesX);
  walkAroundAvoidingObstacles.registerParameter("cont-walk-around-avoiding-obstacles.speed-y",walkAroundAvoidingObstaclesY);

  turnToBall.addRule(new TurnToPoint(interfaces,knownBallPosRobotRelative,true));
  turnToBall.addRule(new AvoidBorder(interfaces));
  turnToBall.addRule(new AvoidOwnPenaltyArea(interfaces));
  turnToBall.addRule(new AvoidTeammates(interfaces));
  turnToBall.addRule(new AvoidObstacles(interfaces));

  turnToBall.registerParameter("cont-turn-to-ball.max-rotation-speed",turnToBallMaxRotationSpeed);

  turn.addRule(new TurnToAngle(interfaces,turnRotationSpeed,true));
  turn.addRule(new AvoidBorder(interfaces));
  turn.addRule(new AvoidOwnPenaltyArea(interfaces));
  turn.addRule(new AvoidTeammates(interfaces));
  turn.addRule(new AvoidObstacles(interfaces));

  turn.registerParameter("cont-turn.rotation-speed",turnRotationSpeed);

  goalieClear.setMaxRotationSpeed(fromDegrees(120.0));
  goalieClear.setMaxSpeed(150.0);
  goalieClear.addRule(new GoBetweenBallAndOwnGoalClear(interfaces));
  goalieClear.addRule(new TurnToPoint(interfaces,knownBallPosRobotRelative,true));

  goalieClear.registerParameter("cont-goalie-clear.max-speed",goalieClearMaxSpeed);

  goalieGuard.setMaxRotationSpeed(fromDegrees(120.0));
  goalieGuard.setMaxSpeed(150.0);
  goalieGuard.addRule(new GoBetweenBallAndOwnGoal(interfaces));
  goalieGuard.addRule(new TurnToPointGoalie(interfaces,knownBallPosRobotRelative,true));
  goalieGuard.addRule(new AvoidBorder(interfaces));

  goalieGuard.registerParameter("cont-goalie-guard.max-speed",goalieGuardMaxSpeed);

  goaliePosition.setMaxRotationSpeed(fromDegrees(120.0));
  goaliePosition.addRule(new GoToPoint(interfaces,goaliePositionX,goaliePositionY, false));
  goaliePosition.addRule(new TurnToAngle(interfaces,goaliePositionAngle,false));
  goaliePosition.addRule(new AvoidPoint(interfaces,knownBallPosRobotRelative,true));
  goaliePosition.addRule(new AvoidBorder(interfaces));

  goaliePosition.registerParameter("cont-goalie-position.x",goaliePositionX);
  goaliePosition.registerParameter("cont-goalie-position.y",goaliePositionY);
  goaliePosition.registerParameter("cont-goalie-position.max-speed",goaliePositionMaxSpeed);
  goaliePosition.registerParameter("cont-goalie-position.angle",goaliePositionAngle);

  goalieReturn.setMaxRotationSpeed(fromDegrees(120.0));
  goalieReturn.addRule(new GoToPoint(interfaces,ownGoalRearPanelMiddle, false));
  goalieReturn.addRule(new TurnToPoint(interfaces,knownBallPosRobotRelative,true));
  goalieReturn.addRule(new AvoidPoint(interfaces,knownBallPosRobotRelative,true));
  goalieReturn.addRule(new AvoidBorder(interfaces));
  goalieReturn.addRule(new AvoidObstacles(interfaces));

  supporterPosition.setMaxRotationSpeed(fromDegrees(120.0));
  // go to destination but no need to reach it exactly
  supporterPosition.addRule(new GoToPoint(interfaces,supporterPositionX,supporterPositionY,false,1.0,0.0,500.0));
  // turn to ball
  supporterPosition.addRule(new TurnToPoint(interfaces,knownBallPosRobotRelative,true));
  // avoid running into opponent goal area
  supporterPosition.addRule(new AvoidPoint(interfaces,opponentGoalRearPanelMiddle,false,3.0,600.0,850.0));
  // avoid running over ball
  supporterPosition.addRule(new AvoidPoint(interfaces,knownBallPosRobotRelative,true));
  supporterPosition.addRule(new AvoidBorder(interfaces));
  supporterPosition.addRule(new AvoidOwnPenaltyArea(interfaces));
  supporterPosition.addRule(new AvoidTeammates(interfaces));
  supporterPosition.addRule(new AvoidObstacles(interfaces));

  supporterPosition.registerParameter("cont-supporter-position.x",supporterPositionX);
  supporterPosition.registerParameter("cont-supporter-position.y",supporterPositionY);
  supporterPosition.registerParameter("cont-supporter-position.max-speed",supporterPositionMaxSpeed);

  offensiveSupporterPosition.setMaxRotationSpeed(fromDegrees(120.0));
  // go to destination but no need to reach it exactly
  offensiveSupporterPosition.addRule(new GoToPoint(interfaces,offensiveSupporterPositionX,offensiveSupporterPositionY,false,1.0,0.0,500.0));
  // turn to ball
  offensiveSupporterPosition.addRule(new TurnToPoint(interfaces,knownPointBetweenBallandGoal,true));
  // avoid running into opponent goal area
  offensiveSupporterPosition.addRule(new AvoidPoint(interfaces,opponentGoalRearPanelMiddle,false,3.0,600.0,850.0));
  // avoid running over ball
  offensiveSupporterPosition.addRule(new AvoidPoint(interfaces,knownBallPosRobotRelative,true));
  offensiveSupporterPosition.addRule(new AvoidBorder(interfaces));
  offensiveSupporterPosition.addRule(new AvoidOwnPenaltyArea(interfaces));
  offensiveSupporterPosition.addRule(new AvoidTeammates(interfaces));
  offensiveSupporterPosition.addRule(new AvoidObstacles(interfaces));

  offensiveSupporterPosition.registerParameter("cont-offensive-supporter-position.x",offensiveSupporterPositionX);
  offensiveSupporterPosition.registerParameter("cont-offensive-supporter-position.y",offensiveSupporterPositionY);
  offensiveSupporterPosition.registerParameter("cont-offensive-supporter-position.max-speed",offensiveSupporterPositionMaxSpeed);

}


void ATH2004ERS7ContinuousBasicBehaviors::registerBasicBehaviors(Xabsl2Engine& engine)
{
  engine.registerBasicBehavior(goToPoint);
  engine.registerBasicBehavior(goForwardToPoint);
  engine.registerBasicBehavior(goForwardToPointTurning);
  engine.registerBasicBehavior(goToBall);
  engine.registerBasicBehavior(dribble);
  engine.registerBasicBehavior(turnWithBall);
  engine.registerBasicBehavior(walkAvoidingObstacles);
  engine.registerBasicBehavior(walkAroundAvoidingObstacles);
  engine.registerBasicBehavior(turnToBall);
  engine.registerBasicBehavior(turn);
  engine.registerBasicBehavior(goalieClear);
  engine.registerBasicBehavior(goalieGuard);
  engine.registerBasicBehavior(goaliePosition);
  engine.registerBasicBehavior(goalieReturn);
  engine.registerBasicBehavior(supporterPosition);
  engine.registerBasicBehavior(offensiveSupporterPosition);
}

void ATH2004ERS7ContinuousBasicBehaviors::update()
{
 
  knownBallPos = ballPosition.getKnownPosition(
    BallModel::behaviorControlTimeAfterWhichCommunicatedBallsAreAccepted);

  knownBallPosRobotRelative = Geometry::vectorTo(robotPose, knownBallPos);
  
    
  /* Point between ball and goal calculation*/
  Vector2<double>rP(robotPose.translation);
  double relAngleToBall = Vector2<double>(knownBallPos -rP).angle();
  double relAngleToGoal = Vector2<double>(Vector2<double>(xPosOpponentGoal,0)-rP).angle();
  if(fabs( relAngleToBall - relAngleToGoal) < pi/2)
  {
    double desAngle = ((relAngleToBall + relAngleToGoal) /2) - (fabs(relAngleToBall) < fabs(relAngleToGoal) ? relAngleToBall:relAngleToGoal);
    knownPointBetweenBallandGoal = Geometry::vectorTo(robotPose, Vector2<double>(rP.x+cos(desAngle),rP.y+sin(desAngle)));
  }

  else
  {
     knownPointBetweenBallandGoal = knownBallPosRobotRelative;
  }
  
   

  walkAvoidingObstacles.setWalkType(MotionRequest::normal);
  switch((int)walkAvoidingObstaclesType)
  {
    case 0: walkAvoidingObstacles.setWalkType(MotionRequest::normal); break;
    case 1: walkAvoidingObstacles.setWalkType(MotionRequest::upsideDown); break;
    case 2: walkAvoidingObstacles.setWalkType(MotionRequest::turnWithBall); break;
    case 3: walkAvoidingObstacles.setWalkType(MotionRequest::turnWithBall2); break;
    case 4: walkAvoidingObstacles.setWalkType(MotionRequest::turnWithBall3); break;
    case 5: walkAvoidingObstacles.setWalkType(MotionRequest::turnWithBall4); break;
    case 6: walkAvoidingObstacles.setWalkType(MotionRequest::turnWithBall5); break;
    case 7: walkAvoidingObstacles.setWalkType(MotionRequest::dash); break;
  }

  goToPoint.setMaxSpeed(goToPointMaxSpeed);
  goForwardToPoint.setMaxSpeed(goForwardToPointMaxSpeed);
  goForwardToPointTurning.setMaxSpeed(goForwardToPointTurningMaxSpeed);
  turnWithBall.setMaxRotationSpeed(fromDegrees(turnWithBallMaxRotationSpeed));
  turnWithBall.setMaxSpeed(fromDegrees(turnWithBallMaxRotationSpeed)*turnWithBallRotationCenter);
  turnToBall.setMaxRotationSpeed(turnToBallMaxRotationSpeed);
  goalieClear.setMaxSpeed(goalieClearMaxSpeed);
  goalieGuard.setMaxSpeed(goalieGuardMaxSpeed);
  goaliePosition.setMaxSpeed(goaliePositionMaxSpeed);
  supporterPosition.setMaxSpeed(supporterPositionMaxSpeed);
  offensiveSupporterPosition.setMaxSpeed(offensiveSupporterPositionMaxSpeed);

  turnWithBallSpeedY = turnWithBallRotationCenter*fromDegrees(turnWithBallAngle);

  // set the evaluation mode
  // this is only done for localization dependent behaviors, others keep discrete mode
  goToPoint.setEvaluationMode(ContinuousBasicBehavior::discrete);
  goForwardToPoint.setEvaluationMode(ContinuousBasicBehavior::discrete);
  goForwardToPointTurning.setEvaluationMode(ContinuousBasicBehavior::discrete);
  goToBall.setEvaluationMode(ContinuousBasicBehavior::discrete);
  dribble.setEvaluationMode(ContinuousBasicBehavior::discrete);
  turnWithBall.setEvaluationMode(ContinuousBasicBehavior::discrete);
  turnToBall.setEvaluationMode(ContinuousBasicBehavior::discrete);
  turn.setEvaluationMode(ContinuousBasicBehavior::discrete);
  goalieClear.setEvaluationMode(ContinuousBasicBehavior::discrete);
  goalieGuard.setEvaluationMode(ContinuousBasicBehavior::discrete);
  goaliePosition.setEvaluationMode(ContinuousBasicBehavior::discrete);
  goalieReturn.setEvaluationMode(ContinuousBasicBehavior::discrete);
  supporterPosition.setEvaluationMode(ContinuousBasicBehavior::discrete);
  offensiveSupporterPosition.setEvaluationMode(ContinuousBasicBehavior::discrete);
}

/*
* Change Log
* 
* $Log: ATH2004ERS7ContinuousBasicBehaviors.cpp,v $
* Revision 1.10  2004/05/04 10:48:58  loetzsch
* replaced all enums
* xxxBehaviorControlTimeAfterWhichCommunicatedBallsAreAccepted
* by
* behaviorControlTimeAfterWhichCommunicatedBallsAreAccepted
* (this mechanism was neither fully implemented nor used)
*
* Revision 1.9  2004/04/15 16:51:50  spranger
* removed turnToPosBetweenBallAndGoal/turn-to-pos-between-ball-and-goal
* added sane calculations for offensive-supporter-position
*
* Revision 1.8  2004/04/14 19:56:24  spranger
* sanitise
*
* Revision 1.7  2004/04/14 16:40:11  spranger
* no message
*
* Revision 1.6  2004/04/14 16:39:48  spranger
* no message
*
* Revision 1.5  2004/04/14 15:51:36  spranger
* removed potential error ;)
*
* Revision 1.4  2004/04/07 13:44:33  risler
* ddd checkin after go04 - second part fixed walktype
*
* Revision 1.3  2004/03/26 15:48:05  spranger
* added turnToPosBetweenBallAndGoal and offensiveSupporterPosition
*
* Revision 1.2  2004/03/19 22:44:22  juengel
* changed turnWithBall
*
* Revision 1.1  2004/03/16 14:00:18  juengel
* Integrated Improvments from "Gnne"
* -ATH2004ERS7Behavior
* -ATHHeadControl
* -KickSelectionTable
* -KickEditor
*
* Revision 1.1  2004/03/06 12:52:11  loetzsch
* cloned ATH2004BehaviorControl into ATH2004ERS7BehaviorControl
*
* Revision 1.1.1.1  2004/03/05 10:10:11  loetzsch
* created local cvs for Gnne
*
* Revision 1.2  2004/02/03 13:20:47  spranger
* renamed all references to  class BallPosition to BallModel (possibly changed include files)
*
* Revision 1.1  2003/10/26 22:49:34  loetzsch
* created ATH2004ERS7BehaviorControl from GT2003BehaviorControl
*  - strongly simplified option graph
*  - moved some symbols from GT2003 to CommonXabsl2Symbols
*  - moved some basic behaviors from GT2003 to CommonXabsl2BasicBehaviors
*
* cloned ATH2004ERS7 three times (BB2004, DDD2004, MSH2004)
*
*/

