/** 
* @file ATH2004SimpleBasicBehaviors.cpp
*
* Implementation of basic behaviors defined in simple-basic-behaviors.xml.
*
* @author Uwe Dffert
* @author Jan Hoffmann
* @author Matthias Jngel
* @author Martin Ltzsch
* @author Max Risler
*/

#include "Tools/Math/PIDsmoothedValue.h"
#include "ATH2004SimpleBasicBehaviors.h"
#include "Tools/Math/Geometry.h"
#include "Tools/Math/Common.h"

#include "Tools/Debugging/Debugging.h"

//needed for dribbleInCenterOfField and getBehindBall2
#include "Tools/FieldDimensions.h"

void ATH2004SimpleBasicBehaviors::registerBasicBehaviors(Xabsl2Engine& engine)
{
  engine.registerBasicBehavior(basicBehaviorAvoidObstacles);
  engine.registerBasicBehavior(basicBehaviorGetBehindBall);
  engine.registerBasicBehavior(basicBehaviorGoToBall);
  engine.registerBasicBehavior(basicBehaviorGoToBallWithDirection);
  engine.registerBasicBehavior(basicBehaviorGoToPoint);
  engine.registerBasicBehavior(basicBehaviorGoToPointFast);
  engine.registerBasicBehavior(basicBehaviorGoToPointPrecisely);
  engine.registerBasicBehavior(basicBehaviorGoToPointAndAvoidObstacles);
  engine.registerBasicBehavior(basicBehaviorDribbleAtBorder);
  engine.registerBasicBehavior(basicBehaviorDribbleInCenterOfField);
  engine.registerBasicBehavior(basicBehaviorGetBehindBall2);
}

void ATH2004BasicBehaviorGoToBallWithDirection::execute()
{
  //accelerationRestrictor.saveLastWalkParameters();
  
  if (maxSpeed == 0)
  {
    maxSpeed = 200;
  }
  
  Vector2<double> ball = ballPosition.getKnownPosition(
    BallModel::behaviorControlTimeAfterWhichCommunicatedBallsAreAccepted);
  
  Vector2<double> pointToReach(ball.x-cos(direction)*distanceX-sin(direction)*distanceY,
    ball.y-sin(direction)*distanceX-cos(direction)*distanceY);
  
  
  double destinationDistance = Geometry::distanceTo(robotPose.getPose(),pointToReach);
  double destinationX = destinationDistance * cos(Geometry::angleTo(robotPose.getPose(),pointToReach));
  double destinationY = destinationDistance * sin(Geometry::angleTo(robotPose.getPose(),pointToReach));
  
  /*
  Vector2<double> pointToReachMirrored(ball.x-cos(direction)*distanceX+sin(direction)*distanceY,
  ball.y-sin(direction)*distanceX+cos(direction)*distanceY);
  
    double destinationDistanceMirrored = Geometry::distanceTo(robotPose.getPose(),pointToReach);
    double destinationYMirrored = destinationDistanceMirrored * sin(Geometry::angleTo(robotPose.getPose(),pointToReachMirrored));
    
      if (fabs(destinationYMirrored) < fabs(destinationY))
      {
      destinationY = destinationYMirrored;
      }
  */
  
  if(destinationDistance < 1)
  {
    destinationDistance = 1;
  }
  
  double minimalTime=destinationDistance/maxSpeed;
  
  double angleDiff=direction-robotPose.rotation;
  if (angleDiff<-pi)
  {
    angleDiff += pi2;
  }
  else if (angleDiff >=pi)
  {
    angleDiff -= pi2;
  }
  
  double maxTurnSpeed = fromDegrees(60);
  double turnTime = angleDiff/maxTurnSpeed;
  if (turnTime>minimalTime)
  {
    minimalTime=turnTime;
  }
  minimalTime += 0.2; //get slower near to destination
  
  motionRequest.motionType = MotionRequest::walk;
  motionRequest.walkType = MotionRequest::normal;
  
  motionRequest.walkParams.translation.x = destinationX / minimalTime;
  motionRequest.walkParams.translation.y = destinationY / minimalTime;
  
  motionRequest.walkParams.rotation = angleDiff;
  
  if (motionRequest.walkParams.rotation > maxTurnSpeed)
    motionRequest.walkParams.rotation = maxTurnSpeed;
  if (motionRequest.walkParams.rotation < -maxTurnSpeed) 
    motionRequest.walkParams.rotation = -maxTurnSpeed;
  
  if ((fabs(toDegrees(angleDiff))<10)&&(destinationDistance<10))
  {
    motionRequest.motionType = MotionRequest::stand;
  }
}

void ATH2004BasicBehaviorGoToBall::execute()
{
  //accelerationRestrictor.saveLastWalkParameters();
  
  if (maxSpeed == 0) maxSpeed = 200;
  
  double maxSpeedX = maxSpeed;
  double maxSpeedY = maxSpeed;
  
  double maxTurnSpeed = fromDegrees(60);
  
  double timeToReachX, timeToReachY, timeToTurnToAngle, estimatedTimeToReachDestination;
  
  Vector2<double> ball = ballPosition.getKnownPosition(
    BallModel::behaviorControlTimeAfterWhichCommunicatedBallsAreAccepted);
  double ballDistance = Geometry::distanceTo(robotPose.getPose(),ball);
  double ballX = ballDistance * cos(Geometry::angleTo(robotPose.getPose(),ball)) + 50;
  double ballY = ballDistance * sin(Geometry::angleTo(robotPose.getPose(),ball));
  
  if(ballDistance == 0)
    ballDistance = 1;
  double ballAngle = Geometry::angleTo(robotPose.getPose(),ball);
  
  double diffX = distanceAtEnd / ballDistance * ballX;
  double diffY = distanceAtEnd / ballDistance * ballY;
  
  Vector2<double> destination(ballX - diffX, ballY - diffY);
  double distanceToDestination;
  distanceToDestination = fabs(ballDistance - distanceAtEnd);
  
  timeToReachX = fabs(destination.x) / maxSpeedX;
  timeToReachY = fabs(destination.y) / maxSpeedY;
  timeToTurnToAngle = ballAngle / maxTurnSpeed;
  estimatedTimeToReachDestination = 
    (timeToReachX < timeToReachY) ? timeToReachY : timeToReachX;
  estimatedTimeToReachDestination = 
    (estimatedTimeToReachDestination < timeToTurnToAngle) ? timeToTurnToAngle : estimatedTimeToReachDestination;
  
  motionRequest.motionType = MotionRequest::walk;
  motionRequest.walkType = MotionRequest::normal;
  
  motionRequest.walkParams.translation.x = 
    destination.x / estimatedTimeToReachDestination;
  motionRequest.walkParams.translation.y = 
    destination.y / estimatedTimeToReachDestination;
  
  
  motionRequest.walkParams.rotation = ballAngle;
  
  if (motionRequest.walkParams.rotation > maxTurnSpeed)
    motionRequest.walkParams.rotation = maxTurnSpeed;
  if (motionRequest.walkParams.rotation < -maxTurnSpeed) 
    motionRequest.walkParams.rotation = -maxTurnSpeed;
  
  
  if ((fabs(toDegrees(ballAngle))<10)&&(distanceToDestination<20))
  {
    motionRequest.motionType = MotionRequest::stand;
  }
  else
  {
    //    accelerationRestrictor.restrictAccelerations(250,250,180);
  }
  
}

void ATH2004BasicBehaviorGetBehindBall::execute()
{
  // This was adapted from the XabslSkillGetBehindBall written by Max Risler.
  
  accelerationRestrictor.saveLastWalkParameters();
  
  double dir = fromDegrees(angle); 
  
  Vector2<double> ball = ballPosition.getKnownPosition(
    BallModel::behaviorControlTimeAfterWhichCommunicatedBallsAreAccepted);
  
  double destX = ball.x;
  double destY = ball.y;
  
  double posX = robotPose.getPose().translation.x;
  double posY = robotPose.getPose().translation.y;
  double posDir = robotPose.getPose().getAngle();
  
  Vector2<double> destPosDiff = ball - robotPose.getPose().translation;
  
  double absDestDir = atan2(destPosDiff.y, destPosDiff.x);
  double relDestDir = normalize(absDestDir - posDir);
  double destDist = destPosDiff.abs();
  
  if (destDist > 1000.0)
  {
    gotoPoint(destX,destY,maxSpeed);
  }
  else if (destDist >distance* 1.2)
  {
    // calculate destination point
    double px,py;
    
    px = destX -distance* cos(dir);
    py = destY -distance* sin(dir);
    
    double distToP = sqrt((posX-px)*(posX-px) + (posY-py)*(posY-py));
    
    if (destDist < distToP)
    {
      // dest is closer than destination point
      // walk to tangent
      double a = acos(distance/destDist);
      
      if ((leftRight == 0 && normalize(dir - absDestDir) > 0) ||
        leftRight > 0)
      {
        px = destX -distance* cos(absDestDir + a);
        py = destY -distance* sin(absDestDir + a);
      } else {
        px = destX -distance* cos(absDestDir - a);
        py = destY -distance* sin(absDestDir - a);
      }
      // distance is distance to tangent plus distance on perimeter
      distToP = sqrt((posX-px)*(posX-px) + (posY-py)*(posY-py));
      distToP +=distance* normalize(fabs((dir - absDestDir)));
    }
    gotoPoint(px,py,maxSpeed);
  }
  else
  {
    // circle around dest
    double diffAngle = normalize(posDir-dir);
    if (diffAngle > pi_2 && leftRight > 0) diffAngle -= pi2;
    if (diffAngle < -pi_2 && leftRight < 0) diffAngle += pi2;
    
    motionRequest.walkParams.translation.x = destDist - distance;
    motionRequest.walkParams.translation.y = diffAngle * distance;
    motionRequest.walkParams.rotation = relDestDir;
  }
  
  motionRequest.motionType = MotionRequest::walk;
  motionRequest.walkType = MotionRequest::normal;
  
  accelerationRestrictor.restrictAccelerations(300,300,180);
}

void ATH2004BasicBehaviorGetBehindBall::gotoPoint(double x, double y, double maxSpeed)
{
  double maxTurnSpeed = fromDegrees(60);
  
  double timeToReachX, timeToReachY, estimatedTimeToReachDestination;
  double distance = Geometry::distanceTo(robotPose.getPose(),Vector2<double>(x,y));
  double distanceX = distance
    * cos(Geometry::angleTo(robotPose.getPose(),Vector2<double>(x,y)));
  double distanceY = distance
    * sin(Geometry::angleTo(robotPose.getPose(),Vector2<double>(x,y)));
  
  if(distance == 0)
    distance = 1;
  double angle = Geometry::angleTo(robotPose.getPose(),ballPosition.getKnownPosition(
    BallModel::behaviorControlTimeAfterWhichCommunicatedBallsAreAccepted));
  
  Vector2<double> destination(distanceX, distanceY);
  
  timeToReachX = fabs(destination.x) / maxSpeed;
  timeToReachY = fabs(destination.y) / maxSpeed;
  
  estimatedTimeToReachDestination = 
    (timeToReachX < timeToReachY) ? timeToReachY : timeToReachX;
  
  motionRequest.motionType = MotionRequest::walk;
  motionRequest.walkType = MotionRequest::normal;
  
  motionRequest.walkParams.translation.x = 
    destination.x / estimatedTimeToReachDestination;
  motionRequest.walkParams.translation.y = 
    destination.y / estimatedTimeToReachDestination;
  
  motionRequest.walkParams.rotation = angle;
  
  if (motionRequest.walkParams.rotation > maxTurnSpeed)
    motionRequest.walkParams.rotation = maxTurnSpeed;
  if (motionRequest.walkParams.rotation < -maxTurnSpeed) 
    motionRequest.walkParams.rotation = -maxTurnSpeed;
}



void ATH2004BasicBehaviorGoToPointAndAvoidObstacles::execute()
{
  double 
    obstacleAvoidanceDistance,
    distanceToDestination = Geometry::distanceTo(robotPose.getPose(),Vector2<double>(x,y)),
    angleToDestination = Geometry::angleTo(robotPose.getPose(),Vector2<double>(x,y)),
    widthOfCorridor = 250,
    targetX, 
    targetAngle, 
    freeSpaceInFront;
  int mode = 0;

  if (farAwayAvoidance == 0)
    farAwayAvoidance = 300;

  // to do: make the following some function of 
  // the distance to the destination to make sure
  // the destination is reached!!
  obstacleAvoidanceDistance = distanceToDestination;
  if (obstacleAvoidanceDistance > 500)
    obstacleAvoidanceDistance = 500;
  
  if (maxSpeed == 0) maxSpeed = 200;
  
  // derive the forward speed of the robot
  // from the free range in front of the robot
  targetX = freeSpaceInFront = obstaclesModel.getDistanceInCorridor(0.0, widthOfCorridor);
  targetX -= 300; 
  
  // perform clipping
  if (targetX > maxSpeed) targetX = maxSpeed;
  if (targetX < 0) targetX = 0;
  
  // default: head to where there's more free space
  double leftForward = obstaclesModel.getTotalFreeSpaceInSector(.7, 1.39, obstacleAvoidanceDistance);
  double rightForward = obstaclesModel.getTotalFreeSpaceInSector(-.7, 1.39, obstacleAvoidanceDistance);
  targetAngle =	farAwayAvoidance*2*(leftForward - rightForward) / (leftForward + rightForward);
  
  // if there's space in front, turn towards goal
  // also turn for goal if corridor in the direction of the goal is free
  if (freeSpaceInFront > obstacleAvoidanceDistance)
  {
    // determine the next free sector in the desired direction
    double nextFree = obstaclesModel.getAngleOfNextFreeSector(pi/4, angleToDestination, (int )obstacleAvoidanceDistance);
    
    // check if there is enough space in the direction of "nextFree"
    if (obstaclesModel.getDistanceInCorridor(nextFree, widthOfCorridor) > obstacleAvoidanceDistance)
      {
      targetAngle = nextFree;
      mode = 1;
      }
    }
  if (obstaclesModel.getDistanceInCorridor(angleToDestination, widthOfCorridor) > obstacleAvoidanceDistance)
    {
    targetAngle = angleToDestination;
    mode = 2;
  }
  
  //OUTPUT(idText, text, "mode " << mode << ", a: " << targetAngle << ", v: " << targetX);
  
  if(obstaclesModel.getDistanceInCorridor((double)sgn(targetAngle) * pi/2, 300) < 100)
  {
    // if no turning is done, at least the forward speed should be 
    // high!!
    targetX = 50;
    targetAngle = 0;  
  }
  
  motionRequest.motionType = MotionRequest::walk;
  motionRequest.walkType = MotionRequest::normal;
  motionRequest.walkParams.translation.x = speedX.approximateVal(targetX);
  motionRequest.walkParams.translation.y = speedY.getVal();
  motionRequest.walkParams.rotation = speedPhi.approximateVal(targetAngle);
}








void ATH2004BasicBehaviorAvoidObstacles::execute()
{
  if(turnDistance == 0) turnDistance = 300;
  if(walkSpeed == 0) walkSpeed = 250;
  if(turnSpeed == 0) turnSpeed = 1.2;
  
  double distanceToObstacle = 200;
  if(stopDistance) distanceToObstacle = stopDistance;
  
  static PIDsmoothedValue speedPhi(0, .1, 0, 0, -pi/4, pi/4, pi/10);
  static PIDsmoothedValue speedX(0, .1, 0, 0, -walkSpeed, walkSpeed, 100);
  static PIDsmoothedValue speedY(0, .1, 0, 0, -walkSpeed/2, walkSpeed/2, 100);
  static double turnDirection = 0;
  speedX.setMax(walkSpeed);
  
  speedX.approximateVal(obstaclesModel.corridorInFront - distanceToObstacle);
  
  /*    
  // if something is in the way but far away
  // or it is close and turning was not set...
  if(((obstaclesModel.distance[ObstaclesModel::front] < turnDistance) && 
  (obstaclesModel.distance[ObstaclesModel::front] >= distanceToObstacle)) 
  ||
  ((obstaclesModel.distance[ObstaclesModel::front] < turnDistance) && 
  (turnDirection == 0)))
  {
  if(obstaclesModel.distance[ObstaclesModel::frontLeft] < obstaclesModel.distance[ObstaclesModel::frontRight])
  turnDirection = -turnSpeed;
  else 
  turnDirection = turnSpeed;
  }
  // if nothing is in the way, walk forward
  else if (obstaclesModel.distance[ObstaclesModel::front] > turnDistance) 
  {
  }
  */  
  turnDirection = 0;
  if(obstaclesModel.corridorInFront < turnDistance) 
  {
    if(obstaclesModel.getDistanceInMajorDirection(ObstaclesModel::frontLeft) < obstaclesModel.getDistanceInMajorDirection(ObstaclesModel::frontRight))
      speedY.approximateVal(-walkSpeed);
    else if (obstaclesModel.getDistanceInMajorDirection(ObstaclesModel::frontLeft) > obstaclesModel.getDistanceInMajorDirection(ObstaclesModel::frontLeft))
      speedY.approximateVal(walkSpeed);
    else if (obstaclesModel.getDistanceInMajorDirection(ObstaclesModel::frontLeft) == obstaclesModel.getDistanceInMajorDirection(ObstaclesModel::frontRight))
      speedY.approximateVal(0);
  }
  else  
    speedY.approximateVal(0);
  
  motionRequest.motionType = MotionRequest::walk;
  motionRequest.walkType = MotionRequest::normal;
  motionRequest.walkParams.translation.x = speedX.getVal();
  motionRequest.walkParams.translation.y = speedY.getVal();
  motionRequest.walkParams.rotation = speedPhi.approximateVal(turnDirection);
}


void ATH2004BasicBehaviorGoToPoint::execute()
{
  accelerationRestrictor.saveLastWalkParameters();
  
  Vector2<double> destination(x,y);
  const Vector2<double>& self = robotPose.translation;
  
  if (maxSpeed==0) maxSpeed = 120;
  
  double maxTurnSpeed = fromDegrees(40);
  
  double distanceToDestination = Geometry::distanceTo(self,destination);
  
  double angleDifference = normalize(fromDegrees(destinationAngle) - robotPose.rotation);
  
  double angleToDestination = Geometry::angleTo(robotPose,destination);
  
  
  motionRequest.motionType = MotionRequest::walk;
  motionRequest.walkType = MotionRequest::normal;
  
  //this time does not necessarily include time for turning!:
  double estimatedTimeToReachDestination;
  if (distanceToDestination > 200)
  {
    estimatedTimeToReachDestination = (distanceToDestination+200)/ maxSpeed;
    
    motionRequest.walkParams.translation.x = cos(angleToDestination) * maxSpeed;
    motionRequest.walkParams.translation.y = sin(angleToDestination) * maxSpeed;
  }
  else
  {
    estimatedTimeToReachDestination = 2*distanceToDestination / maxSpeed;
    if (distanceToDestination > 30)
    {
      motionRequest.walkParams.translation.x = 
        cos(angleToDestination) * maxSpeed*distanceToDestination/200;
      motionRequest.walkParams.translation.y = 
        sin(angleToDestination) * maxSpeed*distanceToDestination/200;
    }
    else
    {
      motionRequest.walkParams.translation.x = 0;
      motionRequest.walkParams.translation.y = 0;
    }
  }
  if (estimatedTimeToReachDestination==0)
  {
    estimatedTimeToReachDestination = 0.001;
  }
  
  if (fabs(toDegrees(angleDifference)) > 20)
  {
    motionRequest.walkParams.rotation = 
      angleDifference / estimatedTimeToReachDestination;
    if (motionRequest.walkParams.rotation > maxTurnSpeed)
    {
      motionRequest.walkParams.rotation = maxTurnSpeed;
    }
    if (motionRequest.walkParams.rotation < -maxTurnSpeed) 
    {
      motionRequest.walkParams.rotation = -maxTurnSpeed;
    }
  }
  else
  {
    motionRequest.walkParams.rotation = 2*angleDifference;
  }
  
  
  
  if ((fabs(toDegrees(angleDifference))<4)&&(distanceToDestination<25))
  {
    motionRequest.walkParams.translation.x = 0;
    motionRequest.walkParams.translation.y = 0;
    motionRequest.walkParams.rotation = 0;
  }
  
  accelerationRestrictor.restrictAccelerations(150,150,100);
  
  if (fabs(motionRequest.walkParams.translation.x) < 10
    && fabs(motionRequest.walkParams.translation.y) < 10
    && fabs(motionRequest.walkParams.rotation) < fromDegrees(10)
    && fabs(distanceToDestination) < 40
    && fabs(angleToDestination) < fromDegrees(5))
  {
    motionRequest.motionType = MotionRequest::stand;
  }
}

void ATH2004BasicBehaviorGoToPointFast::execute()
{
  accelerationRestrictor.saveLastWalkParameters();
  
  Vector2<double> destination(x,y);
  const Vector2<double>& self = robotPose.translation;
  
  if (maxSpeed==0) maxSpeed = 120;
  
  double maxTurnSpeed = fromDegrees(40);
  
  double distanceToDestination = Geometry::distanceTo(self,destination);
  
  double angleDifference = normalize(fromDegrees(destinationAngle) - robotPose.rotation);
  
  double angleToDestination = Geometry::angleTo(robotPose,destination);
  
  if (distanceToDestination < 200) maxSpeed = 100;
  
  motionRequest.motionType = MotionRequest::walk;
  motionRequest.walkType = MotionRequest::normal;
  
  //this time does not necessarily include time for turning!:
  double estimatedTimeToReachDestination;
  
  estimatedTimeToReachDestination = (distanceToDestination+200)/ maxSpeed;
  
  motionRequest.walkParams.translation.x = cos(angleToDestination) * maxSpeed;
  motionRequest.walkParams.translation.y = sin(angleToDestination) * maxSpeed;
  
  if (fabs(toDegrees(angleDifference)) > 20)
  {
    motionRequest.walkParams.rotation = 
      angleDifference / estimatedTimeToReachDestination;
    if (motionRequest.walkParams.rotation > maxTurnSpeed)
    {
      motionRequest.walkParams.rotation = maxTurnSpeed;
    }
    if (motionRequest.walkParams.rotation < -maxTurnSpeed) 
    {
      motionRequest.walkParams.rotation = -maxTurnSpeed;
    }
  }
  else
  {
    motionRequest.walkParams.rotation = 2*angleDifference;
  }
  
  // accelerationRestrictor.restrictAccelerations(300,300,100);
}

void ATH2004BasicBehaviorGoToPointPrecisely::execute()
{
  Pose2D pose = robotPose + Pose2D(Vector2<double>(-105,0));
  
  accelerationRestrictor.saveLastWalkParameters();
  
  Vector2<double> destination = (Pose2D(fromDegrees(destinationAngle), Vector2<double>(x,y)) + Pose2D(Vector2<double>(-105,0))).translation;
  const Vector2<double>& self = pose.translation;
  
  if (maxSpeed==0) maxSpeed = 120;
  
  double maxTurnSpeed = fromDegrees(40);
  
  double distanceToDestination = Geometry::distanceTo(self,destination);
  
  double angleDifference = normalize(fromDegrees(destinationAngle) - pose.rotation);
  
  double angleToDestination = Geometry::angleTo(pose,destination);
  
  
  motionRequest.motionType = MotionRequest::walk;
  motionRequest.walkType = MotionRequest::normal;
  
  //this time does not necessarily include time for turning!:
  double estimatedTimeToReachDestination;
  if (distanceToDestination > 200)
  {
    estimatedTimeToReachDestination = (distanceToDestination+200)/ maxSpeed;
    
    motionRequest.walkParams.translation.x = cos(angleToDestination) * maxSpeed;
    motionRequest.walkParams.translation.y = sin(angleToDestination) * maxSpeed;
  }
  else
  {
    estimatedTimeToReachDestination = 2*distanceToDestination / maxSpeed;
    motionRequest.walkParams.translation.x = 
      cos(angleToDestination) * maxSpeed*distanceToDestination/200;
    motionRequest.walkParams.translation.y = 
      sin(angleToDestination) * maxSpeed*distanceToDestination/200;
    const double minSpeed = 30;
    if(fabs(motionRequest.walkParams.translation.x) > fabs(motionRequest.walkParams.translation.y) &&
      fabs(motionRequest.walkParams.translation.x) < minSpeed)
    {
      motionRequest.walkParams.translation.x = motionRequest.walkParams.translation.x > 0 ? minSpeed : -minSpeed;
      motionRequest.walkParams.translation.x *= minSpeed / fabs(motionRequest.walkParams.translation.x);
    }
    else if(fabs(motionRequest.walkParams.translation.x) <= fabs(motionRequest.walkParams.translation.y) &&
      fabs(motionRequest.walkParams.translation.y) < minSpeed)
    {
      motionRequest.walkParams.translation.y = motionRequest.walkParams.translation.y > 0 ? minSpeed : -minSpeed;
      motionRequest.walkParams.translation.x *= minSpeed / fabs(motionRequest.walkParams.translation.y);
    }
    
  }
  if (estimatedTimeToReachDestination==0)
  {
    estimatedTimeToReachDestination = 0.001;
  }
  
  if (fabs(toDegrees(angleDifference)) > 20)
  {
    motionRequest.walkParams.rotation = 
      angleDifference / estimatedTimeToReachDestination;
    if (motionRequest.walkParams.rotation > maxTurnSpeed)
    {
      motionRequest.walkParams.rotation = maxTurnSpeed;
    }
    if (motionRequest.walkParams.rotation < -maxTurnSpeed) 
    {
      motionRequest.walkParams.rotation = -maxTurnSpeed;
    }
  }
  else
  {
    motionRequest.walkParams.rotation = 2*angleDifference;
  }
  
  accelerationRestrictor.restrictAccelerations(150,150,100);
  
  if (fabs(motionRequest.walkParams.translation.x) < 10
    && fabs(motionRequest.walkParams.translation.y) < 10
    && fabs(motionRequest.walkParams.rotation) < fromDegrees(10)
    && fabs(distanceToDestination) < 40
    && fabs(angleToDestination) < fromDegrees(5))
  {
    motionRequest.motionType = MotionRequest::stand;
  }
}


void ATH2004BasicBehaviorDribbleAtBorder::execute()
{
  
  
  //Ballposition
  Vector2<double> ball = ballPosition.seen;
  
  double ballDistance = Geometry::distanceTo(robotPose.getPose(),ball);
  // unused double ballX = ballDistance * cos(Geometry::angleTo(robotPose.getPose(),ball));
  double ballY = ballDistance * sin(Geometry::angleTo(robotPose.getPose(),ball));
  
  /* Some requests */
  motionRequest.walkType = MotionRequest::normal;
  motionRequest.walkParams.translation.x = 200;
  
  /* setting the angle of the robot depending on its position to the opponents ground line */
  double destinationAngle;
  destinationAngle = (robotPose.translation.x - xPosHalfWayLine) * 90 / (xPosOpponentGroundline - xPosHalfWayLine);
  
  if (robotPose.translation.y < 0)
  {
    /* robot is on the left side of field and uses the left foot for dribbling*/  
    motionRequest.walkParams.translation.y = ballY + 70;
  }
  else
  {
    /* robot is on the left side of field  and uses the right foot for dribbling*/  
    motionRequest.walkParams.translation.y = ballY - 70;
  }
  
  
  
  {
    
    
    if (motionRequest.walkParams.translation.y > 100) 
    {
      motionRequest.walkParams.translation.y = 100;
      
      if (robotPose.translation.x > xPosOpponentPenaltyArea)
      {
        motionRequest.walkParams.rotation = - robotPose.getAngle() + fromDegrees(destinationAngle);
      }
      else
      {
        motionRequest.walkParams.rotation = 0;
      }
      
    }
    
    
    
    if (motionRequest.walkParams.translation.y < -100)
    {
      motionRequest.walkParams.translation.y = -100;
      
      
      if (robotPose.translation.x > xPosOpponentPenaltyArea)
      {
        
        motionRequest.walkParams.rotation = - robotPose.getAngle() + fromDegrees(- destinationAngle);
      }
      else
      {
        motionRequest.walkParams.rotation = 0;
      }
      
    }
    
    //motionRequest.walkParams.translation.y = 0;
    motionRequest.motionType = MotionRequest::walk;
  }
}

void ATH2004BasicBehaviorDribbleInCenterOfField::execute()
{
  if(!((int)speed))
    speed = 300;
  //Ballposition
  Vector2<double> ball = ballPosition.seen;

  Vector2<double> goalPos(2000,0);
  

  double ballDistance = Geometry::distanceTo(robotPose.getPose(),ball);
  double ballX = ballDistance * cos(Geometry::angleTo(robotPose.getPose(),ball));
  double ballY = ballDistance * sin(Geometry::angleTo(robotPose.getPose(),ball));
  double sideOffset = 76;

  if(!side)
    side = 1;
  
  if((ball.y < -500))
    side = 1;
  
  if((ball.y > 500))
    side = -1;

  if(modi != 2)
	direction = 361;
  
  if(robotPose.translation.x > 1800)
	  speed = 100;
	else
		if(robotPose.translation.x > 1600)
			speed = 200;
			else
				speed = 300;

  double targetX = ballX - 80;
  double targetY = ballY + sideOffset * side;
  
  double scale = 1;
  double threshold = 80;
  
  if(fabs(targetY) > threshold)
    scale = 0;
  else
    scale = (1-fabs(targetY/threshold))*threshold;
  
  
  double targetDistance = sqrt(sqr(targetX)+sqr(targetY));
  
  double unscale = (sqr(targetX)*(sqr(targetDistance)+scale))/(sqr(targetX)+scale)-sqr(targetDistance);
  
  double speedX = speed * (sqr(targetX/(targetDistance+1+unscale)))*sgn(targetX);
  double speedY = speed * (sqr(targetY/(targetDistance+1+scale)))*sgn(targetY);
  
  // ---------rotation calculation-------
  switch((int)modi){
	// modi 1: dribble parallel to x-axis in direction of opponent goal
	case 1: rotationM = - robotPose.getAngle();
			break;
	// modi 2: fix the init-direction of robot as the dribble-direction
	case 2: if(direction == 361) // 361 is the init-value if direction
				direction = robotPose.getAngle();

			rotationM = direction - robotPose.getAngle();
			break;

	// modi 3: dribble in the direction of opponent goal
	case 3: if(sqrt(sqr(targetY)+sqr(targetX)) < 250)
				rotationM = ((goalPos-ball).angle()-robotPose.getAngle());
			else
				if(sqrt(sqr(targetY)+sqr(targetX)) > 350)
					rotationM = - robotPose.getAngle();
				else
					if(!rotationM)
						rotationM = - robotPose.getAngle();
			break;
	//dribble without direction correction
	default: rotationM = 0;
			 break;  // :)
  }//end switch
  // ------------------------------------

  if(fabs(targetY) < 60)
    speedX = speed - speedY;
  

  motionRequest.walkParams.translation.x = speedX;
  motionRequest.walkParams.translation.y = speedY;
  motionRequest.walkParams.rotation = rotationM;
  
  motionRequest.motionType = MotionRequest::walk;
}//end DribbleInCenterOfField

void ATH2004BasicBehaviorGetBehindBall2::execute()
{
  Vector2<double> oppositeGoal(xPosOpponentGroundline, yPosCenterGoal);
/*
  if(!((int)speed))
    speed = 300;
  //Ballposition
  Vector2<double> ball = ballPosition.seen;  
  
  double sideOffset = 76;

  if((ball.y < -500))
    side = 1;
  
  if((ball.y > 500))
    side = -1;
  
  double targetX = ballX - 60;
  double targetY = ballY + sideOffset * side;
*/
  
  motionRequest.walkParams.translation.x = 0;
  motionRequest.walkParams.translation.y = 0;
  motionRequest.walkParams.rotation = 0;
  
  motionRequest.motionType = MotionRequest::walk;
}//end GetBehindBall2

 /*
 * Change Log
 * 
 * $Log: ATH2004SimpleBasicBehaviors.cpp,v $
 * Revision 1.19  2004/05/04 10:48:58  loetzsch
 * replaced all enums
 * xxxBehaviorControlTimeAfterWhichCommunicatedBallsAreAccepted
 * by
 * behaviorControlTimeAfterWhichCommunicatedBallsAreAccepted
 * (this mechanism was neither fully implemented nor used)
 *
 * Revision 1.18  2004/03/16 14:00:17  juengel
 * Integrated Improvments from "Gnne"
 * -ATH2004ERS7Behavior
 * -ATHHeadControl
 * -KickSelectionTable
 * -KickEditor
 *
 * Revision 1.2  2004/03/15 17:11:38  hoffmann
 * - added ATH2004HeadControl
 * - added ATH2004LEDControl
 * - headmotiontester shows "tilt2"
 * - motion process updates odometry while no new robotPose is received, added to motion request
 * - some ui adjustments
 * - added member function to "field" to find out if robot is in own penalty area for use in the obstacles locator
 *
 * Revision 1.17  2004/03/02 10:10:10  mellmann
 * 3 dribble-modes in DribbleInCenterOfField
 *
 * Revision 1.16  2004/02/03 13:20:47  spranger
 * renamed all references to  class BallPosition to BallModel (possibly changed include files)
 *
 * Revision 1.15  2004/01/23 16:32:54  roefer
 * Warnings removed
 *
 * Revision 1.14  2004/01/13 18:33:26  heinze
 * worked at dribble-at-border
 *
 * Revision 1.13  2004/01/08 13:51:48  loetzsch
 * added go-to-point-fast
 *
 * Revision 1.12  2003/12/17 13:08:40  heinze
 * worked at dribble-at-border
 * - trying to fix problems
 *
 * Revision 1.11  2003/12/11 17:45:48  heinze
 * worked at dribble-at-border
 * - added: turning with corner
 *
 * Revision 1.9  2003/12/11 17:11:29  mellmann
 * worked on dribbleInCenterOfField
 *
 * Revision 1.8  2003/12/09 21:44:53  mellmann
 * completed DribbleInCenterOfField
 *
 * Revision 1.7  2003/12/09 16:29:06  jhoffman
 * removed OUTPUTs
 *
 * Revision 1.6  2003/12/04 17:22:54  mellmann
 * Added "GetBehindBall2" and "DribbleInCenterOfField".
 *
 * Revision 1.5  2003/12/04 13:46:29  heinze
 * worked at dribble-at-border
 *
 * Revision 1.4  2003/11/25 19:53:36  heinze
 * added basic behavior dribble-at-border
 *
 * Revision 1.3  2003/10/28 13:57:45  dueffert
 * unused evolution removed
 *
 * Revision 1.2  2003/10/28 13:25:52  dueffert
 * spelling improved
 *
 * Revision 1.1  2003/10/26 22:49:34  loetzsch
 * created ATH2004BehaviorControl from GT2003BehaviorControl
 *  - strongly simplified option graph
 *  - moved some symbols from GT2003 to CommonXabsl2Symbols
 *  - moved some basic behaviors from GT2003 to CommonXabsl2BasicBehaviors
 *
 * cloned ATH2004 three times (BB2004, DDD2004, MSH2004)
 *
*/

