/**
* @file ATH2004ContinuousRules/GoBetweenBallAndOwnGoalClear.h
*
* @author Dirk Thomas
*/

#ifndef __ATH2004GoBetweenBallAndOwnGoalClear_h_
#define __ATH2004GoBetweenBallAndOwnGoalClear_h_

#include "Tools/ContinuousBasicBehaviors/ContinuousRule.h"
#include "Tools/FieldDimensions.h"

namespace ATH2004ContinuousRules
{

/**@class GoBetweenBallAndOwnGoal
 */
class GoBetweenBallAndOwnGoalClear: public ContinuousRule{
private:
	/** the distance above which the result vector is maxForce in mm*/
  double maxForceLength;
  /** the distance below which the result vector is 0 in mm*/
  double minDistance;
  /** the maximum attracting force */
  double maxForce;

  /** the distance from the ball */
  double distanceToBall;

public:

  /** constructor
   */
  GoBetweenBallAndOwnGoalClear(
    const BehaviorControlInterfaces& interfaces,
    double maxForce = 1.0,
    double minDistance = 0.0,
    double maxForceLength = 350.0,
    double distanceToBall = 100.0
		)
    :
    ContinuousRule(interfaces,"GoBetweenBallAndOwnGoalClear"),
    maxForce(maxForce),
    minDistance(minDistance),
    maxForceLength(maxForceLength),
    distanceToBall(distanceToBall)
  {};

	/** executes the basic behavior rule
   * @param robotPose the current robots pose at which the rule is to be evaluated
   * @param walk the direction and speed of the suggested motion coded as an vector (output)
	 * @param ra the rotation angle, the direction the robot should be directed (output)
	 * @param rweight the rotation weight the weight of the suggested rotation (output)
	 */
  virtual void execute(const RobotPose& robotPose,
                       Vector2<double>& walk,
                       double& ra, double& rweight)
  {
    
    walk = ballPosition.seen - robotPose.translation;

    Vector2<double> ballToGoal(xPosOwnGroundline+50, yPosCenterGoal);
    ballToGoal -= ballPosition.seen;
    ballToGoal.normalize();

    // position between ball and own goal with a given distance to the ball
    ballToGoal *= distanceToBall;
		walk += ballToGoal;

    double distance = walk.abs();
    walk.normalize();

    walk *= getAttractingForce(distance, maxForce, minDistance, maxForceLength);

    ra = 0;
    rweight = 0;

  }
};

}


#endif //__ATH2004GoBetweenBallAndOwnGoal_h_


/*
* Change log:
*
* $Log: GoBetweenBallAndOwnGoalClear.h,v $
* Revision 1.3  2004/03/08 00:58:41  roefer
* Interfaces should be const
*
* Revision 1.2  2003/10/28 15:04:09  loetzsch
* fixed doxygen bugs
*
* Revision 1.1  2003/10/26 22:49:34  loetzsch
* created ATH2004BehaviorControl from GT2003BehaviorControl
*  - strongly simplified option graph
*  - moved some symbols from GT2003 to CommonXabsl2Symbols
*  - moved some basic behaviors from GT2003 to CommonXabsl2BasicBehaviors
*
* cloned ATH2004 three times (BB2004, DDD2004, MSH2004)
*
*/
