/**
 * @file MSH2004BallLocatorElement.cpp
 * 
 * This file contains a container class for ball-localization.
 *
 * @author <a href="mailto:grypho@tempus-vivit.net">Carsten Schumann</a>
 */
#include "MSH2004BallLocatorElement.h"
#include "Tools/Debugging/Debugging.h"

MSH2004BallLocatorElement::MSH2004BallLocatorElement(const BallPercept& ballPercept, const RobotPose& robotPose, const OdometryData& odometryPose,const long currentFrame){
  this->systemTime=SystemCall::getCurrentSystemTime();
  this->ballPercept = ballPercept;
  this->pose=robotPose;
  this->correctedPose=robotPose;
  this->odometry=odometryPose;
  this->ballMovement.x=0;
  this->ballMovement.y=0;
  this->ballMovementPerMS.x=0;
  this->ballMovementPerMS.y=0;
  this->ballOnField.x=0;
  this->ballOnField.y=0;
  this->systemFrame=currentFrame;
  this->maxAgeMS=3000;
}

Vector2<double> MSH2004BallLocatorElement::getBallpositionOnField(){
	return this->ballOnField;
}

Pose2D MSH2004BallLocatorElement::getCorrectedPose(){
	return this->correctedPose;
}

Pose2D MSH2004BallLocatorElement::getPose(){
	return this->pose;
}

OdometryData MSH2004BallLocatorElement::getOdometry(){
	return this->odometry;
}

Vector2<double> MSH2004BallLocatorElement::getBallMovement(){
	return this->ballMovement;
}
Vector2<double> MSH2004BallLocatorElement::getBallMovementPerMS(){
	return this->ballMovementPerMS;
}

void MSH2004BallLocatorElement::clearTempData(){
  	this->ballMovement.x=0;
    this->ballMovement.y=0;
  	this->ballMovementPerMS.x=0;
    this->ballMovementPerMS.y=0;
    this->ballOnField.x=0;
    this->ballOnField.y=0;
    this->correctedPose.translation.x=0;
    this->correctedPose.translation.y=0;
}

double MSH2004BallLocatorElement::getRotationShiftoff(){
	return this->getOdometry().rotation-this->getPose().rotation;
}

long MSH2004BallLocatorElement::getAge(int currentFrame){
	#ifdef APERIOS1_3_2
		return SystemCall::getCurrentSystemTime()-this->systemTime;
	#else
	#ifdef _WIN32
		return (currentFrame-this->systemFrame)*100; //100ms per frame = 10 fps
	#else 
		#error "Unknown Operating System"
	#endif
	#endif
}

void MSH2004BallLocatorElement::recalculatePose(MSH2004BallLocatorElement* futuralElement,Matrix2x2<double>& shiftoffMatrix){
  //divide futural position into three ordinates
  double robotx=futuralElement->getCorrectedPose().translation.x;
  double roboty=futuralElement->getCorrectedPose().translation.y;
  double robotr=futuralElement->getCorrectedPose().rotation;
    // get ringbuffer odometry at entry i and i-1. 

	const OdometryData& odo_f=futuralElement->getOdometry();
    const OdometryData& odo_now=this->getOdometry();

    // calculate movement that is needeed to move the robot from now to the futural position
    double deltax=odo_f.translation.x - odo_now.translation.x;
    double deltay=odo_f.translation.y - odo_now.translation.y;
    double deltar=odo_f.rotation - odo_now.rotation;

    //use movement and transform coordinate system to coordinate system of field. 
    //after that, apply movement to (by selflocator calculated) robot position to
    //get robot position at entry i-1

    robotx+=deltax*shiftoffMatrix.c[0].x+deltay*shiftoffMatrix.c[1].x;
    roboty+=deltax*shiftoffMatrix.c[0].y+deltay*shiftoffMatrix.c[1].y;
    robotr+=deltar;

    //save this into buffer for further calculations
    this->correctedPose.translation.x=robotx;
    this->correctedPose.translation.y=roboty;
    this->correctedPose.rotation=robotr;

}
void MSH2004BallLocatorElement::recalculateBallOnField(){
  //~ double ballAngle=this->ballPercept.getAngle();
  //~ double ballDistance=this->ballPercept.getDistance();

  double ballAngle=this->ballPercept.getAngleIntrinsicBased();
  double ballDistance=this->ballPercept.getDistanceIntrinsicBased();

  this->ballOnField.x=ballDistance*cos(ballAngle+this->correctedPose.rotation)+this->correctedPose.translation.x;
  this->ballOnField.y=ballDistance*sin(ballAngle+this->correctedPose.rotation)+this->correctedPose.translation.y;

}

void MSH2004BallLocatorElement::recalculateBallMovement(MSH2004BallLocatorElement* futuralElement,int currentFrame){
  const Vector2<double>& ballOnFieldFrom=this->ballOnField;
  const Vector2<double>& ballOnFieldTo=futuralElement->getBallpositionOnField();
  long perceptTime=this->getAge(currentFrame);
  long perceptTimeLast=futuralElement->getAge(currentFrame);
  if (perceptTime != perceptTimeLast){

    ballMovement.x=(ballOnFieldTo.x-ballOnFieldFrom.x);
    ballMovement.y=(ballOnFieldTo.y-ballOnFieldFrom.y);
    ballMovementPerMS=ballMovement/(perceptTime-perceptTimeLast);
    setMovementValidity(1.0);
  } else {
    setMovementValidity(0.0);
  }
}

bool MSH2004BallLocatorElement::entryRecent(int currentFrame){
	return this->getAge(currentFrame)<this->maxAgeMS;
}

bool MSH2004BallLocatorElement::movementInvertedTo(MSH2004BallLocatorElement* other){
  const Vector2<double>& m1=this->ballMovement;
  if (m1.abs()==0){
    return true;
  }
  const Vector2<double>& m2=other->getBallMovement();
  
  double coslambda=(m1.x*m2.x + m1.y*m2.y) / (sqrt(sqr(m1.x)+sqr(m1.y))*sqrt(sqr(m2.x)+sqr(m2.y)));
  double lengthratio=m2.abs()/m1.abs();
  

/*  if (coslambda<-0.8 && coslambda>-0.96){
    OUTPUT(idText,text,"lambda="<<coslambda);
  }*/
  return (coslambda<-0.9) && (lengthratio > 0.5) && (lengthratio < 2.0);



}


double MSH2004BallLocatorElement::getMovementValidity(){
	return this->movementValidity;
}

void MSH2004BallLocatorElement::setMovementValidity(double value){
	this->movementValidity=value;
}

/*
 * Change log : 
 * $Log: MSH2004BallLocatorElement.cpp,v $
 * Revision 1.4  2004/04/08 15:33:00  wachter
 * GT04 checkin of Microsoft-Hellounds
 *
 * Revision 1.3  2004/01/29 14:52:37  schumann
 * changed constant
 *
 * Revision 1.2  2004/01/22 20:08:00  schumann
 * added validity for movement
 * added elemination of jumping ball
 * improved calculation of ball movement
 *
 * Revision 1.1  2004/01/21 11:48:52  schumann
 * Added motion extrapolation and cleaned up balllocator
 *
*/
