/**
* @file KalmanBallLocator.cpp
* Contains a BallLocator implementation using Kalman filtering
*
* @author <a href="mailto:stefanuhrig@gmx.net">Stefan Uhrig</a>
*/
//------------------------------------------------------------------------------
#include "KalmanBallLocator.h"
#include "Tools/Debugging/Debugging.h"
#include "Tools/Actorics/RobotDimensions.h"
#include "Tools/FieldDimensions.h"
#include "Platform/SystemCall.h"
#include "Tools/Debugging/DebugDrawings.h"
#include "Tools/Debugging/KalmanDebugData.h"
#include "Platform/GTAssert.h"
#include "KalmanProcessModels/FixedPositionModel.h"
#include "KalmanProcessModels/ConstantSpeedModel.h"
//------------------------------------------------------------------------------
double KalmanBallLocator::dFieldDiagonalLength =
  1.2*sqrt((2.0*xPosOpponentGoal)*(2.0*xPosOpponentGoal) +
           (2.0*yPosLeftFlags)*(2.0*yPosLeftFlags));
//------------------------------------------------------------------------------
KalmanBallLocator::KalmanBallLocator(BallLocatorInterfaces& interfaces)
 : BallLocator(interfaces),
   ballSideDetector(ballPercept, robotPose, calibrationRequest, ballPosition)

{
  // Initialise member variables
  bSendProcessModelStates = false;
  ballWasSeen = false;
  x_abs = 0.0;
  y_abs = 0.0;
  x_rel = 0.0;
  y_rel = 0.0;
  vx_abs = 0.0;
  vy_abs = 0.0;
  vx_rel = 0.0;
  vy_rel = 0.0;
  
  // Add the models the filter is to work with
  // addModel(new KalmanFixedPositionModel);
  addModel(new KalmanConstantSpeedModel);
}
//------------------------------------------------------------------------------
KalmanBallLocator::~KalmanBallLocator()
{
  // Free the models
  destroyModels();
}
//------------------------------------------------------------------------------
void KalmanBallLocator::execute()
{
  // ++ own detection used ++
  // execute ballSideDetector
  // ballSideDetector.execute();
  
  // calculate error (why?)
  double ballMeasurementError = 159.1 * tan(openingAngleHeight_ERS210/2)/176*3;
  ballPosition.seen.distanceError = 
    ballMeasurementError *
    ballPercept.getDistance() * ballPercept.getDistance() /
    (ballRadius * 159.1 /* focal length ers219 */);
  ballPosition.seen.angleError = 0.05;
  
  // Distinguish between seen and unseen ball
  if (ballPercept.ballWasSeen)
  {
    handleSeenBall();
    ballWasSeen = true;
  }
  else
  {
    handleUnseenBall();
    ballWasSeen = false;
  }

  // Set frame number
  ballPosition.frameNumber = cameraMatrix.frameNumber;

  // Set ball state (none, rollByRight, rollByLeft)
  setBallState();

  // Send the results of the process models as debug messages
  if (bSendProcessModelStates)
    sendProcessModelStates();

  // Draw the ball position on the field
  drawBallPosition();
}
//------------------------------------------------------------------------------
void KalmanBallLocator::handleSeenBall()
{
  int i;

  // If ball wasn't seen the last time, set the time the ball was seen first
  if (!ballWasSeen)
    ballPosition.seen.timeWhenFirstSeenConsecutively = timeOfImageProcessing;

  // Time in seconds
  double time = (double)timeOfImageProcessing / 1000.0;

  // Get seen coordinates
  Vector2<double> ballOffset;
  ballPercept.getOffsetSizeBased(ballOffset);

  // Ignore impossible coordinates
  if (isNonSensePos(ballOffset.x, ballOffset.y))
  {
    handleUnseenBall();
    return;
  }

  // Relative coordinates to robot in meter
  x_rel = ballOffset.x / 1000.0;
  y_rel = ballOffset.y / 1000.0;

  // Determine change of odometry in meter
  /*
  Vector2<double> dTranslation =
    odometryData.translation - lastOdometry.translation;
  lastOdometry = odometryData;
  dTranslation.x /= 1000.0;
  dTranslation.y /= 1000.0;
  */

  // ++odometry is already relative++
  /*
  // Adapt translation to robot rotation
  double trans_x_rel = dTranslation.x*cos(robotPose.rotation) -
                       dTranslation.y*sin(robotPose.rotation); 
  double trans_y_rel = dTranslation.x*sin(robotPose.rotation) +
                       dTranslation.y*cos(robotPose.rotation);
  dTranslation.x = trans_x_rel;
  dTranslation.y = trans_y_rel;
  */

  // Update process models
  double liklihoodSum = 0.0;
  try
  {
    for (i = 0; i < (int)kalmanModels.size(); ++i)
    {
      // Send change of odometry to model
      kalmanModels[i]->adapt(lastOdometry, odometryData);

      // Do update and save result
      kalmanUpdateResults[i] = kalmanModels[i]->update(time, x_rel, y_rel);
      liklihoodSum += kalmanUpdateResults[i].liklihood;

      // Reset process model if result is not good
      if (kalmanModels[i]->isNonSensePos(kalmanUpdateResults[i].state.x,
                                         kalmanUpdateResults[i].state.y))
        kalmanModels[i]->reset();
    }
  }
  catch (...)
  {
    // Here we should never end, but better safe than sorry
    OUTPUT(idText, text,
           "Unknown Exception in KalmanBallLocator::handleSeenBall");
    setUnknownResult();
    return;
  }
  lastOdometry = odometryData;

  // Get weighted position and speed
  if (liklihoodSum != 0.0)
  {
    x_rel = 0.0;
    y_rel = 0.0;
    vx_rel = 0.0;
    vy_rel = 0.0;
    double probability;

    std::vector<KalmanUpdateResult>::iterator pos;
    for (pos = kalmanUpdateResults.begin();
         pos != kalmanUpdateResults.end();
         ++pos)
    {
      if (pos->liklihood != 0.0)
        probability = pos->liklihood = pos->liklihood / liklihoodSum;
      else
        probability = 0.0;
      x_rel += pos->state.x * probability;
      y_rel += pos->state.y * probability;
      vx_rel += pos->state.vx * probability;
      vy_rel += pos->state.vy * probability;
    }

    // Change meter to millimeter and meter per second to millimeter per second
    x_rel *= 1000.0;
    y_rel *= 1000.0;
    vx_rel *= 1000.0;
    vy_rel *= 1000.0;

    // Save last seen position
    last_seen_x_rel = x_rel;
    last_seen_y_rel = y_rel;

    // Translate relative coordinates to robot into field coordinates
    ballOffset = Geometry::relative2FieldCoord(robotPose, x_rel, y_rel);
    x_abs = ballOffset.x;
    y_abs = ballOffset.y;
    vx_abs =  vx_rel*cos(robotPose.rotation) + vy_rel*sin(robotPose.rotation); 
    vy_abs = -vx_rel*sin(robotPose.rotation) + vy_rel*cos(robotPose.rotation);
    
    // Set seen ball data
    ballPosition.seen = ballOffset;
    
    // ++changed to relative values++
    ballPosition.seen.speed.x = vx_rel;
    ballPosition.seen.speed.y = vy_rel;

    // Set the propagated data, too
    ballPosition.propagated.timeOfObservation = timeOfImageProcessing;
    ballPosition.propagated.x = ballOffset.x;
    ballPosition.propagated.y = ballOffset.y;
    ballPosition.propagated.positionWhenLastObserved = ballOffset;
    
    // ++changed to relative values++
    ballPosition.propagated.observedSpeed.x = vx_rel;
    ballPosition.propagated.observedSpeed.y = vy_rel;
  }
  else
  {
    // Probably process models are not initialised yet

    // Change meter to millimeter
    x_rel *= 1000.0;
    y_rel *= 1000.0;

    // Translate relative coordinates to robot into field coordinates
    ballOffset = Geometry::relative2FieldCoord(robotPose, x_rel, y_rel);
    x_abs = ballOffset.x;
    y_abs = ballOffset.y;
    vx_rel = 0.0;
    vy_rel = 0.0;
    vx_abs = 0.0;
    vy_abs = 0.0;
    
    // Set seen (=measured) ball position, speed is unknown
    ballPosition.seen = ballOffset;
    ballPosition.seen.speed.x = 0.0;
    ballPosition.seen.speed.y = 0.0;
  }

  // Set times
  ballPosition.seen.timeWhenLastSeen = timeOfImageProcessing;
  ballPosition.seen.timeUntilSeenConsecutively = timeOfImageProcessing;
}
//------------------------------------------------------------------------------
void KalmanBallLocator::handleUnseenBall()
{
  // Use predict function of the models to predict the position and
  // speed of the ball

  // Time in seconds
  double time = (double)timeOfImageProcessing / 1000.0;
  
  // Determine change of odometry in meter
  /*
  Vector2<double> dTranslation =
    odometryData.translation - lastOdometry.translation;
  lastOdometry = odometryData;
  dTranslation.x /= 1000.0;
  dTranslation.y /= 1000.0;
  */

  // ++odometry is already relative++
  /*
  // Adapt translation to robot rotation
  double trans_x_rel = dTranslation.x*cos(robotPose.rotation) -
                       dTranslation.y*sin(robotPose.rotation); 
  double trans_y_rel = dTranslation.x*sin(robotPose.rotation) +
                       dTranslation.y*cos(robotPose.rotation);
  dTranslation.x = trans_x_rel;
  dTranslation.y = trans_y_rel;
  */

  int i;
  double liklihoodSum = 0.0;
  try
  {
    for (i = 0; i < (int)kalmanModels.size(); ++i)
    {
      // Send change of odometry to model
      kalmanModels[i]->adapt(lastOdometry, odometryData);

      // Predict next state
      kalmanPredictResults[i] = kalmanModels[i]->predict(time);
      liklihoodSum += kalmanPredictResults[i].liklihood;

      // Reset process model if results are not good
      if (kalmanModels[i]->isNonSensePos(kalmanPredictResults[i].state.x,
                                         kalmanPredictResults[i].state.y))
        kalmanModels[i]->reset();
    }
  }
  catch (...)
  {
    // Here we should never end, but better safe than sorry
    OUTPUT(idText, text,
           "Unknown Exception in KalmanBallLocator::handleUnseenBall");
    setUnknownResult();
    return;
  }
  lastOdometry = odometryData;

  // Get weighted position and speed
  if (liklihoodSum != 0.0)
  {
    x_rel = 0.0;
    y_rel = 0.0;
    vx_rel = 0.0;
    vy_rel = 0.0;
    double probability;

    std::vector<KalmanPredictResult>::iterator pos;
    for (pos = kalmanPredictResults.begin();
         pos != kalmanPredictResults.end();
         ++pos)
    {
      if (pos->liklihood != 0.0)
        probability = pos->liklihood = pos->liklihood / liklihoodSum;
      else
        probability = 0.0;
      x_rel += pos->state.x * probability;
      y_rel += pos->state.y * probability;
      vx_rel += pos->state.vx * probability;
      vy_rel += pos->state.vy * probability;
    }

    // Change meter to millimeter and meter per second to millimeter per second
    x_rel *= 1000.0;
    y_rel *= 1000.0;
    vx_rel *= 1000.0;
    vy_rel *= 1000.0;

    // Translate relative coordinates to robot into field coordinates
    Vector2<double> ballOffset =
      Geometry::relative2FieldCoord(robotPose, x_rel, y_rel);
    x_abs = ballOffset.x;
    y_abs = ballOffset.y;
    vx_abs =  vx_rel*cos(robotPose.rotation) + vy_rel*sin(robotPose.rotation); 
    vy_abs = -vx_rel*sin(robotPose.rotation) + vy_rel*cos(robotPose.rotation);

    // Set estimated position
    ballPosition.propagated.x = ballOffset.x;
    ballPosition.propagated.y = ballOffset.y;
  }
  else
  {
    // Probably process models are not initialised yet
    
    // Get time difference in seconds
    double dt = (timeOfImageProcessing -
                 ballPosition.propagated.timeOfObservation) / 1000.0;
    
    // Assume the ball is moving equally and set estimated position
    ballPosition.propagated.x =
      ballPosition.propagated.positionWhenLastObserved.x + 
      ballPosition.propagated.observedSpeed.x*dt;
    ballPosition.propagated.y =
      ballPosition.propagated.positionWhenLastObserved.y + 
      ballPosition.propagated.observedSpeed.y*dt;
  }

  // write last seen position to seen ball
  Vector2<double> ballOnField = Geometry::relative2FieldCoord(robotPose, last_seen_x_rel, last_seen_y_rel);
  ballPosition.seen.x = ballOnField.x;
  ballPosition.seen.y = ballOnField.y;
}
//------------------------------------------------------------------------------
void KalmanBallLocator::setBallState()
{
  // Ball must be approaching faster than 50 millimeters per second
  if (vx_rel > -50.0)
  {
    ballPosition.ballState = BallModel::none;
    return;
  }

  // Ball must be nearer than 1 meter
  double dist = sqrt(x_rel*x_rel + y_rel*y_rel);
  if (dist > 1000.0)
  {
    ballPosition.ballState = BallModel::none;
    return;
  }

  // Calculate y-position (side) of ball when the ball has the same height
  // as the robot und set appropriate ball state
  double y = y_rel - (vy_rel/vx_rel)*x_rel;
  if ((y < -100.0) && (y > -500.0))
  {
    ballPosition.ballState = BallModel::ballRollsByRight;
  }
  else if ((y > 100.0) && (y < 500.0))
  {
    ballPosition.ballState = BallModel::ballRollsByLeft;
  }
  else if ((y >= -100.0) && (y <= 100.0))
  {
    ballPosition.ballState = BallModel::ballRollsMiddle;
  }
  else 
  {
    ballPosition.ballState = BallModel::none;
  }    
}
//------------------------------------------------------------------------------
void KalmanBallLocator::setUnknownResult()
{
  // Sets the result to the last known results
  ballPosition.propagated.x = x_abs;
  ballPosition.propagated.y = y_abs;
}
//------------------------------------------------------------------------------
bool KalmanBallLocator::isNonSensePos(double x, double y)
{
  /*
  double diag = sqrt((2*xPosOpponentGoal)*(2*xPosOpponentGoal) +
                     (2*yPosLeftFlags)*(2*yPosLeftFlags));*/

  double dist = sqrt(x*x + y*y);
  if (dist > dFieldDiagonalLength)
    return true;
  else
    return false;
}
//------------------------------------------------------------------------------
void KalmanBallLocator::addModel(KalmanProcessModelBase* pModel)
{
  // Add the model and make room in the result vectors
  kalmanModels.push_back(pModel);
  kalmanUpdateResults.push_back(KalmanUpdateResult());
  kalmanPredictResults.push_back(KalmanPredictResult());
}
//------------------------------------------------------------------------------
void KalmanBallLocator::destroyModels()
{
  // Free the models
  std::vector<KalmanProcessModelBase*>::iterator pos;
  for (pos = kalmanModels.begin(); pos != kalmanModels.end(); ++pos)
    delete *pos;
}
//------------------------------------------------------------------------------
bool KalmanBallLocator::handleMessage(InMessage& message)
{
  // Handle only KalmanData messages
  if (message.getMessageID() != idKalmanData)
    return false;

  // Get type of action
  char action;
  message.bin >> action;

  switch (action)
  {
    case KalmanDebugData::idRequestParameters:
    {
      // Parameters of the process models were requested

      // Start message and send number of models
      getDebugOut().bin << (char)KalmanDebugData::idSendingParameters 
                        << kalmanModels.size();
      
      // Send the data of the models
      std::vector<KalmanProcessModelBase*>::iterator pos;
      for (pos = kalmanModels.begin(); pos != kalmanModels.end(); ++pos)
        (*pos)->toStream(getDebugOut().bin);
      
      // Finish message
      getDebugOut().finishMessage(idKalmanData);
    }
    return true;

    case KalmanDebugData::idSetParameters:
    {
      // The parameters of a model are to be set

      // Read index of the model
      int modelIndex;
      message.bin >> modelIndex;
      
      // Set data of the model
      kalmanModels[modelIndex]->fromStream(message.bin);
    }
    return true;
  }

  // Message is unknown, reset position in queue
  message.resetReadPosition();
  return false;
}
//------------------------------------------------------------------------------
void KalmanBallLocator::sendProcessModelStates()
{
  // Get debug stream
  OutMessage& stream = getDebugOut();

  // Write ID of message
  stream.bin << (char)KalmanDebugData::idFilterState;
  
  // Time stamp of the data
  stream.bin << timeOfImageProcessing;

  // Write robot position
  stream.bin << robotPose;

  // Write relative coordinates and speeds
  // (in millimeters and millimeters per second)
  stream.bin << x_rel << y_rel;
  stream.bin << vx_rel << vy_rel;

  // Write if ball was seen or not
  stream.bin << (char)ballWasSeen;

  // Write number of models
  stream.bin << kalmanUpdateResults.size();
  
  // Write result of models
  std::vector<KalmanUpdateResult>::const_iterator urpos;
  std::vector<KalmanPredictResult>::const_iterator prpos;
  std::vector<KalmanProcessModelBase*>::const_iterator mpos;
  for (mpos = kalmanModels.begin(),
         urpos = kalmanUpdateResults.begin(),
         prpos = kalmanPredictResults.begin();
       mpos != kalmanModels.end();
       ++mpos, ++urpos, ++prpos)
  {
    
    if (ballWasSeen)
    {
      // Write position in millimeters
      stream.bin << urpos->state.x*1000.0 << urpos->state.y*1000.0;
      // Write speed in millimeters per second
      stream.bin << urpos->state.vx*1000.0 << urpos->state.vy*1000.0;
      // Write probability (weight was converted to probability)
      stream.bin << urpos->liklihood;
    }
    else
    {
      // Write position in millimeters
      stream.bin << prpos->state.x*1000.0 << prpos->state.y*1000.0;
      // Write speed in millimeters per second
      stream.bin << prpos->state.vx*1000.0 << prpos->state.vy*1000.0;
      // Write probability (weight was converted to probability)
      stream.bin << prpos->liklihood;
    }

    // Write dimensioni of state
    int n = (*mpos)->getStateDim();
    stream.bin << n;
    
    // Write internal covarinace matrix of process model
    double* M = new double[n*n];
    (*mpos)->getP(M);
    int i;
    for (i = 0; i < n*n; ++i)
      stream.bin << M[i];
    delete [] M;
  }
  
  // Finish message
  stream.finishMessage(idKalmanData);
}
//------------------------------------------------------------------------------
void KalmanBallLocator::drawBallPosition()
{
  // Draw global positions
  if (ballWasSeen)
  {
    CIRCLE(ballLocatorField, x_abs, y_abs, 90, 3, 0, Drawings::orange);
    LINE(ballLocatorField, x_abs, y_abs, 
         x_abs + vx_abs*3, y_abs + vy_abs*3,
         3, 0, Drawings::orange);
  }
  else
  {
    CIRCLE(ballLocatorField, x_abs, y_abs, 90, 3, 0, Drawings::blue);
    LINE(ballLocatorField, x_abs, y_abs, 
         x_abs + vx_abs*3, y_abs + vy_abs*3,
         3, 0, Drawings::blue);
  }

  // Draw positions of models
  std::vector<KalmanUpdateResult>::const_iterator urpos;
  std::vector<KalmanPredictResult>::const_iterator prpos;
  std::vector<KalmanProcessModelBase*>::const_iterator mpos;
  for (mpos = kalmanModels.begin(),
         urpos = kalmanUpdateResults.begin(),
         prpos = kalmanPredictResults.begin();
       mpos != kalmanModels.end();
       ++mpos, ++urpos, ++prpos)
  {
    
    if (ballWasSeen)
    {
      // Calculate field coordinates
      Vector2<double> ballOffset =
        Geometry::relative2FieldCoord(robotPose, urpos->state.x*1000.0,
                                      urpos->state.y*1000.0);
      CIRCLE(ballLocatorField, ballOffset.x, ballOffset.y,
             60, 2, 0, Drawings::yellowOrange);
    }
    else
    {
      // Calculate field coordinates
      Vector2<double> ballOffset =
        Geometry::relative2FieldCoord(robotPose, prpos->state.x*1000.0,
                                      prpos->state.y*1000.0);
      CIRCLE(ballLocatorField, ballOffset.x, ballOffset.y,
             60, 2, 0, Drawings::skyblue);
    }
  }  
  
  DEBUG_DRAWING_FINISHED(ballLocatorField);
}
//------------------------------------------------------------------------------

/*
 * Change log :
 * $Log: KalmanBallLocator.cpp,v $
 * Revision 1.8  2004/05/03 15:32:25  uhrig
 * Added additional ball state and disabled sending of Kalman filter process model states.
 *
 * Revision 1.7  2004/04/27 09:09:22  uhrig
 * Using own side ball detection again
 *
 * Revision 1.6  2004/04/22 14:28:02  roefer
 * .NET errors/warnings removed
 *
 * Revision 1.5  2004/04/20 14:12:16  uhrig
 *  - odometry processing corrected
 *  - fixed position model disabled
 *  - covariance matrices tuned
 *
 * Revision 1.4  2004/04/07 12:28:56  risler
 * ddd checkin after go04 - first part
 *
 * Revision 1.5  2004/03/31 00:09:46  risler
 * set last seen position when ball is not seen
 *
 * Revision 1.4  2004/03/30 15:48:04  risler
 * various bugfixes from stefan
 *
 * Revision 1.1.1.1  2004/03/29 08:28:51  Administrator
 * initial transfer from tamara
 *
 * Revision 1.3  2004/03/27 21:51:32  risler
 * process model message sending disabled
 *
 * Revision 1.2  2004/03/15 12:28:52  risler
 * change log added
 *
 *
 */
