/**
* @file DDD2004BallLocator.cpp
* 
* This file contains the default class for ball-localization.
*
* @author <a href="mailto:brunn@sim.tu-darmstadt.de">Ronnie Brunn</a>
* @author <a href="mailto:mkunz@sim.tu-darmstadt.de">Michael Kunz</a>
*/

#include "DDD2004BallLocator.h"
#include "Tools/FieldDimensions.h"
#include "Platform/SystemCall.h"
#include "Tools/Debugging/DebugDrawings.h"
#include <math.h>
#include "Tools/Math/Matrix2x2.h"
#include "Platform/GTAssert.h"


DDD2004BallLocator::DDD2004BallLocator(const BallLocatorInterfaces& interfaces)
: BallLocator(interfaces),
ballPosX (0, 0.9, 0, 0, -xPosOpponentGoal, xPosOpponentGoal, 500000),
ballPosY (0, 0.9, 0, 0, -yPosLeftFlags, yPosLeftFlags, 500000),
ballSpeedX(0, 0.3, 0, 0, -500, 500, 100000),
ballSpeedY(0, 0.3, 0, 0, -500, 500, 100000),
ballAngle(0, 0.9, 0, 0, -pi, pi, 500000),
ballDistance(0, 0.5, 0, 0, 0, 10000),
consecutiveFramesBallSeen(0),
consecutiveFramesBallNotSeen(0),
ballSideDetector(ballPercept, robotPose, calibrationRequest, ballPosition)
{
  lastTimeBallSeen = 0;
}

/*
smoothing in the PID smoothed value should not be step based
but time based!
*/
void DDD2004BallLocator::execute()
{
  ballSideDetector.execute();

  double timeDiff = (double )(timeOfImageProcessing - lastTimeBallSeen)/1000; // 1/msec -> 1/sec
  lastTimeBallSeen = timeOfImageProcessing;
	Vector2<double> ballOnField;

  compensateOdometry();

  // calculate error
  double const ballMeasurementError = 159.1 * tan(openingAngleHeight_ERS210/2)/176*3;
  ballPosition.seen.distanceError = 
    ballMeasurementError *
    ballPercept.getDistanceSizeBased() * ballPercept.getDistanceSizeBased() /
    (ballRadius * 159.1 /* focal length ers219 */);
  ballPosition.seen.angleError = 0.05;

  // ball not seen
  if(!ballPercept.ballWasSeen)
  {
    consecutiveFramesBallSeen = 0;
    consecutiveFramesBallNotSeen++;

	  Vector2<double> propSpeed = ballPosition.propagated.getPropagatedSpeed(timeOfImageProcessing);
	  Vector2<double> propBallPos = ballPosition.propagated.getPropagatedPosition(timeOfImageProcessing);
		    
	  ballSpeedX.resetTo(propSpeed.x);
    ballSpeedY.resetTo(propSpeed.y);

    ballPosX.resetTo(propBallPos.x); 
    ballPosY.resetTo(propBallPos.y); 
		ballAngle.resetTo(propBallPos.angle());
		ballDistance.resetTo(propBallPos.abs());

    ballOnField = Geometry::relative2FieldCoord(robotPose, propBallPos.x, propBallPos.y);
    CIRCLE(ballLocatorField, ballOnField.x, ballOnField.y, 90, 3, 0, Drawings::blue);

    // move seen to compensate for jumping localization:
    ballOnField = Geometry::relative2FieldCoord(robotPose, lastBallSeen.x, lastBallSeen.y);
    ballPosition.seen.x = ballOnField.x;
    ballPosition.seen.y = ballOnField.y;
  }
  else // ball was seen
  {
    // if there is a gap of n frames during which the ball was not seen
    // reset the "timeWhenLastSeenConsecutively"
    if (consecutiveFramesBallNotSeen > 5)
      ballPosition.seen.timeWhenFirstSeenConsecutively = timeOfImageProcessing;

    Vector2<double> ballOffset;
    ballPercept.getOffsetSizeBased(ballOffset);

    // calc ball position:
    if (consecutiveFramesBallNotSeen > 10) // if ball has not been seen for a while but is seen now, directly set to percept...
    {
      ballPosX.resetTo(ballOffset.x); 
      ballPosY.resetTo(ballOffset.y);
			ballAngle.resetTo(ballPercept.getAngleSizeBased());
			ballDistance.resetTo(ballPercept.getDistanceSizeBased());

      ballSpeedX.reset();
      ballSpeedY.reset();
			
			//ballOnField = Geometry::relative2FieldCoord(robotPose, ballPosX.getVal(), ballPosY.getVal());
			//debug 

			ballOnField = Geometry::relative2FieldCoord(robotPose, ballDistance.getVal()*cos(ballAngle.getVal()), ballDistance.getVal()*sin(ballAngle.getVal()));
			CIRCLE(ballLocatorField, ballOnField.x, ballOnField.y, 200, 3, 0, Drawings::red);  
    }
    else  // if it has been seen recently, perform smoothing...
    {
      // calc ball speed:
      if ((consecutiveFramesBallSeen > 1) && timeDiff > 0)
      {
        double vx = (ballOffset.x - ballPosX.getVal())/timeDiff;
        double vy = (ballOffset.y - ballPosY.getVal())/timeDiff;

        double deltaDistance  = ballPercept.getDistanceSizeBased()
                                 - ballDistance.getVal();
        double deltaAngle     = ballPercept.getAngleSizeBased()  - ballAngle.getVal();

        // compare if the deltas are greater than 
        // the standard deviation (error) of the 
        // ball measurement:
        if ((fabs(deltaDistance) < ballPosition.seen.distanceError) && 
          (fabs(deltaAngle) < ballPosition.seen.angleError))
        {
          //OUTPUT(idText, text, "---");
          ballSpeedX.resetTo(0.0);
          ballSpeedY.resetTo(0.0);
        }
        else if (consecutiveFramesBallSeen > 1)
        {
          ballSpeedX.approximateVal(vx);
          ballSpeedY.approximateVal(vy);
        }
        else
        {
          ballSpeedX.resetTo(vx);
          ballSpeedY.resetTo(vy);
        }
		    
				
				ballPosition.propagated.timeOfObservation = timeOfImageProcessing;
				ballPosition.propagated.observedSpeed.x = ballSpeedX.getVal();
				ballPosition.propagated.observedSpeed.y = ballSpeedY.getVal();
      }
        
      // calc position:
      ballPosX.approximateVal(ballOffset.x); 
      ballPosY.approximateVal(ballOffset.y);  


      // old version
      ballAngle.approximateVal(ballPercept.getAngleSizeBased()); 
      ballDistance.approximateVal(ballPercept.getDistanceSizeBased());  
      
      lastBallSeen.x = ballDistance.getVal()*cos(ballAngle.getVal());//ballPosX.getVal();
      lastBallSeen.y = ballDistance.getVal()*sin(ballAngle.getVal());//ballPosY.getVal();

			/* 2do: make this angle/dist */
			ballPosition.propagated.positionWhenLastObserved.x = ballPosX.getVal();
			ballPosition.propagated.positionWhenLastObserved.y = ballPosY.getVal();
      
      Vector2<double> ballOnField = Geometry::relative2FieldCoord(robotPose, ballDistance.getVal()*cos(ballAngle.getVal()), ballDistance.getVal()*sin(ballAngle.getVal()));
      ballPosition.seen.x = ballOnField.x;
      ballPosition.seen.y = ballOnField.y;
      ballPosition.propagated.x  = ballPosition.seen.x;
      ballPosition.propagated.y  = ballPosition.seen.y;
    }
    consecutiveFramesBallSeen++;
    consecutiveFramesBallNotSeen = 0;

    ballPosition.seen.timeWhenLastSeen = timeOfImageProcessing;



    if (SystemCall::getTimeSince(ballPosition.seen.timeWhenFirstSeenConsecutively) > 350)
    {
      ballPosition.seen.timeUntilSeenConsecutively = timeOfImageProcessing; 
    }
  }
	
  ballPosition.seen.speed = Vector2<double>(ballSpeedX.getVal(), ballSpeedY.getVal()); 
  Vector2<double> speedOnField(ballPosition.seen.x + ballSpeedX.getVal(), ballPosition.seen.y + ballSpeedY.getVal());
  //OUTPUT(idText, text, " " << ballPosition.seen.speed.x << ", " << ballPosition.seen.speed.y);
  Vector2<double> robotToBall(ballPosition.seen.x, ballPosition.seen.y);
  robotToBall -= robotPose.translation;
  robotToBall.normalize();
  robotToBall *= ballPosition.seen.distanceError;

//  ballPosition.seen.speed.x = ballSpeedX.getVal()*cos(robotPose.rotation) + ballSpeedY.getVal()*sin(robotPose.rotation); 
//  ballPosition.seen.speed.y = -ballSpeedX.getVal()*sin(robotPose.rotation) + ballSpeedY.getVal()*cos(robotPose.rotation);

  CIRCLE(ballLocatorField, ballPosition.seen.x, ballPosition.seen.y, 50, 3, 0, Drawings::orange);

  LINE(ballLocatorField, 
    ballPosition.seen.x - robotToBall.x, ballPosition.seen.y - robotToBall.y, 
    ballPosition.seen.x + robotToBall.x, ballPosition.seen.y + robotToBall.y, 
    2, 0, Drawings::orange);

  robotToBall.normalize();
  robotToBall *= (ballPercept.getDistanceSizeBased() * ballPosition.seen.angleError);

  LINE(ballLocatorField, 
    ballPosition.seen.x + robotToBall.y, ballPosition.seen.y - robotToBall.x, 
    ballPosition.seen.x - robotToBall.y, ballPosition.seen.y + robotToBall.x, 
    2, 0, Drawings::orange);

  ARROW(ballLocatorField, ballPosition.seen.x, ballPosition.seen.y, speedOnField.x, speedOnField.y, 1, 3, Drawings::red);
  /*
  // ball intercept point

  // achtung: division by zero if b1=b2=0!!!


  double theta, 
    b1 = ballPosX.getVal() - robotPose.translation.x,
    b2 = ballPosY.getVal() - robotPose.translation.y, 
    v1 = ballSpeedX.getVal(), 
    v2 = ballSpeedY.getVal(),
    maxSpeed = 150;

  theta = asin((b2*b2*v1 - b1*b2*v2 + b1*sqrt(b2*b2*(maxSpeed*maxSpeed - v1*v1) + 2*b1*b2*v1*v2 + b1*b1*(maxSpeed*maxSpeed - v2*v2)))/
          (maxSpeed*(b1*b1 + b2*b2)));

  LINE(ballLocatorField, robotPose.translation.x, robotPose.translation.y, 
    robotPose.translation.x + sin(theta)*500, robotPose.translation.y + cos(theta)*500, 
    3, 0, Drawings::blue); 

  theta -= pi;
  theta *= -1;

  LINE(ballLocatorField, robotPose.translation.x, robotPose.translation.y, 
    robotPose.translation.x + sin(theta)*500, robotPose.translation.y + cos(theta)*500, 
    3, 0, Drawings::blue); 
  */
  DEBUG_DRAWING_FINISHED(ballLocatorField);  
}




void DDD2004BallLocator::compensateOdometry()
{
  lastBallSeen = (lastRobotOdometry + Pose2D(lastBallSeen) - odometryData).translation;
  Vector2<double> b = (lastRobotOdometry + Pose2D(Vector2<double>(ballPosX.getVal(), ballPosY.getVal()))
                       - odometryData).translation;
  ballPosX.setTo(b.x);
  ballPosY.setTo(b.y);
  lastRobotOdometry = odometryData;
}        


bool DDD2004BallLocator::handleMessage(InMessage& message)
{
  bool handled = false;
  
  switch(message.getMessageID())
  {
  case idGenericDebugData:
    {
      GenericDebugData d;
      message.bin >> d;
      if(d.id == GenericDebugData::ballLocatorPIDs)
      {
        OUTPUT(idText,text,"generic debug message handled by module DDD2004BallLocator");
        
        ballPosX.setWeightP(d.data[0]);
        ballPosX.setWeightI(d.data[1]);
        ballPosX.setWeightD(d.data[2]);
        
        ballPosY.setWeightP(d.data[0]);
        ballPosY.setWeightI(d.data[1]);
        ballPosY.setWeightD(d.data[2]);
        
        ballSpeedX.setWeightP(d.data[3]);
        ballSpeedX.setWeightI(d.data[4]);
        ballSpeedX.setWeightD(d.data[5]);
        
        ballSpeedY.setWeightP(d.data[3]);
        ballSpeedY.setWeightI(d.data[4]);
        ballSpeedY.setWeightD(d.data[5]);
      }
      handled = true;
      break;
    }
  }
  return handled;
}

/*
* Change log :
* 
* $Log: DDD2004BallLocator.cpp,v $
* Revision 1.1  2004/04/07 12:28:56  risler
* ddd checkin after go04 - first part
*
* Revision 1.2  2004/03/30 14:21:30  Papa Schlumpf
* only use size-based offsets, angles and distances
*
* Revision 1.1  2004/03/30 14:00:34  Papa Schlumpf
* added DDD2004BallLocator
*
*
*/
